<?php
// Start session if not already started
require_once '../includes/session_start.php';

// Include authentication functions
require_once '../auth/functions.php';

// Require student role to access this page
requireRole('student');

// Include database connection
require_once '../config/db_connect.php';

// Get student ID
$student_id = $_SESSION['user_id'];

// Get material ID
if (!isset($_GET['id']) || empty($_GET['id'])) {
    $_SESSION['error'] = "Material ID not provided.";
    header("Location: courses.php");
    exit();
}

$material_id = intval($_GET['id']);

// Get material details and check if student is enrolled in the course
$materialQuery = "
    SELECT m.*, c.title as course_title, c.course_id,
           (SELECT COUNT(*) FROM enrollments WHERE course_id = m.course_id AND student_id = ?) as is_enrolled
    FROM course_materials m
    JOIN courses c ON m.course_id = c.course_id
    WHERE m.material_id = ? AND m.status = 'active'
";

$stmt = $conn->prepare($materialQuery);
if (!$stmt) {
    $_SESSION['error'] = "Database error. Please try again later.";
    header("Location: courses.php");
    exit();
}

$stmt->bind_param("ii", $student_id, $material_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    $_SESSION['error'] = "Material not found or you don't have permission to access it.";
    header("Location: courses.php");
    exit();
}

$material = $result->fetch_assoc();

// Check if student is enrolled in the course
if ($material['is_enrolled'] <= 0) {
    $_SESSION['error'] = "You must be enrolled in this course to view its materials.";
    header("Location: course-detail.php?id=" . $material['course_id']);
    exit();
}

// Record the view in material_views if not already viewed
$viewCheckQuery = "SELECT view_id FROM material_views WHERE material_id = ? AND student_id = ?";
$viewCheckStmt = $conn->prepare($viewCheckQuery);
$viewCheckStmt->bind_param("ii", $material_id, $student_id);
$viewCheckStmt->execute();
$viewResult = $viewCheckStmt->get_result();

if ($viewResult->num_rows === 0) {
    // Insert new view record
    $insertViewQuery = "INSERT INTO material_views (material_id, student_id, view_date) VALUES (?, ?, NOW())";
    $insertStmt = $conn->prepare($insertViewQuery);
    $insertStmt->bind_param("ii", $material_id, $student_id);
    $insertStmt->execute();
}

// Handle different file types for display
$fileType = $material['file_type'] ?? '';
$filePath = '../uploads/materials/' . $material['file_path'];
$fileUrl = '../uploads/materials/' . $material['file_path'];
$embedType = 'unknown';

// Determine how to display the content based on file type
if (strpos($fileType, 'pdf') !== false) {
    $embedType = 'pdf';
} else if (strpos($fileType, 'video') !== false) {
    $embedType = 'video';
} else if (strpos($fileType, 'audio') !== false) {
    $embedType = 'audio';
} else if (strpos($fileType, 'image') !== false) {
    $embedType = 'image';
} else if (strpos($fileType, 'text') !== false) {
    $embedType = 'text';
}

// Set page title
$pageTitle = "View: " . $material['title'];

// Include header
include_once 'components/header.php';

// Include navigation
include_once 'components/navbar.php';
?>

<div class="container-fluid pt-4">
    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <?= $_SESSION['success']; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <?= $_SESSION['error']; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>
    
    <div class="row">
        <div class="col-12">
            <div class="card shadow-sm mb-4">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <div>
                        <h5 class="mb-0"><?= htmlspecialchars($material['title']) ?></h5>
                        <p class="text-muted small mb-0">
                            <i class='bx bx-book me-1'></i> 
                            <a href="course-detail.php?id=<?= $material['course_id'] ?>" class="text-reset">
                                <?= htmlspecialchars($material['course_title']) ?>
                            </a>
                        </p>
                    </div>
                    <a href="download_resource.php?id=<?= $material_id ?>" class="btn btn-primary">
                        <i class='bx bx-download me-1'></i> Download
                    </a>
                </div>
                <div class="card-body">
                    <?php if (!empty($material['description'])): ?>
                        <div class="mb-4">
                            <h6>Description:</h6>
                            <p><?= nl2br(htmlspecialchars($material['description'])) ?></p>
                        </div>
                    <?php endif; ?>
                    
                    <!-- File viewer based on file type -->
                    <div class="file-viewer mt-3">
                        <?php if ($embedType === 'pdf' && file_exists($filePath)): ?>
                            <div class="ratio ratio-16x9" style="min-height: 600px;">
                                <iframe src="<?= $fileUrl ?>" allowfullscreen></iframe>
                            </div>
                        <?php elseif ($embedType === 'video' && file_exists($filePath)): ?>
                            <div class="ratio ratio-16x9">
                                <video controls>
                                    <source src="<?= $fileUrl ?>" type="<?= $fileType ?>">
                                    Your browser does not support the video tag.
                                </video>
                            </div>
                        <?php elseif ($embedType === 'audio' && file_exists($filePath)): ?>
                            <audio controls class="w-100">
                                <source src="<?= $fileUrl ?>" type="<?= $fileType ?>">
                                Your browser does not support the audio element.
                            </audio>
                        <?php elseif ($embedType === 'image' && file_exists($filePath)): ?>
                            <div class="text-center">
                                <img src="<?= $fileUrl ?>" class="img-fluid" alt="<?= htmlspecialchars($material['title']) ?>">
                            </div>
                        <?php elseif ($embedType === 'text' && file_exists($filePath)): ?>
                            <div class="p-3 bg-light">
                                <pre><?= htmlspecialchars(file_get_contents($filePath)) ?></pre>
                            </div>
                        <?php else: ?>
                            <div class="alert alert-warning">
                                <i class='bx bx-error-circle me-2'></i>
                                This file type cannot be previewed directly in the browser. 
                                Please use the download button to access the file.
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                <div class="card-footer bg-white d-flex justify-content-between align-items-center">
                    <div class="text-muted small">
                        <span><i class='bx bx-calendar me-1'></i> Added: <?= date('M d, Y', strtotime($material['date_added'])) ?></span>
                        <span class="ms-3"><i class='bx bx-file me-1'></i> Type: <?= htmlspecialchars($fileType) ?></span>
                        <?php if (!empty($material['file_size'])): ?>
                            <span class="ms-3"><i class='bx bx-data me-1'></i> Size: 
                            <?php
                                $fileSize = $material['file_size'];
                                if ($fileSize < 1024) {
                                    echo $fileSize . ' B';
                                } else if ($fileSize < 1024 * 1024) {
                                    echo round($fileSize / 1024, 1) . ' KB';
                                } else if ($fileSize < 1024 * 1024 * 1024) {
                                    echo round($fileSize / (1024 * 1024), 1) . ' MB';
                                } else {
                                    echo round($fileSize / (1024 * 1024 * 1024), 1) . ' GB';
                                }
                            ?>
                            </span>
                        <?php endif; ?>
                    </div>
                    <a href="course-detail.php?id=<?= $material['course_id'] ?>#materials" class="btn btn-outline-secondary btn-sm">
                        Back to Materials
                    </a>
                </div>
            </div>
        </div>
    </div>
</div>

<?php
// Include footer
include_once 'components/footer.php';
?>
