<?php
/**
 * Database Migration: Add Grade Weighting Columns to Courses Table
 * This script adds assignment_weight and quiz_weight columns to enable teachers
 * to specify how much each assessment type contributes to the final grade.
 */

require_once '../config/db_connect.php';

echo "<h2>🎯 Grade Weighting Migration for Courses Table</h2>";
echo "<div style='font-family: Arial, sans-serif; max-width: 800px; margin: 20px;'>";

try {
    // Check if columns already exist
    $check_query = "SHOW COLUMNS FROM courses LIKE 'assignment_weight'";
    $result = $conn->query($check_query);
    
    if ($result->num_rows > 0) {
        echo "<div style='background: #fff3cd; border: 1px solid #ffeaa7; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
        echo "<h3>⚠️ Columns Already Exist</h3>";
        echo "<p>The grade weighting columns already exist in the courses table.</p>";
        echo "</div>";
    } else {
        echo "<div style='background: #d4edda; border: 1px solid #c3e6cb; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
        echo "<h3>🔧 Adding Grade Weighting Columns</h3>";
        
        // Add the new columns
        $alter_query = "
            ALTER TABLE courses 
            ADD COLUMN assignment_weight DECIMAL(5,2) DEFAULT 70.00 COMMENT 'Percentage weight for assignments in final grade',
            ADD COLUMN quiz_weight DECIMAL(5,2) DEFAULT 30.00 COMMENT 'Percentage weight for quizzes in final grade'
        ";
        
        if ($conn->query($alter_query) === TRUE) {
            echo "<p>✅ <strong>Successfully added grade weighting columns:</strong></p>";
            echo "<ul>";
            echo "<li><strong>assignment_weight:</strong> DECIMAL(5,2) DEFAULT 70.00</li>";
            echo "<li><strong>quiz_weight:</strong> DECIMAL(5,2) DEFAULT 30.00</li>";
            echo "</ul>";
            
            // Show the current table structure
            $describe_result = $conn->query("DESCRIBE courses");
            echo "<h4>📋 Updated Courses Table Structure:</h4>";
            echo "<table border='1' style='border-collapse: collapse; width: 100%; margin: 10px 0;'>";
            echo "<tr style='background: #f8f9fa;'><th style='padding: 8px;'>Field</th><th style='padding: 8px;'>Type</th><th style='padding: 8px;'>Null</th><th style='padding: 8px;'>Key</th><th style='padding: 8px;'>Default</th><th style='padding: 8px;'>Extra</th></tr>";
            
            while ($row = $describe_result->fetch_assoc()) {
                $bg_color = in_array($row['Field'], ['assignment_weight', 'quiz_weight']) ? 'background: #e7f3ff;' : '';
                echo "<tr style='{$bg_color}'>";
                echo "<td style='padding: 8px;'>" . htmlspecialchars($row['Field']) . "</td>";
                echo "<td style='padding: 8px;'>" . htmlspecialchars($row['Type']) . "</td>";
                echo "<td style='padding: 8px;'>" . htmlspecialchars($row['Null']) . "</td>";
                echo "<td style='padding: 8px;'>" . htmlspecialchars($row['Key']) . "</td>";
                echo "<td style='padding: 8px;'>" . htmlspecialchars($row['Default'] ?? 'NULL') . "</td>";
                echo "<td style='padding: 8px;'>" . htmlspecialchars($row['Extra']) . "</td>";
                echo "</tr>";
            }
            echo "</table>";
            
        } else {
            throw new Exception("Error adding columns: " . $conn->error);
        }
        echo "</div>";
    }
    
    // Update existing courses to have proper weighting if they don't
    echo "<div style='background: #e2e3e5; border: 1px solid #c6c8ca; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
    echo "<h3>🔄 Updating Existing Courses</h3>";
    
    $update_query = "
        UPDATE courses 
        SET assignment_weight = 70.00, quiz_weight = 30.00 
        WHERE assignment_weight IS NULL OR quiz_weight IS NULL
    ";
    
    if ($conn->query($update_query) === TRUE) {
        $affected_rows = $conn->affected_rows;
        echo "<p>✅ Updated {$affected_rows} courses with default grade weighting (70% assignments, 30% quizzes)</p>";
    } else {
        echo "<p>❌ Error updating existing courses: " . $conn->error . "</p>";
    }
    echo "</div>";
    
    // Show sample data
    echo "<div style='background: #f8f9fa; border: 1px solid #dee2e6; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
    echo "<h3>📊 Current Courses with Grade Weighting</h3>";
    
    $sample_query = "
        SELECT course_id, title, assignment_weight, quiz_weight, 
               (assignment_weight + quiz_weight) as total_weight
        FROM courses 
        ORDER BY course_id DESC 
        LIMIT 10
    ";
    
    $sample_result = $conn->query($sample_query);
    
    if ($sample_result && $sample_result->num_rows > 0) {
        echo "<table border='1' style='border-collapse: collapse; width: 100%; margin: 10px 0;'>";
        echo "<tr style='background: #f8f9fa;'>";
        echo "<th style='padding: 8px;'>Course ID</th>";
        echo "<th style='padding: 8px;'>Title</th>";
        echo "<th style='padding: 8px;'>Assignment Weight</th>";
        echo "<th style='padding: 8px;'>Quiz Weight</th>";
        echo "<th style='padding: 8px;'>Total Weight</th>";
        echo "<th style='padding: 8px;'>Status</th>";
        echo "</tr>";
        
        while ($row = $sample_result->fetch_assoc()) {
            $total = floatval($row['total_weight']);
            $status_color = $total == 100 ? '#28a745' : '#dc3545';
            $status_text = $total == 100 ? 'Valid (100%)' : 'Invalid (' . $total . '%)';
            
            echo "<tr>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($row['course_id']) . "</td>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($row['title']) . "</td>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($row['assignment_weight']) . "%</td>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($row['quiz_weight']) . "%</td>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($row['total_weight']) . "%</td>";
            echo "<td style='padding: 8px; color: {$status_color}; font-weight: bold;'>{$status_text}</td>";
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "<p>No courses found in the database.</p>";
    }
    echo "</div>";
    
    echo "<div style='background: #d1ecf1; border: 1px solid #bee5eb; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
    echo "<h3>🎯 Next Steps</h3>";
    echo "<ol>";
    echo "<li><strong>Update Course Creation Form:</strong> Add assignment and quiz weight input fields</li>";
    echo "<li><strong>Update Processing Logic:</strong> Handle the new weighting fields in course creation</li>";
    echo "<li><strong>Add Validation:</strong> Ensure assignment_weight + quiz_weight = 100%</li>";
    echo "<li><strong>Update Grade Calculation:</strong> Use these weights in final grade computation</li>";
    echo "<li><strong>Test Thoroughly:</strong> Verify all grade calculations work correctly</li>";
    echo "</ol>";
    echo "</div>";
    
} catch (Exception $e) {
    echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
    echo "<h3>❌ Migration Error</h3>";
    echo "<p>Error occurred during migration: " . htmlspecialchars($e->getMessage()) . "</p>";
    echo "</div>";
}

echo "</div>";

// Close the database connection
$conn->close();
?>
