<?php
/**
 * AJAX Video Detection Handler
 * Detects and processes video URLs from various platforms
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Include required files
require_once '../../includes/config.php';
require_once '../../includes/VideoEmbedder.php';

try {
    // Check if request is POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Invalid request method');
    }
    
    // Get the video URL
    $videoUrl = isset($_POST['video_url']) ? trim($_POST['video_url']) : '';
    
    if (empty($videoUrl)) {
        echo json_encode([
            'success' => false,
            'message' => 'Video URL is required'
        ]);
        exit;
    }
    
    // Validate URL format
    if (!filter_var($videoUrl, FILTER_VALIDATE_URL)) {
        echo json_encode([
            'success' => false,
            'message' => 'Please enter a valid URL'
        ]);
        exit;
    }
    
    // DEBUG: Log AJAX request with unique marker
    error_log("🚀 AJAX_DEBUG_REQUEST_URL: " . $videoUrl);
    
    // Detect video using VideoEmbedder class
    $videoData = VideoEmbedder::detectVideo($videoUrl);
    
    if (!$videoData || !$videoData['is_valid']) {
        echo json_encode([
            'success' => false,
            'message' => $videoData['error'] ?? 'Unsupported video platform or invalid URL'
        ]);
        exit;
    }
    
    // DEBUG: Log VideoEmbedder response with unique marker
    error_log("📋 AJAX_DEBUG_VIDEODATA_RECEIVED: " . json_encode($videoData));
    
    // Sanitize the embed code for security
    $sanitizedEmbedCode = VideoEmbedder::sanitizeEmbedCode($videoData['embed_code']);
    
    // DEBUG: Log sanitized embed code with unique marker
    error_log("🔒 AJAX_DEBUG_SANITIZED_EMBED: " . $sanitizedEmbedCode);
    
    $responseData = [
        'success' => true,
        'message' => 'Video detected successfully from ' . $videoData['platform_name'],
        'data' => [
            'platform' => $videoData['platform'],
            'platform_name' => $videoData['platform_name'],
            'video_id' => $videoData['video_id'],
            'original_url' => $videoData['original_url'],
            'embed_code' => $sanitizedEmbedCode,
            'thumbnail_url' => $videoData['thumbnail_url'],
            'is_valid' => $videoData['is_valid'],
            // Include rich metadata for auto-fill functionality
            'title' => $videoData['title'] ?? null,
            'description' => $videoData['description'] ?? null,
            'author_name' => $videoData['author_name'] ?? null,
            'author_url' => $videoData['author_url'] ?? null,
            'duration' => $videoData['duration'] ?? null,
            'view_count' => $videoData['view_count'] ?? null,
            'upload_date' => $videoData['upload_date'] ?? null,
            'debug_marker' => 'AJAX_RESPONSE_SUCCESS_' . time()
        ]
    ];
    
    // DEBUG: Log final AJAX response with unique marker
    error_log("📤 AJAX_DEBUG_FINAL_RESPONSE: " . json_encode($responseData));
    
    echo json_encode($responseData);
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => 'Error processing video URL: ' . $e->getMessage()
    ]);
}
?>
