<?php
/**
 * AJAX endpoint for detecting YouTube video duration
 * Called from upload-video.php JavaScript
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Initialize response
$response = [
    'success' => false,
    'duration' => null,
    'method' => null,
    'message' => ''
];

try {
    // Only allow POST requests
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Only POST requests allowed');
    }
    
    // Get video ID from request
    $videoId = isset($_POST['video_id']) ? trim($_POST['video_id']) : '';
    
    if (empty($videoId)) {
        throw new Exception('Video ID is required');
    }
    
    // Validate YouTube video ID format
    if (!preg_match('/^[a-zA-Z0-9_-]{11}$/', $videoId)) {
        throw new Exception('Invalid YouTube video ID format');
    }
    
    error_log("🎬 YouTube Duration Detection: Starting for video ID: " . $videoId);
    
    // Method 1: Try YouTube Data API v3 (requires API key)
    $apiKey = 'AIzaSyDummyKeyForTesting'; // Replace with real API key
    
    if ($apiKey !== 'AIzaSyDummyKeyForTesting') {
        error_log("🔑 Trying YouTube Data API v3...");
        
        $url = "https://www.googleapis.com/youtube/v3/videos?part=contentDetails&id=" . urlencode($videoId) . "&key=" . $apiKey;
        
        $context = stream_context_create([
            'http' => [
                'timeout' => 10,
                'method' => 'GET',
                'header' => 'User-Agent: PanaditeAcademy/1.0'
            ]
        ]);
        
        $apiResponse = @file_get_contents($url, false, $context);
        
        if ($apiResponse) {
            $data = json_decode($apiResponse, true);
            
            if (isset($data['items'][0]['contentDetails']['duration'])) {
                $iso8601Duration = $data['items'][0]['contentDetails']['duration'];
                $readableDuration = convertISO8601ToReadable($iso8601Duration);
                
                $response['success'] = true;
                $response['duration'] = $readableDuration;
                $response['method'] = 'YouTube Data API v3';
                $response['message'] = 'Duration detected successfully';
                
                error_log("✅ YouTube API success: " . $readableDuration);
                echo json_encode($response);
                exit;
            }
        }
        
        error_log("⚠️ YouTube Data API failed or no data returned");
    }
    
    // Method 2: Try scraping YouTube page (fallback method)
    error_log("🌐 Trying YouTube page scraping fallback...");
    
    $youtubeUrl = "https://www.youtube.com/watch?v=" . $videoId;
    
    // Try with CURL first for better reliability
    if (function_exists('curl_init')) {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $youtubeUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 15);
        curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/119.0.0.0 Safari/537.36');
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language: en-US,en;q=0.5',
            'Cache-Control: no-cache',
            'Pragma: no-cache'
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $html = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        error_log("🌐 CURL response: HTTP " . $httpCode . ", HTML length: " . strlen($html));
    } else {
        // Fallback to file_get_contents
        $context = stream_context_create([
            'http' => [
                'timeout' => 15,
                'method' => 'GET',
                'header' => [
                    'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/119.0.0.0 Safari/537.36',
                    'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
                    'Accept-Language: en-US,en;q=0.5'
                ]
            ]
        ]);
        
        $html = @file_get_contents($youtubeUrl, false, $context);
        error_log("🌐 file_get_contents HTML length: " . strlen($html));
    }
    
    if ($html && strlen($html) > 1000) {
        // Enhanced patterns to extract duration
        $patterns = [
            '/"lengthSeconds":"(\d+)"/',                    // Most common JSON pattern
            '/"length_seconds":"(\d+)"/',                   // Alternative JSON pattern
            '/lengthSeconds["\s]*:[\s"]*(\d+)/',           // Loose JSON pattern
            '/"approxDurationMs":"(\d+)"/',                 // Milliseconds pattern
            '/approxDurationMs["\s]*:[\s"]*"(\d+)"/',      // Loose milliseconds pattern
            '/"durationSeconds":"(\d+)"/',                  // Direct duration pattern
            '/duration["\s]*:["\s]*(\d+)["\s]*,/',         // Generic duration pattern
            '/"simpleText":"(\d+:\d+)"/',                   // Human readable duration
            '/"text":"(\d+:\d+:\d+)"/',                     // Full time format
            '/"text":"(\d+:\d+)"/',                         // Simple time format
        ];
        
        foreach ($patterns as $index => $pattern) {
            if (preg_match($pattern, $html, $matches)) {
                $seconds = 0;
                
                // Handle different match formats
                if (strpos($matches[1], ':') !== false) {
                    // Time format (MM:SS or HH:MM:SS)
                    $timeParts = explode(':', $matches[1]);
                    if (count($timeParts) == 2) {
                        $seconds = ($timeParts[0] * 60) + $timeParts[1];
                    } elseif (count($timeParts) == 3) {
                        $seconds = ($timeParts[0] * 3600) + ($timeParts[1] * 60) + $timeParts[2];
                    }
                } elseif ($index == 3 || $index == 4) {
                    // Milliseconds - convert to seconds
                    $seconds = intval($matches[1]) / 1000;
                } else {
                    // Direct seconds
                    $seconds = intval($matches[1]);
                }
                
                if ($seconds > 0) {
                    $readableDuration = formatDurationFromSeconds(round($seconds));
                    
                    $response['success'] = true;
                    $response['duration'] = $readableDuration;
                    $response['method'] = 'YouTube Page Scraping';
                    $response['message'] = 'Duration detected via page scraping (pattern ' . ($index + 1) . ')';
                    
                    error_log("✅ YouTube scraping success: " . $readableDuration . " (pattern " . ($index + 1) . ": " . $pattern . ")");
                    echo json_encode($response);
                    exit;
                }
            }
        }
        
        error_log("⚠️ YouTube page scraping failed - no duration patterns matched");
        
        // Debug: Save HTML to check what we got
        $debugFile = '/tmp/youtube_debug_' . $videoId . '.html';
        if (is_writable(dirname($debugFile))) {
            file_put_contents($debugFile, substr($html, 0, 5000));
            error_log("🔍 Debug HTML saved to: " . $debugFile);
        }
        
        // Show HTML snippet for debugging
        error_log("HTML snippet (first 800 chars): " . substr($html, 0, 800));
        
        // Look for any duration-related content
        if (preg_match_all('/duration|length|time/i', $html, $durationMatches, PREG_OFFSET_CAPTURE)) {
            $contexts = [];
            foreach (array_slice($durationMatches[0], 0, 5) as $match) {
                $pos = $match[1];
                $context = substr($html, max(0, $pos - 50), 150);
                $contexts[] = $context;
            }
            error_log("🔍 Duration contexts found: " . implode(' | ', $contexts));
        }
        
    } else {
        error_log("❌ Failed to fetch YouTube page or got empty response");
    }
    
    // Method 3: Simple string search for duration (more reliable)
    error_log("🔍 Trying simple string search method...");
    
    if ($html && strlen($html) > 1000) {
        // Look for common duration indicators without complex regex
        $searchStrings = [
            'lengthSeconds',
            'approxDurationMs', 
            'duration',
            'length_seconds'
        ];
        
        foreach ($searchStrings as $searchStr) {
            $pos = strpos($html, $searchStr);
            if ($pos !== false) {
                // Extract surrounding context
                $context = substr($html, $pos, 200);
                error_log("🔍 Found '$searchStr' context: " . $context);
                
                // Extract numbers after the search string
                if (preg_match('/' . preg_quote($searchStr) . '["\s:]*"?(\d+)"?/', $context, $matches)) {
                    $value = intval($matches[1]);
                    
                    // Convert based on the field type
                    if (strpos($searchStr, 'Ms') !== false) {
                        // Milliseconds to seconds
                        $seconds = $value / 1000;
                    } else {
                        // Already in seconds
                        $seconds = $value;
                    }
                    
                    if ($seconds > 0 && $seconds < 86400) { // Max 24 hours
                        $readableDuration = formatDurationFromSeconds(round($seconds));
                        
                        $response['success'] = true;
                        $response['duration'] = $readableDuration;
                        $response['method'] = 'Simple String Search';
                        $response['message'] = 'Duration detected via simple search (' . $searchStr . ')';
                        
                        error_log("✅ Simple search success: " . $readableDuration . " from " . $searchStr);
                        echo json_encode($response);
                        exit;
                    }
                }
            }
        }
    }
    
    // Method 4: Try YouTube oEmbed API (confirms video exists)
    error_log("🔗 Trying YouTube oEmbed API...");
    
    $oembedUrl = "https://www.youtube.com/oembed?url=https://www.youtube.com/watch?v=" . urlencode($videoId) . "&format=json";
    
    $oembedContext = stream_context_create([
        'http' => [
            'timeout' => 5,
            'method' => 'GET',
            'header' => 'User-Agent: PanaditeAcademy/1.0'
        ]
    ]);
    
    $oembedResponse = @file_get_contents($oembedUrl, false, $oembedContext);
    
    if ($oembedResponse) {
        $oembedData = json_decode($oembedResponse, true);
        
        if ($oembedData && isset($oembedData['title'])) {
            // oEmbed doesn't provide duration, but we can confirm video exists
            error_log("✅ Video exists (confirmed via oEmbed): " . $oembedData['title']);
            
            // Since video exists, provide a more helpful message
            $response['success'] = false;
            $response['method'] = 'oEmbed Verified';
            $response['message'] = 'Video found but duration detection failed. You may need to configure YouTube Data API key for reliable duration detection.';
        } else {
            error_log("⚠️ oEmbed returned invalid data");
        }
    } else {
        error_log("❌ oEmbed API failed - video may not exist or be private");
    }
    
    // Method 5: Try YouTube RSS feed (sometimes contains duration)
    error_log("📺 Trying YouTube RSS feed method...");
    
    $rssUrl = "https://www.youtube.com/feeds/videos.xml?video_id=" . $videoId;
    
    $rssResponse = @file_get_contents($rssUrl, false, $oembedContext);
    
    if ($rssResponse && strlen($rssResponse) > 100) {
        error_log("📺 RSS response length: " . strlen($rssResponse));
        
        // Look for duration in RSS XML
        if (preg_match('/<media:content[^>]*duration="(\d+)"/', $rssResponse, $matches)) {
            $seconds = intval($matches[1]);
            if ($seconds > 0) {
                $readableDuration = formatDurationFromSeconds($seconds);
                
                $response['success'] = true;
                $response['duration'] = $readableDuration;
                $response['method'] = 'YouTube RSS Feed';
                $response['message'] = 'Duration detected via RSS feed';
                
                error_log("✅ RSS feed success: " . $readableDuration);
                echo json_encode($response);
                exit;
            }
        }
        
        error_log("⚠️ RSS feed method failed - no duration found");
    } else {
        error_log("❌ RSS feed failed or empty");
    }
    
    // Method 6: Provide manual estimation option (last resort)
    error_log("🔧 All automatic methods failed, providing manual option");
    
    $response['success'] = false;
    $response['method'] = 'Manual Required';
    $response['message'] = 'Automatic duration detection failed. You can manually check the video duration and enter it during upload, or configure YouTube Data API key for reliable detection.';
    
    // If all methods failed
    // throw new Exception('All duration detection methods failed. Video may be private, unavailable, or require YouTube Data API key.');
    
} catch (Exception $e) {
    error_log("❌ YouTube Duration Detection Error: " . $e->getMessage());
    
    $response['success'] = false;
    $response['message'] = $e->getMessage();
    $response['method'] = 'Error';
}

echo json_encode($response);

/**
 * Convert ISO 8601 duration format (PT4M13S) to readable format
 */
function convertISO8601ToReadable($iso8601) {
    try {
        $interval = new DateInterval($iso8601);
        $seconds = ($interval->h * 3600) + ($interval->i * 60) + $interval->s;
        return formatDurationFromSeconds($seconds);
    } catch (Exception $e) {
        error_log("❌ ISO8601 conversion failed: " . $e->getMessage());
        return null;
    }
}

/**
 * Convert seconds to readable format (MM:SS or HH:MM:SS)
 */
function formatDurationFromSeconds($seconds) {
    $hours = floor($seconds / 3600);
    $minutes = floor(($seconds % 3600) / 60);
    $secs = $seconds % 60;
    
    if ($hours > 0) {
        return sprintf('%d:%02d:%02d', $hours, $minutes, $secs);
    } else {
        return sprintf('%d:%02d', $minutes, $secs);
    }
}
?>
