<?php
/**
 * AJAX Handler for Teacher Dashboard Data
 * Returns JSON response with course, video, earnings and enrollment data
 */

// Start session and include required files
session_start();
require_once '../../config/database.php';
require_once '../../auth/functions.php';

// Check if user is logged in and is a teacher
if (!isLoggedIn() || !hasRole('teacher')) {
    echo json_encode([
        'success' => false,
        'message' => 'Unauthorized access'
    ]);
    exit;
}

// Get current user ID
$userId = $_SESSION['user_id'];

try {
    $data = [];
    
    // Get course count
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM courses WHERE teacher_id = ?");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    $courseCount = $result->fetch_assoc()['count'];
    $data['courseCount'] = $courseCount;
    
    // Get video count
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM videos WHERE teacher_id = ?");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    $videoCount = $result->fetch_assoc()['count'];
    $data['videoCount'] = $videoCount;
    
    // Get earnings data
    $stmt = $conn->prepare("
        SELECT SUM(p.amount) as total_earnings 
        FROM payments p 
        JOIN videos v ON p.video_id = v.video_id 
        WHERE v.teacher_id = ? AND p.status = 'completed'
    ");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    $totalEarnings = $result->fetch_assoc()['total_earnings'] ?: 0;
    $data['totalEarnings'] = $totalEarnings;
    
    // Get purchase count
    $stmt = $conn->prepare("
        SELECT COUNT(*) as count 
        FROM payments p 
        JOIN videos v ON p.video_id = v.video_id 
        WHERE v.teacher_id = ? AND p.status = 'completed'
    ");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    $purchaseCount = $result->fetch_assoc()['count'];
    $data['purchaseCount'] = $purchaseCount;
    
    // Get monthly earnings
    $currentMonth = date('Y-m-01');
    $nextMonth = date('Y-m-01', strtotime('+1 month'));
    
    $stmt = $conn->prepare("
        SELECT SUM(p.amount) as monthly_earnings 
        FROM payments p 
        JOIN videos v ON p.video_id = v.video_id 
        WHERE v.teacher_id = ? 
        AND p.status = 'completed' 
        AND p.payment_date >= ? 
        AND p.payment_date < ?
    ");
    $stmt->bind_param("iss", $userId, $currentMonth, $nextMonth);
    $stmt->execute();
    $result = $stmt->get_result();
    $monthlyEarnings = $result->fetch_assoc()['monthly_earnings'] ?: 0;
    $data['monthlyEarnings'] = $monthlyEarnings;
    
    // Get earnings data for chart (last 30 days)
    $thirtyDaysAgo = date('Y-m-d', strtotime('-30 days'));
    $today = date('Y-m-d');
    
    $earningsData = [];
    $stmt = $conn->prepare("
        SELECT DATE(p.payment_date) as date, SUM(p.amount) as amount 
        FROM payments p 
        JOIN videos v ON p.video_id = v.video_id 
        WHERE v.teacher_id = ? 
        AND p.status = 'completed' 
        AND p.payment_date >= ? 
        AND p.payment_date <= ? 
        GROUP BY DATE(p.payment_date)
        ORDER BY date ASC
    ");
    $stmt->bind_param("iss", $userId, $thirtyDaysAgo, $today);
    $stmt->execute();
    $result = $stmt->get_result();
    
    while ($row = $result->fetch_assoc()) {
        $earningsData[] = [
            'date' => date('M d', strtotime($row['date'])),
            'amount' => (float) $row['amount']
        ];
    }
    $data['earningsData'] = $earningsData;
    
    // Get recent courses
    $courses = [];
    $stmt = $conn->prepare("
        SELECT c.*, 
               (SELECT COUNT(*) FROM enrollments e WHERE e.course_id = c.course_id) AS student_count 
        FROM courses c 
        WHERE c.teacher_id = ? 
        ORDER BY c.created_at DESC 
        LIMIT 8
    ");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    while ($row = $result->fetch_assoc()) {
        // Make sure the thumbnail path is correct
        if ($row['thumbnail'] && !str_starts_with($row['thumbnail'], 'http') && !str_starts_with($row['thumbnail'], '/')) {
            $row['thumbnail'] = '/Panadite academy/assets/uploads/thumbnails/' . $row['thumbnail'];
        }
        
        // Limit description length
        $row['description'] = mb_substr($row['description'], 0, 120) . (mb_strlen($row['description']) > 120 ? '...' : '');
        
        $courses[] = $row;
    }
    $data['recentCourses'] = $courses;
    
    // Get recent videos
    $videos = [];
    $stmt = $conn->prepare("
        SELECT v.*, c.title as course_title,
               (SELECT COUNT(*) FROM payments p WHERE p.video_id = v.video_id AND p.status = 'completed') AS purchases
        FROM videos v
        LEFT JOIN courses c ON v.course_id = c.course_id
        WHERE v.teacher_id = ?
        ORDER BY v.created_at DESC
        LIMIT 10
    ");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    while ($row = $result->fetch_assoc()) {
        // Make sure the thumbnail path is correct
        if ($row['thumbnail'] && !str_starts_with($row['thumbnail'], 'http') && !str_starts_with($row['thumbnail'], '/')) {
            $row['thumbnail'] = '/Panadite academy/assets/uploads/thumbnails/' . $row['thumbnail'];
        }
        
        $videos[] = $row;
    }
    $data['recentVideos'] = $videos;
    
    // Get recent student enrollments
    $enrollments = [];
    $stmt = $conn->prepare("
        SELECT u.first_name, u.last_name, c.title as course_title, e.enrolled_date
        FROM enrollments e
        JOIN users u ON e.student_id = u.user_id
        JOIN courses c ON e.course_id = c.course_id
        WHERE c.teacher_id = ?
        ORDER BY e.enrolled_date DESC
        LIMIT 5
    ");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    while ($row = $result->fetch_assoc()) {
        $enrollments[] = $row;
    }
    $data['recentEnrollments'] = $enrollments;
    
    // Get unread notifications count
    $stmt = $conn->prepare("
        SELECT COUNT(*) as count
        FROM notifications
        WHERE user_id = ? AND is_read = 0
    ");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    $unreadNotifications = $result->fetch_assoc()['count'];
    $data['unreadNotifications'] = $unreadNotifications;
    
    // Return success response with data
    echo json_encode([
        'success' => true,
        'data' => $data
    ]);
    
} catch (Exception $e) {
    // Return error response
    echo json_encode([
        'success' => false,
        'message' => 'Error loading dashboard data: ' . $e->getMessage()
    ]);
}
