<?php
/**
 * AJAX Handler for Updating Course Status
 * Updates the is_published status of a course and returns JSON response
 */

// Start session and include required files
session_start();
require_once '../../config/database.php';
require_once '../../auth/functions.php';

// Check if user is logged in and is a teacher
if (!isLoggedIn() || !hasRole('teacher')) {
    echo json_encode([
        'success' => false,
        'message' => 'Unauthorized access'
    ]);
    exit;
}

// Check if request is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode([
        'success' => false,
        'message' => 'Invalid request method'
    ]);
    exit;
}

// Get current user ID
$userId = $_SESSION['user_id'];

// Get POST data
$courseId = isset($_POST['course_id']) ? intval($_POST['course_id']) : 0;
$isPublished = isset($_POST['is_published']) ? intval($_POST['is_published']) : 0;

// Validate input
if ($courseId <= 0) {
    echo json_encode([
        'success' => false,
        'message' => 'Invalid course ID'
    ]);
    exit;
}

try {
    // Check if course belongs to the teacher
    $stmt = $conn->prepare("SELECT course_id FROM courses WHERE course_id = ? AND teacher_id = ?");
    $stmt->bind_param("ii", $courseId, $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        echo json_encode([
            'success' => false,
            'message' => 'Course not found or you do not have permission to update it'
        ]);
        exit;
    }
    
    // Update course publish status
    $stmt = $conn->prepare("UPDATE courses SET is_published = ? WHERE course_id = ? AND teacher_id = ?");
    $stmt->bind_param("iii", $isPublished, $courseId, $userId);
    $stmt->execute();
    
    if ($stmt->affected_rows > 0) {
        echo json_encode([
            'success' => true,
            'message' => $isPublished ? 'Course published successfully' : 'Course unpublished successfully',
            'data' => [
                'course_id' => $courseId,
                'is_published' => $isPublished
            ]
        ]);
    } else {
        echo json_encode([
            'success' => false,
            'message' => 'No changes made to course status'
        ]);
    }
    
} catch (Exception $e) {
    // Return error response
    echo json_encode([
        'success' => false,
        'message' => 'Error updating course status: ' . $e->getMessage()
    ]);
}
