// Update calendar grid based on view
function updateCalendarGrid(calendarData, viewType) {
    const calendarMain = document.getElementById('calendar-main');
    if (!calendarMain) return;
    
    // Clear current calendar view
    while (calendarMain.firstChild) {
        calendarMain.removeChild(calendarMain.firstChild);
    }
    
    // Update based on view type
    switch(viewType) {
        case 'month':
            renderMonthView(calendarData, calendarMain);
            break;
        case 'week':
            renderWeekView(calendarData, calendarMain);
            break;
        case 'day':
            renderDayView(calendarData, calendarMain);
            break;
        case 'list':
            renderListView(calendarData, calendarMain);
            break;
        default:
            renderMonthView(calendarData, calendarMain);
    }
}

// Render month view
function renderMonthView(calendarData, container) {
    // Create month view table
    const table = document.createElement('table');
    table.className = 'calendar-table month-view';
    
    // Create table header with days of week
    const thead = document.createElement('thead');
    const headerRow = document.createElement('tr');
    const daysOfWeek = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
    
    daysOfWeek.forEach(day => {
        const th = document.createElement('th');
        th.textContent = day;
        headerRow.appendChild(th);
    });
    
    thead.appendChild(headerRow);
    table.appendChild(thead);
    
    // Create table body with days
    const tbody = document.createElement('tbody');
    let date = 1;
    const daysInMonth = calendarData.daysInMonth;
    const firstDayOfMonth = calendarData.firstDayOfMonth;
    
    // Create calendar rows
    for (let i = 0; i < 6; i++) {
        // Break if we've already displayed all days
        if (date > daysInMonth) break;
        
        const row = document.createElement('tr');
        
        // Create cells for each day of the week
        for (let j = 0; j < 7; j++) {
            const cell = document.createElement('td');
            
            // Add day number if it's within the current month
            if ((i === 0 && j < firstDayOfMonth) || date > daysInMonth) {
                // Empty cell for days outside the current month
                cell.className = 'empty';
            } else {
                // Cell for days in the current month
                cell.className = 'day-cell';
                
                // Create day number
                const dayNumber = document.createElement('div');
                dayNumber.className = 'day-number';
                dayNumber.textContent = date;
                
                // Highlight today
                if (date === calendarData.today && 
                    calendarData.month === calendarData.currentMonth && 
                    calendarData.year === calendarData.currentYear) {
                    dayNumber.classList.add('today');
                }
                
                cell.appendChild(dayNumber);
                
                // Add events for this day
                const eventsContainer = document.createElement('div');
                eventsContainer.className = 'day-events';
                
                // Find events for this day
                const dayEvents = calendarData.events.filter(event => {
                    const eventDate = new Date(event.event_date);
                    return eventDate.getDate() === date;
                });
                
                // Add event items
                dayEvents.forEach(event => {
                    const eventElement = document.createElement('div');
                    eventElement.className = `calendar-event ${event.type}`;
                    eventElement.innerHTML = `<span class="event-title">${event.title}</span>`;
                    
                    // Add click event to show modal
                    eventElement.addEventListener('click', () => {
                        showEventModal(
                            event.title,
                            formatTimeRange(event.start_time, event.end_time),
                            event.course_title,
                            event.description
                        );
                    });
                    
                    eventsContainer.appendChild(eventElement);
                });
                
                cell.appendChild(eventsContainer);
                
                // Increment date
                date++;
            }
            
            row.appendChild(cell);
        }
        
        tbody.appendChild(row);
    }
    
    table.appendChild(tbody);
    container.appendChild(table);
}

// Render week view
function renderWeekView(calendarData, container) {
    const weekView = document.createElement('div');
    weekView.className = 'calendar-week-view';
    
    // Create header with time column and days
    const header = document.createElement('div');
    header.className = 'week-header';
    
    // Add time column header
    const timeHeader = document.createElement('div');
    timeHeader.className = 'time-header';
    timeHeader.textContent = 'Time';
    header.appendChild(timeHeader);
    
    // Get current week days based on today's date
    const today = new Date(calendarData.year, calendarData.month - 1, calendarData.today);
    const dayOfWeek = today.getDay();
    const weekStart = new Date(today);
    weekStart.setDate(today.getDate() - dayOfWeek); // Start from Sunday
    
    // Add day headers
    for (let i = 0; i < 7; i++) {
        const day = new Date(weekStart);
        day.setDate(weekStart.getDate() + i);
        
        const dayHeader = document.createElement('div');
        dayHeader.className = 'day-header';
        dayHeader.innerHTML = `
            <div class="day-name">${day.toLocaleDateString('en-US', { weekday: 'short' })}</div>
            <div class="day-date">${day.getDate()}</div>
        `;
        
        // Highlight today
        if (i === dayOfWeek) {
            dayHeader.classList.add('today');
        }
        
        header.appendChild(dayHeader);
    }
    
    weekView.appendChild(header);
    
    // Create time grid
    const timeGrid = document.createElement('div');
    timeGrid.className = 'week-time-grid';
    
    // Add time slots (8AM to 8PM)
    for (let hour = 8; hour <= 20; hour++) {
        const timeRow = document.createElement('div');
        timeRow.className = 'time-row';
        
        // Add time label
        const timeLabel = document.createElement('div');
        timeLabel.className = 'time-label';
        timeLabel.textContent = `${hour > 12 ? hour - 12 : hour}${hour >= 12 ? 'PM' : 'AM'}`;
        timeRow.appendChild(timeLabel);
        
        // Add day columns
        for (let i = 0; i < 7; i++) {
            const dayCol = document.createElement('div');
            dayCol.className = 'day-column';
            
            // Get the date for this column
            const day = new Date(weekStart);
            day.setDate(weekStart.getDate() + i);
            
            // Find events for this time slot and day
            const dayEvents = calendarData.events.filter(event => {
                const eventDate = new Date(event.event_date);
                const eventHour = parseInt(event.start_time.split(':')[0]);
                return eventDate.getDate() === day.getDate() && 
                       eventDate.getMonth() === day.getMonth() && 
                       eventHour === hour;
            });
            
            // Add events
            dayEvents.forEach(event => {
                const eventElement = document.createElement('div');
                eventElement.className = `calendar-event ${event.type}`;
                eventElement.innerHTML = `<span class="event-title">${event.title}</span>`;
                
                // Add click event to show modal
                eventElement.addEventListener('click', () => {
                    showEventModal(
                        event.title,
                        formatTimeRange(event.start_time, event.end_time),
                        event.course_title,
                        event.description
                    );
                });
                
                dayCol.appendChild(eventElement);
            });
            
            timeRow.appendChild(dayCol);
        }
        
        timeGrid.appendChild(timeRow);
    }
    
    weekView.appendChild(timeGrid);
    container.appendChild(weekView);
}

// Render day view
function renderDayView(calendarData, container) {
    const dayView = document.createElement('div');
    dayView.className = 'calendar-day-view';
    
    // Get selected date (default to today if not specified)
    const selectedDate = new Date(calendarData.year, calendarData.month - 1, calendarData.today);
    
    // Create day header
    const header = document.createElement('div');
    header.className = 'day-view-header';
    header.innerHTML = `
        <h3>${selectedDate.toLocaleDateString('en-US', { weekday: 'long' })}</h3>
        <h4>${selectedDate.toLocaleDateString('en-US', { month: 'long', day: 'numeric' })}</h4>
    `;
    dayView.appendChild(header);
    
    // Create time schedule
    const schedule = document.createElement('div');
    schedule.className = 'day-schedule';
    
    // Add time slots (8AM to 8PM)
    for (let hour = 8; hour <= 20; hour++) {
        const timeSlot = document.createElement('div');
        timeSlot.className = 'time-slot';
        
        // Add time label
        const timeLabel = document.createElement('div');
        timeLabel.className = 'time-label';
        timeLabel.textContent = `${hour > 12 ? hour - 12 : hour}${hour >= 12 ? 'PM' : 'AM'}`;
        timeSlot.appendChild(timeLabel);
        
        // Add content area
        const slotContent = document.createElement('div');
        slotContent.className = 'slot-content';
        
        // Find events for this time slot
        const slotEvents = calendarData.events.filter(event => {
            const eventDate = new Date(event.event_date);
            const eventHour = parseInt(event.start_time.split(':')[0]);
            return eventDate.getDate() === selectedDate.getDate() && 
                   eventDate.getMonth() === selectedDate.getMonth() && 
                   eventDate.getFullYear() === selectedDate.getFullYear() && 
                   eventHour === hour;
        });
        
        // Add events
        slotEvents.forEach(event => {
            const eventElement = document.createElement('div');
            eventElement.className = `calendar-event ${event.type}`;
            eventElement.innerHTML = `
                <div class="event-time">${formatTimeRange(event.start_time, event.end_time)}</div>
                <div class="event-title">${event.title}</div>
                <div class="event-location">${event.course_title}</div>
            `;
            
            // Add click event to show modal
            eventElement.addEventListener('click', () => {
                showEventModal(
                    event.title,
                    formatTimeRange(event.start_time, event.end_time),
                    event.course_title,
                    event.description
                );
            });
            
            slotContent.appendChild(eventElement);
        });
        
        timeSlot.appendChild(slotContent);
        schedule.appendChild(timeSlot);
    }
    
    dayView.appendChild(schedule);
    container.appendChild(dayView);
}

// Render list view
function renderListView(calendarData, container) {
    const listView = document.createElement('div');
    listView.className = 'calendar-list-view';
    
    // Sort events by date
    const events = [...calendarData.events].sort((a, b) => {
        return new Date(a.event_date) - new Date(b.event_date);
    });
    
    // Group events by date
    const eventsByDate = {};
    events.forEach(event => {
        const dateKey = event.event_date;
        if (!eventsByDate[dateKey]) {
            eventsByDate[dateKey] = [];
        }
        eventsByDate[dateKey].push(event);
    });
    
    // Create list items for each date
    for (const dateKey in eventsByDate) {
        const dateEvents = eventsByDate[dateKey];
        const date = new Date(dateKey);
        
        // Create date header
        const dateHeader = document.createElement('div');
        dateHeader.className = 'list-date-header';
        dateHeader.innerHTML = `
            <div class="date-day">${date.getDate()}</div>
            <div class="date-details">
                <div class="date-weekday">${date.toLocaleDateString('en-US', { weekday: 'long' })}</div>
                <div class="date-month">${date.toLocaleDateString('en-US', { month: 'long' })}</div>
            </div>
        `;
        listView.appendChild(dateHeader);
        
        // Create event items
        const eventsList = document.createElement('div');
        eventsList.className = 'events-list';
        
        dateEvents.forEach(event => {
            const eventItem = document.createElement('div');
            eventItem.className = `list-event-item ${event.type}`;
            
            // Add event badge
            let badgeText = '';
            switch (event.type) {
                case 'class':
                    badgeText = 'Class';
                    break;
                case 'assignment':
                    badgeText = 'Due';
                    break;
                case 'exam':
                    badgeText = 'Exam';
                    break;
                default:
                    badgeText = 'Event';
            }
            
            eventItem.innerHTML = `
                <div class="event-time">${formatTimeRange(event.start_time, event.end_time)}</div>
                <div class="event-details">
                    <div class="event-title">
                        <span class="event-badge ${event.type}">${badgeText}</span>
                        ${event.title}
                    </div>
                    <div class="event-location">${event.course_title}</div>
                </div>
            `;
            
            // Add click event to show modal
            eventItem.addEventListener('click', () => {
                showEventModal(
                    event.title,
                    formatTimeRange(event.start_time, event.end_time),
                    event.course_title,
                    event.description
                );
            });
            
            eventsList.appendChild(eventItem);
        });
        
        listView.appendChild(eventsList);
    }
    
    // If no events, show message
    if (Object.keys(eventsByDate).length === 0) {
        const noEvents = document.createElement('div');
        noEvents.className = 'no-events-message';
        noEvents.innerHTML = `
            <p>No events scheduled for this month</p>
            <button class="btn-primary">Add Event</button>
        `;
        listView.appendChild(noEvents);
    }
    
    container.appendChild(listView);
}

// Update events display
function updateEventsDisplay(events) {
    const upcomingEventsContainer = document.querySelector('.upcoming-events-content');
    if (!upcomingEventsContainer) return;
    
    // Clear current events
    upcomingEventsContainer.innerHTML = '';
    
    if (events.length > 0) {
        // Add new events
        events.forEach(event => {
            // Determine badge text based on event type
            let badgeText = 'Event';
            let badgeClass = '';
            
            switch (event.type) {
                case 'class':
                    badgeText = 'Class';
                    badgeClass = 'recurring';
                    break;
                case 'assignment':
                    badgeText = 'Due';
                    badgeClass = 'important';
                    break;
                case 'exam':
                    badgeText = 'Exam';
                    badgeClass = 'reminder';
                    break;
            }
            
            const eventDate = new Date(event.event_date);
            const formattedDate = eventDate.toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' });
            
            const eventHtml = `
                <div class="event-item" onclick="showEventModal('${event.title}', '${formatTimeRange(event.start_time, event.end_time)}', '${event.course_title}', '${event.description}')">
                    <div class="event-title">
                        <span class="event-badge ${badgeClass}">${badgeText}</span>
                        ${event.title}
                    </div>
                    <div class="event-course">
                        <i class="fas fa-bookmark"></i>
                        ${event.course_title}
                    </div>
                    <div class="event-date">
                        <i class="fas fa-calendar"></i>
                        ${formattedDate}
                    </div>
                </div>
            `;
            
            upcomingEventsContainer.innerHTML += eventHtml;
        });
    } else {
        // Show no events message
        upcomingEventsContainer.innerHTML = `
            <div class="no-events">
                <p>No upcoming events scheduled for the upcoming days</p>
                <button class="btn-primary">Add Event</button>
            </div>
        `;
    }
}

// Helper function to format time range
function formatTimeRange(startTime, endTime) {
    // Convert 24-hour times to AM/PM format
    function formatTime(timeStr) {
        const [hours, minutes] = timeStr.split(':');
        const hour = parseInt(hours);
        const ampm = hour >= 12 ? 'PM' : 'AM';
        const hour12 = hour % 12 || 12;
        return `${hour12}:${minutes} ${ampm}`;
    }
    
    const formattedStart = formatTime(startTime);
    const formattedEnd = formatTime(endTime);
    
    return `${formattedStart} - ${formattedEnd}`;
}
