<?php
require_once '../config/db_connect.php';

// Turn on all error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<!DOCTYPE html>
<html>
<head>
    <title>Database Thumbnail Path Analysis</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        table { border-collapse: collapse; width: 100%; }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
        th { background-color: #f2f2f2; }
        .error { color: red; font-weight: bold; }
        .success { color: green; font-weight: bold; }
        .warning { color: orange; font-weight: bold; }
        .container { margin-bottom: 30px; }
    </style>
</head>
<body>
    <h1>Database Thumbnail Path Analysis</h1>";

// 1. Check all videos and their thumbnail paths
$query = "SELECT video_id, title, thumbnail, video_url FROM videos ORDER BY video_id DESC";
$result = $conn->query($query);

echo "<div class='container'>
    <h2>Video Records in Database</h2>";

if ($result && $result->num_rows > 0) {
    echo "<table>
        <tr>
            <th>ID</th>
            <th>Title</th>
            <th>Thumbnail Path</th>
            <th>Expected Format</th>
            <th>File Exists</th>
            <th>Status</th>
        </tr>";
    
    while ($row = $result->fetch_assoc()) {
        echo "<tr>
            <td>{$row['video_id']}</td>
            <td>" . htmlspecialchars($row['title']) . "</td>
            <td>" . (empty($row['thumbnail']) ? "<span class='error'>NULL</span>" : htmlspecialchars($row['thumbnail'])) . "</td>";
        
        // Expected format
        echo "<td>uploads/thumbnails/filename.jpg</td>";
        
        // Check if file exists
        if (!empty($row['thumbnail'])) {
            $full_path = __DIR__ . '/../' . $row['thumbnail'];
            $file_exists = file_exists($full_path);
            echo "<td>" . ($file_exists ? "<span class='success'>Yes</span>" : "<span class='error'>No</span>") . "</td>";
            
            // Status
            if ($file_exists) {
                echo "<td class='success'>Valid thumbnail</td>";
            } else {
                echo "<td class='error'>File not found</td>";
            }
        } else {
            echo "<td class='error'>N/A</td>";
            echo "<td class='error'>Missing thumbnail path</td>";
        }
        
        echo "</tr>";
    }
    
    echo "</table>";
} else {
    echo "<p>No videos found in the database.</p>";
}

echo "</div>";

// 2. Check if there are files in uploads/thumbnails that are not used in the database
$thumbnail_dir = __DIR__ . '/../uploads/thumbnails/';
echo "<div class='container'>
    <h2>Thumbnails Directory Analysis</h2>";

if (is_dir($thumbnail_dir)) {
    $files = array_diff(scandir($thumbnail_dir), ['.', '..']);
    
    echo "<p>Found " . count($files) . " files in uploads/thumbnails directory</p>";
    
    $used_thumbnails = [];
    $result = $conn->query("SELECT thumbnail FROM videos WHERE thumbnail IS NOT NULL");
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            if (!empty($row['thumbnail'])) {
                $used_thumbnails[] = basename($row['thumbnail']);
            }
        }
    }
    
    echo "<table>
        <tr>
            <th>Filename</th>
            <th>Full Path</th>
            <th>Used in Database</th>
        </tr>";
    
    foreach ($files as $file) {
        if (is_file($thumbnail_dir . $file)) {
            $used = in_array($file, $used_thumbnails);
            echo "<tr>
                <td>" . htmlspecialchars($file) . "</td>
                <td>uploads/thumbnails/" . htmlspecialchars($file) . "</td>
                <td>" . ($used ? "<span class='success'>Yes</span>" : "<span class='warning'>No</span>") . "</td>
            </tr>";
        }
    }
    
    echo "</table>";
} else {
    echo "<p class='error'>The directory uploads/thumbnails does not exist!</p>";
}

echo "</div>";

// 3. Check if the thumbnail paths in the database follow the correct format
$query = "SELECT video_id, thumbnail FROM videos WHERE thumbnail IS NOT NULL";
$result = $conn->query($query);

echo "<div class='container'>
    <h2>Path Format Analysis</h2>
    <p>Checking if thumbnail paths follow the correct format: <code>uploads/thumbnails/filename.jpg</code></p>";

if ($result && $result->num_rows > 0) {
    $correct_format = 0;
    $incorrect_format = 0;
    $incorrect_paths = [];
    
    while ($row = $result->fetch_assoc()) {
        $path = $row['thumbnail'];
        $expected_prefix = 'uploads/thumbnails/';
        
        if (strpos($path, $expected_prefix) === 0) {
            $correct_format++;
        } else {
            $incorrect_format++;
            $incorrect_paths[] = [
                'video_id' => $row['video_id'],
                'current_path' => $path
            ];
        }
    }
    
    echo "<p>Results:</p>
    <ul>
        <li>Correct format: <span class='success'>{$correct_format}</span></li>
        <li>Incorrect format: <span class='error'>{$incorrect_format}</span></li>
    </ul>";
    
    if ($incorrect_format > 0) {
        echo "<h3>Incorrect Paths</h3>
        <table>
            <tr>
                <th>Video ID</th>
                <th>Current Path</th>
            </tr>";
        
        foreach ($incorrect_paths as $path) {
            echo "<tr>
                <td>{$path['video_id']}</td>
                <td>{$path['current_path']}</td>
            </tr>";
        }
        
        echo "</table>";
        
        // Add fix button
        echo "<form method='post'>
            <input type='submit' name='fix_paths' value='Fix Incorrect Paths' style='padding: 10px; margin-top: 10px; background-color: #4CAF50; color: white; border: none; cursor: pointer;'>
        </form>";
        
        // Handle path fixing
        if (isset($_POST['fix_paths'])) {
            $success_count = 0;
            
            foreach ($incorrect_paths as $path) {
                $video_id = $path['video_id'];
                $current_path = $path['current_path'];
                
                // Get just the filename from the path
                $filename = basename($current_path);
                $new_path = 'uploads/thumbnails/' . $filename;
                
                // Update the database
                $update_query = "UPDATE videos SET thumbnail = ? WHERE video_id = ?";
                $stmt = $conn->prepare($update_query);
                $stmt->bind_param("si", $new_path, $video_id);
                
                if ($stmt->execute()) {
                    $success_count++;
                }
            }
            
            echo "<p class='success'>Updated {$success_count} paths to the correct format.</p>";
            echo "<p>Refresh page to see results.</p>";
        }
    }
} else {
    echo "<p>No thumbnails to analyze in the database.</p>";
}

echo "</div>";

// 4. Fix missing thumbnails
echo "<div class='container'>
    <h2>Fix Missing Thumbnails</h2>
    <p>This will assign thumbnails from uploads/thumbnails to videos without thumbnails</p>
    
    <form method='post'>
        <input type='submit' name='assign_thumbnails' value='Assign Thumbnails to Videos' style='padding: 10px; background-color: #2196F3; color: white; border: none; cursor: pointer;'>
    </form>";

if (isset($_POST['assign_thumbnails'])) {
    // Get all videos without thumbnails
    $query = "SELECT video_id, title FROM videos WHERE thumbnail IS NULL OR thumbnail = ''";
    $result = $conn->query($query);
    
    if ($result && $result->num_rows > 0) {
        $videos_to_update = [];
        while ($row = $result->fetch_assoc()) {
            $videos_to_update[] = $row;
        }
        
        // Get all thumbnails
        $thumbnails = [];
        if (is_dir($thumbnail_dir)) {
            $files = array_diff(scandir($thumbnail_dir), ['.', '..']);
            foreach ($files as $file) {
                if (is_file($thumbnail_dir . $file)) {
                    $thumbnails[] = $file;
                }
            }
        }
        
        $updated = 0;
        
        if (!empty($thumbnails)) {
            foreach ($videos_to_update as $index => $video) {
                // Choose a thumbnail based on video ID to ensure consistent selection
                $thumbnail_index = $video['video_id'] % count($thumbnails);
                $thumbnail = $thumbnails[$thumbnail_index];
                
                $thumbnail_path = 'uploads/thumbnails/' . $thumbnail;
                
                $update_query = "UPDATE videos SET thumbnail = ? WHERE video_id = ?";
                $stmt = $conn->prepare($update_query);
                $stmt->bind_param("si", $thumbnail_path, $video['video_id']);
                
                if ($stmt->execute()) {
                    $updated++;
                }
            }
        }
        
        echo "<p class='success'>Updated {$updated} videos with thumbnails.</p>";
        echo "<p>Refresh page to see results.</p>";
    } else {
        echo "<p>No videos without thumbnails found.</p>";
    }
}

echo "</div>";

echo "</body>
</html>";
?>
