<?php
// Include the centralized teacher layout
require_once __DIR__ . '/components/teacher_layout.php';

include_once "../includes/functions.php";
require_once "../auth/functions.php";
require_once "../includes/id_encryption.php"; // Include ID encryption utility

// Error handling and reporting during development
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Fatal error handler to capture and display errors
function shutdown_handler() {
    $error = error_get_last();
    if ($error !== NULL && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        echo '<pre>';
        print_r($error);
        echo '</pre>';
    }
}
register_shutdown_function('shutdown_handler');

// Get teacher data with fallback for emergency/test accounts
function courses_get_teacher($conn, $user_id) {
    if ($user_id == 999) {
        return [
            'user_id' => 999,
            'username' => 'teacher',
            'first_name' => 'Emergency',
            'last_name' => 'Teacher',
            'email' => 'emergency@example.com',
            'bio' => 'Emergency teacher account',
            'qualifications' => 'Emergency credentials',
            'expertise' => 'System testing',
            'status' => 'active'
        ];
    }
    
    if (!$conn) {
        error_log('Database connection not available in get_teacher');
        return [
            'user_id' => $user_id,
            'username' => 'teacher',
            'first_name' => 'Unknown',
            'last_name' => 'Teacher',
            'email' => 'unknown@example.com',
            'bio' => 'Teacher profile unavailable',
            'qualifications' => 'Unknown',
            'expertise' => 'Unknown',
            'status' => 'active'
        ];
    }
    
    try {
        // Debug message to see connection info
        error_log("DB Connection Status: " . ($conn ? "Connected" : "Not Connected"));
        
        // Check if teacher_profiles table exists and what columns it has
        $check_table = $conn->query("SHOW TABLES LIKE 'teacher_profiles'");
        if ($check_table && $check_table->num_rows == 0) {
            error_log("teacher_profiles table does not exist");
        }
        
        // Simpler query first to check if user exists
        $user_check = $conn->query("SELECT * FROM users WHERE user_id = {$user_id}");
        if (!$user_check || $user_check->num_rows == 0) {
            error_log("User with ID {$user_id} not found");
        } else {
            error_log("User with ID {$user_id} found");
        }
        
        // Prepare statement
        $stmt = $conn->prepare("SELECT u.*, tp.bio, tp.qualifications, tp.expertise, tp.status 
                              FROM users u 
                              LEFT JOIN teacher_profiles tp ON u.user_id = tp.teacher_id 
                              WHERE u.user_id = ?");
                              
        // Check if prepare was successful
        if ($stmt === false) {
            error_log("Prepare failed: (" . $conn->errno . ") " . $conn->error);
            // Fall back to a simpler query without join
            $stmt = $conn->prepare("SELECT * FROM users WHERE user_id = ?");
            if ($stmt === false) {
                error_log("Even simple prepare failed: (" . $conn->errno . ") " . $conn->error);
                throw new Exception("Database prepare error");
            }
        }
        
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result && $result->num_rows > 0) {
            return $result->fetch_assoc();
        }
    } catch (Exception $e) {
        error_log("Error getting teacher: " . $e->getMessage());
    }
    
    return [
        'user_id' => $user_id,
        'username' => 'teacher',
        'first_name' => 'Unknown',
        'last_name' => 'Teacher',
        'email' => 'unknown@example.com',
        'bio' => 'Teacher profile unavailable',
        'qualifications' => 'Unknown',
        'expertise' => 'Unknown',
        'status' => 'active'
    ];
}

// Get teacher courses with fallback data
function get_teacher_courses($conn, $teacher_id) {
    if ($teacher_id == 999) {
        return [
            ['course_id' => 1, 'title' => 'Introduction to PHP', 'description' => 'Learn PHP basics', 'image' => 'default.jpg', 'status' => 'published', 'created_at' => '2023-05-15', 'students' => 24],
            ['course_id' => 2, 'title' => 'Advanced JavaScript', 'description' => 'Master JS concepts', 'image' => 'default.jpg', 'status' => 'draft', 'created_at' => '2023-06-01', 'students' => 12],
            ['course_id' => 3, 'title' => 'Web Security', 'description' => 'Learn to secure web apps', 'image' => 'default.jpg', 'status' => 'published', 'created_at' => '2023-04-20', 'students' => 18]
        ];
    }
    
    if (!$conn) {
        return [];
    }
    
    $courses = [];
    
    try {
        // Check if courses table exists
        $check_table = $conn->query("SHOW TABLES LIKE 'courses'");
        if ($check_table && $check_table->num_rows == 0) {
            error_log("courses table does not exist");
            return [];
        }
        
        // Check if enrollments table exists for the subquery
        $check_enrollments = $conn->query("SHOW TABLES LIKE 'enrollments'");
        
        // If enrollments table doesn't exist, use simpler query without student count
        if ($check_enrollments && $check_enrollments->num_rows == 0) {
            error_log("enrollments table does not exist, using simpler query");
            $stmt = $conn->prepare("SELECT c.* FROM courses c WHERE c.teacher_id = ? ORDER BY c.created_at DESC");
        } else {
            $stmt = $conn->prepare("SELECT c.*, 
                                  (SELECT COUNT(*) FROM enrollments e WHERE e.course_id = c.course_id) as students,
                                  (SELECT AVG(rating) FROM course_reviews cr WHERE cr.course_id = c.course_id) as avg_rating,
                                  (SELECT COUNT(*) FROM course_reviews cr WHERE cr.course_id = c.course_id) as review_count
                                  FROM courses c 
                                  WHERE c.teacher_id = ? 
                                  ORDER BY c.created_at DESC");
        }
        
        // Check if prepare was successful
        if ($stmt === false) {
            error_log("Courses prepare failed: (" . $conn->errno . ") " . $conn->error);
            return [];
        }
        
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result && $result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                // If students key doesn't exist, set it to 0
                if (!isset($row['students'])) {
                    $row['students'] = 0;
                }
                $courses[] = $row;
            }
        }
    } catch (Exception $e) {
        error_log("Error getting courses: " . $e->getMessage());
    }
    
    return $courses;
}

// Fetch teacher data and courses
$teacher_id = $_SESSION['user_id'];
$teacher = courses_get_teacher($conn, $teacher_id);
$courses = get_teacher_courses($conn, $teacher_id);
?>

<?php $current_page = 'courses.php'; // Set current page for menu highlighting ?>

<!-- We'll remove this duplicate header as we'll use the improved one below -->

<!-- Display Session Messages -->
<?php if (isset($_SESSION['success_message'])): ?>
                <div class="row mb-3">
                    <div class="col-12">
                        <div class="alert alert-success alert-dismissible fade show" role="alert">
                            <i class="fas fa-check-circle me-2"></i>
                            <?php echo $_SESSION['success_message']; unset($_SESSION['success_message']); ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
                
                <?php if (isset($_SESSION['error_message'])): ?>
                <div class="row mb-3">
                    <div class="col-12">
                        <div class="alert alert-danger alert-dismissible fade show" role="alert">
                            <i class="fas fa-exclamation-circle me-2"></i>
                            <?php echo $_SESSION['error_message']; unset($_SESSION['error_message']); ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Page Header -->
<div class="row mb-4">
    <div class="col-12">
        <div class="card border-0" style="border-radius: 16px; box-shadow: 0 10px 20px rgba(0,0,0,0.05); overflow: hidden;">
            <div class="card-body p-4">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h4 class="mb-1 fw-bold">My Courses</h4>
                        <p class="text-muted mb-0">Manage your educational content</p>
                    </div>
                    <a href="create-course.php" class="btn btn-primary px-4 py-2 rounded-pill">
                        <svg class="svg-inline--fa fa-plus me-2" style="width: 16px; height: 16px;" aria-hidden="true" focusable="false" data-prefix="fas" data-icon="plus" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path fill="currentColor" d="M256 80c0-17.7-14.3-32-32-32s-32 14.3-32 32V224H48c-17.7 0-32 14.3-32 32s14.3 32 32 32H192V432c0 17.7 14.3 32 32 32s32-14.3 32-32V288H400c17.7 0 32-14.3 32-32s-14.3-32-32-32H256V80z"></path></svg>
                        Create New Course
                    </a>
                </div>
            </div>
        </div>
    </div>
</div>
                
                <!-- Course List -->
<div class="row">
    <div class="col-12">
        <div class="card border-0" style="border-radius: 16px; box-shadow: 0 6px 12px rgba(0,0,0,0.03); overflow: hidden;">
            <div class="card-body p-4">
                <div class="d-flex align-items-center justify-content-between mb-4">
                    <h5 class="fw-bold mb-0">Your Courses</h5>
                    <div class="d-flex">
                        <div class="input-group me-2" style="width: 250px;">
                            <span class="input-group-text border-end-0 bg-light">
                                <svg class="svg-inline--fa fa-search" style="width: 14px; height: 14px; color: #6c757d;" aria-hidden="true" focusable="false" data-prefix="fas" data-icon="search" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path fill="currentColor" d="M416 208c0 45.9-14.9 88.3-40 122.7L502.6 457.4c12.5 12.5 12.5 32.8 0 45.3s-32.8 12.5-45.3 0L330.7 376c-34.4 25.2-76.8 40-122.7 40C93.1 416 0 322.9 0 208S93.1 0 208 0S416 93.1 416 208zM208 352a144 144 0 1 0 0-288 144 144 0 1 0 0 288z"></path></svg>
                            </span>
                            <input type="text" class="form-control bg-light border-start-0" id="courseSearch" placeholder="Search courses...">
                        </div>
                        <select class="form-select bg-light" style="width: auto;">
                            <option value="all">All Status</option>
                            <option value="published">Published</option>
                            <option value="draft">Draft</option>
                        </select>
                    </div>
                </div>
                <div class="table-responsive">
                    <table class="table align-middle">
                        <thead>
                            <tr>
                                <th scope="col" style="font-weight: 500; color: #6c757d; font-size: 0.875rem;">Course</th>
                                <th scope="col" style="font-weight: 500; color: #6c757d; font-size: 0.875rem;">Status</th>
                                <th scope="col" style="font-weight: 500; color: #6c757d; font-size: 0.875rem;">Students</th>
                                <th scope="col" style="font-weight: 500; color: #6c757d; font-size: 0.875rem;">Rating</th>
                                <th scope="col" style="font-weight: 500; color: #6c757d; font-size: 0.875rem; text-align: right;">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            try {
                                // Make sure we have a valid database connection
                                if (!$conn) {
                                    // Attempt to reconnect
                                    include_once "../config/db_connect.php";
                                    if (!$conn) {
                                        throw new Exception("Database connection is not available");
                                    }
                                }
                                
                                // Fetch the teacher's courses from database with student count and ratings
                                $coursesQuery = "SELECT c.*, 
                                                (SELECT COUNT(*) FROM enrollments e WHERE e.course_id = c.course_id) as students,
                                                (SELECT AVG(rating) FROM course_reviews cr WHERE cr.course_id = c.course_id) as avg_rating,
                                                (SELECT COUNT(*) FROM course_reviews cr WHERE cr.course_id = c.course_id) as review_count
                                                FROM courses c 
                                                WHERE c.teacher_id = ? 
                                                ORDER BY c.created_at DESC";
                                $stmt = $conn->prepare($coursesQuery);
                                
                                if ($stmt === false) {
                                    throw new Exception("Failed to prepare statement: " . $conn->error);
                                }
                                
                                $stmt->bind_param('i', $teacher_id);
                                
                                if (!$stmt->execute()) {
                                    throw new Exception("Failed to execute query: " . $stmt->error);
                                }
                                
                                $result = $stmt->get_result();
                            
                                if ($result && $result->num_rows > 0) {
                                    while ($course = $result->fetch_assoc()) {
                                        // Get the category name
                                        $categoryName = "";
                                        if (!empty($course['category_id'])) {
                                            $catQuery = "SELECT category_name FROM categories WHERE category_id = ?";
                                            $catStmt = $conn->prepare($catQuery);
                                            $catStmt->bind_param('i', $course['category_id']);
                                            $catStmt->execute();
                                            $catResult = $catStmt->get_result();
                                            if ($catResult->num_rows > 0) {
                                                $categoryName = $catResult->fetch_assoc()['category_name'];
                                            }
                                        }
                                        // Output course row HTML
                                        ?>
                            <tr>
                                <td>
                                    <div class="d-flex align-items-center">
                                        <div style="width: 60px; height: 40px; overflow: hidden; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.05); position: relative; margin-right: 16px;">
                                            <img 
                                                src="<?php echo !empty($course['thumbnail']) ? $course['thumbnail'] : '../assets/images/placeholder.png'; ?>" 
                                                alt="<?php echo htmlspecialchars($course['title']); ?>" 
                                                style="width: 100%; height: 100%; object-fit: cover;"
                                            >
                                        </div>
                                        <div>
                                            <h6 class="mb-0 fw-medium"><?php echo htmlspecialchars($course['title']); ?></h6>
                                            <div class="d-flex align-items-center">
                                                <span class="badge text-bg-light rounded-pill py-1 px-2 me-2" style="font-size: 0.7rem; font-weight: 500;"><?php echo !empty($categoryName) ? htmlspecialchars($categoryName) : 'Uncategorized'; ?></span>
                                                <span class="text-muted" style="font-size: 0.7rem;"><?php echo date('M d, Y', strtotime($course['created_at'])); ?></span>
                                            </div>
                                        </div>
                                    </div>
                                </td>
                                <td>
                                    <?php if ($course['is_published']): ?>
                                    <span class="badge rounded-pill px-3 py-2" style="background-color: rgba(25, 135, 84, 0.1); color: #198754; font-weight: 500; font-size: 0.75rem;">
                                        <svg class="svg-inline--fa fa-circle me-1" style="width: 6px; height: 6px;" aria-hidden="true" focusable="false" data-prefix="fas" data-icon="circle" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path fill="currentColor" d="M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512z"></path></svg>
                                        Published
                                    </span>
                                    <?php else: ?>
                                    <span class="badge rounded-pill px-3 py-2" style="background-color: rgba(255, 193, 7, 0.1); color: #fd7e14; font-weight: 500; font-size: 0.75rem;">
                                        <svg class="svg-inline--fa fa-circle me-1" style="width: 6px; height: 6px;" aria-hidden="true" focusable="false" data-prefix="fas" data-icon="circle" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path fill="currentColor" d="M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512z"></path></svg>
                                        Draft
                                    </span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span class="fw-medium"><?php echo isset($course['students']) ? $course['students'] : 0; ?></span> <span class="text-muted">enrolled</span>
                                </td>
                                <td>
                                    <div class="d-flex align-items-center">
                                        <?php 
                                        $rating = isset($course['avg_rating']) && $course['avg_rating'] ? number_format($course['avg_rating'], 1) : null;
                                        $review_count = isset($course['review_count']) ? $course['review_count'] : 0;
                                        ?>
                                        <?php if ($rating): ?>
                                            <span class="me-1 fw-medium"><?php echo $rating; ?></span>
                                            <svg class="svg-inline--fa fa-star" style="width: 14px; height: 14px; color: #ffc107;" aria-hidden="true" focusable="false" data-prefix="fas" data-icon="star" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path fill="currentColor" d="M316.9 18C311.6 7 300.4 0 288.1 0s-23.4 7-28.8 18L195 150.3 51.4 171.5c-12 1.8-22 10.2-25.7 21.7s-.7 24.2 7.9 32.7L137.8 329 113.2 474.7c-2 12 3 24.2 12.9 31.3s23 8 33.8 2.3l128.3-68.5 128.3 68.5c10.8 5.7 23.9 4.9 33.8-2.3s14.9-19.3 12.9-31.3L438.5 329 542.7 225.9c8.6-8.5 11.7-21.2 7.9-32.7s-13.7-19.9-25.7-21.7L381.2 150.3 316.9 18z"></path></svg>
                                            <small class="text-muted ms-1">(<?php echo $review_count; ?>)</small>
                                        <?php else: ?>
                                            <span class="me-1 fw-medium text-muted">No reviews</span>
                                            <svg class="svg-inline--fa fa-star" style="width: 14px; height: 14px; color: #dee2e6;" aria-hidden="true" focusable="false" data-prefix="fas" data-icon="star" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path fill="currentColor" d="M316.9 18C311.6 7 300.4 0 288.1 0s-23.4 7-28.8 18L195 150.3 51.4 171.5c-12 1.8-22 10.2-25.7 21.7s-.7 24.2 7.9 32.7L137.8 329 113.2 474.7c-2 12 3 24.2 12.9 31.3s23 8 33.8 2.3l128.3-68.5 128.3 68.5c10.8 5.7 23.9 4.9 33.8-2.3s14.9-19.3 12.9-31.3L438.5 329 542.7 225.9c8.6-8.5 11.7-21.2 7.9-32.7s-13.7-19.9-25.7-21.7L381.2 150.3 316.9 18z"></path></svg>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td class="text-end">
                                    <div class="d-inline-flex">
                                        <a href="<?php echo IdEncryption::generateUrl('view-course.php', $course['course_id']); ?>" class="btn btn-sm btn-light rounded-circle me-1" title="View Course">
                                            <svg class="svg-inline--fa fa-eye" style="width: 14px; height: 14px;" aria-hidden="true" focusable="false" data-prefix="fas" data-icon="eye" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path fill="currentColor" d="M288 32c-80.8 0-145.5 36.8-192.6 80.6C48.6 156 17.3 208 2.5 243.7c-3.3 7.9-3.3 16.7 0 24.6C17.3 304 48.6 356 95.4 399.4C142.5 443.2 207.2 480 288 480s145.5-36.8 192.6-80.6c46.8-43.5 78.1-95.4 93-131.1c3.3-7.9 3.3-16.7 0-24.6c-14.9-35.7-46.2-87.7-93-131.1C433.5 68.8 368.8 32 288 32zM144 256a144 144 0 1 1 288 0 144 144 0 1 1 -288 0zm144-64c0 35.3-28.7 64-64 64c-7.1 0-13.9-1.2-20.3-3.3c-5.5-1.8-11.9 1.6-11.7 7.4c.3 6.9 1.3 13.8 3.2 20.7c13.7 51.2 66.4 81.6 117.6 67.9s81.6-66.4 67.9-117.6c-11.1-41.5-47.8-69.4-88.6-71.1c-5.8-.2-9.2 6.1-7.4 11.7c2.1 6.4 3.3 13.2 3.3 20.3z"></path></svg>
                                        </a>
                                        <a href="<?php echo IdEncryption::generateUrl('manage-content.php', $course['course_id']); ?>" class="btn btn-sm btn-light rounded-circle me-1" title="Manage Content">
                                            <svg class="svg-inline--fa fa-book" style="width: 14px; height: 14px;" aria-hidden="true" focusable="false" data-prefix="fas" data-icon="book" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path fill="currentColor" d="M96 0C43 0 0 43 0 96V416c0 53 43 96 96 96H384h32c17.7 0 32-14.3 32-32s-14.3-32-32-32V384c17.7 0 32-14.3 32-32V32c0-17.7-14.3-32-32-32H384 96zm0 384H352v64H96c-17.7 0-32-14.3-32-32s14.3-32 32-32zm32-240c0-8.8 7.2-16 16-16H336c8.8 0 16 7.2 16 16s-7.2 16-16 16H144c-8.8 0-16-7.2-16-16s7.2-16 16-16zm16 48H336c8.8 0 16 7.2 16 16s-7.2 16-16 16H144c-8.8 0-16-7.2-16-16s7.2-16 16-16z"></path></svg>
                                        </a>
                                        <a href="analytics.php" class="btn btn-sm btn-light rounded-circle me-1" title="View Analytics">
                                            <svg class="svg-inline--fa fa-chart-line" style="width: 14px; height: 14px;" aria-hidden="true" focusable="false" data-prefix="fas" data-icon="chart-line" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path fill="currentColor" d="M64 64c0-17.7-14.3-32-32-32S0 46.3 0 64V400c0 44.2 35.8 80 80 80H480c17.7 0 32-14.3 32-32s-14.3-32-32-32H80c-8.8 0-16-7.2-16-16V64zm406.6 86.6c12.5-12.5 12.5-32.8 0-45.3s-32.8-12.5-45.3 0L320 210.7l-57.4-57.4c-12.5-12.5-32.8-12.5-45.3 0l-112 112c-12.5 12.5-12.5 32.8 0 45.3s32.8 12.5 45.3 0L240 221.3l57.4 57.4c12.5 12.5 32.8 12.5 45.3 0l128-128z"></path></svg>
                                        </a>
                                        <div class="dropdown d-inline-block">
                                            <button class="btn btn-sm btn-light rounded-circle" type="button" data-bs-toggle="dropdown" title="More Options">
                                                <svg class="svg-inline--fa fa-ellipsis-vertical" style="width: 14px; height: 14px;" aria-hidden="true" focusable="false" data-prefix="fas" data-icon="ellipsis-vertical" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 128 512"><path fill="currentColor" d="M64 360a56 56 0 1 0 0 112 56 56 0 1 0 0-112zm0-160a56 56 0 1 0 0 112 56 56 0 1 0 0-112zM120 96A56 56 0 1 0 8 96a56 56 0 1 0 112 0z"></path></svg>
                                            </button>
                                            <ul class="dropdown-menu dropdown-menu-end">
                                                <li><a class="dropdown-item" href="#" data-course-id="<?php echo $course['course_id']; ?>" onclick="deleteCourse(<?php echo $course['course_id']; ?>)"><span class="text-danger"><svg class="svg-inline--fa fa-trash-can me-2" style="width: 14px; height: 14px;" aria-hidden="true" focusable="false" data-prefix="fas" data-icon="trash-can" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path fill="currentColor" d="M135.2 17.7L128 32H32C14.3 32 0 46.3 0 64S14.3 96 32 96H416c17.7 0 32-14.3 32-32s-14.3-32-32-32H320l-7.2-14.3C307.4 6.8 296.3 0 284.2 0H163.8c-12.1 0-23.2 6.8-28.6 17.7zM416 128H32L53.2 467c1.6 25.3 22.6 45 47.9 45H346.9c25.3 0 46.3-19.7 47.9-45L416 128z"></path></svg> Delete</span></a></li>
                                            </ul>
                                        </div>
                                    </div>
                                </td>
                            </tr>
                            <?php 
                                    } // End while loop for course rows
                                } else { // No courses found
                            ?>
                            <tr>
                                <td colspan="5" class="text-center py-5">
                                    <div class="py-4">
                                        <svg class="svg-inline--fa fa-folder-open mb-3" style="width: 48px; height: 48px; color: #dee2e6;" aria-hidden="true" focusable="false" data-prefix="fas" data-icon="folder-open" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path fill="currentColor" d="M88.7 223.8L0 375.8V96C0 60.7 28.7 32 64 32H181.5c17 0 33.3 6.7 45.3 18.7l26.5 26.5c12 12 28.3 18.7 45.3 18.7H416c35.3 0 64 28.7 64 64v32H144c-22.8 0-43.8 12.1-55.3 31.8zm27.6 16.1C122.1 230 132.6 224 144 224H544c11.5 0 22 6.1 27.7 16.1s5.7 22.2-.1 32.1l-112 192C453.9 474 443.4 480 432 480H32c-11.5 0-22-6.1-27.7-16.1s-5.7-22.2 .1-32.1l112-192z"></path></svg>
                                        <h5 class="mb-2">No courses found</h5>
                                        <p class="text-muted mb-4">Start sharing your knowledge by creating your first course</p>
                                        <a href="create-course.php" class="btn btn-primary rounded-pill px-4">
                                            <svg class="svg-inline--fa fa-plus me-2" style="width: 16px; height: 16px;" aria-hidden="true" focusable="false" data-prefix="fas" data-icon="plus" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path fill="currentColor" d="M256 80c0-17.7-14.3-32-32-32s-32 14.3-32 32V224H48c-17.7 0-32 14.3-32 32s14.3 32 32 32H192V432c0 17.7 14.3 32 32 32s32-14.3 32-32V288H400c17.7 0 32-14.3 32-32s-14.3-32-32-32H256V80z"></path></svg>
                                            Create New Course
                                        </a>
                                    </div>
                                </td>
                            </tr>
                            <?php 
                                } // End else for no courses found
                            } catch (Exception $e) {
                                echo '<tr><td colspan="5" class="text-center py-3 text-danger">Error: ' . htmlspecialchars($e->getMessage()) . '</td></tr>';
                                error_log("Error in courses.php: " . $e->getMessage());
                            }
                            ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
/* Modern Courses Page Styling - Panadite Theme */
* {
    font-family: 'Inter', 'Poppins', sans-serif !important;
}

.container-fluid {
    padding: 0 30px !important;
}

/* Page Header Styling */
h1.h3, h2, h3, h4, h5 {
    font-weight: 700 !important;
    color: #1f2937 !important;
    letter-spacing: -0.025em !important;
}

h1.h3 {
    font-size: 28px !important;
    margin-bottom: 8px !important;
}

.text-muted {
    color: #6b7280 !important;
    font-size: 16px !important;
    font-weight: 400 !important;
}

/* Modern Card Design */
.card {
    border: none !important;
    border-radius: 16px !important;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08) !important;
    background: #ffffff !important;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1) !important;
    overflow: hidden !important;
}

.card:hover {
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.12) !important;
    transform: translateY(-2px) !important;
}

.card-body {
    padding: 24px !important;
}

/* Button Styling */
.btn {
    border-radius: 12px !important;
    padding: 12px 24px !important;
    font-weight: 600 !important;
    font-size: 14px !important;
    transition: all 0.2s ease !important;
    border: none !important;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1) !important;
}

.btn-primary {
    background: linear-gradient(135deg, #4361ee, #5a72f0) !important;
    color: white !important;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #3b4de8, #4361ee) !important;
    transform: translateY(-1px) !important;
    box-shadow: 0 4px 12px rgba(67, 97, 238, 0.3) !important;
}

.btn-outline-primary {
    border: 2px solid #4361ee !important;
    color: #4361ee !important;
    background: transparent !important;
}

.btn-outline-primary:hover {
    background: #4361ee !important;
    color: white !important;
    transform: translateY(-1px) !important;
    box-shadow: 0 4px 12px rgba(67, 97, 238, 0.3) !important;
}

.btn-sm {
    padding: 8px 16px !important;
    font-size: 12px !important;
}

.btn-outline-secondary {
    border: 1px solid #e5e7eb !important;
    color: #6b7280 !important;
    background: #f9fafb !important;
}

.btn-outline-secondary:hover {
    background: #e5e7eb !important;
    color: #374151 !important;
    border-color: #d1d5db !important;
}

.btn-outline-danger {
    border: 1px solid #ef4444 !important;
    color: #ef4444 !important;
    background: #fef2f2 !important;
}

.btn-outline-danger:hover {
    background: #ef4444 !important;
    color: white !important;
}

/* Search and Filter Styling */
.input-group {
    border-radius: 12px !important;
    overflow: hidden !important;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05) !important;
}

.form-control, .form-select {
    border: 2px solid #e5e7eb !important;
    border-radius: 12px !important;
    padding: 12px 16px !important;
    font-size: 14px !important;
    transition: all 0.2s ease !important;
    background: #ffffff !important;
}

.form-control:focus, .form-select:focus {
    border-color: #4361ee !important;
    box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.1) !important;
    outline: none !important;
}

.input-group .form-control {
    border-radius: 12px 0 0 12px !important;
}

.input-group .btn {
    border-radius: 0 12px 12px 0 !important;
}

.form-control::placeholder {
    color: #9ca3af !important;
}

/* Table Styling */
.table-responsive {
    border-radius: 12px !important;
    overflow: hidden !important;
}

.table {
    margin-bottom: 0 !important;
    background: white !important;
}

.table th {
    font-weight: 600 !important;
    color: #374151 !important;
    font-size: 14px !important;
    text-transform: uppercase !important;
    letter-spacing: 0.05em !important;
    padding: 16px 20px !important;
    border-bottom: 2px solid #f3f4f6 !important;
    background: #f8fafc !important;
}

.table td {
    padding: 20px !important;
    border-bottom: 1px solid #f1f5f9 !important;
    vertical-align: middle !important;
}

.table tbody tr {
    transition: all 0.2s ease !important;
}

.table tbody tr:hover {
    background: #f8fafc !important;
}

/* Course Image Styling */
.course-image {
    border-radius: 8px !important;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1) !important;
    transition: transform 0.2s ease !important;
}

.course-image:hover {
    transform: scale(1.05) !important;
}

/* Status Badge Styling */
.badge {
    padding: 6px 12px !important;
    border-radius: 20px !important;
    font-weight: 600 !important;
    font-size: 11px !important;
    text-transform: uppercase !important;
    letter-spacing: 0.05em !important;
}

.bg-success {
    background: linear-gradient(135deg, #10b981, #059669) !important;
}

.bg-warning {
    background: linear-gradient(135deg, #f59e0b, #d97706) !important;
}

.bg-secondary {
    background: linear-gradient(135deg, #6b7280, #4b5563) !important;
}

/* Course Title and Info */
.fw-bold {
    font-weight: 700 !important;
    color: #1f2937 !important;
}

.text-secondary {
    color: #6b7280 !important;
    font-size: 13px !important;
}

/* Action Buttons */
.dropdown-toggle::after {
    border: none !important;
    font-family: 'boxicons' !important;
    content: '\e928' !important;
    font-size: 16px !important;
}

.dropdown-menu {
    border: none !important;
    border-radius: 12px !important;
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15) !important;
    padding: 8px !important;
    margin-top: 8px !important;
}

.dropdown-item {
    border-radius: 8px !important;
    padding: 10px 16px !important;
    font-size: 14px !important;
    transition: all 0.2s ease !important;
    display: flex !important;
    align-items: center !important;
}

.dropdown-item i {
    margin-right: 8px !important;
    width: 16px !important;
}

.dropdown-item:hover {
    background: #f8fafc !important;
    color: #1f2937 !important;
}

/* Empty State Styling */
.py-5 {
    padding: 48px 0 !important;
}

.py-4 {
    padding: 32px 0 !important;
}

.svg-inline--fa {
    color: #d1d5db !important;
}

.text-center h5 {
    color: #374151 !important;
    font-size: 20px !important;
    margin-bottom: 8px !important;
}

/* Rating Stars */
.text-warning {
    color: #f59e0b !important;
}

/* Responsive Design */
@media (max-width: 992px) {
    .container-fluid {
        padding: 0 20px !important;
    }
    
    .d-flex.align-items-center.justify-content-between {
        flex-direction: column !important;
        align-items: stretch !important;
        gap: 16px !important;
    }
    
    .d-flex.me-2 {
        margin-right: 0 !important;
        margin-bottom: 12px !important;
    }
}

@media (max-width: 768px) {
    .card-body {
        padding: 20px !important;
    }
    
    h1.h3 {
        font-size: 24px !important;
    }
    
    .table-responsive {
        font-size: 13px !important;
    }
    
    .table th, .table td {
        padding: 12px 8px !important;
    }
    
    .course-image {
        width: 40px !important;
        height: 40px !important;
    }
}

/* Animation for loading states */
@keyframes fadeIn {
    from { opacity: 0; transform: translateY(20px); }
    to { opacity: 1; transform: translateY(0); }
}

.table tbody tr {
    animation: fadeIn 0.3s ease-out !important;
}

/* Success/Error Messages */
.alert {
    border: none !important;
    border-radius: 12px !important;
    padding: 16px 20px !important;
    font-weight: 500 !important;
    margin-bottom: 24px !important;
}

.alert-success {
    background: linear-gradient(135deg, #ecfdf5, #d1fae5) !important;
    color: #065f46 !important;
}

.alert-danger {
    background: linear-gradient(135deg, #fef2f2, #fecaca) !important;
    color: #991b1b !important;
}

/* Course Stats */
.text-primary {
    color: #4361ee !important;
    font-weight: 600 !important;
}

/* Improved spacing */
.mb-3 {
    margin-bottom: 24px !important;
}

.mb-4 {
    margin-bottom: 32px !important;
}

/* Fix horizontal scrolling only - let natural vertical scrolling work */
body {
    overflow-x: hidden !important;
}

.main-content {
    padding: 20px ;
    
}

.container-fluid {
    padding-left: 24px !important;
    padding-right: 24px !important;
 
}

.row {
    margin-left: 0 !important;
    margin-right: 0 !important;
}

.col-12 {
    padding-left: 12px !important;
    padding-right: 12px !important;
}

/* Custom scrollbar for table */
.table-responsive::-webkit-scrollbar {
    height: 8px;
}

.table-responsive::-webkit-scrollbar-track {
    background: #f1f5f9;
    border-radius: 4px;
}

.table-responsive::-webkit-scrollbar-thumb {
    background: #cbd5e1;
    border-radius: 4px;
}

.table-responsive::-webkit-scrollbar-thumb:hover {
    background: #94a3b8;
}

/* Action Buttons Styling - Make them visible and beautiful */
.btn-light {
    background: #f8fafc !important;
    border: 1px solid #e2e8f0 !important;
    color: #64748b !important;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1) !important;
}

.btn-light:hover {
    background: #e2e8f0 !important;
    border-color: #cbd5e1 !important;
    color: #475569 !important;
    transform: translateY(-1px) !important;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.15) !important;
}

.btn-light:focus {
    background: #e2e8f0 !important;
    border-color: #4361ee !important;
    color: #475569 !important;
    box-shadow: 0 0 0 2px rgba(67, 97, 238, 0.2) !important;
}

/* Rounded circle action buttons */
.btn.rounded-circle {
    width: 36px !important;
    height: 36px !important;
    padding: 0 !important;
    display: flex !important;
    align-items: center !important;
    justify-content: center !important;
    border-radius: 50% !important;
}

/* Action buttons container */
.d-inline-flex {
    gap: 4px !important;
}

/* SVG icons in action buttons */
.btn .svg-inline--fa {
    width: 14px !important;
    height: 14px !important;
    color: inherit !important;
}

/* Dropdown button specific styling */
.dropdown-toggle.btn-light {
    background: #f8fafc !important;
    border: 1px solid #e2e8f0 !important;
}

.dropdown-toggle.btn-light:hover {
    background: #e2e8f0 !important;
}

/* Make sure dropdown menu items are visible */
.dropdown-menu .dropdown-item {
    color: #374151 !important;
    font-size: 13px !important;
}

.dropdown-menu .dropdown-item:hover {
    background: #f8fafc !important;
    color: #1f2937 !important;
}

.dropdown-menu .dropdown-item .text-danger {
    color: #ef4444 !important;
}

.dropdown-menu .dropdown-item:hover .text-danger {
    color: #dc2626 !important;
}
</style>

<?php
// Include the centralized teacher layout end
require_once __DIR__ . '/components/teacher_layout_end.php';
?>

<!-- Page-specific scripts -->
<script>
    // Delete course functionality
    function deleteCourse(courseId) {
        if (confirm('Are you sure you want to delete this course? This action cannot be undone.')) {
            // Create a form to submit the delete request
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = 'delete-course.php';
            
            const input = document.createElement('input');
            input.type = 'hidden';
            input.name = 'course_id';
            input.value = courseId;
            
            form.appendChild(input);
            document.body.appendChild(form);
            form.submit();
        }
    }
</script>
