<?php
// Include the centralized teacher layout
require_once __DIR__ . '/components/teacher_layout.php';
require_once '../auth/functions.php';
requireRole('teacher');
require_once '../config/db_connect.php';
require_once '../includes/id_encryption.php';

// Get current teacher ID
$teacher_id = $_SESSION['user_id'];

// Check if this is for a specific video (coming from content workflow)
$video_id = null;
$video_info = null;
$encrypted_video_id = $_GET['video_id'] ?? '';
$from_workflow = !empty($encrypted_video_id);

if ($from_workflow) {
    $video_id = IdEncryption::decrypt($encrypted_video_id);
    
    // Get video information for context
    $video_stmt = $pdo->prepare("
        SELECT v.video_id, v.title as video_title, v.course_id, c.title as course_title 
        FROM videos v 
        JOIN courses c ON v.course_id = c.course_id 
        WHERE v.video_id = ? AND v.teacher_id = ?
    ");
    $video_stmt->execute([$video_id, $teacher_id]);
    $video_info = $video_stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$video_info) {
        $_SESSION['error_message'] = 'Video not found or access denied.';
        header('Location: upload-video.php');
        exit();
    }
}

// Get teacher's courses
$courses_stmt = $pdo->prepare("SELECT course_id, title FROM courses WHERE teacher_id = ? ORDER BY title");
$courses_stmt->execute([$teacher_id]);
$courses = $courses_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get teacher's videos for optional video selection (using correct query from videos.php)
$videos_stmt = $pdo->prepare("
    SELECT v.video_id, v.title as video_title, v.course_id, c.title as course_title 
    FROM videos v 
    LEFT JOIN courses c ON v.course_id = c.course_id 
    WHERE c.teacher_id = ? 
    ORDER BY v.created_at DESC
");
$videos_stmt->execute([$teacher_id]);
$videos = $videos_stmt->fetchAll(PDO::FETCH_ASSOC);

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $quiz_title = trim($_POST['quiz_title']);
    $quiz_description = trim($_POST['quiz_description']);
    $course_id = intval($_POST['course_id']);
    $time_limit = intval($_POST['time_limit']);
    $passing_score = intval($_POST['passing_score']);
    $ai_grading_enabled = isset($_POST['ai_grading_enabled']) ? 1 : 0;
    $auto_grade_mcq = isset($_POST['auto_grade_mcq']) ? 1 : 0;
    $feedback_enabled = isset($_POST['feedback_enabled']) ? 1 : 0;
    $contributes_to_final_grade = isset($_POST['contributes_to_final_grade']) ? 1 : 0;
    
    // Handle video-specific quiz (from workflow)
    $form_video_id = !empty($_POST['video_id']) ? intval($_POST['video_id']) : null;
    
    // Validate required fields
    if (empty($quiz_title) || empty($course_id)) {
        $error = "Quiz title and course are required.";
    } else {
        try {
            $pdo->beginTransaction();
            
            // Insert quiz (with video_id if applicable)
            $quiz_stmt = $pdo->prepare("
                INSERT INTO quizzes (teacher_id, course_id, video_id, title, description, time_limit, passing_score, 
                                   ai_grading_enabled, auto_grade_mcq, feedback_enabled, contributes_to_final_grade, 
                                   is_published, created_at, updated_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 0, NOW(), NOW())
            ");
            
            $quiz_stmt->execute([
                $teacher_id, $course_id, $form_video_id, $quiz_title, $quiz_description, 
                $time_limit, $passing_score, $ai_grading_enabled, $auto_grade_mcq, $feedback_enabled, $contributes_to_final_grade
            ]);
            
            $quiz_id = $pdo->lastInsertId();
            
            // Process questions
            if (isset($_POST['questions']) && is_array($_POST['questions'])) {
                foreach ($_POST['questions'] as $index => $question_data) {
                    $question_text = trim($question_data['text']);
                    $question_type = $question_data['type'];
                    $points = intval($question_data['points']);
                    
                    if (empty($question_text)) continue;
                    
                    // Insert question
                    $question_stmt = $pdo->prepare("
                        INSERT INTO quiz_questions (quiz_id, question_text, question_type, points, created_at)
                        VALUES (?, ?, ?, ?, NOW())
                    ");
                    
                    $question_stmt->execute([$quiz_id, $question_text, $question_type, $points]);
                    $question_id = $pdo->lastInsertId();
                    
                    // Insert options for multiple choice and true/false
                    if ($question_type === 'multiple_choice' && isset($question_data['options'])) {
                        foreach ($question_data['options'] as $option_index => $option_text) {
                            if (empty(trim($option_text))) continue;
                            
                            $is_correct = isset($question_data['correct_option']) && 
                                         $question_data['correct_option'] == $option_index ? 1 : 0;
                            
                            $option_stmt = $pdo->prepare("
                                INSERT INTO quiz_question_options (question_id, option_text, is_correct, created_at)
                                VALUES (?, ?, ?, NOW())
                            ");
                            
                            $option_stmt->execute([$question_id, trim($option_text), $is_correct]);
                        }
                    } elseif ($question_type === 'true_false') {
                        // Insert True/False options
                        $correct_answer = $question_data['correct_answer'] ?? 'true';
                        
                        $true_stmt = $pdo->prepare("
                            INSERT INTO quiz_question_options (question_id, option_text, is_correct, created_at)
                            VALUES (?, 'True', ?, NOW())
                        ");
                        $true_stmt->execute([$question_id, $correct_answer === 'true' ? 1 : 0]);
                        
                        $false_stmt = $pdo->prepare("
                            INSERT INTO quiz_question_options (question_id, option_text, is_correct, created_at)
                            VALUES (?, 'False', ?, NOW())
                        ");
                        $false_stmt->execute([$question_id, $correct_answer === 'false' ? 1 : 0]);
                    }
                }
            }
            
            $pdo->commit();
            
            // Handle workflow completion if this is a video-specific quiz
            if ($from_workflow && $form_video_id) {
                // Mark quiz step as completed in workflow session
                if (isset($_SESSION['content_workflow'])) {
                    if (!isset($_SESSION['content_workflow']['completed_steps'])) {
                        $_SESSION['content_workflow']['completed_steps'] = [];
                    }
                    $_SESSION['content_workflow']['completed_steps'][] = 'quiz';
                    $_SESSION['content_workflow']['current_step'] = $_SESSION['content_workflow']['current_step'] + 1;
                }
                
                $_SESSION['success_message'] = "Quiz created successfully and linked to your video!";
                header('Location: content-creation-workflow.php?video_id=' . $encrypted_video_id);
                exit();
            } else {
                $success = "Quiz created successfully!";
            }
            
        } catch (Exception $e) {
            $pdo->rollBack();
            $error = "Error creating quiz: " . $e->getMessage();
        }
    }
}

$page_title = "Create New Quiz";
?>

<!-- Video Filtering Script -->
<script>
let lastSelectedCourseId = '';

// Initially hide all videos on page load
document.addEventListener('DOMContentLoaded', function() {
    const videoSelect = document.querySelector('select[name="video_id"]');
    if (videoSelect) {
        const videoOptions = videoSelect.querySelectorAll('option');
        videoOptions.forEach(option => {
            if (option.value !== '') {
                option.style.display = 'none'; // Hide all videos initially
            }
        });
    }
});

function filterVideosByCourse() {
    const courseSelect = document.querySelector('select[name="course_id"]');
    const videoSelect = document.querySelector('select[name="video_id"]');
    
    if (!courseSelect || !videoSelect) return;
    
    const selectedCourseId = courseSelect.value;
    const videoOptions = videoSelect.querySelectorAll('option');
    
    // Only reset video selection if course actually changed
    if (lastSelectedCourseId !== selectedCourseId) {
        videoSelect.value = '';
        lastSelectedCourseId = selectedCourseId;
    }
    
    videoOptions.forEach(option => {
        if (option.value === '') {
            option.style.display = 'block'; // Always show "Select a video" option
            return;
        }
        
        const videoCourseId = option.getAttribute('data-course-id');
        
        if (!selectedCourseId) {
            // Hide all videos if no course selected
            option.style.display = 'none';
        } else if (videoCourseId === selectedCourseId) {
            // Show videos that match selected course
            option.style.display = 'block';
        } else {
            // Hide videos that don't match selected course
            option.style.display = 'none';
        }
    });
}
</script>

<!-- Additional CSS for quiz creation -->
    
    <style>
        :root {
            --panadite-primary: #2563eb;
            --panadite-secondary: #1d4ed8;
            --panadite-light: #dbeafe;
            --panadite-dark: #1e40af;
            --panadite-accent: #f59e0b;
            --text-primary: #1f2937;
            --text-secondary: #6b7280;
            --border-color: #e5e7eb;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --error-color: #ef4444;
            --bg-light: #f8fafc;
        }

        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            background: linear-gradient(135deg, var(--bg-light) 0%, #e2e8f0 100%);
            color: var(--text-primary);
            line-height: 1.6;
            min-height: 100vh;
        }

        .page-header {
            background: linear-gradient(135deg, var(--panadite-primary) 0%, var(--panadite-secondary) 100%);
            color: white;
            padding: 2rem 0;
            margin-bottom: 2rem;
            box-shadow: 0 4px 20px rgba(37, 99, 235, 0.15);
        }

        .header-content {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 1rem;
        }

        .page-title {
            font-size: 2.5rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
            text-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }

        .page-subtitle {
            font-size: 1.1rem;
            opacity: 0.9;
            font-weight: 400;
        }

        .container {
            max-width: 1000px;
            margin: 0 auto;
            padding: 0 1rem 2rem;
        }

        .form-card {
            background: white;
            border-radius: 20px;
            overflow: hidden;
            box-shadow: 0 6px 25px rgba(0,0,0,0.08);
            margin-bottom: 2rem;
        }

        .form-header {
            padding: 2rem;
            background: linear-gradient(135deg, var(--bg-light) 0%, #e2e8f0 50%);
            border-bottom: 1px solid var(--border-color);
        }

        .form-body {
            padding: 2rem;
        }

        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-label {
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
            display: block;
        }

        .form-control {
            width: 100%;
            padding: 0.75rem 1rem;
            border: 2px solid var(--border-color);
            border-radius: 8px;
            font-size: 0.95rem;
            transition: all 0.3s ease;
        }

        .form-control:focus {
            outline: none;
            border-color: var(--panadite-primary);
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }

        .form-select {
            width: 100%;
            padding: 0.75rem 1rem;
            border: 2px solid var(--border-color);
            border-radius: 8px;
            font-size: 0.95rem;
            background-color: white;
        }

        .form-check {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            margin-bottom: 1rem;
        }

        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            font-size: 0.95rem;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }

        .btn-primary {
            background: var(--panadite-primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--panadite-secondary);
        }

        .btn-success {
            background: var(--success-color);
            color: white;
        }

        .btn-outline {
            background: transparent;
            color: var(--panadite-primary);
            border: 2px solid var(--panadite-primary);
        }

        .btn-outline:hover {
            background: var(--panadite-primary);
            color: white;
        }

        .questions-container {
            margin-top: 2rem;
        }

        .question-card {
            background: white;
            border: 2px solid var(--border-color);
            border-radius: 12px;
            padding: 1.5rem;
            margin-bottom: 1rem;
            position: relative;
        }

        .question-header {
            display: flex;
            justify-content: between;
            align-items: center;
            margin-bottom: 1rem;
        }

        .question-number {
            background: var(--panadite-primary);
            color: white;
            width: 30px;
            height: 30px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            margin-right: 1rem;
        }

        .remove-question {
            background: var(--error-color);
            color: white;
            border: none;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .options-container {
            margin-top: 1rem;
        }

        .option-group {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            margin-bottom: 0.5rem;
        }

        .option-input {
            flex: 1;
            padding: 0.5rem;
            border: 1px solid var(--border-color);
            border-radius: 6px;
        }

        .radio-input {
            width: 20px;
            height: 20px;
        }

        .alert {
            padding: 1rem 1.5rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .alert-success {
            background: #d1fae5;
            color: #059669;
            border: 1px solid #a7f3d0;
        }

        .alert-danger {
            background: #fee2e2;
            color: #dc2626;
            border: 1px solid #fca5a5;
        }

        @media (max-width: 768px) {
            .page-title {
                font-size: 2rem;
            }
            
            .form-body {
                padding: 1.5rem;
            }
        }
    </style>

    <div class="page-header">
        <div class="header-content">
            <h1 class="page-title">
                <i class="bx bx-plus-circle"></i>
                Create New Quiz
            </h1>
            <p class="page-subtitle">Build dynamic quizzes with multiple question types and auto-marking</p>
        </div>
    </div>

    <div class="container">
        <!-- Messages -->
        <?php if (isset($success)): ?>
            <div class="alert alert-success">
                <i class="bx bx-check-circle"></i>
                <?= htmlspecialchars($success) ?>
                <a href="quizzes.php" style="margin-left: 1rem;">Go to Quiz Management</a>
            </div>
        <?php endif; ?>

        <?php if (isset($error)): ?>
            <div class="alert alert-danger">
                <i class="bx bx-error-circle"></i>
                <?= htmlspecialchars($error) ?>
            </div>
        <?php endif; ?>

        <!-- Quiz Creation Form -->
        <form method="POST" id="quizForm">
            <!-- Basic Information -->
            <div class="form-card">
                <div class="form-header">
                    <h3><i class="bx bx-info-circle"></i> Quiz Information</h3>
                </div>
                <div class="form-body">
                    <div class="row">
                        <div class="col-md-8">
                            <div class="form-group">
                                <label class="form-label">Quiz Title *</label>
                                <input type="text" name="quiz_title" class="form-control" required 
                                       placeholder="Enter quiz title...">
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label class="form-label">Course *</label>
                                <select name="course_id" class="form-select" required onchange="filterVideosByCourse()">
                                    <option value="">Select Course</option>
                                    <?php foreach ($courses as $course): ?>
                                        <option value="<?= $course['course_id'] ?>">
                                            <?= htmlspecialchars($course['title']) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>

                    <!-- Video Selection (Optional) -->
                    <div class="form-group">
                        <label for="video_id" class="form-label">
                            <i class="bx bx-video me-2" style="color: #667eea;"></i>Link to Specific Video (Optional)
                        </label>
                        <select id="video_id" name="video_id" class="form-select">
                            <option value="">Select a video (optional)</option>
                            <?php foreach ($videos as $video): ?>
                                <option value="<?= $video['video_id'] ?>" data-course-id="<?= $video['course_id'] ?? '' ?>">
                                    <?= htmlspecialchars($video['video_title']) ?> (<?= htmlspecialchars($video['course_title']) ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <small class="form-text text-muted">
                            <i class="bx bx-info-circle me-1"></i>
                            Select a video to create a video-specific quiz, or leave blank for a general course quiz.
                        </small>
                    </div>

                    <div class="form-group">
                        <label class="form-label">Description</label>
                        <textarea name="quiz_description" class="form-control" rows="3" 
                                  placeholder="Provide a brief description of this quiz..."></textarea>
                    </div>

                    <div class="row">
                        <div class="col-md-4">
                            <div class="form-group">
                                <label class="form-label">Time Limit (minutes)</label>
                                <input type="number" name="time_limit" class="form-control" value="30" min="1" max="300">
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label class="form-label">Passing Score (%)</label>
                                <input type="number" name="passing_score" class="form-control" value="60" min="1" max="100">
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="feedback_enabled">Quiz Settings</label>
                                <div class="form-check">
                                    <input type="checkbox" class="form-check-input" id="ai_grading_enabled" name="ai_grading_enabled" checked>
                                    <label class="form-check-label" for="ai_grading_enabled">
                                        <i class="bx bx-brain"></i> Enable AI Grading
                                    </label>
                                </div>
                                <div class="form-check">
                                    <input type="checkbox" class="form-check-input" id="auto_grade_mcq" name="auto_grade_mcq" checked>
                                    <label class="form-check-label" for="auto_grade_mcq">
                                        <i class="bx bx-check-square"></i> Auto-grade Multiple Choice
                                    </label>
                                </div>
                                <div class="form-check">
                                    <input type="checkbox" class="form-check-input" id="feedback_enabled" name="feedback_enabled" checked>
                                    <label class="form-check-label" for="feedback_enabled">
                                        <i class="bx bx-comment-detail"></i> Enable Feedback
                                    </label>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Final Grade Contribution Section -->
            <div class="card mb-4" style="border-left: 4px solid #f59e0b;">
                <div class="card-header" style="background: linear-gradient(135deg, #fef3e2 0%, #fde68a 100%); border-bottom: 1px solid #f59e0b;">
                    <h5 class="card-title mb-0" style="color: #d97706;">
                        <i class="bx bx-medal" style="font-size: 1.2rem; margin-right: 0.5rem;"></i>
                        Grade Settings
                    </h5>
                </div>
                <div class="card-body" style="background: #fffbeb;">
                    <div class="form-check" style="padding: 1rem; background: white; border-radius: 8px; border: 1px solid #f59e0b;">
                        <input type="checkbox" class="form-check-input" id="contributes_to_final_grade" name="contributes_to_final_grade" checked style="transform: scale(1.2);">
                        <label class="form-check-label" for="contributes_to_final_grade" style="font-size: 1.1rem; font-weight: 600; color: #d97706; margin-left: 0.5rem;">
                            <i class="bx bx-medal" style="margin-right: 0.5rem;"></i>
                            Contributes to Final Grade
                        </label>
                        <div style="margin-top: 0.5rem; margin-left: 2rem;">
                            <small class="text-muted" style="display: block; line-height: 1.4;">
                                <strong>Checked:</strong> This quiz will count toward students' final grades<br>
                                <strong>Unchecked:</strong> Practice quiz only - won't affect final grades
                            </small>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Questions Section -->
            <div class="form-card">
                <div class="form-header">
                    <h3><i class="bx bx-help-circle"></i> Quiz Questions</h3>
                    <button type="button" class="btn btn-primary" onclick="addQuestion()">
                        <i class="bx bx-plus"></i> Add Question
                    </button>
                </div>
                <div class="form-body">
                    <div id="questionsContainer">
                        <!-- Questions will be added here dynamically -->
                    </div>
                </div>
            </div>

            <!-- Submit Buttons -->
            <div class="d-flex gap-3">
                <button type="submit" class="btn btn-success">
                    <i class="bx bx-check"></i> Create Quiz
                </button>
                <a href="quizzes.php" class="btn btn-outline">
                    <i class="bx bx-arrow-back"></i> Cancel
                </a>
            </div>
        </form>
    </div>

    <script>
        let questionCount = 0;

        function addQuestion() {
            questionCount++;
            const questionHtml = `
                <div class="question-card" id="question-${questionCount}">
                    <div class="question-header">
                        <div class="d-flex align-items-center">
                            <span class="question-number">${questionCount}</span>
                            <h5>Question ${questionCount}</h5>
                        </div>
                        <button type="button" class="remove-question" onclick="removeQuestion(${questionCount})">
                            <i class="bx bx-x"></i>
                        </button>
                    </div>

                    <div class="form-group">
                        <label class="form-label">Question Text *</label>
                        <textarea name="questions[${questionCount}][text]" class="form-control" rows="2" required
                                  placeholder="Enter your question..."></textarea>
                    </div>

                    <div class="row">
                        <div class="col-md-8">
                            <div class="form-group">
                                <label class="form-label">Question Type *</label>
                                <select name="questions[${questionCount}][type]" class="form-select" 
                                        onchange="updateQuestionType(${questionCount}, this.value)" required>
                                    <option value="multiple_choice">Multiple Choice</option>
                                    <option value="true_false">True/False</option>
                                    <option value="short_answer">Short Answer</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label class="form-label">Points</label>
                                <input type="number" name="questions[${questionCount}][points]" class="form-control" 
                                       value="1" min="1" max="100">
                            </div>
                        </div>
                    </div>

                    <div id="options-${questionCount}" class="options-container">
                        ${generateMultipleChoiceOptions(questionCount)}
                    </div>
                </div>
            `;

            document.getElementById('questionsContainer').insertAdjacentHTML('beforeend', questionHtml);
        }

        function removeQuestion(questionId) {
            document.getElementById(`question-${questionId}`).remove();
            renumberQuestions();
        }

        function renumberQuestions() {
            const questions = document.querySelectorAll('.question-card');
            questions.forEach((question, index) => {
                const newNumber = index + 1;
                question.querySelector('.question-number').textContent = newNumber;
                question.querySelector('h5').textContent = `Question ${newNumber}`;
            });
        }

        function updateQuestionType(questionId, type) {
            const optionsContainer = document.getElementById(`options-${questionId}`);
            
            switch(type) {
                case 'multiple_choice':
                    optionsContainer.innerHTML = generateMultipleChoiceOptions(questionId);
                    break;
                case 'true_false':
                    optionsContainer.innerHTML = generateTrueFalseOptions(questionId);
                    break;
                case 'short_answer':
                    optionsContainer.innerHTML = `<p class="text-muted">Students will type their answer in a text field.</p>`;
                    break;
            }
        }

        function generateMultipleChoiceOptions(questionId) {
            return `
                <label class="form-label">Answer Options (select correct answer):</label>
                <div class="option-group">
                    <input type="radio" name="questions[${questionId}][correct_option]" value="0" class="radio-input" checked>
                    <input type="text" name="questions[${questionId}][options][]" class="option-input" 
                           placeholder="Option 1" required>
                </div>
                <div class="option-group">
                    <input type="radio" name="questions[${questionId}][correct_option]" value="1" class="radio-input">
                    <input type="text" name="questions[${questionId}][options][]" class="option-input" 
                           placeholder="Option 2" required>
                </div>
                <div class="option-group">
                    <input type="radio" name="questions[${questionId}][correct_option]" value="2" class="radio-input">
                    <input type="text" name="questions[${questionId}][options][]" class="option-input" 
                           placeholder="Option 3">
                </div>
                <div class="option-group">
                    <input type="radio" name="questions[${questionId}][correct_option]" value="3" class="radio-input">
                    <input type="text" name="questions[${questionId}][options][]" class="option-input" 
                           placeholder="Option 4">
                </div>
                <button type="button" class="btn btn-outline btn-sm mt-2" onclick="addOption(${questionId})">
                    <i class="bx bx-plus"></i> Add Option
                </button>
            `;
        }

        function generateTrueFalseOptions(questionId) {
            return `
                <label class="form-label">Correct Answer:</label>
                <div class="form-check">
                    <input type="radio" name="questions[${questionId}][correct_answer]" value="true" checked>
                    <label>True</label>
                </div>
                <div class="form-check">
                    <input type="radio" name="questions[${questionId}][correct_answer]" value="false">
                    <label>False</label>
                </div>
            `;
        }

        function addOption(questionId) {
            const optionsContainer = document.getElementById(`options-${questionId}`);
            const optionGroups = optionsContainer.querySelectorAll('.option-group');
            const optionCount = optionGroups.length;
            
            const newOption = `
                <div class="option-group">
                    <input type="radio" name="questions[${questionId}][correct_option]" value="${optionCount}" class="radio-input">
                    <input type="text" name="questions[${questionId}][options][]" class="option-input" 
                           placeholder="Option ${optionCount + 1}">
                    <button type="button" class="btn btn-outline btn-sm" onclick="removeOption(this)">
                        <i class="bx bx-x"></i>
                    </button>
                </div>
            `;
            
            optionsContainer.querySelector('.btn').insertAdjacentHTML('beforebegin', newOption);
        }

        function removeOption(button) {
            button.closest('.option-group').remove();
        }

        // Add first question on load
        document.addEventListener('DOMContentLoaded', function() {
            addQuestion();
        });
    </script>

<?php include_once __DIR__ . '/components/teacher_layout_end.php'; ?>
