<?php
session_start();
include_once "../config/db_connect.php";

// Check if user is logged in and is a teacher or admin
if (!isset($_SESSION['user_id']) || ($_SESSION['user_role'] !== 'teacher' && $_SESSION['user_role'] !== 'admin')) {
    header("Location: ../login.php");
    exit;
}

// Create quizzes table
$createQuizTable = "CREATE TABLE IF NOT EXISTS quizzes (
    quiz_id INT AUTO_INCREMENT PRIMARY KEY,
    teacher_id INT NOT NULL,
    course_id INT NOT NULL,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    time_limit INT DEFAULT 30,
    passing_score INT DEFAULT 60,
    is_published TINYINT(1) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (teacher_id) REFERENCES users(user_id) ON DELETE CASCADE,
    FOREIGN KEY (course_id) REFERENCES courses(course_id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";

// Create quiz questions table
$createQuestionsTable = "CREATE TABLE IF NOT EXISTS quiz_questions (
    question_id INT AUTO_INCREMENT PRIMARY KEY,
    quiz_id INT NOT NULL,
    question_text TEXT NOT NULL,
    question_type ENUM('multiple_choice', 'true_false', 'short_answer') NOT NULL DEFAULT 'multiple_choice',
    points INT NOT NULL DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (quiz_id) REFERENCES quizzes(quiz_id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";

// Create quiz question options table
$createOptionsTable = "CREATE TABLE IF NOT EXISTS quiz_question_options (
    option_id INT AUTO_INCREMENT PRIMARY KEY,
    question_id INT NOT NULL,
    option_text TEXT NOT NULL,
    is_correct TINYINT(1) NOT NULL DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (question_id) REFERENCES quiz_questions(question_id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";

// Create quiz submissions table
$createSubmissionsTable = "CREATE TABLE IF NOT EXISTS quiz_submissions (
    submission_id INT AUTO_INCREMENT PRIMARY KEY,
    quiz_id INT NOT NULL,
    student_id INT NOT NULL,
    score DECIMAL(5,2) NOT NULL DEFAULT 0,
    total_points INT NOT NULL DEFAULT 0,
    start_time TIMESTAMP NULL DEFAULT NULL,
    end_time TIMESTAMP NULL DEFAULT NULL,
    status ENUM('in_progress', 'completed', 'expired') NOT NULL DEFAULT 'in_progress',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (quiz_id) REFERENCES quizzes(quiz_id) ON DELETE CASCADE,
    FOREIGN KEY (student_id) REFERENCES users(user_id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";

// Create quiz answers table
$createAnswersTable = "CREATE TABLE IF NOT EXISTS quiz_answers (
    answer_id INT AUTO_INCREMENT PRIMARY KEY,
    submission_id INT NOT NULL,
    question_id INT NOT NULL,
    selected_option_id INT NULL,
    text_answer TEXT,
    is_correct TINYINT(1) DEFAULT 0,
    points_earned DECIMAL(5,2) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (submission_id) REFERENCES quiz_submissions(submission_id) ON DELETE CASCADE,
    FOREIGN KEY (question_id) REFERENCES quiz_questions(question_id) ON DELETE CASCADE,
    FOREIGN KEY (selected_option_id) REFERENCES quiz_question_options(option_id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";

// Execute create table queries with error handling
$tables = [
    "quizzes" => $createQuizTable,
    "quiz_questions" => $createQuestionsTable,
    "quiz_question_options" => $createOptionsTable,
    "quiz_submissions" => $createSubmissionsTable,
    "quiz_answers" => $createAnswersTable
];

echo "<h2>Quiz System Database Setup</h2>\n";

$all_success = true;
foreach ($tables as $table_name => $query) {
    echo "<h3>Creating $table_name table:</h3>";
    try {
        if ($conn->query($query)) {
            echo "<div style='color: green;'>Table $table_name created successfully or already exists</div>";
        } else {
            echo "<div style='color: red;'>Error creating $table_name table: " . $conn->error . "</div>";
            $all_success = false;
        }
    } catch (Exception $e) {
        echo "<div style='color: red;'>Error creating $table_name table: " . $e->getMessage() . "</div>";
        $all_success = false;
    }
}

if ($all_success) {
    echo "<div style='margin-top: 20px; color: green; font-weight: bold;'>All quiz system tables created successfully!</div>";
    echo "<div style='margin-top: 10px;'><a href='quizzes.php'>Go to Quizzes Page</a></div>";
} else {
    echo "<div style='margin-top: 20px; color: red; font-weight: bold;'>Some errors occurred during table creation. Please review the errors above.</div>";
}

$conn->close();
?>
