<?php
/**
 * 🔍 DEEP UPLOAD ANALYSIS - COMPREHENSIVE DEBUGGING
 */

require_once __DIR__ . '/../includes/config.php';

echo "<html><head><title>🔍 Deep Upload Analysis</title>";
echo "<script src='https://code.jquery.com/jquery-3.6.0.min.js'></script>";
echo "</head><body style='font-family: monospace; padding: 20px; background: #f8f9fa;'>";
echo "<h1>🔍 DEEP UPLOAD ANALYSIS & DEBUGGING</h1>";
echo "<hr>";

echo "<div id='testResults' style='background: white; padding: 20px; border-radius: 8px; margin: 20px 0;'></div>";

echo "<h2>🎯 LIVE UPLOAD FORM TESTING</h2>";
echo "<div style='background: #fff3cd; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
echo "<p><strong>Instructions:</strong></p>";
echo "<ol>";
echo "<li>Keep this page open in one tab</li>";
echo "<li>Open upload-video.php in another tab</li>";
echo "<li>Try to upload a video</li>";
echo "<li>Come back here to see what actually happened</li>";
echo "</ol>";
echo "</div>";

echo "<script>";
echo "
let testResults = document.getElementById('testResults');
let testCount = 0;

function log(message, type = 'info') {
    testCount++;
    const colors = {
        info: '#17a2b8',
        success: '#28a745', 
        error: '#dc3545',
        warning: '#ffc107'
    };
    
    const icons = {
        info: 'ℹ️',
        success: '✅',
        error: '❌', 
        warning: '⚠️'
    };
    
    const timestamp = new Date().toLocaleTimeString();
    testResults.innerHTML += `<div style='margin: 5px 0; color: ${colors[type]};'>[${timestamp}] ${icons[type]} ${message}</div>`;
    console.log(`[${timestamp}] ${message}`);
}

$(document).ready(function() {
    log('🚀 Starting comprehensive upload analysis...', 'info');
    
    // 1. TEST: Check upload page accessibility
    log('📡 Testing upload page accessibility...', 'info');
    
    fetch('upload-video.php')
        .then(response => {
            if (response.ok) {
                log('✅ Upload page accessible', 'success');
                return response.text();
            } else {
                log('❌ Upload page not accessible: ' + response.status, 'error');
                throw new Error('Upload page not accessible');
            }
        })
        .then(html => {
            // Parse the HTML content
            const parser = new DOMParser();
            const doc = parser.parseFromString(html, 'text/html');
            
            // Check for key elements
            const form = doc.getElementById('videoUploadForm');
            const submitBtn = doc.querySelector('button[type=\"submit\"]');
            const videoUrl = doc.getElementById('videoUrl');
            
            log('📝 Form element: ' + (form ? '✅ Found' : '❌ Missing'), form ? 'success' : 'error');
            log('🔘 Submit button: ' + (submitBtn ? '✅ Found' : '❌ Missing'), submitBtn ? 'success' : 'error');
            log('🔗 Video URL input: ' + (videoUrl ? '✅ Found' : '❌ Missing'), videoUrl ? 'success' : 'error');
            
            if (form) {
                log('📋 Form action: ' + form.getAttribute('action'), 'info');
                log('📋 Form method: ' + form.getAttribute('method'), 'info');
                log('📋 Form enctype: ' + form.getAttribute('enctype'), 'info');
            }
            
            // Check for JavaScript
            const scripts = doc.querySelectorAll('script');
            log('📜 Script tags found: ' + scripts.length, 'info');
            
            let hasJQuery = false;
            let hasAjaxHandler = false;
            
            scripts.forEach(script => {
                const content = script.textContent || script.innerText || '';
                if (content.includes('jquery') || content.includes('$')) {
                    hasJQuery = true;
                }
                if (content.includes('videoUploadForm') && content.includes('submit')) {
                    hasAjaxHandler = true;
                }
            });
            
            log('📚 jQuery detected: ' + (hasJQuery ? '✅ Yes' : '❌ No'), hasJQuery ? 'success' : 'error');
            log('🔄 AJAX handler detected: ' + (hasAjaxHandler ? '✅ Yes' : '❌ No'), hasAjaxHandler ? 'success' : 'error');
        })
        .catch(error => {
            log('❌ Error analyzing upload page: ' + error.message, 'error');
        });
    
    // 2. TEST: Backend processing endpoint
    setTimeout(() => {
        log('🔍 Testing backend processing endpoint...', 'info');
        
        const testData = new FormData();
        testData.append('form_type', 'video_upload');
        testData.append('title', 'Debug Test Video');
        testData.append('course_id', '1');
        testData.append('video_type', 'url');
        testData.append('video_url', 'https://youtu.be/4sosXZ4y-5');
        testData.append('detected_platform', 'youtube');
        testData.append('video_id_external', '4sosXZ4y-5');
        testData.append('debug_mode', '1');
        
        fetch('process-video-with-content.php', {
            method: 'POST',
            body: testData
        })
        .then(response => {
            log('📡 Backend response status: ' + response.status, response.ok ? 'success' : 'error');
            log('📡 Backend response type: ' + response.headers.get('content-type'), 'info');
            
            return response.text();
        })
        .then(responseText => {
            log('📄 Backend response length: ' + responseText.length + ' characters', 'info');
            
            // Try to parse as JSON
            try {
                const jsonResponse = JSON.parse(responseText);
                log('✅ Backend returns valid JSON', 'success');
                log('📋 Response success: ' + (jsonResponse.success ? '✅' : '❌'), jsonResponse.success ? 'success' : 'error');
                log('💬 Response message: ' + (jsonResponse.message || 'No message'), 'info');
                
                if (jsonResponse.redirect) {
                    log('🔄 Response redirect: ' + jsonResponse.redirect, 'info');
                }
                
            } catch (e) {
                log('❌ Backend does NOT return valid JSON', 'error');
                log('📄 Raw response preview: ' + responseText.substring(0, 200) + '...', 'warning');
                
                // Check if it's HTML (redirect response)
                if (responseText.includes('<html>') || responseText.includes('<DOCTYPE')) {
                    log('🔄 Backend is returning HTML (likely redirect)', 'warning');
                } else if (responseText.includes('error') || responseText.includes('Error')) {
                    log('❌ Backend response contains errors', 'error');
                }
            }
        })
        .catch(error => {
            log('❌ Backend request failed: ' + error.message, 'error');
        });
    }, 2000);
    
    // 3. TEST: Session and authentication
    setTimeout(() => {
        log('🔐 Testing session and authentication...', 'info');
        
        fetch('check-session.php')
            .then(response => response.text())
            .then(html => {
                if (html.includes('AUTHENTICATION IS VALID')) {
                    log('✅ Authentication is valid', 'success');
                } else if (html.includes('AUTHENTICATION FAILED')) {
                    log('❌ Authentication failed', 'error');
                } else {
                    log('⚠️ Authentication status unclear', 'warning');
                }
                
                // Check for specific session data
                if (html.includes('User ID:') && html.includes('User Role:')) {
                    log('📋 Session data appears to be present', 'success');
                } else {
                    log('❌ Session data missing', 'error');
                }
            })
            .catch(error => {
                log('❌ Session check failed: ' + error.message, 'error');
            });
    }, 3000);
    
    // 4. TEST: Database connectivity
    setTimeout(() => {
        log('🗄️ Testing database connectivity...', 'info');
        
        fetch('debug-video-upload.php')
            .then(response => response.text())
            .then(html => {
                if (html.includes('✅ CONNECTED')) {
                    log('✅ Database connection working', 'success');
                } else {
                    log('❌ Database connection issues detected', 'error');
                }
                
                if (html.includes('SUCCESS! Video inserted')) {
                    log('✅ Database video insertion works', 'success');
                } else {
                    log('❌ Database video insertion failing', 'error');
                }
            })
            .catch(error => {
                log('❌ Database test failed: ' + error.message, 'error');
            });
    }, 4000);
    
    // 5. FINAL ANALYSIS
    setTimeout(() => {
        log('🎯 Analysis complete. Results summary:', 'info');
        log('📊 Total tests run: ' + testCount, 'info');
        
        const errorCount = testResults.innerHTML.split('❌').length - 1;
        const successCount = testResults.innerHTML.split('✅').length - 1;
        
        log('✅ Successful checks: ' + successCount, 'success');
        log('❌ Failed checks: ' + errorCount, errorCount > 0 ? 'error' : 'success');
        
        if (errorCount > 0) {
            log('🔧 Issues detected that need fixing before upload will work', 'warning');
        } else {
            log('🎉 All checks passed - upload should be working', 'success');
        }
    }, 5000);
});
";
echo "</script>";

echo "<h2>🎯 MANUAL DEBUGGING STEPS</h2>";
echo "<div style='background: white; padding: 20px; border-radius: 8px;'>";
echo "<ol>";
echo "<li><strong>Go to upload page:</strong> <a href='upload-video.php' target='_blank'>upload-video.php</a></li>";
echo "<li><strong>Open browser console (F12)</strong></li>";
echo "<li><strong>Try video URL:</strong> https://youtu.be/4sosXZ4y-5</li>";
echo "<li><strong>Click 'Detect Video'</strong> - check console</li>";
echo "<li><strong>Fill required fields</strong></li>";
echo "<li><strong>Click 'Upload Video'</strong> - watch console carefully</li>";
echo "<li><strong>Report any console errors or network failures</strong></li>";
echo "</ol>";
echo "</div>";

echo "<h2>🔍 NEXT DEBUGGING LINKS</h2>";
echo "<div style='margin: 20px 0;'>";
echo "<a href='upload-video.php' target='_blank' style='display: inline-block; padding: 10px 20px; background: #007bff; color: white; text-decoration: none; border-radius: 3px; margin: 5px;'>🎬 Upload Page</a>";
echo "<a href='check-session.php' target='_blank' style='display: inline-block; padding: 10px 20px; background: #17a2b8; color: white; text-decoration: none; border-radius: 3px; margin: 5px;'>🔐 Check Session</a>";
echo "<a href='debug-video-upload.php' target='_blank' style='display: inline-block; padding: 10px 20px; background: #28a745; color: white; text-decoration: none; border-radius: 3px; margin: 5px;'>🔍 Debug Upload</a>";
echo "</div>";

echo "</body></html>";
?>
