<?php
/**
 * 🔍 COMPREHENSIVE VIDEO UPLOAD DEBUG SCRIPT
 * This script helps debug the entire video upload flow
 */

session_start();
ini_set('display_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/VideoEmbedder.php';

echo "<html><head><title>🔍 Video Upload Debug</title></head><body style='font-family: monospace; padding: 20px;'>";
echo "<h1>🔍 VIDEO UPLOAD DEBUG ANALYSIS</h1>";
echo "<hr>";

// 1. TEST VIDEO DETECTION API
echo "<h2>1. 🎬 TESTING VIDEO DETECTION API</h2>";
$test_url = "https://youtu.be/4sosXZ4y-5";
echo "<strong>Test URL:</strong> $test_url<br>";

try {
    $detection_result = VideoEmbedder::detectVideo($test_url);
    echo "<strong>✅ Detection Result:</strong><br>";
    echo "<pre>" . print_r($detection_result, true) . "</pre>";
} catch (Exception $e) {
    echo "<strong>❌ Detection Error:</strong> " . $e->getMessage() . "<br>";
}

echo "<hr>";

// 2. CHECK DATABASE TABLES
echo "<h2>2. 🗄️ DATABASE TABLES ANALYSIS</h2>";

// Check videos table structure
echo "<h3>Videos Table Structure:</h3>";
try {
    $result = $conn->query("DESCRIBE videos");
    if ($result) {
        echo "<table border='1' style='border-collapse: collapse; margin: 10px 0;'>";
        echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th></tr>";
        while ($row = $result->fetch_assoc()) {
            echo "<tr>";
            echo "<td>{$row['Field']}</td>";
            echo "<td>{$row['Type']}</td>";
            echo "<td>{$row['Null']}</td>";
            echo "<td>{$row['Key']}</td>";  
            echo "<td>{$row['Default']}</td>";
            echo "</tr>";
        }
        echo "</table>";
    }
} catch (Exception $e) {
    echo "<strong>❌ Table Error:</strong> " . $e->getMessage() . "<br>";
}

// Check for video_url and platform columns
echo "<h3>Video URL/Platform Column Check:</h3>";
try {
    $columns = [];
    $result = $conn->query("SHOW COLUMNS FROM videos");
    while ($row = $result->fetch_assoc()) {
        $columns[] = $row['Field'];
    }
    
    $url_columns = ['video_url', 'external_url', 'platform', 'video_id_external', 'embed_code'];
    foreach ($url_columns as $col) {
        $exists = in_array($col, $columns) ? "✅ EXISTS" : "❌ MISSING";
        echo "<strong>$col:</strong> $exists<br>";
    }
} catch (Exception $e) {
    echo "<strong>❌ Column Check Error:</strong> " . $e->getMessage() . "<br>";
}

echo "<hr>";

// 3. CHECK RECENT UPLOADS
echo "<h2>3. 📊 RECENT UPLOAD ATTEMPTS</h2>";
try {
    $teacher_id = $_SESSION['user_id'] ?? 0;
    $result = $conn->query("SELECT id, title, description, video_path, video_url, platform, video_id_external, created_at 
                           FROM videos 
                           WHERE teacher_id = $teacher_id 
                           ORDER BY created_at DESC 
                           LIMIT 5");
    
    if ($result && $result->num_rows > 0) {
        echo "<table border='1' style='border-collapse: collapse; margin: 10px 0;'>";
        echo "<tr><th>ID</th><th>Title</th><th>Type</th><th>Platform</th><th>External ID</th><th>Created</th></tr>";
        while ($row = $result->fetch_assoc()) {
            $type = !empty($row['video_path']) ? 'FILE' : (!empty($row['video_url']) ? 'URL' : 'UNKNOWN');
            echo "<tr>";
            echo "<td>{$row['id']}</td>";
            echo "<td>" . substr($row['title'], 0, 30) . "</td>";
            echo "<td><strong>$type</strong></td>";
            echo "<td>{$row['platform']}</td>";
            echo "<td>{$row['video_id_external']}</td>";
            echo "<td>{$row['created_at']}</td>";
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "<strong>ℹ️ No recent uploads found.</strong><br>";
    }
} catch (Exception $e) {
    echo "<strong>❌ Recent Uploads Error:</strong> " . $e->getMessage() . "<br>";
}

echo "<hr>";

// 4. SIMULATE FORM SUBMISSION
echo "<h2>4. 🧪 SIMULATE FORM SUBMISSION</h2>";
echo "<form method='POST' style='background: #f0f0f0; padding: 15px; border-radius: 5px;'>";
echo "<h3>Test Video Upload Submission:</h3>";
echo "<strong>Video Title:</strong> <input type='text' name='title' value='Debug Test Video' style='width: 200px;'><br><br>";
echo "<strong>Course ID:</strong> <input type='number' name='course_id' value='1' style='width: 100px;'><br><br>";
echo "<strong>Video Type:</strong> <select name='video_type'>";
echo "<option value='url'>URL</option>";
echo "<option value='file'>File</option>";
echo "</select><br><br>";
echo "<strong>Video URL:</strong> <input type='text' name='video_url' value='$test_url' style='width: 300px;'><br><br>";
echo "<strong>Platform:</strong> <input type='text' name='detected_platform' value='youtube' style='width: 100px;'><br><br>";
echo "<strong>Video ID:</strong> <input type='text' name='video_id_external' value='4sosXZ4y-5' style='width: 150px;'><br><br>";
echo "<input type='hidden' name='form_type' value='video_upload'>";
echo "<input type='hidden' name='debug_test' value='1'>";
echo "<button type='submit' style='padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 3px;'>🧪 Test Upload</button>";
echo "</form>";

// Process test submission
if ($_POST && isset($_POST['debug_test'])) {
    echo "<h3>🔍 PROCESSING TEST SUBMISSION:</h3>";
    echo "<strong>POST Data Received:</strong><br>";
    echo "<pre>";
    foreach ($_POST as $key => $value) {
        echo "$key = " . (is_array($value) ? print_r($value, true) : $value) . "\n";
    }
    echo "</pre>";
    
    // Validate required fields
    $title = trim($_POST['title'] ?? '');
    $course_id = intval($_POST['course_id'] ?? 0);
    $video_type = trim($_POST['video_type'] ?? '');
    $video_url = trim($_POST['video_url'] ?? '');
    $platform = trim($_POST['detected_platform'] ?? '');
    $video_id_external = trim($_POST['video_id_external'] ?? '');
    
    echo "<strong>Validation Results:</strong><br>";
    echo "Title: " . ($title ? "✅ '$title'" : "❌ Empty") . "<br>";
    echo "Course ID: " . ($course_id > 0 ? "✅ $course_id" : "❌ Invalid") . "<br>";
    echo "Video Type: " . ($video_type ? "✅ '$video_type'" : "❌ Empty") . "<br>";
    echo "Video URL: " . ($video_url ? "✅ '$video_url'" : "❌ Empty") . "<br>";
    echo "Platform: " . ($platform ? "✅ '$platform'" : "❌ Empty") . "<br>";
    echo "Video ID: " . ($video_id_external ? "✅ '$video_id_external'" : "❌ Empty") . "<br>";
    
    // Test database insertion
    if ($title && $course_id > 0 && $video_type === 'url' && $video_url && $platform) {
        echo "<br><strong>🧪 ATTEMPTING DATABASE INSERT:</strong><br>";
        try {
            $teacher_id = $_SESSION['user_id'] ?? 999; // Use test teacher ID
            $description = "Debug test video";
            
            $sql = "INSERT INTO videos (teacher_id, course_id, title, description, video_url, platform, video_id_external, video_type, is_published, created_at) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, 1, NOW())";
            
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("iissssss", $teacher_id, $course_id, $title, $description, $video_url, $platform, $video_id_external, $video_type);
            
            if ($stmt->execute()) {
                $video_id = $conn->insert_id;
                echo "✅ <strong>SUCCESS!</strong> Video inserted with ID: $video_id<br>";
                echo "This means the database structure is correct and the issue is elsewhere.<br>";
            } else {
                echo "❌ <strong>INSERT FAILED:</strong> " . $stmt->error . "<br>";
            }
        } catch (Exception $e) {
            echo "❌ <strong>DATABASE ERROR:</strong> " . $e->getMessage() . "<br>";
        }
    } else {
        echo "<br>❌ <strong>VALIDATION FAILED - CANNOT TEST INSERT</strong><br>";
    }
}

echo "<hr>";

// 5. CHECK SESSION AND PERMISSIONS
echo "<h2>5. 👤 SESSION & PERMISSIONS</h2>";
echo "<strong>Session Data:</strong><br>";
echo "User ID: " . ($_SESSION['user_id'] ?? 'NOT SET') . "<br>";
echo "User Role: " . ($_SESSION['user_role'] ?? 'NOT SET') . "<br>";
echo "Username: " . ($_SESSION['username'] ?? 'NOT SET') . "<br>";

// 6. FRONTEND DEBUG SCRIPT
echo "<hr>";
echo "<h2>6. 🖥️ FRONTEND DEBUG SCRIPT</h2>";
echo "<div style='background: #f8f8f8; padding: 15px; border-radius: 5px;'>";
echo "<strong>Add this to your browser console on upload-video.php:</strong><br>";
echo "<textarea style='width: 100%; height: 200px; font-family: monospace;'>";
echo "// 🔍 DEBUG FORM SUBMISSION
console.log('🔍 DEBUGGING FORM SUBMISSION');

// Check form data before submission
$('#videoUploadForm').on('submit', function(e) {
    e.preventDefault(); // Prevent actual submission for debugging
    
    console.log('📝 FORM DATA CHECK:');
    console.log('Title:', $('#videoTitle').val());
    console.log('Course:', $('[name=\"course_id\"]').val());
    console.log('Video Type:', $('[name=\"video_type\"]:checked').val());
    console.log('Video URL:', $('#videoUrl').val());
    console.log('Detected Platform:', $('#detectedPlatformInput').val());
    console.log('Video ID External:', $('#videoIdExternal').val());
    console.log('Embed Code:', $('#embedCodeInput').val());
    
    // Check hidden fields
    console.log('🔍 HIDDEN FIELDS:');
    $('input[type=\"hidden\"]').each(function() {
        console.log($(this).attr('name') + ':', $(this).val());
    });
    
    // Serialize form data
    console.log('📋 SERIALIZED FORM:', $(this).serialize());
    
    // Remove this preventDefault to allow actual submission
    // $(this).off('submit').submit();
});";
echo "</textarea>";
echo "</div>";

echo "<br><a href='upload-video.php' style='display: inline-block; padding: 10px 20px; background: #28a745; color: white; text-decoration: none; border-radius: 3px;'>← Back to Upload Page</a>";
echo "</body></html>";
?>
