<?php
/**
 * Debug Script: Enrollment & Progress Investigation
 * This script thoroughly examines the database structure for student enrollments and progress tracking
 */

// Start output buffering to catch any errors
ob_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Try different database connection paths
$db_paths = [
    __DIR__ . '/config/database.php',
    __DIR__ . '/includes/config.php',
    __DIR__ . '/includes/database.php',
    __DIR__ . '/config/config.php'
];

$conn = null;
$db_connected = false;

foreach ($db_paths as $path) {
    if (file_exists($path)) {
        try {
            require_once $path;
            if (isset($conn) && $conn) {
                $db_connected = true;
                break;
            }
        } catch (Exception $e) {
            // Continue to next path
            continue;
        }
    }
}

// If no connection found, try manual connection
if (!$db_connected) {
    // Try common database connection settings
    $servername = "localhost";
    $username = "root";
    $password = "";
    $dbname = "panadite_academy";
    
    try {
        $conn = new mysqli($servername, $username, $password, $dbname);
        if ($conn->connect_error) {
            throw new Exception("Connection failed: " . $conn->connect_error);
        }
        $db_connected = true;
    } catch (Exception $e) {
        $db_error = $e->getMessage();
    }
}

// Get course ID from URL parameter
$course_id = isset($_GET['course_id']) ? (int)$_GET['course_id'] : 13;

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Debug: Enrollment & Progress - Course <?= $course_id ?></title>
    <style>
        body { 
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; 
            margin: 20px; 
            background: #f5f5f5; 
        }
        .container { 
            max-width: 1200px; 
            margin: 0 auto; 
            background: white; 
            padding: 20px; 
            border-radius: 8px; 
            box-shadow: 0 2px 10px rgba(0,0,0,0.1); 
        }
        h1, h2 { 
            color: #333; 
            border-bottom: 2px solid #007bff; 
            padding-bottom: 10px; 
        }
        h3 { 
            color: #666; 
            margin-top: 25px; 
        }
        .debug-section { 
            margin: 20px 0; 
            padding: 15px; 
            border: 1px solid #ddd; 
            border-radius: 5px; 
            background: #f9f9f9; 
        }
        .success { 
            background: #d4edda; 
            border-color: #c3e6cb; 
            color: #155724; 
        }
        .error { 
            background: #f8d7da; 
            border-color: #f5c6cb; 
            color: #721c24; 
        }
        .info { 
            background: #d1ecf1; 
            border-color: #bee5eb; 
            color: #0c5460; 
        }
        .warning { 
            background: #fff3cd; 
            border-color: #ffeaa7; 
            color: #856404; 
        }
        table { 
            width: 100%; 
            border-collapse: collapse; 
            margin: 10px 0; 
        }
        th, td { 
            padding: 8px 12px; 
            text-align: left; 
            border: 1px solid #ddd; 
        }
        th { 
            background: #f8f9fa; 
            font-weight: bold; 
        }
        .field-name { 
            font-family: 'Courier New', monospace; 
            background: #e9ecef; 
            padding: 2px 6px; 
            border-radius: 3px; 
        }
        pre { 
            background: #f8f9fa; 
            border: 1px solid #e9ecef; 
            border-radius: 4px; 
            padding: 10px; 
            overflow-x: auto; 
            font-size: 12px; 
            max-height: 300px;
            overflow-y: auto;
        }
        .nav-links {
            margin-bottom: 20px;
            padding: 10px;
            background: #e9ecef;
            border-radius: 5px;
        }
        .nav-links a {
            margin-right: 15px;
            color: #007bff;
            text-decoration: none;
            font-weight: bold;
        }
        .nav-links a:hover {
            text-decoration: underline;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔍 Debug: Enrollment & Progress Investigation</h1>
        
        <div class="nav-links">
            <strong>Quick Links:</strong>
            <a href="teacher/view-course.php?id=<?= $course_id ?>">← Back to Course View</a>
            <a href="?course_id=<?= $course_id ?>&refresh=1">🔄 Refresh Debug</a>
            <a href="?course_id=<?= $course_id ?>&show_all_tables=1">📋 Show All Tables</a>
        </div>
        
        <div class="info debug-section">
            <strong>🎯 Debugging Course ID:</strong> <?= $course_id ?><br>
            <strong>⏰ Timestamp:</strong> <?= date('Y-m-d H:i:s') ?><br>
            <strong>🌐 Script Location:</strong> <?= __FILE__ ?><br>
            <strong>📁 Root Directory:</strong> <?= __DIR__ ?>
        </div>

        <?php
        
        echo "<h2>📊 1. Database Connection Test</h2>";
        
        if (!$db_connected || !$conn) {
            echo '<div class="error debug-section">';
            echo '<strong>❌ DATABASE CONNECTION FAILED</strong><br>';
            if (isset($db_error)) {
                echo "Error: " . htmlspecialchars($db_error) . "<br>";
            }
            echo '<br>Tried the following paths:<ul>';
            foreach ($db_paths as $path) {
                $exists = file_exists($path) ? '✅' : '❌';
                echo "<li>$exists " . htmlspecialchars($path) . "</li>";
            }
            echo '</ul>';
            echo '<strong>Manual Connection Attempt:</strong><br>';
            echo "Server: localhost, User: root, DB: panadite_academy<br>";
            echo '</div>';
            
            // Still show some basic info even without DB
            echo '<div class="warning debug-section">';
            echo '<strong>⚠️ Continuing with limited information...</strong>';
            echo '</div>';
        } else {
            echo '<div class="success debug-section"><strong>✅ Database Connected Successfully</strong></div>';
        }

        if ($db_connected && $conn) {
            echo "<h2>🏫 2. Course Information</h2>";
            
            $course_stmt = $conn->prepare("SELECT * FROM courses WHERE course_id = ?");
            if ($course_stmt) {
                $course_stmt->bind_param("i", $course_id);
                if ($course_stmt->execute()) {
                    $course_result = $course_stmt->get_result();
                    $course = $course_result->fetch_assoc();
                    
                    if ($course) {
                        echo '<div class="success debug-section">';
                        echo "<strong>✅ Course Found:</strong> " . htmlspecialchars($course['title'] ?? 'N/A') . "<br>";
                        echo "<strong>📚 Course ID:</strong> " . htmlspecialchars($course['course_id'] ?? 'N/A') . "<br>";
                        echo "<strong>👨‍🏫 Instructor:</strong> " . htmlspecialchars($course['instructor_name'] ?? 'N/A') . "<br>";
                        echo "<strong>📊 Status:</strong> " . htmlspecialchars($course['status'] ?? 'N/A') . "<br>";
                        echo "<strong>💰 Price:</strong> $" . htmlspecialchars($course['price'] ?? '0') . "<br>";
                        echo '</div>';
                        
                        echo "<h4>📄 Full Course Data:</h4>";
                        echo "<pre>" . htmlspecialchars(print_r($course, true)) . "</pre>";
                    } else {
                        echo '<div class="error debug-section"><strong>❌ Course not found with ID: ' . $course_id . '</strong></div>';
                    }
                }
            } else {
                echo '<div class="error debug-section"><strong>❌ Failed to prepare course query</strong></div>';
            }

            // Show all tables if requested
            if (isset($_GET['show_all_tables'])) {
                echo "<h2>📋 All Database Tables</h2>";
                $tables_result = $conn->query("SHOW TABLES");
                if ($tables_result) {
                    echo '<div class="info debug-section">';
                    echo '<strong>📊 Available Tables:</strong><br><br>';
                    echo '<table><tr><th>Table Name</th><th>Row Count</th></tr>';
                    while ($table = $tables_result->fetch_row()) {
                        $table_name = $table[0];
                        $count_result = $conn->query("SELECT COUNT(*) as count FROM `$table_name`");
                        $count = $count_result ? $count_result->fetch_assoc()['count'] : 'N/A';
                        echo "<tr><td><span class='field-name'>$table_name</span></td><td>$count</td></tr>";
                    }
                    echo '</table>';
                    echo '</div>';
                }
            }

            echo "<h2>📋 3. Enrollment Tables Investigation</h2>";
            
            // Check for enrollment tables
            $enrollment_tables = ['course_enrollments', 'enrollments', 'student_enrollments', 'course_students', 'user_courses'];
            
            foreach ($enrollment_tables as $table) {
                echo "<h3>🔍 Table: <span class='field-name'>$table</span></h3>";
                
                // Check if table exists
                $check_table = $conn->query("SHOW TABLES LIKE '$table'");
                if ($check_table && $check_table->num_rows > 0) {
                    echo '<div class="success debug-section"><strong>✅ Table exists</strong></div>';
                    
                    // Get table structure
                    $describe = $conn->query("DESCRIBE $table");
                    if ($describe) {
                        echo "<h4>📝 Table Structure:</h4>";
                        echo "<table><tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th></tr>";
                        while ($field = $describe->fetch_assoc()) {
                            echo "<tr>";
                            echo "<td><span class='field-name'>" . htmlspecialchars($field['Field']) . "</span></td>";
                            echo "<td>" . htmlspecialchars($field['Type']) . "</td>";
                            echo "<td>" . htmlspecialchars($field['Null']) . "</td>";
                            echo "<td>" . htmlspecialchars($field['Key']) . "</td>";
                            echo "<td>" . htmlspecialchars($field['Default'] ?? 'NULL') . "</td>";
                            echo "</tr>";
                        }
                        echo "</table>";
                    }
                    
                    // Get row count
                    $count_result = $conn->query("SELECT COUNT(*) as total FROM $table");
                    if ($count_result) {
                        $total = $count_result->fetch_assoc()['total'];
                        echo "<div class='info debug-section'><strong>📊 Total rows:</strong> $total</div>";
                    }
                    
                    // Get course-specific data
                    $course_count_result = $conn->query("SELECT COUNT(*) as course_total FROM $table WHERE course_id = $course_id");
                    if ($course_count_result) {
                        $course_total = $course_count_result->fetch_assoc()['course_total'];
                        echo "<div class='info debug-section'><strong>🎯 Rows for Course $course_id:</strong> $course_total</div>";
                        
                        if ($course_total > 0) {
                            // Get sample data
                            $sample_result = $conn->query("SELECT * FROM $table WHERE course_id = $course_id LIMIT 3");
                            if ($sample_result) {
                                $sample_data = $sample_result->fetch_all(MYSQLI_ASSOC);
                                
                                echo "<h4>📄 Sample Data:</h4>";
                                echo "<pre>" . htmlspecialchars(print_r($sample_data, true)) . "</pre>";
                            }
                        }
                    }
                    
                } else {
                    echo '<div class="error debug-section"><strong>❌ Table does not exist</strong></div>';
                }
            }

            echo "<h2>🎓 4. Student Data Analysis</h2>";
            
            // Find working enrollment table and get student details
            $working_enrollment_data = [];
            $working_table = null;
            
            foreach ($enrollment_tables as $table) {
                $check_table = $conn->query("SHOW TABLES LIKE '$table'");
                if ($check_table && $check_table->num_rows > 0) {
                    $enrollment_result = $conn->query("SELECT * FROM $table WHERE course_id = $course_id");
                    if ($enrollment_result && $enrollment_result->num_rows > 0) {
                        $working_enrollment_data = $enrollment_result->fetch_all(MYSQLI_ASSOC);
                        $working_table = $table;
                        break;
                    }
                }
            }
            
            if (!empty($working_enrollment_data)) {
                echo '<div class="success debug-section">';
                echo "<strong>✅ Found " . count($working_enrollment_data) . " enrollments in:</strong> <span class='field-name'>$working_table</span>";
                echo '</div>';
                
                echo "<h4>👥 Enrolled Students Analysis:</h4>";
                
                foreach ($working_enrollment_data as $index => $enrollment) {
                    echo "<h5>🎓 Student #" . ($index + 1) . ":</h5>";
                    
                    // Show enrollment data
                    echo "<strong>📋 Enrollment Data:</strong>";
                    echo "<pre>" . htmlspecialchars(print_r($enrollment, true)) . "</pre>";
                    
                    // Try to get user details
                    $student_id = $enrollment['student_id'] ?? $enrollment['user_id'] ?? null;
                    if ($student_id) {
                        $user_result = $conn->query("SELECT * FROM users WHERE user_id = $student_id");
                        if ($user_result && $user_result->num_rows > 0) {
                            $user_data = $user_result->fetch_assoc();
                            echo "<strong>👤 User Details:</strong>";
                            echo "<pre>" . htmlspecialchars(print_r($user_data, true)) . "</pre>";
                        } else {
                            echo "<div class='warning debug-section'>⚠️ No user data found for student_id: $student_id</div>";
                        }
                    } else {
                        echo "<div class='error debug-section'>❌ No student ID found in enrollment data</div>";
                    }
                }
                
            } else {
                echo '<div class="error debug-section"><strong>❌ No enrollment data found for course ' . $course_id . '</strong></div>';
            }

            echo "<h2>📈 5. Progress Tracking Tables</h2>";
            
            // Check for progress-related tables
            $progress_tables = ['video_progress', 'course_progress', 'student_progress', 'lesson_progress', 'user_progress', 'quiz_attempts'];
            
            foreach ($progress_tables as $table) {
                echo "<h3>📊 Progress Table: <span class='field-name'>$table</span></h3>";
                
                $check_table = $conn->query("SHOW TABLES LIKE '$table'");
                if ($check_table && $check_table->num_rows > 0) {
                    echo '<div class="success debug-section"><strong>✅ Table exists</strong></div>';
                    
                    // Get table structure
                    $describe = $conn->query("DESCRIBE $table");
                    if ($describe) {
                        echo "<h4>📝 Structure:</h4>";
                        echo "<table><tr><th>Field</th><th>Type</th></tr>";
                        while ($field = $describe->fetch_assoc()) {
                            echo "<tr>";
                            echo "<td><span class='field-name'>" . htmlspecialchars($field['Field']) . "</span></td>";
                            echo "<td>" . htmlspecialchars($field['Type']) . "</td>";
                            echo "</tr>";
                        }
                        echo "</table>";
                    }
                    
                    // Check for course-specific data
                    $progress_result = $conn->query("SELECT COUNT(*) as total FROM $table WHERE course_id = $course_id");
                    if ($progress_result) {
                        $progress_total = $progress_result->fetch_assoc()['total'];
                        echo "<div class='info debug-section'><strong>📊 Records for Course $course_id:</strong> $progress_total</div>";
                        
                        if ($progress_total > 0) {
                            $sample_progress = $conn->query("SELECT * FROM $table WHERE course_id = $course_id LIMIT 3");
                            if ($sample_progress) {
                                $sample_data = $sample_progress->fetch_all(MYSQLI_ASSOC);
                                echo "<h4>📄 Sample Data:</h4>";
                                echo "<pre>" . htmlspecialchars(print_r($sample_data, true)) . "</pre>";
                            }
                        }
                    }
                    
                } else {
                    echo '<div class="error debug-section"><strong>❌ Table does not exist</strong></div>';
                }
            }

            echo "<h2>🎥 6. Course Content</h2>";
            
            // Check videos
            $videos_result = $conn->query("SELECT COUNT(*) as video_count FROM videos WHERE course_id = $course_id");
            if ($videos_result) {
                $video_count = $videos_result->fetch_assoc()['video_count'];
                echo "<div class='info debug-section'><strong>🎥 Total Videos:</strong> $video_count</div>";
                
                if ($video_count > 0) {
                    $sample_videos = $conn->query("SELECT * FROM videos WHERE course_id = $course_id LIMIT 2");
                    if ($sample_videos) {
                        $video_data = $sample_videos->fetch_all(MYSQLI_ASSOC);
                        echo "<h4>📄 Sample Videos:</h4>";
                        echo "<pre>" . htmlspecialchars(print_r($video_data, true)) . "</pre>";
                    }
                }
            }

            echo "<h2>🔧 7. Fix Recommendations</h2>";
            
            echo '<div class="success debug-section">';
            echo '<h4>🛠️ Based on this analysis:</h4>';
            echo '<ol>';
            
            if (!empty($working_enrollment_data)) {
                echo '<li><strong>✅ Enrollment Table:</strong> Use <span class="field-name">' . $working_table . '</span></li>';
                
                // Analyze date fields
                $date_fields = [];
                foreach (array_keys($working_enrollment_data[0]) as $field) {
                    if (stripos($field, 'date') !== false || stripos($field, 'created') !== false || stripos($field, 'time') !== false) {
                        $date_fields[] = $field;
                    }
                }
                
                if (!empty($date_fields)) {
                    echo '<li><strong>📅 Date Fields Available:</strong> ';
                    foreach ($date_fields as $field) {
                        echo "<span class='field-name'>$field</span> ";
                    }
                    echo '</li>';
                } else {
                    echo '<li><strong>⚠️ No date fields found</strong> - May need to add enrollment_date column</li>';
                }
                
                // Analyze ID fields
                $id_fields = [];
                foreach (array_keys($working_enrollment_data[0]) as $field) {
                    if (stripos($field, 'student') !== false || stripos($field, 'user') !== false) {
                        $id_fields[] = $field;
                    }
                }
                
                if (!empty($id_fields)) {
                    echo '<li><strong>🆔 Student ID Fields:</strong> ';
                    foreach ($id_fields as $field) {
                        echo "<span class='field-name'>$field</span> ";
                    }
                    echo '</li>';
                }
            }
            
            echo '</ol>';
            echo '</div>';
        }

        ?>
        
        <div class="nav-links" style="margin-top: 30px;">
            <strong>🔗 Quick Actions:</strong>
            <a href="teacher/view-course.php?id=<?= $course_id ?>">← Back to Course View</a>
            <a href="?course_id=<?= $course_id ?>&refresh=<?= time() ?>">🔄 Refresh</a>
            <a href="?course_id=<?= $course_id ?>&show_all_tables=1">📋 Show All Tables</a>
        </div>
        
        <div class="info debug-section" style="margin-top: 20px;">
            <strong>💡 Debug Complete!</strong> Use the information above to fix the enrollment and progress display issues.
        </div>
        
    </div>
</body>
</html>
