<?php
/**
 * Debug Script for Assignment Submissions File Storage
 * This script analyzes the database to understand file storage issues
 */

require_once 'includes/session_start.php';
require_once 'includes/db_connection.php';

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>File Submissions Debug - Panadite Academy</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        .debug-box { background: #f8f9fa; border: 1px solid #dee2e6; padding: 15px; margin: 10px 0; }
        .json-raw { background: #fff3cd; padding: 10px; font-family: monospace; font-size: 12px; }
        .file-count { font-weight: bold; color: #198754; }
        .truncated { color: #dc3545; font-weight: bold; }
        .success { color: #198754; }
        .error { color: #dc3545; }
    </style>
</head>
<body>
<div class="container mt-4">
    <h1>🔍 File Submissions Database Debug</h1>
    <p class="text-muted">Analyzing assignment submissions and file storage patterns...</p>

    <?php
    try {
        // 1. Get all assignment submissions with files
        echo "<div class='debug-box'>";
        echo "<h3>📊 Assignment Submissions Overview</h3>";
        
        $overview_query = "SELECT 
            COUNT(*) as total_submissions,
            COUNT(CASE WHEN file_url IS NOT NULL AND file_url != '' THEN 1 END) as submissions_with_files,
            COUNT(CASE WHEN CHAR_LENGTH(file_url) = 255 THEN 1 END) as truncated_submissions
            FROM assignment_submissions";
        
        $result = $conn->query($overview_query);
        $stats = $result->fetch_assoc();
        
        echo "<ul class='list-unstyled'>";
        echo "<li><strong>Total Submissions:</strong> " . $stats['total_submissions'] . "</li>";
        echo "<li><strong>Submissions with Files:</strong> <span class='file-count'>" . $stats['submissions_with_files'] . "</span></li>";
        echo "<li><strong>Truncated (255 chars):</strong> <span class='truncated'>" . $stats['truncated_submissions'] . "</span></li>";
        echo "</ul>";
        echo "</div>";

        // 2. Analyze each submission with files
        echo "<div class='debug-box'>";
        echo "<h3>📁 Detailed File Analysis</h3>";
        
        $submissions_query = "SELECT 
            s.submission_id,
            s.assignment_id,
            s.student_id,
            s.file_url,
            CHAR_LENGTH(s.file_url) as url_length,
            a.title as assignment_title,
            u.first_name,
            u.last_name
            FROM assignment_submissions s
            JOIN assignments a ON s.assignment_id = a.assignment_id
            JOIN users u ON s.student_id = u.user_id
            WHERE s.file_url IS NOT NULL AND s.file_url != ''
            ORDER BY s.submission_id DESC";
        
        $result = $conn->query($submissions_query);
        
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                echo "<div class='card mb-3'>";
                echo "<div class='card-header'>";
                echo "<strong>Submission ID:</strong> " . $row['submission_id'] . " | ";
                echo "<strong>Assignment:</strong> " . htmlspecialchars($row['assignment_title']) . " | ";
                echo "<strong>Student:</strong> " . htmlspecialchars($row['first_name'] . ' ' . $row['last_name']);
                echo "</div>";
                echo "<div class='card-body'>";
                
                // Character length analysis
                $is_truncated = ($row['url_length'] == 255);
                echo "<p><strong>File URL Length:</strong> <span class='" . ($is_truncated ? 'truncated' : 'success') . "'>" . $row['url_length'] . " characters</span>";
                if ($is_truncated) {
                    echo " <span class='badge bg-danger'>TRUNCATED!</span>";
                }
                echo "</p>";
                
                // Raw JSON display
                echo "<p><strong>Raw File URL Data:</strong></p>";
                echo "<div class='json-raw'>" . htmlspecialchars($row['file_url']) . "</div>";
                
                // JSON parsing analysis
                $files = json_decode($row['file_url'], true);
                $json_error = json_last_error();
                
                echo "<p><strong>JSON Parse Status:</strong> ";
                if ($json_error === JSON_ERROR_NONE && is_array($files)) {
                    echo "<span class='success'>✅ SUCCESS - " . count($files) . " files detected</span>";
                    
                    echo "<ul class='mt-2'>";
                    foreach ($files as $index => $file) {
                        echo "<li><strong>File " . ($index + 1) . ":</strong> ";
                        if (isset($file['original_name'])) {
                            echo htmlspecialchars($file['original_name']);
                        } else {
                            echo "Unknown filename";
                        }
                        if (isset($file['file_path'])) {
                            echo " → " . htmlspecialchars($file['file_path']);
                        }
                        echo "</li>";
                    }
                    echo "</ul>";
                } else {
                    echo "<span class='error'>❌ FAILED - " . json_last_error_msg() . "</span>";
                    
                    // Try to fix truncated JSON
                    if ($is_truncated && substr($row['file_url'], 0, 1) === '[') {
                        echo "<br><strong>🔧 Attempting to fix truncated JSON...</strong>";
                        
                        $last_brace = strrpos($row['file_url'], '}');
                        if ($last_brace !== false) {
                            $fixed_json = substr($row['file_url'], 0, $last_brace + 1) . ']';
                            $fixed_files = json_decode($fixed_json, true);
                            
                            if (is_array($fixed_files)) {
                                echo "<br><span class='success'>✅ RECOVERY SUCCESS - " . count($fixed_files) . " files recovered</span>";
                                
                                echo "<ul class='mt-2'>";
                                foreach ($fixed_files as $index => $file) {
                                    echo "<li><strong>Recovered File " . ($index + 1) . ":</strong> ";
                                    if (isset($file['original_name'])) {
                                        echo htmlspecialchars($file['original_name']);
                                    }
                                    echo "</li>";
                                }
                                echo "</ul>";
                            } else {
                                echo "<br><span class='error'>❌ RECOVERY FAILED</span>";
                            }
                        }
                    }
                }
                
                echo "</div>";
                echo "</div>";
            }
        } else {
            echo "<p class='text-muted'>No submissions with files found.</p>";
        }
        echo "</div>";

        // 3. Database schema analysis
        echo "<div class='debug-box'>";
        echo "<h3>🗄️ Database Schema Analysis</h3>";
        
        $schema_query = "DESCRIBE assignment_submissions";
        $result = $conn->query($schema_query);
        
        echo "<table class='table table-sm'>";
        echo "<thead><tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr></thead>";
        echo "<tbody>";
        
        while ($row = $result->fetch_assoc()) {
            echo "<tr>";
            echo "<td><strong>" . $row['Field'] . "</strong></td>";
            echo "<td>";
            if ($row['Field'] === 'file_url') {
                echo "<span class='error'>" . $row['Type'] . "</span> ⚠️ TOO SMALL!";
            } else {
                echo $row['Type'];
            }
            echo "</td>";
            echo "<td>" . $row['Null'] . "</td>";
            echo "<td>" . $row['Key'] . "</td>";
            echo "<td>" . ($row['Default'] ?? 'NULL') . "</td>";
            echo "<td>" . $row['Extra'] . "</td>";
            echo "</tr>";
        }
        echo "</tbody>";
        echo "</table>";
        echo "</div>";

        // 4. Recommendations
        echo "<div class='debug-box'>";
        echo "<h3>💡 Recommendations</h3>";
        echo "<div class='alert alert-warning'>";
        echo "<h5>Issues Found:</h5>";
        echo "<ul>";
        if ($stats['truncated_submissions'] > 0) {
            echo "<li><strong>Field Size Limitation:</strong> " . $stats['truncated_submissions'] . " submission(s) have truncated file data due to VARCHAR(255) limit.</li>";
        }
        echo "<li><strong>Recommended Fix:</strong> Change file_url field from VARCHAR(255) to TEXT to support multiple files.</li>";
        echo "</ul>";
        echo "</div>";
        
        echo "<div class='alert alert-info'>";
        echo "<h5>Migration Command:</h5>";
        echo "<code>ALTER TABLE assignment_submissions MODIFY COLUMN file_url TEXT;</code>";
        echo "</div>";
        echo "</div>";

    } catch (Exception $e) {
        echo "<div class='alert alert-danger'>";
        echo "<strong>Error:</strong> " . $e->getMessage();
        echo "</div>";
    }
    ?>

    <div class="mt-4">
        <a href="teacher/assignments.php" class="btn btn-primary">← Back to Assignments</a>
        <a href="teacher/view-submissions.php?id=41" class="btn btn-secondary">View Submissions</a>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
