<?php
// Root level debug for library file paths
// Include database connection
require_once __DIR__ . '/includes/db_connection.php';

echo "<h2>🔍 Library File Path Debug Tool (Root Level)</h2>";
echo "<style>
    body{font-family:Arial;padding:20px;background:#f5f5f5;} 
    .debug-section{background:white;padding:20px;margin:20px 0;border-radius:8px;box-shadow:0 2px 4px rgba(0,0,0,0.1);} 
    .file-item{background:#f9f9f9;padding:15px;margin:10px 0;border-radius:5px;border-left:4px solid #007cba;} 
    .success{color:#28a745;font-weight:bold;} 
    .error{color:#dc3545;font-weight:bold;} 
    .warning{color:#ffc107;font-weight:bold;}
    .info{color:#17a2b8;font-weight:bold;}
    .path{background:#e9ecef;padding:5px;border-radius:3px;font-family:monospace;font-size:12px;}
</style>";

// Session Information
echo "<div class='debug-section'>";
echo "<h3>🔐 Session Information</h3>";
session_start();
if (isset($_SESSION['user_id'])) {
    echo "<span class='success'>✅ Session Active</span><br>";
    echo "<strong>User ID:</strong> " . $_SESSION['user_id'] . "<br>";
    echo "<strong>User Role:</strong> " . ($_SESSION['user_role'] ?? 'Not set') . "<br>";
    echo "<strong>User Name:</strong> " . ($_SESSION['user_name'] ?? 'Not set') . "<br>";
    $teacher_id = $_SESSION['user_id'];
} else {
    echo "<span class='error'>❌ No Active Session</span><br>";
    echo "<em>This might be why you got 'Access denied' in the teacher directory</em><br>";
    $teacher_id = null;
}
echo "</div>";

// Database Connection Test
echo "<div class='debug-section'>";
echo "<h3>🗄️ Database Connection</h3>";
if (isset($conn) && $conn->ping()) {
    echo "<span class='success'>✅ Database Connected</span><br>";
} else {
    echo "<span class='error'>❌ Database Connection Failed</span><br>";
}
echo "</div>";

// Directory Structure Check
echo "<div class='debug-section'>";
echo "<h3>📁 Directory Structure</h3>";

$upload_dir = __DIR__ . '/uploads/library_submissions/';
echo "<strong>Upload Directory:</strong><br>";
echo "<div class='path'>" . $upload_dir . "</div>";
echo "<strong>Exists:</strong> " . (is_dir($upload_dir) ? "<span class='success'>✅ YES</span>" : "<span class='error'>❌ NO</span>") . "<br>";
echo "<strong>Writable:</strong> " . (is_writable($upload_dir) ? "<span class='success'>✅ YES</span>" : "<span class='error'>❌ NO</span>") . "<br>";

if (is_dir($upload_dir)) {
    $files = scandir($upload_dir);
    $files = array_filter($files, function($f) { return $f !== '.' && $f !== '..'; });
    echo "<strong>Files Count:</strong> " . count($files) . "<br>";
    
    if (count($files) > 0) {
        echo "<strong>Files in Directory:</strong><br>";
        foreach ($files as $file) {
            $file_path = $upload_dir . $file;
            $size = file_exists($file_path) ? number_format(filesize($file_path) / 1024, 1) . ' KB' : 'Unknown';
            $web_accessible = 'uploads/library_submissions/' . $file;
            echo "• <strong>{$file}</strong> ({$size}) - <a href='{$web_accessible}' target='_blank'>Test Access</a><br>";
        }
    }
}
echo "</div>";

// Library Submissions Table Check
if (isset($conn) && $conn->ping()) {
    echo "<div class='debug-section'>";
    echo "<h3>📚 Library Submissions Data</h3>";
    
    // Check if table exists
    $result = $conn->query("SHOW TABLES LIKE 'library_submissions'");
    if ($result->num_rows > 0) {
        echo "<span class='success'>✅ Table Exists</span><br>";
        
        // Get all submissions
        $result = $conn->query("SELECT id, teacher_id, title, file_path, thumbnail_path, submission_date FROM library_submissions ORDER BY submission_date DESC LIMIT 10");
        
        if ($result->num_rows > 0) {
            echo "<strong>Recent Submissions:</strong><br>";
            
            while ($row = $result->fetch_assoc()) {
                echo "<div class='file-item'>";
                echo "<h4>📄 " . htmlspecialchars($row['title']) . " (ID: {$row['id']})</h4>";
                echo "<strong>Teacher ID:</strong> " . $row['teacher_id'] . "<br>";
                echo "<strong>Submitted:</strong> " . $row['submission_date'] . "<br>";
                
                // Document file check
                echo "<h5>Document File:</h5>";
                echo "<strong>DB Path:</strong> <div class='path'>" . htmlspecialchars($row['file_path']) . "</div>";
                
                $full_doc_path = __DIR__ . '/' . $row['file_path'];
                echo "<strong>Full Server Path:</strong> <div class='path'>" . $full_doc_path . "</div>";
                echo "<strong>File Exists:</strong> " . (file_exists($full_doc_path) ? "<span class='success'>✅ YES</span>" : "<span class='error'>❌ NO</span>") . "<br>";
                
                $web_doc_path = $row['file_path'];
                echo "<strong>Web Access Test:</strong> <a href='{$web_doc_path}' target='_blank'>{$web_doc_path}</a><br>";
                
                // Thumbnail check
                if (!empty($row['thumbnail_path'])) {
                    echo "<h5>Thumbnail:</h5>";
                    echo "<strong>DB Path:</strong> <div class='path'>" . htmlspecialchars($row['thumbnail_path']) . "</div>";
                    
                    $full_thumb_path = __DIR__ . '/' . $row['thumbnail_path'];
                    echo "<strong>Full Server Path:</strong> <div class='path'>" . $full_thumb_path . "</div>";
                    echo "<strong>File Exists:</strong> " . (file_exists($full_thumb_path) ? "<span class='success'>✅ YES</span>" : "<span class='error'>❌ NO</span>") . "<br>";
                    
                    $web_thumb_path = $row['thumbnail_path'];
                    echo "<strong>Web Access Test:</strong> <a href='{$web_thumb_path}' target='_blank'>{$web_thumb_path}</a><br>";
                    
                    if (file_exists($full_thumb_path)) {
                        echo "<strong>Thumbnail Preview:</strong><br>";
                        echo "<img src='{$web_thumb_path}' style='max-width:100px;max-height:100px;border:1px solid #ccc;margin:5px 0;' onerror=\"this.style.display='none'; this.nextElementSibling.style.display='block';\">";
                        echo "<div style='display:none;color:red;'>❌ Thumbnail failed to load via web</div>";
                    }
                } else {
                    echo "<h5>Thumbnail:</h5>";
                    echo "<span class='warning'>⚠️ No thumbnail uploaded</span><br>";
                }
                
                echo "</div>";
            }
        } else {
            echo "<span class='warning'>⚠️ No submissions found in database</span><br>";
        }
    } else {
        echo "<span class='error'>❌ library_submissions table not found</span><br>";
    }
    echo "</div>";
}

// PHP Configuration
echo "<div class='debug-section'>";
echo "<h3>⚙️ PHP Configuration</h3>";
echo "<strong>Upload Max Size:</strong> " . ini_get('upload_max_filesize') . "<br>";
echo "<strong>Post Max Size:</strong> " . ini_get('post_max_size') . "<br>";
echo "<strong>Max File Uploads:</strong> " . ini_get('max_file_uploads') . "<br>";
echo "<strong>File Uploads Enabled:</strong> " . (ini_get('file_uploads') ? "<span class='success'>✅ YES</span>" : "<span class='error'>❌ NO</span>") . "<br>";
echo "</div>";

// Suggested Fixes
echo "<div class='debug-section'>";
echo "<h3>🔧 Suggested Fixes</h3>";
echo "<ul>";
echo "<li><strong>Session Issue:</strong> If session is not active, log in as teacher first</li>";
echo "<li><strong>File Access:</strong> Files should be accessible via <code>uploads/library_submissions/filename.ext</code></li>";
echo "<li><strong>Download Links:</strong> Should use relative paths from current directory</li>";
echo "<li><strong>Thumbnail Display:</strong> Should use same relative paths</li>";
echo "</ul>";
echo "</div>";
?>

<p><a href="teacher/library_submissions.php">← Back to Library Submissions</a></p>
