<?php
/**
 * Grade Calculation Demonstration
 * Shows how individual course grades are calculated properly
 */

require_once '../config/db_connect.php';
require_once '../includes/course_grade_calculator.php';

echo "<h2>🎯 Individual Course Grade Calculation Demo</h2>";
echo "<div style='font-family: Arial, sans-serif; max-width: 1000px; margin: 20px;'>";

// Example scenarios
echo "<h3>📚 How Grade Calculation Works</h3>";
echo "<div style='background: #e8f5e8; padding: 20px; border-radius: 8px; margin: 15px 0;'>";
echo "<h4>🔧 System Logic:</h4>";
echo "<ol>";
echo "<li><strong>Teachers set course weighting:</strong> e.g., 70% assignments + 30% quizzes = 100%</li>";
echo "<li><strong>Teachers mark which items contribute:</strong> Some assignments/quizzes have 'contributes_to_final_grade' = 1</li>";
echo "<li><strong>System calculates averages:</strong> Only from contributing items</li>";
echo "<li><strong>System applies weights:</strong> Maintains 100% total with smart fallbacks</li>";
echo "</ol>";
echo "</div>";

// Scenario examples
$scenarios = [
    [
        'title' => 'Standard Course (70/30 split)',
        'course_weight' => ['assignment' => 70, 'quiz' => 30],
        'assignments' => [
            ['title' => 'Assignment 1', 'contributes' => true, 'grade' => 85],
            ['title' => 'Assignment 2', 'contributes' => true, 'grade' => 90],
            ['title' => 'Practice Assignment', 'contributes' => false, 'grade' => 75],
        ],
        'quizzes' => [
            ['title' => 'Quiz 1', 'contributes' => true, 'score' => 88],
            ['title' => 'Quiz 2', 'contributes' => true, 'score' => 92],
            ['title' => 'Practice Quiz', 'contributes' => false, 'score' => 70],
        ]
    ],
    [
        'title' => 'Assignment-Only Course',
        'course_weight' => ['assignment' => 70, 'quiz' => 30],
        'assignments' => [
            ['title' => 'Project 1', 'contributes' => true, 'grade' => 88],
            ['title' => 'Project 2', 'contributes' => true, 'grade' => 91],
        ],
        'quizzes' => [] // No quizzes
    ],
    [
        'title' => 'Quiz-Heavy Course',
        'course_weight' => ['assignment' => 40, 'quiz' => 60],
        'assignments' => [
            ['title' => 'Final Project', 'contributes' => true, 'grade' => 85],
        ],
        'quizzes' => [
            ['title' => 'Quiz 1', 'contributes' => true, 'score' => 90],
            ['title' => 'Quiz 2', 'contributes' => true, 'score' => 87],
            ['title' => 'Quiz 3', 'contributes' => true, 'score' => 93],
            ['title' => 'Practice Quiz', 'contributes' => false, 'score' => 65],
        ]
    ]
];

foreach ($scenarios as $index => $scenario) {
    echo "<div style='background: #f8f9fa; border: 1px solid #dee2e6; border-radius: 8px; padding: 20px; margin: 15px 0;'>";
    echo "<h4 style='color: #495057;'>📖 Scenario " . ($index + 1) . ": {$scenario['title']}</h4>";
    
    // Course Configuration
    echo "<div style='background: #fff3cd; padding: 10px; border-radius: 5px; margin: 10px 0;'>";
    echo "<strong>Course Configuration:</strong> ";
    echo "Assignments: {$scenario['course_weight']['assignment']}% | ";
    echo "Quizzes: {$scenario['course_weight']['quiz']}%";
    echo "</div>";
    
    // Assignments
    echo "<div style='background: #d4edda; padding: 10px; border-radius: 5px; margin: 10px 0;'>";
    echo "<strong>📝 Assignments:</strong><br>";
    if (empty($scenario['assignments'])) {
        echo "&nbsp;&nbsp;• <em>No assignments in this course</em><br>";
    } else {
        $contributing_assignments = 0;
        $assignment_total = 0;
        $assignment_count = 0;
        
        foreach ($scenario['assignments'] as $assignment) {
            $status = $assignment['contributes'] ? 'COUNTS TO FINAL' : 'Practice Only';
            $color = $assignment['contributes'] ? '#28a745' : '#6c757d';
            
            echo "&nbsp;&nbsp;• {$assignment['title']}: {$assignment['grade']}% ";
            echo "<span style='color: {$color}; font-weight: bold;'>({$status})</span><br>";
            
            if ($assignment['contributes']) {
                $assignment_total += $assignment['grade'];
                $assignment_count++;
            }
        }
        
        if ($assignment_count > 0) {
            $assignment_avg = $assignment_total / $assignment_count;
            echo "&nbsp;&nbsp;<strong>Average of Contributing Assignments: " . round($assignment_avg, 2) . "%</strong>";
        }
    }
    echo "</div>";
    
    // Quizzes
    echo "<div style='background: #d1ecf1; padding: 10px; border-radius: 5px; margin: 10px 0;'>";
    echo "<strong>🧠 Quizzes:</strong><br>";
    if (empty($scenario['quizzes'])) {
        echo "&nbsp;&nbsp;• <em>No quizzes in this course</em><br>";
    } else {
        $contributing_quizzes = 0;
        $quiz_total = 0;
        $quiz_count = 0;
        
        foreach ($scenario['quizzes'] as $quiz) {
            $status = $quiz['contributes'] ? 'COUNTS TO FINAL' : 'Practice Only';
            $color = $quiz['contributes'] ? '#28a745' : '#6c757d';
            
            echo "&nbsp;&nbsp;• {$quiz['title']}: {$quiz['score']}% ";
            echo "<span style='color: {$color}; font-weight: bold;'>({$status})</span><br>";
            
            if ($quiz['contributes']) {
                $quiz_total += $quiz['score'];
                $quiz_count++;
            }
        }
        
        if ($quiz_count > 0) {
            $quiz_avg = $quiz_total / $quiz_count;
            echo "&nbsp;&nbsp;<strong>Average of Contributing Quizzes: " . round($quiz_avg, 2) . "%</strong>";
        }
    }
    echo "</div>";
    
    // Final Calculation
    echo "<div style='background: #e2e3e5; padding: 15px; border-radius: 5px; margin: 10px 0; border-left: 4px solid #6f42c1;'>";
    echo "<h5 style='color: #6f42c1; margin-bottom: 10px;'>🧮 Final Grade Calculation:</h5>";
    
    // Calculate based on scenario
    $has_assignments = !empty($scenario['assignments']);
    $has_quizzes = !empty($scenario['quizzes']);
    
    if ($has_assignments && $has_quizzes) {
        // Both exist - use configured weights
        $assignment_avg = 0;
        $assignment_count = 0;
        foreach ($scenario['assignments'] as $a) {
            if ($a['contributes']) {
                $assignment_avg += $a['grade'];
                $assignment_count++;
            }
        }
        $assignment_avg = $assignment_count > 0 ? $assignment_avg / $assignment_count : 0;
        
        $quiz_avg = 0;
        $quiz_count = 0;
        foreach ($scenario['quizzes'] as $q) {
            if ($q['contributes']) {
                $quiz_avg += $q['score'];
                $quiz_count++;
            }
        }
        $quiz_avg = $quiz_count > 0 ? $quiz_avg / $quiz_count : 0;
        
        $assignment_contribution = ($assignment_avg * $scenario['course_weight']['assignment']) / 100;
        $quiz_contribution = ($quiz_avg * $scenario['course_weight']['quiz']) / 100;
        $final_grade = $assignment_contribution + $quiz_contribution;
        
        echo "<strong>Method:</strong> Standard weighted calculation<br>";
        echo "<strong>Assignment Component:</strong> " . round($assignment_avg, 2) . "% × {$scenario['course_weight']['assignment']}% = " . round($assignment_contribution, 2) . "%<br>";
        echo "<strong>Quiz Component:</strong> " . round($quiz_avg, 2) . "% × {$scenario['course_weight']['quiz']}% = " . round($quiz_contribution, 2) . "%<br>";
        echo "<strong>Final Grade:</strong> " . round($assignment_contribution, 2) . "% + " . round($quiz_contribution, 2) . "% = <span style='font-size: 1.2em; color: #28a745; font-weight: bold;'>" . round($final_grade, 2) . "%</span><br>";
        echo "<strong>Weight Verification:</strong> {$scenario['course_weight']['assignment']}% + {$scenario['course_weight']['quiz']}% = 100% ✅";
        
    } elseif ($has_assignments && !$has_quizzes) {
        // Only assignments - use 100% assignment weight
        $assignment_avg = 0;
        $assignment_count = 0;
        foreach ($scenario['assignments'] as $a) {
            if ($a['contributes']) {
                $assignment_avg += $a['grade'];
                $assignment_count++;
            }
        }
        $assignment_avg = $assignment_count > 0 ? $assignment_avg / $assignment_count : 0;
        
        echo "<strong>Method:</strong> Assignment-only grading (no quizzes available)<br>";
        echo "<strong>Assignment Component:</strong> " . round($assignment_avg, 2) . "% × 100% = " . round($assignment_avg, 2) . "%<br>";
        echo "<strong>Quiz Component:</strong> N/A × 0% = 0%<br>";
        echo "<strong>Final Grade:</strong> <span style='font-size: 1.2em; color: #28a745; font-weight: bold;'>" . round($assignment_avg, 2) . "%</span><br>";
        echo "<strong>Weight Verification:</strong> 100% assignments + 0% quizzes = 100% ✅";
        
    } elseif (!$has_assignments && $has_quizzes) {
        // Only quizzes - use 100% quiz weight
        $quiz_avg = 0;
        $quiz_count = 0;
        foreach ($scenario['quizzes'] as $q) {
            if ($q['contributes']) {
                $quiz_avg += $q['score'];
                $quiz_count++;
            }
        }
        $quiz_avg = $quiz_count > 0 ? $quiz_avg / $quiz_count : 0;
        
        echo "<strong>Method:</strong> Quiz-only grading (no assignments available)<br>";
        echo "<strong>Assignment Component:</strong> N/A × 0% = 0%<br>";
        echo "<strong>Quiz Component:</strong> " . round($quiz_avg, 2) . "% × 100% = " . round($quiz_avg, 2) . "%<br>";
        echo "<strong>Final Grade:</strong> <span style='font-size: 1.2em; color: #28a745; font-weight: bold;'>" . round($quiz_avg, 2) . "%</span><br>";
        echo "<strong>Weight Verification:</strong> 0% assignments + 100% quizzes = 100% ✅";
    }
    
    echo "</div>";
    echo "</div>";
}

// Summary
echo "<div style='background: #d1ecf1; border: 1px solid #bee5eb; padding: 20px; border-radius: 8px; margin: 20px 0;'>";
echo "<h3 style='color: #0c5460;'>🎯 Key Benefits of This System:</h3>";
echo "<ul>";
echo "<li><strong>Always 100%:</strong> Weights automatically adjust to ensure proper grading</li>";
echo "<li><strong>Flexible Assessment:</strong> Teachers can mix graded and practice content</li>";
echo "<li><strong>Independent Courses:</strong> Each course calculates grades separately</li>";
echo "<li><strong>Smart Fallbacks:</strong> Works even if only assignments OR quizzes exist</li>";
echo "<li><strong>Transparent:</strong> Students see exactly what counts toward their grade</li>";
echo "</ul>";
echo "</div>";

echo "<div style='background: #fff3cd; border: 1px solid #ffeaa7; padding: 15px; border-radius: 8px; margin: 20px 0;'>";
echo "<h4>⚠️ Implementation Notes:</h4>";
echo "<ul>";
echo "<li>The <code>contributes_to_final_grade</code> columns in assignments and quizzes tables control what counts</li>";
echo "<li>Course-level <code>assignment_weight</code> and <code>quiz_weight</code> control the distribution</li>";
echo "<li>The <code>CourseGradeCalculator</code> class handles all the complex logic automatically</li>";
echo "<li>Each course is completely independent - perfect for multiple course enrollment</li>";
echo "</ul>";
echo "</div>";

echo "</div>";
?>
