<?php
/**
 * Secure Download Script for Teacher Feedback Files
 * Ensures only authorized users can download teacher feedback files
 */

session_start();

// Debug mode - enable to see what's failing
$debug = true;

if ($debug) {
    echo "DEBUG: Starting download script<br>";
    echo "DEBUG: Session data: " . print_r($_SESSION, true) . "<br>";
}

// Include database connection and authentication
require_once __DIR__ . '/../includes/db_connection.php';
require_once __DIR__ . '/../includes/auth_functions.php';

if ($debug) {
    echo "DEBUG: Files included successfully<br>";
    echo "DEBUG: Checking authentication...<br>";
}

// Check if user is logged in and has teacher role
if (!auth_isLoggedIn() || !auth_hasPermission('teacher')) {
    if ($debug) {
        echo "DEBUG: Authentication failed<br>";
        echo "DEBUG: Logged in: " . (auth_isLoggedIn() ? 'YES' : 'NO') . "<br>";
        echo "DEBUG: Has teacher permission: " . (auth_hasPermission('teacher') ? 'YES' : 'NO') . "<br>";
    }
    die("Unauthorized access");
}

if ($debug) {
    echo "DEBUG: Authentication successful<br>";
}

$file_name = $_GET['file'] ?? '';
$submission_id = intval($_GET['submission'] ?? 0);

if (empty($file_name) || $submission_id <= 0) {
    die("Invalid request");
}

// Verify user has permission to access this file
$permission_query = "
    SELECT s.assignment_id, a.course_id, c.teacher_id, s.teacher_feedback_files
    FROM assignment_submissions s
    JOIN assignments a ON s.assignment_id = a.assignment_id
    JOIN courses c ON a.course_id = c.course_id
    WHERE s.submission_id = ? AND c.teacher_id = ?
";

$stmt = $conn->prepare($permission_query);
$stmt->bind_param("ii", $submission_id, $_SESSION['user_id']);
$stmt->execute();
$result = $stmt->get_result()->fetch_assoc();

if (!$result) {
    die("Access denied");
}

// Verify file exists in teacher feedback files JSON
$teacher_files = json_decode($result['teacher_feedback_files'], true) ?: [];
$file_found = false;
$original_name = '';

foreach ($teacher_files as $file) {
    if ($file['file_path'] === $file_name) {
        $file_found = true;
        $original_name = $file['original_name'];
        break;
    }
}

if (!$file_found) {
    die("File not found");
}

// Build file path
$file_path = __DIR__ . '/../uploads/teacher_feedback/' . $file_name;

// Debug information (remove in production)
error_log("Teacher feedback download attempt:");
error_log("File name: " . $file_name);
error_log("File path: " . $file_path);
error_log("File exists: " . (file_exists($file_path) ? 'YES' : 'NO'));

if (!file_exists($file_path)) {
    // Check if directory exists
    $dir_path = __DIR__ . '/../uploads/teacher_feedback/';
    error_log("Directory path: " . $dir_path);
    error_log("Directory exists: " . (is_dir($dir_path) ? 'YES' : 'NO'));
    
    // List files in directory if it exists
    if (is_dir($dir_path)) {
        $files = scandir($dir_path);
        error_log("Files in directory: " . implode(', ', $files));
    }
    
    die("File does not exist on server. Check error logs for details.");
}

// Get file info
$file_size = filesize($file_path);
$file_extension = pathinfo($original_name, PATHINFO_EXTENSION);
$mime_type = 'application/octet-stream';

// Set appropriate MIME type
$mime_types = [
    'pdf' => 'application/pdf',
    'doc' => 'application/msword',
    'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
    'txt' => 'text/plain',
    'jpg' => 'image/jpeg',
    'jpeg' => 'image/jpeg',
    'png' => 'image/png',
    'gif' => 'image/gif',
    'mp4' => 'video/mp4',
    'mov' => 'video/quicktime',
    'avi' => 'video/x-msvideo',
    'mp3' => 'audio/mpeg',
    'wav' => 'audio/wav',
    'xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
    'xls' => 'application/vnd.ms-excel',
    'ppt' => 'application/vnd.ms-powerpoint',
    'pptx' => 'application/vnd.openxmlformats-officedocument.presentationml.presentation'
];

if (isset($mime_types[strtolower($file_extension)])) {
    $mime_type = $mime_types[strtolower($file_extension)];
}

// Check if this should be displayed inline (for viewing) or as download
$is_inline_view = isset($_GET['inline']) && $_GET['inline'] === '1';
$is_image = in_array(strtolower($file_extension), ['jpg', 'jpeg', 'png', 'gif']);
$is_video = in_array(strtolower($file_extension), ['mp4', 'mov', 'avi', 'webm']);
$is_audio = in_array(strtolower($file_extension), ['mp3', 'wav']);
$is_pdf = strtolower($file_extension) === 'pdf';

// Send appropriate headers
header('Content-Type: ' . $mime_type);

if ($is_inline_view || $is_image || $is_video || $is_audio || $is_pdf) {
    // Display inline for viewing
    header('Content-Disposition: inline; filename="' . $original_name . '"');
} else {
    // Force download for other file types
    header('Content-Disposition: attachment; filename="' . $original_name . '"');
}

header('Content-Length: ' . $file_size);
header('Cache-Control: public, max-age=3600'); // Allow caching for media files
header('Accept-Ranges: bytes'); // Support range requests for video

// Output file
readfile($file_path);
exit;
?>
