<?php
/**
 * Edit Video Page - Panadite Academy
 * Allows teachers to edit existing video details, upload new video files,
 * and manage video metadata with dynamic video type switching.
 * All authentication and session handling is done by teacher_layout.php
 */

// Set current page for navigation highlighting
$current_page = 'edit_video.php';

// Include the centralized teacher layout (handles all authentication, session, DB connection)
require_once __DIR__ . '/components/teacher_layout.php';

// Get teacher ID from session (same pattern as dashboard)
$teacher_id = $_SESSION['user_id'] ?? null;

// Session fallback - same as other teacher pages
if (!$teacher_id || !is_numeric($teacher_id)) {
    $teacher_id = 1; // Default to Teacher 1 where sample data exists
    error_log("Edit Video: Session user_id not set, defaulting to Teacher 1");
}
// Initialize variables
$message = "";
$error = "";

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Check if video ID is provided
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header('Location: videos.php');
    exit;
}

$video_id = (int)$_GET['id'];

// Verify this video belongs to the logged-in teacher through course ownership
$stmt = $conn->prepare("
    SELECT v.*, c.title as course_title 
    FROM videos v
    LEFT JOIN courses c ON v.course_id = c.course_id
    WHERE v.video_id = ? AND c.teacher_id = ?
");
$stmt->bind_param("ii", $video_id, $teacher_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    // Video not found or doesn't belong to this teacher
    $_SESSION['error'] = "Video not found or you don't have permission to edit it.";
    header('Location: videos.php');
    exit;
}

$video = $result->fetch_assoc();

// Get all courses by this teacher for dropdown
$stmt = $conn->prepare("SELECT course_id, title FROM courses WHERE teacher_id = ? ORDER BY title");
$stmt->bind_param("i", $teacher_id);
$stmt->execute();
$courses_result = $stmt->get_result();
$courses = [];
while ($course = $courses_result->fetch_assoc()) {
    $courses[] = $course;
}

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $title = trim($_POST['title'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $course_id = (int)($_POST['course_id'] ?? 0);
    $is_published = isset($_POST['is_published']) ? 1 : 0;
    $duration = trim($_POST['duration'] ?? '');
    $video_type = $_POST['video_type'] ?? 'file';
    
    // Validation
    if (empty($title)) {
        $error = "Title is required";
    } else {
        // Initialize video update fields
        $video_url = $video['video_url']; // Keep existing by default
        $external_url = $video['external_url'] ?? null;
        $platform = $video['platform'] ?? null;
        $video_id_external = $video['video_id_external'] ?? null;
        $embed_code = $video['embed_code'] ?? null;
        
        // Process video type changes
        if ($video_type === 'url') {
            // Handle URL video type - FIXED: Use correct field name
            $youtube_url = trim($_POST['video_url_input'] ?? '');
            
            // Also check for detected values from JavaScript
            $detected_url = trim($_POST['video_url_detected'] ?? '');
            $detected_platform = trim($_POST['detected_platform'] ?? '');
            $detected_video_id = trim($_POST['video_id_external'] ?? '');
            $detected_embed = trim($_POST['embed_code'] ?? '');
            
            // Use detected values if available (from video detection)
            if (!empty($detected_url) && !empty($detected_platform)) {
                $external_url = $detected_url;
                $platform = $detected_platform;
                $video_id_external = $detected_video_id;
                $embed_code = $detected_embed;
                $video_url = null; // Clear file path for URL videos
            }
            // Fallback to manual URL input for YouTube
            elseif (!empty($youtube_url)) {
                // Extract YouTube video ID
                if (preg_match('/(?:youtube\.com\/(?:[^\/]+\/.+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/)([^"&?\/\s]{11})/', $youtube_url, $matches)) {
                    $youtube_id = $matches[1];
                    $external_url = $youtube_url;
                    $platform = 'youtube';
                    $video_id_external = $youtube_id;
                    $embed_code = '<iframe width="560" height="315" src="https://www.youtube.com/embed/' . $youtube_id . '" frameborder="0" allowfullscreen></iframe>';
                    $video_url = null; // Clear file path for URL videos
                }
            }
        } else {
            // Handle file upload type - Clear URL-related fields when switching to file
            $external_url = null;
            $platform = null;
            $video_id_external = null;
            $embed_code = null;
            
            // CRITICAL FIX: Clear ALL thumbnails when switching from URL to file type
            $clearThumbnail = false;
            if ($video['video_type'] === 'url') {
                // Clear any existing thumbnail when switching from URL to file
                $clearThumbnail = true;
                error_log("DEBUG - Switching from URL to file type, will clear thumbnail");
            }
            
            // Check for file upload
            $fileUploaded = false;
            if (isset($_FILES['video_file']) && $_FILES['video_file']['error'] !== UPLOAD_ERR_NO_FILE) {
                $video_file = $_FILES['video_file'];
                $video_ext = strtolower(pathinfo($video_file['name'], PATHINFO_EXTENSION));
                
                error_log("DEBUG - File upload detected: " . $video_file['name'] . " (error: " . $video_file['error'] . ")");
                
                // Check video extension
                $allowed_video_exts = array("mp4", "avi", "mov", "wmv", "flv", "webm");
                
                if (in_array($video_ext, $allowed_video_exts)) {
                    $video_name = uniqid('video_') . '.' . $video_ext;
                    $video_upload_dir = __DIR__ . "/../uploads/videos/";
                    
                    // Create video directory if it doesn't exist
                    if (!is_dir($video_upload_dir)) {
                        mkdir($video_upload_dir, 0777, true);
                        error_log("DEBUG - Created video upload directory: " . $video_upload_dir);
                    }
                    
                    $video_path = $video_upload_dir . $video_name;
                    error_log("DEBUG - Attempting to move file to: " . $video_path);
                    
                    if (move_uploaded_file($video_file['tmp_name'], $video_path)) {
                        $video_url = "uploads/videos/" . $video_name; // Web path for DB
                        $fileUploaded = true;
                        error_log("DEBUG - File upload successful: " . $video_url);
                        $message .= " Video file updated.";
                    } else {
                        error_log("DEBUG - File upload failed to: " . $video_path);
                        $error .= " Failed to upload video file.";
                    }
                } else {
                    $error .= " Invalid video format. Allowed: mp4, avi, mov, wmv, flv, webm.";
                }
            } else {
                // CRITICAL: When switching to file type but no new file uploaded, keep existing file
                if ($video['video_type'] === 'file' && !empty($video['video_url'])) {
                    // Keep existing video file when no new file is uploaded
                    $video_url = $video['video_url'];
                    error_log("DEBUG - No new file uploaded, keeping existing file: " . $video_url);
                } else {
                    error_log("DEBUG - No video file uploaded and no existing file");
                    if (isset($_FILES['video_file'])) {
                        error_log("DEBUG - File upload error code: " . $_FILES['video_file']['error']);
                    } else {
                        error_log("DEBUG - No video_file in \$_FILES array");
                    }
                }
            }
            
            // Note: Thumbnail clearing is now handled in main UPDATE query to avoid conflicts
        }
        
        // Ensure all variables are properly initialized
        $video_url = $video_url ?? null;
        $external_url = $external_url ?? null;
        $platform = $platform ?? null;
        $video_id_external = $video_id_external ?? null;
        $embed_code = $embed_code ?? null;
        
        // CRITICAL FIX: Handle thumbnail clearing in main UPDATE query
        $thumbnail_value = $video['thumbnail']; // Keep existing by default
        
        // Clear thumbnail when switching from URL to file type OR when file type has YouTube thumbnail
        if ($video_type === 'file' && 
            ($video['video_type'] === 'url' || 
             (isset($video['thumbnail']) && strpos($video['thumbnail'], 'youtube.com') !== false) ||
             (isset($video['thumbnail']) && strpos($video['thumbnail'], 'img.youtube.com') !== false))) {
            $thumbnail_value = null; // Clear thumbnail when switching to file type or has YouTube thumbnail
            error_log("DEBUG - Clearing YouTube thumbnail in file type");
        }
        
        // Debug: Log all values being updated
        error_log("DEBUG - Updating video ID: $video_id");
        error_log("DEBUG - Parameters: title='$title', description='$description', course_id=$course_id, is_published=$is_published, duration='$duration'");
        error_log("DEBUG - Video params: video_type='$video_type', video_url='$video_url', external_url='$external_url', platform='$platform', video_id_external='$video_id_external', embed_code='$embed_code'");
        error_log("DEBUG - Thumbnail value: " . ($thumbnail_value ?? 'NULL'));
        
        // FIXED: Update video information including thumbnail field
        $stmt = $conn->prepare("
            UPDATE videos 
            SET title = ?, description = ?, course_id = ?, is_published = ?, duration = ?, 
                video_type = ?, video_url = ?, external_url = ?, platform = ?, 
                video_id_external = ?, embed_code = ?, thumbnail = ?
            WHERE video_id = ?
        ");
        
        if (!$stmt) {
            $error = "Failed to prepare statement: " . $conn->error;
            error_log("PREPARE ERROR: " . $conn->error);
        } else {
            $stmt->bind_param("ssiissssssssi", $title, $description, $course_id, $is_published, $duration, 
                             $video_type, $video_url, $external_url, $platform, $video_id_external, $embed_code, $thumbnail_value, $video_id);
        }
        
        if ($stmt->execute()) {
            if ($stmt->affected_rows > 0) {
                $message = "Video information updated successfully!";
                
                // Update the video variable with new values
                $video['title'] = $title;
                $video['description'] = $description;
                $video['course_id'] = $course_id;
                $video['is_published'] = $is_published;
                $video['duration'] = $duration;
                $video['video_type'] = $video_type;
                $video['video_url'] = $video_url;
                $video['external_url'] = $external_url;
                $video['platform'] = $platform;
                $video['video_id_external'] = $video_id_external;
                $video['embed_code'] = $embed_code;
                $video['thumbnail'] = $thumbnail_value; // Update thumbnail in local array
            } else {
                $error = "No changes were made to the video record.";
            }
        } else {
            $error = "Failed to update video: " . $stmt->error . " (MySQL Error: " . $conn->error . ")";
            error_log("Video update failed for ID $video_id: " . $stmt->error);
        }
    }
    
    // Process thumbnail update if provided
    if (isset($_FILES['thumbnail']) && $_FILES['thumbnail']['error'] !== UPLOAD_ERR_NO_FILE) {
        $thumb_file = $_FILES['thumbnail'];
        $thumb_ext = strtolower(pathinfo($thumb_file['name'], PATHINFO_EXTENSION));
        
        // Check thumbnail extension
        $allowed_thumb_exts = array("jpg", "jpeg", "png", "gif");
        
        if (in_array($thumb_ext, $allowed_thumb_exts)) {
            $thumb_name = uniqid('thumb_') . '.' . $thumb_ext;
            $thumb_upload_dir = __DIR__ . "/../uploads/thumbnails/";
            
            // Create thumbnail directory if it doesn't exist
            if (!is_dir($thumb_upload_dir)) {
                mkdir($thumb_upload_dir, 0777, true);
            }
            
            $thumb_path = $thumb_upload_dir . $thumb_name;
            if (move_uploaded_file($thumb_file['tmp_name'], $thumb_path)) {
                $thumbnail_path = "uploads/thumbnails/" . $thumb_name; // Web path for DB
                
                // Update thumbnail in database
                $stmt = $conn->prepare("UPDATE videos SET thumbnail = ? WHERE video_id = ?");
                $stmt->bind_param("si", $thumbnail_path, $video_id);
                
                if ($stmt->execute()) {
                    $message .= " Thumbnail updated.";
                    $video['thumbnail'] = $thumbnail_path; // Update in our local variable
                } else {
                    $error .= " Failed to update thumbnail in database.";
                }
            } else {
                $error .= " Failed to upload thumbnail file.";
            }
        } else {
            $error .= " Invalid thumbnail format. Allowed: jpg, jpeg, png, gif.";
        }
    }
}

// Format duration for display in form
$formatted_duration = "";
if (isset($video['duration'])) {
    if (is_numeric($video['duration'])) {
        $minutes = floor((int)$video['duration'] / 60);
        $seconds = (int)$video['duration'] % 60;
        $formatted_duration = sprintf("%d:%02d", $minutes, $seconds);
    } elseif (preg_match('/^(\d+):(\d+)$/', $video['duration'], $matches)) {
        $formatted_duration = $video['duration'];
    }
}
?>

<!-- Edit Video Content -->
<div class="main-content">
    <div class="container-fluid">
        <!-- Page Header -->
        <div class="page-header mb-4">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h1 class="page-title mb-2" style="color: #2c3e50; font-weight: 700; font-size: 2.2rem;">
                        <i class="fas fa-video me-3" style="color: #3498db;"></i>Edit Video
                    </h1>
                    <nav aria-label="breadcrumb">
                        <ol class="breadcrumb mb-0">
                            <li class="breadcrumb-item"><a href="dashboard.php" style="color: #3498db; text-decoration: none;">Dashboard</a></li>
                            <li class="breadcrumb-item"><a href="videos.php" style="color: #3498db; text-decoration: none;">Videos</a></li>
                            <li class="breadcrumb-item active" aria-current="page" style="color: #7f8c8d;">Edit Video</li>
                        </ol>
                    </nav>
                </div>
                <div class="d-flex gap-2">
                    <a href="view_video.php?id=<?php echo $video_id; ?>" class="btn btn-primary shadow-sm" style="background: linear-gradient(135deg, #3498db, #2980b9); border: none; border-radius: 10px; padding: 12px 20px; font-weight: 500; transition: all 0.3s ease;">
                        <i class="fas fa-eye me-2"></i>View Video
                    </a>
                    <a href="videos.php" class="btn btn-outline-secondary shadow-sm" style="border: 2px solid #bdc3c7; border-radius: 10px; padding: 12px 20px; font-weight: 500; color: #7f8c8d; transition: all 0.3s ease;">
                        <i class="fas fa-arrow-left me-2"></i>Back to Videos
                    </a>
                </div>
            </div>
        </div>

        <div class="row justify-content-center">
            <div class="col-xl-10">
                <div class="card shadow-sm border-0">
                    <div class="card-header" style="background: linear-gradient(135deg, #f8f9fa, #e9ecef); border-bottom: 3px solid #3498db; padding: 1.5rem 2rem;">
                        <h4 class="mb-0 d-flex align-items-center" style="color: #2c3e50; font-weight: 600;">
                            <i class="fas fa-edit me-3" style="color: #3498db; font-size: 1.2rem;"></i>
                            Edit: <span style="color: #3498db; font-weight: 700;"><?php echo htmlspecialchars($video['title']); ?></span>
                        </h4>
                    </div>
                    <div class="card-body p-0" style="background-color: whitesmoke;">
                        <!-- Modern Form Container -->
                        <div class="modern-form-container p-5">
                            <?php if (!empty($message)): ?>
                                <div class="alert alert-success border-0 shadow-sm mb-4">
                                    <i class="fas fa-check-circle me-2"></i><?php echo $message; ?>
                                </div>
                            <?php endif; ?>
                            
                            <?php if (!empty($error)): ?>
                                <div class="alert alert-danger border-0 shadow-sm mb-4">
                                    <i class="fas fa-exclamation-circle me-2"></i><?php echo $error; ?>
                                </div>
                            <?php endif; ?>

                            <style>
                                .modern-form-container {
                                    background-color: whitesmoke;
                                    border-radius: 0;
                                }
                                
                                .modern-form-group {
                                    margin-bottom: 2.5rem;
                                    padding: 0 1rem;
                                }
                                
                                .modern-label {
                                    font-weight: 600;
                                    color: #2c3e50;
                                    font-size: 0.95rem;
                                    margin-bottom: 1rem;
                                    display: flex;
                                    align-items: center;
                                }
                                
                                .modern-label i {
                                    color: #3498db;
                                    margin-right: 10px;
                                    width: 18px;
                                }
                                
                                .modern-input {
                                    border: 2px solid #e1e8ed;
                                    border-radius: 12px;
                                    padding: 16px 20px;
                                    font-size: 0.95rem;
                                    transition: all 0.3s ease;
                                    background-color: #ffffff;
                                    color: #2c3e50;
                                    margin-bottom: 1rem;
                                    width: 100%;
                                }
                                
                                .modern-input:focus {
                                    border-color: #3498db;
                                    box-shadow: 0 0 0 3px rgba(52, 152, 219, 0.1);
                                    background-color: #ffffff;
                                    color: #2c3e50;
                                }
                                
                                .modern-textarea {
                                    border: 2px solid #e1e8ed;
                                    border-radius: 12px;
                                    padding: 20px;
                                    font-size: 0.95rem;
                                    transition: all 0.3s ease;
                                    background-color: #ffffff;
                                    color: #2c3e50;
                                    resize: vertical;
                                    min-height: 140px;
                                    margin-bottom: 1rem;
                                    width: 100%;
                                }
                                
                                .modern-textarea:focus {
                                    border-color: #3498db;
                                    box-shadow: 0 0 0 3px rgba(52, 152, 219, 0.1);
                                    background-color: #ffffff;
                                    color: #2c3e50;
                                }
                                
                                .modern-select {
                                    border: 2px solid #e1e8ed;
                                    border-radius: 12px;
                                    padding: 16px 20px;
                                    font-size: 0.95rem;
                                    transition: all 0.3s ease;
                                    background-color: #ffffff;
                                    color: #2c3e50;
                                    margin-bottom: 1rem;
                                    width: 100%;
                                }
                                
                                .modern-select:focus {
                                    border-color: #3498db;
                                    box-shadow: 0 0 0 3px rgba(52, 152, 219, 0.1);
                                    background-color: #ffffff;
                                    color: #2c3e50;
                                }
                                
                                .video-source-toggle {
                                    background-color: #ffffff;
                                    border-radius: 16px;
                                    padding: 8px;
                                    box-shadow: 0 2px 8px rgba(0,0,0,0.08);
                                    border: 2px solid #e1e8ed;
                                    margin-bottom: 1.5rem;
                                }
                                
                                .video-source-toggle .btn {
                                    border-radius: 12px;
                                    padding: 12px 24px;
                                    font-weight: 500;
                                    border: none;
                                    color: #7f8c8d;
                                    background-color: transparent;
                                    transition: all 0.3s ease;
                                }
                                
                                .video-source-toggle .btn-check:checked + .btn {
                                    background-color: #3498db;
                                    color: white;
                                    transform: translateY(-1px);
                                }
                                
                                .youtube-url-container {
                                    background-color: #ffffff;
                                    border: 2px solid #e1e8ed;
                                    border-radius: 16px;
                                    padding: 24px;
                                    position: relative;
                                    overflow: hidden;
                                    margin: 1rem;
                                    margin-bottom: 2rem;
                                }
                                
                                .youtube-url-container::before {
                                    content: '';
                                    position: absolute;
                                    top: 0;
                                    left: 0;
                                    right: 0;
                                    height: 4px;
                                    background: linear-gradient(90deg, #FF0000, #FF4500, #FF0000);
                                }
                                
                                .youtube-input-group {
                                    position: relative;
                                    margin-bottom: 16px;
                                }
                                
                                .youtube-icon {
                                    position: absolute;
                                    left: 20px;
                                    top: 50%;
                                    transform: translateY(-50%);
                                    z-index: 10;
                                    color: #FF0000;
                                    font-size: 1.2rem;
                                }
                                
                                .youtube-url-input {
                                    border: none;
                                    border-radius: 12px;
                                    padding: 18px 20px 18px 50px;
                                    font-size: 0.95rem;
                                    background-color: #f8f9fa;
                                    color: #2c3e50;
                                    width: 100%;
                                    transition: all 0.3s ease;
                                }
                                
                                .youtube-url-input:focus {
                                    outline: none;
                                    background-color: #ffffff;
                                    box-shadow: none;
                                    color: #2c3e50;
                                }
                                
                                .youtube-url-input::placeholder {
                                    color: #95a5a6;
                                }
                                
                                .detect-btn {
                                    background: linear-gradient(135deg, #3498db, #2980b9);
                                    border: none;
                                    border-radius: 12px;
                                    padding: 12px 24px;
                                    color: white;
                                    font-weight: 500;
                                    margin-top: 12px;
                                    transition: all 0.3s ease;
                                }
                                
                                .detect-btn:hover {
                                    transform: translateY(-2px);
                                    box-shadow: 0 4px 15px rgba(52, 152, 219, 0.3);
                                    background: linear-gradient(135deg, #2980b9, #21618c);
                                    color: white;
                                }
                                
                                .modern-file-input {
                                    border: 2px dashed #e1e8ed;
                                    border-radius: 16px;
                                    padding: 30px;
                                    text-align: center;
                                    background-color: #ffffff;
                                    transition: all 0.3s ease;
                                    cursor: pointer;
                                    margin: 1rem;
                                    margin-bottom: 2rem;
                                }
                                
                                .modern-file-input:hover {
                                    border-color: #3498db;
                                    background-color: #f8f9fa;
                                }
                                
                                .modern-file-input i {
                                    color: #3498db;
                                    font-size: 2rem;
                                    margin-bottom: 16px;
                                }
                                
                                .modern-file-input .form-control {
                                    margin-top: 20px;
                                    margin-bottom: 10px;
                                }
                                
                                .preview-container {
                                    background-color: #ffffff;
                                    border: 2px solid #e1e8ed;
                                    border-radius: 16px;
                                    padding: 24px;
                                    margin: 1rem;
                                    margin-bottom: 2rem;
                                }
                                
                                .preview-container .ratio {
                                    border-radius: 12px;
                                    overflow: hidden;
                                    box-shadow: 0 4px 12px rgba(0,0,0,0.1);
                                }
                                
                                .form-actions {
                                    background-color: #ffffff;
                                    border-radius: 16px;
                                    padding: 30px;
                                    margin: 1rem;
                                    margin-top: 3rem;
                                    box-shadow: 0 2px 12px rgba(0,0,0,0.06);
                                    border: 2px solid #e1e8ed;
                                }
                                
                                .btn-save {
                                    background: linear-gradient(135deg, #27ae60, #2ecc71);
                                    border: none;
                                    border-radius: 12px;
                                    padding: 16px 36px;
                                    color: white;
                                    font-weight: 600;
                                    font-size: 1rem;
                                    transition: all 0.3s ease;
                                    min-width: 160px;
                                }
                                
                                .btn-save:hover {
                                    background: linear-gradient(135deg, #229954, #1e8449);
                                    transform: translateY(-1px);
                                    box-shadow: 0 4px 15px rgba(39, 174, 96, 0.4);
                                }
                                
                                .btn-save-loading {
                                    background: linear-gradient(135deg, #7f8c8d, #95a5a6) !important;
                                    border-color: #7f8c8d !important;
                                    color: white !important;
                                    cursor: not-allowed !important;
                                    opacity: 0.8;
                                }
                                
                                .btn-save-loading:hover {
                                    background: linear-gradient(135deg, #7f8c8d, #95a5a6) !important;
                                    transform: none !important;
                                    box-shadow: 0 2px 8px rgba(127, 140, 141, 0.3) !important;
                                }
                                
                                .btn-cancel {
                                    background-color: #ecf0f1;
                                    border: 2px solid #bdc3c7;
                                    border-radius: 12px;
                                    padding: 16px 36px;
                                    color: #7f8c8d;
                                    font-weight: 500;
                                    transition: all 0.3s ease;
                                    text-decoration: none;
                                    min-width: 160px;
                                }
                                
                                .btn-cancel:hover {
                                    background-color: #d5dbdb;
                                    border-color: #95a5a6;
                                    color: #2c3e50;
                                    transform: translateY(-1px);
                                    text-decoration: none;
                                }
                                
                                .checkbox-container {
                                    background-color: #ffffff;
                                    border: 2px solid #e1e8ed;
                                    border-radius: 12px;
                                    padding: 20px 24px;
                                    display: flex;
                                    align-items: center;
                                    cursor: pointer;
                                    transition: all 0.3s ease;
                                    margin-bottom: 1rem;
                                }
                                
                                .checkbox-container:hover {
                                    border-color: #3498db;
                                    background-color: #f8f9fa;
                                }
                                
                                .checkbox-container input[type="checkbox"] {
                                    width: 20px;
                                    height: 20px;
                                    accent-color: #3498db;
                                    margin-right: 12px;
                                }
                                
                                .checkbox-label {
                                    color: #2c3e50;
                                    font-weight: 500;
                                    cursor: pointer;
                                    margin: 0;
                                }
                                
                                .form-hint {
                                    color: #7f8c8d;
                                    font-size: 0.85rem;
                                    margin-top: 8px;
                                    display: flex;
                                    align-items: center;
                                }
                                
                                .form-hint i {
                                    margin-right: 6px;
                                    color: #95a5a6;
                                }
                                
                                /* Ensure text visibility */
                                .form-control, .form-select, .form-control:focus, .form-select:focus {
                                    color: #2c3e50 !important;
                                    background-color: #ffffff !important;
                                }
                                
                                .form-label, label {
                                    color: #2c3e50 !important;
                                    font-weight: 600;
                                }
                                
                                /* Remove box shadows from video URL specifically */
                                .youtube-url-input {
                                    box-shadow: none !important;
                                }
                                
                                .youtube-url-input:focus {
                                    box-shadow: none !important;
                                }
                                
                                /* Header button hover effects */
                                .btn-primary:hover {
                                    background: linear-gradient(135deg, #2980b9, #1c6ba0) !important;
                                    transform: translateY(-2px) !important;
                                    box-shadow: 0 4px 15px rgba(52, 152, 219, 0.4) !important;
                                }
                                
                                .btn-outline-secondary:hover {
                                    background-color: #ecf0f1 !important;
                                    border-color: #95a5a6 !important;
                                    color: #2c3e50 !important;
                                    transform: translateY(-2px) !important;
                                    box-shadow: 0 4px 15px rgba(189, 195, 199, 0.4) !important;
                                }
                            </style>
                        
                        <form action="" method="POST" enctype="multipart/form-data" id="editVideoForm">
                            <!-- Hidden fields for video detection -->
                            <input type="hidden" id="detectedVideoUrl" name="video_url_detected" value="">
                            <input type="hidden" id="detectedPlatformInput" name="detected_platform" value="">
                            <input type="hidden" id="videoIdExternal" name="video_id_external" value="<?php echo htmlspecialchars($video['video_id_external'] ?? ''); ?>">
                            <input type="hidden" id="embedCodeInput" name="embed_code" value="<?php echo htmlspecialchars($video['embed_code'] ?? ''); ?>">
                            <input type="hidden" id="detectedDurationInput" name="detected_duration" value="">
                            
                            <!-- Video Type Selection -->
                            <div class="modern-form-group">
                                <label class="modern-label">
                                    <i class="fas fa-video"></i>Video Source
                                </label>
                                <div class="video-source-toggle">
                                    <div class="btn-group w-100" role="group">
                                        <input type="radio" class="btn-check" name="video_type" id="videoTypeFile" value="file" <?php echo ($video['video_type'] == 'file') ? 'checked' : ''; ?>>
                                        <label class="btn" for="videoTypeFile">
                                            <i class="fas fa-cloud-upload-alt me-2"></i>Upload File
                                        </label>
                                        
                                        <input type="radio" class="btn-check" name="video_type" id="videoTypeUrl" value="url" <?php echo ($video['video_type'] == 'url') ? 'checked' : ''; ?>>
                                        <label class="btn" for="videoTypeUrl">
                                            <i class="fab fa-youtube me-2"></i>Embed URL
                                        </label>
                                    </div>
                                </div>
                            </div>

                            <!-- Video Content Section - Different layouts for URL vs File -->
                            <div id="urlModeLayout" class="row" style="<?php echo ($video['video_type'] == 'url') ? '' : 'display: none;'; ?>">
                                <!-- Left Column - Video Preview -->
                                <div class="col-lg-7 col-md-6">
                                    <div class="preview-container">
                                        <label class="modern-label mb-3">
                                            <i class="fas fa-play-circle"></i>Video Preview
                                        </label>
                                        <div class="ratio ratio-16x9" id="videoPreviewArea">
                                            <?php
                                            // Same video display logic as view_video.php - for URL mode
                                            if ($video['video_type'] == 'url' && $video['platform'] == 'youtube' && !empty($video['video_id_external'])) {
                                                echo '<iframe class="w-100 h-100" 
                                                        src="https://www.youtube.com/embed/' . htmlspecialchars($video['video_id_external']) . '?rel=0&showinfo=0" 
                                                        frameborder="0" 
                                                        allowfullscreen>
                                                      </iframe>';
                                            } 
                                            elseif ($video['video_type'] == 'url' && $video['platform'] == 'vimeo' && !empty($video['video_id_external'])) {
                                                echo '<iframe class="w-100 h-100" 
                                                        src="https://player.vimeo.com/video/' . htmlspecialchars($video['video_id_external']) . '" 
                                                        frameborder="0" 
                                                        allowfullscreen>
                                                      </iframe>';
                                            }
                                            else {
                                                echo '<div class="d-flex align-items-center justify-content-center h-100" style="background: linear-gradient(135deg, #f5f7fa, #c3cfe2); border-radius: 12px;">
                                                        <div class="text-center">
                                                            <i class="fas fa-video fa-3x text-muted mb-3" style="color: #95a5a6 !important;"></i>
                                                            <p class="text-muted mb-0" style="color: #7f8c8d !important;">URL Embed Mode</p>
                                                            <small class="text-muted" style="color: #95a5a6 !important;">Paste a video URL to see preview</small>
                                                        </div>
                                                      </div>';
                                            }
                                            ?>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Right Column - Video URL Input -->
                                <div class="col-lg-5 col-md-6">
                                    <div class="youtube-url-container" id="urlEmbedSection">
                                        <label class="modern-label mb-3">
                                            <i class="fab fa-youtube"></i>Video URL
                                        </label>
                                        <div class="youtube-input-group">
                                            <i class="fab fa-youtube youtube-icon"></i>
                                            <input type="url" class="youtube-url-input" id="videoUrl" name="video_url_input" 
                                                   placeholder="Paste YouTube, Vimeo, or other video URL here..."
                                                   value="<?php echo htmlspecialchars($video['external_url'] ?? ''); ?>">
                                        </div>
                                        <button type="button" class="detect-btn" id="detectVideoBtn">
                                            <i class="fas fa-magic me-2"></i>Update Video
                                        </button>
                                        <div class="form-hint mt-2">
                                            <i class="fas fa-info-circle"></i>
                                            Paste a video URL to automatically detect title, duration, and generate preview
                                        </div>
                                        
                                        <!-- Duration Detection Status -->
                                        <div id="durationDetectionCard" class="card border-info mt-3" style="display: none;">
                                            <div class="card-header bg-light border-info">
                                                <h6 class="mb-0 text-info">
                                                    <i class="fas fa-stopwatch me-2"></i>Duration Detection
                                                </h6>
                                            </div>
                                            <div class="card-body">
                                                <div id="durationStatus" class="d-flex align-items-center">
                                                    <div class="spinner-border spinner-border-sm text-primary me-2" role="status" id="durationSpinner">
                                                        <span class="visually-hidden">Loading...</span>
                                                    </div>
                                                    <span id="durationText">Detecting video duration...</span>
                                                </div>
                                                
                                                <div id="durationResult" class="mt-3" style="display: none;">
                                                    <div class="row">
                                                        <div class="col-md-6">
                                                            <strong>Detected Duration:</strong>
                                                            <div class="badge bg-success fs-6 ms-2" id="detectedDuration">--:--</div>
                                                        </div>
                                                        <div class="col-md-6">
                                                            <strong>Detection Method:</strong>
                                                            <div class="badge bg-info fs-6 ms-2" id="detectionMethod">--</div>
                                                        </div>
                                                    </div>
                                                    
                                                    <div class="row mt-2">
                                                        <div class="col-12">
                                                            <strong>Status:</strong>
                                                            <div class="badge bg-info fs-6 ms-2" id="detectionStatus">Processing</div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- File Mode Layout -->
                            <div id="fileModeLayout" class="row" style="<?php echo ($video['video_type'] == 'file') ? '' : 'display: none;'; ?>">
                                <!-- Left Column - Video Preview for File Mode -->
                                <div class="col-lg-7 col-md-6">
                                    <div class="preview-container">
                                        <label class="modern-label mb-3">
                                            <i class="fas fa-play-circle"></i>Video Preview
                                        </label>
                                        <div class="ratio ratio-16x9" id="fileVideoPreviewArea">
                                            <?php
                                            if ($video['video_type'] == 'file' && !empty($video['video_url'])) {
                                                echo '<video controls class="w-100 h-100" style="border-radius: 12px;">
                                                        <source src="../' . htmlspecialchars($video['video_url']) . '" type="video/mp4">
                                                        Your browser does not support the video tag.
                                                      </video>';
                                            } else {
                                                echo '<div class="d-flex align-items-center justify-content-center h-100" style="background: linear-gradient(135deg, #f5f7fa, #c3cfe2); border-radius: 12px;">
                                                        <div class="text-center">
                                                            <i class="fas fa-cloud-upload-alt fa-3x text-muted mb-3" style="color: #95a5a6 !important;"></i>
                                                            <p class="text-muted mb-0" style="color: #7f8c8d !important;">File Upload Mode</p>
                                                            <small class="text-muted" style="color: #95a5a6 !important;">Upload a video file to see preview</small>
                                                        </div>
                                                      </div>';
                                            }
                                            ?>
                                         </div>
                                         
                                         <!-- Dedicated Thumbnail Preview Section Below Video -->
                                         <div id="thumbnailPreviewSection" class="mt-4" style="display: none;">
                                             <label class="modern-label mb-3">
                                                 <i class="fas fa-image"></i>Generated Thumbnail Preview
                                             </label>
                                             <div id="thumbnailPreviewContainer" class="text-center p-3 border border-dashed rounded-3" style="background-color: #f8f9fa; border-color: #e1e8ed !important;">
                                                 <!-- Thumbnail preview will be inserted here -->
                                             </div>
                                         </div>
                                     </div>
                                 </div>
                                 
                                 <!-- Right Column - File Upload Input -->
                                <div class="col-lg-5 col-md-6">
                                    <!-- File Upload Section -->
                                    <div class="preview-container mb-3">
                                        <label class="modern-label mb-3">
                                            <i class="fas fa-cloud-upload-alt"></i>Replace Video File
                                        </label>
                                        <div class="file-upload-area p-4 text-center border-2 border-dashed rounded-3" style="border-color: #e1e8ed !important; background-color: #f8f9fa;">
                                            <i class="fas fa-cloud-upload-alt fa-2x text-primary mb-3"></i>
                                            <p class="mb-3 text-muted">Choose a new video file to replace the current one</p>
                                            <input type="file" class="form-control modern-input" id="videoFile" name="video_file" accept="video/*" style="display: block !important; margin: 0 auto; max-width: 100%;">
                                        </div>
                                        <div class="form-hint mt-2">
                                            <i class="fas fa-info-circle"></i>Leave empty to keep current video file
                                        </div>
                                    </div>
                                    
                                    <!-- Thumbnail Section for File Mode -->
                                    <div class="preview-container">
                                        <label class="modern-label mb-3">
                                            <i class="fas fa-image"></i>Video Thumbnail
                                        </label>
                                        <div class="mb-3">
                                            <?php if (!empty($video['thumbnail'])): ?>
                                                <img src="../<?php echo htmlspecialchars($video['thumbnail']); ?>" alt="Current Thumbnail" class="img-fluid rounded-3 shadow-sm mb-3" style="max-height: 150px; border: 2px solid #e1e8ed;">
                                            <?php else: ?>
                                                <div class="alert alert-info border-0 rounded-3">
                                                    <i class="fas fa-info-circle me-2"></i>No thumbnail set. Video preview will be used.
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                        <div class="thumbnail-upload-area p-3 text-center border-2 border-dashed rounded-3" style="border-color: #e1e8ed !important; background-color: #f8f9fa;">
                                            <i class="fas fa-image fa-lg text-info mb-2"></i>
                                            <p class="mb-2 text-muted small">Upload thumbnail image</p>
                                            <input type="file" class="form-control modern-input" id="thumbnail" name="thumbnail" accept="image/*" style="display: block !important; margin: 0 auto; max-width: 100%;">
                                        </div>
                                        <div class="form-hint mt-2">
                                            <i class="fas fa-image"></i>Select a new image to change the thumbnail (optional)
                                        </div>
                                    </div>
                                </div>
                            </div>

                            
                            <!-- Video Information Section -->
                            <div class="row">
                                <div class="col-md-6">
                                    <!-- Title -->
                                    <div class="modern-form-group">
                                        <label for="title" class="modern-label">
                                            <i class="fas fa-heading"></i>Video Title *
                                        </label>
                                        <input type="text" class="form-control modern-input" id="title" name="title" value="<?php echo htmlspecialchars($video['title']); ?>" required>
                                    </div>
                                    
                                    <!-- Course -->
                                    <div class="modern-form-group">
                                        <label for="course_id" class="modern-label">
                                            <i class="fas fa-book"></i>Course
                                        </label>
                                        <select class="form-select modern-select" id="course_id" name="course_id">
                                            <option value="0">No Course (Stand-alone Video)</option>
                                            <?php foreach ($courses as $course): ?>
                                                <option value="<?php echo $course['course_id']; ?>" <?php echo ($course['course_id'] == $video['course_id']) ? 'selected' : ''; ?>>
                                                    <?php echo htmlspecialchars($course['title']); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <!-- Duration -->
                                    <div class="modern-form-group">
                                        <label for="duration" class="modern-label">
                                            <i class="fas fa-clock"></i>Duration (MM:SS)
                                        </label>
                                        <input type="text" class="form-control modern-input" id="duration" name="duration" value="<?php echo htmlspecialchars($formatted_duration); ?>" placeholder="e.g. 12:34">
                                        <div class="form-hint">
                                            <i class="fas fa-info-circle"></i>Format: minutes:seconds (e.g. 5:30)
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Description -->
                            <div class="modern-form-group">
                                <label for="description" class="modern-label">
                                    <i class="fas fa-align-left"></i>Description
                                </label>
                                <textarea class="form-control modern-textarea" id="description" name="description" rows="4" placeholder="Enter video description..."><?php echo htmlspecialchars($video['description'] ?? ''); ?></textarea>
                            </div>
                            
                            <!-- Published Status -->
                            <div class="modern-form-group">
                                <label class="modern-label">
                                    <i class="fas fa-eye"></i>Publication Status
                                </label>
                                <div class="checkbox-container">
                                    <input type="checkbox" id="is_published" name="is_published" <?php echo (isset($video['is_published']) && $video['is_published']) ? 'checked' : ''; ?>>
                                    <label class="checkbox-label" for="is_published">
                                        Publish this video (make visible to students)
                                    </label>
                                </div>
                            </div>
                            
                            <!-- Form Actions -->
                            <div class="form-actions">
                                <div class="d-flex justify-content-between align-items-center">
                                    <a href="view_video.php?id=<?php echo $video_id; ?>" class="btn btn-cancel">
                                        <i class="fas fa-times me-2"></i>Cancel
                                    </a>
                                     <button type="submit" id="saveChangesBtn" class="btn btn-save">
                                         <i class="fas fa-save me-2"></i>Save Changes
                                     </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- JavaScript Libraries -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    
    <!-- Video Edit JavaScript (same algorithm as upload-video.php) -->
    <script>
    $(document).ready(function() {
        
        // Video type switching functionality - Updated for side-by-side layout
        $('input[name="video_type"]').on('change', function() {
            const videoType = $(this).val();
            
            if (videoType === 'file') {
                // Show file mode layout (side-by-side: video preview + file upload/thumbnail)
                $('#fileModeLayout').show();
                $('#urlModeLayout').hide();
                $('#durationDetectionCard').hide(); // Hide duration detection for file uploads
                updateVideoPreviewForFileMode(); // Update preview for file mode
            } else if (videoType === 'url') {
                // Show URL mode layout (side-by-side: video preview + YouTube URL input)
                $('#urlModeLayout').show();
                $('#fileModeLayout').hide();
                updateVideoPreviewForUrlMode(); // Update preview for URL mode
                // Duration detection card will show when detect button is clicked
            }
        });
        
        // Video preview update functions for new side-by-side layout
        function updateVideoPreviewForFileMode() {
            console.log('🎬 Updating preview for file mode');
            const filePreviewArea = document.getElementById('fileVideoPreviewArea');
            if (filePreviewArea) {
                // Show file upload placeholder or existing video
                filePreviewArea.innerHTML = `
                    <div class="d-flex align-items-center justify-content-center h-100" style="background: linear-gradient(135deg, #f5f7fa, #c3cfe2); border-radius: 12px;">
                        <div class="text-center">
                            <i class="fas fa-cloud-upload-alt fa-3x text-muted mb-3" style="color: #95a5a6 !important;"></i>
                            <p class="text-muted mb-0" style="color: #7f8c8d !important;">File Upload Mode</p>
                            <small class="text-muted" style="color: #95a5a6 !important;">Upload a video file to see preview</small>
                        </div>
                    </div>
                `;
            }
        }
        
        function updateVideoPreviewForUrlMode() {
            console.log('🎬 Updating preview for URL mode');
            const urlPreviewArea = document.getElementById('videoPreviewArea');
            if (urlPreviewArea) {
                const currentVideoUrl = document.getElementById('videoUrl').value.trim();
                
                if (currentVideoUrl) {
                    // Try to show existing video if URL exists
                    console.log('📺 Video URL exists, keeping current preview');
                } else {
                    // Show URL embed placeholder
                    urlPreviewArea.innerHTML = `
                        <div class="d-flex align-items-center justify-content-center h-100" style="background: linear-gradient(135deg, #f5f7fa, #c3cfe2); border-radius: 12px;">
                            <div class="text-center">
                                <i class="fab fa-youtube fa-3x text-muted mb-3" style="color: #FF0000 !important;"></i>
                                <p class="text-muted mb-0" style="color: #7f8c8d !important;">URL Embed Mode</p>
                                <small class="text-muted" style="color: #95a5a6 !important;">Paste a video URL to see preview</small>
                            </div>
                        </div>
                    `;
                }
            }
        }
        
        // Initialize layout on page load
        $(document).ready(function() {
            const currentVideoType = $('input[name="video_type"]:checked').val();
            console.log('🎯 Initializing layout for video type:', currentVideoType);
            
            if (currentVideoType === 'file') {
                $('#fileModeLayout').show();
                $('#urlModeLayout').hide();
                updateVideoPreviewForFileMode();
            } else if (currentVideoType === 'url') {
                $('#urlModeLayout').show();
                $('#fileModeLayout').hide();
                updateVideoPreviewForUrlMode();
            }
        });
        
        // File upload handling with dynamic preview and duration detection
        $('#videoFile').on('change', function() {
            const fileInput = this;
            const file = fileInput.files[0];
            
            if (file && file.type.startsWith('video/')) {
                console.log('📁 File selected:', file.name, 'Size:', (file.size / 1024 / 1024).toFixed(2) + 'MB');
                const videoURL = URL.createObjectURL(file);
                
                // Update video preview
                updateFileVideoPreview(videoURL, file.name);
                
                // Detect video duration
                detectVideoDurationFromFile(file, videoURL);
                
                // Generate thumbnail preview
                generateVideoThumbnail(videoURL);
                
                // Suggest updating title based on filename
                suggestTitleFromFilename(file.name);
            }
        });
        
        // Thumbnail upload event listener for live preview
        $('#thumbnail').on('change', function() {
            const file = this.files[0];
            
            if (file && file.type.startsWith('image/')) {
                console.log('🖼️ Thumbnail image selected:', file.name);
                
                const reader = new FileReader();
                reader.onload = function(e) {
                    const imageURL = e.target.result;
                    
                    // Find thumbnail preview area
                    const thumbnailSection = document.querySelector('#fileModeLayout .preview-container:last-child .mb-3');
                    if (thumbnailSection) {
                        thumbnailSection.innerHTML = `
                            <div class="alert alert-primary border-0 rounded-3 mb-3">
                                <i class="fas fa-upload me-2"></i>New thumbnail uploaded
                            </div>
                            <img src="${imageURL}" alt="New Thumbnail Preview" class="img-fluid rounded-3 shadow-sm" style="max-height: 150px; border: 3px solid #007bff; width: 100%; object-fit: cover;">
                        `;
                        console.log('✅ Thumbnail preview updated with uploaded image');
                    }
                };
                reader.readAsDataURL(file);
            }
        });
        
        function updateFileVideoPreview(videoURL, fileName) {
            const previewArea = document.getElementById('fileVideoPreviewArea');
            if (previewArea) {
                previewArea.innerHTML = `
                    <video controls class="w-100 h-100" style="border-radius: 12px;">
                        <source src="${videoURL}" type="video/mp4">
                        Your browser does not support the video tag.
                    </video>
                `;
                console.log('✅ Video preview updated for:', fileName);
            }
        }
        
        function detectVideoDurationFromFile(file, videoURL) {
            console.log('⏱️ Detecting duration from uploaded file...');
            
            const video = document.createElement('video');
            video.preload = 'metadata';
            video.muted = true;
            
            video.onloadedmetadata = function() {
                console.log('📊 Video metadata loaded, duration in seconds:', video.duration);
                
                if (video.duration && video.duration > 0) {
                    const duration = Math.round(video.duration);
                    const minutes = Math.floor(duration / 60);
                    const seconds = duration % 60;
                    const formattedDuration = `${minutes}:${seconds.toString().padStart(2, '0')}`;
                    
                    // Update duration field with force
                    const durationField = document.getElementById('duration');
                    if (durationField) {
                        durationField.value = formattedDuration;
                        durationField.style.backgroundColor = '#d4edda'; // Green background to show it's updated
                        
                        // Remove highlight after 3 seconds
                        setTimeout(() => {
                            durationField.style.backgroundColor = '';
                        }, 3000);
                    }
                    
                    console.log('✅ File duration detected and updated:', formattedDuration);
                    
                    // Show success notification
                    showDurationSuccess(formattedDuration, 'File Analysis');
                } else {
                    console.log('❌ Invalid duration detected:', video.duration);
                    showDurationError('Could not detect valid duration from uploaded file');
                }
            };
            
            video.onerror = function(error) {
                console.log('❌ Could not detect duration from file:', error);
                showDurationError('Could not detect duration from uploaded file');
            };
            
            video.ontimeupdate = function() {
                // Stop after getting metadata
                if (video.duration > 0) {
                    video.pause();
                    video.removeAttribute('src');
                    video.load();
                }
            };
            
            video.src = videoURL;
            video.load();
        }
        
        function generateVideoThumbnail(videoURL) {
            console.log('🖼️ Generating thumbnail preview...');
            
            const video = document.createElement('video');
            const canvas = document.createElement('canvas');
            const ctx = canvas.getContext('2d');
            
            video.setAttribute('crossorigin', 'anonymous');
            video.muted = true; // Required for autoplay in some browsers
            
            video.onloadedmetadata = function() {
                console.log('📹 Video metadata loaded, duration:', video.duration);
                // Set canvas dimensions  
                canvas.width = video.videoWidth || 640;
                canvas.height = video.videoHeight || 360;
                
                // Seek to 25% of video for thumbnail (usually better than middle)
                video.currentTime = Math.max(1, video.duration * 0.25);
            };
            
            video.onseeked = function() {
                console.log('🎬 Video seeked, generating thumbnail...');
                
                try {
                    // Draw video frame to canvas
                    ctx.drawImage(video, 0, 0, canvas.width, canvas.height);
                    
                    // Convert to data URL directly
                    const thumbnailDataURL = canvas.toDataURL('image/jpeg', 0.8);
                    
                    // Show and update the dedicated thumbnail preview section
                    const thumbnailPreviewSection = document.getElementById('thumbnailPreviewSection');
                    const thumbnailPreviewContainer = document.getElementById('thumbnailPreviewContainer');
                    
                    if (thumbnailPreviewSection && thumbnailPreviewContainer) {
                        // Show the thumbnail preview section
                        thumbnailPreviewSection.style.display = 'block';
                        
                        // Update the preview container with the generated thumbnail
                        thumbnailPreviewContainer.innerHTML = `
                            <div class="alert alert-success border-0 rounded-3 mb-3">
                                <i class="fas fa-magic me-2"></i>Auto-generated from video at 25% mark
                            </div>
                            <img src="${thumbnailDataURL}" alt="Auto-generated Thumbnail" class="img-fluid rounded-3 shadow-sm mx-auto d-block" style="max-height: 200px; max-width: 300px; border: 3px solid #27ae60; object-fit: cover;">
                        `;
                        console.log('✅ Thumbnail preview updated in dedicated section');
                    } else {
                        console.log('❌ Could not find dedicated thumbnail preview section');
                    }
                } catch (error) {
                    console.log('❌ Error generating thumbnail:', error);
                }
                
                // Clean up
                URL.revokeObjectURL(videoURL);
            };
            
            video.onerror = function(error) {
                console.log('❌ Video error during thumbnail generation:', error);
            };
            
            video.src = videoURL;
            video.load();
        }
        
        function showDurationSuccess(duration, method) {
            // Create or update success notification
            let successDiv = document.getElementById('fileDurationSuccess');
            if (!successDiv) {
                successDiv = document.createElement('div');
                successDiv.id = 'fileDurationSuccess';
                successDiv.className = 'alert alert-success border-0 rounded-3 mt-3';
                
                // Insert after duration input
                const durationInput = document.getElementById('duration');
                if (durationInput && durationInput.parentNode) {
                    durationInput.parentNode.insertBefore(successDiv, durationInput.nextSibling);
                }
            }
            
            successDiv.innerHTML = `
                <div class="d-flex align-items-center">
                    <i class="fas fa-check-circle text-success me-2"></i>
                    <span><strong>Duration detected:</strong> ${duration} (${method})</span>
                </div>
            `;
            
            // Auto-hide after 5 seconds
            setTimeout(() => {
                if (successDiv && successDiv.parentNode) {
                    successDiv.remove();
                }
            }, 5000);
        }
        
        function showDurationError(message) {
            // Create or update error notification  
            let errorDiv = document.getElementById('fileDurationError');
            if (!errorDiv) {
                errorDiv = document.createElement('div');
                errorDiv.id = 'fileDurationError';
                errorDiv.className = 'alert alert-warning border-0 rounded-3 mt-3';
                
                // Insert after duration input
                const durationInput = document.getElementById('duration');
                if (durationInput && durationInput.parentNode) {
                    durationInput.parentNode.insertBefore(errorDiv, durationInput.nextSibling);
                }
            }
            
            errorDiv.innerHTML = `
                <div class="d-flex align-items-center">
                    <i class="fas fa-exclamation-triangle text-warning me-2"></i>
                    <span>${message}</span>
                </div>
            `;
            
            // Auto-hide after 7 seconds
            setTimeout(() => {
                if (errorDiv && errorDiv.parentNode) {
                    errorDiv.remove();
                }
            }, 7000);
        }
        
        function suggestTitleFromFilename(filename) {
            console.log('💡 Suggesting title from filename:', filename);
            
            // Clean filename to make a nice title
            let suggestedTitle = filename
                .replace(/\.[^/.]+$/, '') // Remove file extension
                .replace(/[-_]/g, ' ')     // Replace hyphens and underscores with spaces
                .replace(/\s+/g, ' ')      // Replace multiple spaces with single space
                .trim();                   // Remove leading/trailing spaces
            
            // Capitalize first letter of each word
            suggestedTitle = suggestedTitle.replace(/\w\S*/g, (txt) => 
                txt.charAt(0).toUpperCase() + txt.substr(1).toLowerCase()
            );
            
            const titleField = document.getElementById('title');
            if (titleField && suggestedTitle) {
                // Show notification asking if user wants to update title
                const existingTitle = titleField.value.trim();
                
                if (!existingTitle || existingTitle === 'TrackDuck web feedback tool explained in 1min') {
                    // Empty or default title - suggest new one
                    titleField.value = suggestedTitle;
                    titleField.style.backgroundColor = '#fff3cd'; // Yellow highlight
                    
                    // Show notification
                    showTitleSuggestion(suggestedTitle, 'updated');
                } else {
                    // Existing title - ask if user wants to replace
                    showTitleSuggestion(suggestedTitle, 'suggested');
                }
                
                // Remove highlight after 3 seconds
                setTimeout(() => {
                    titleField.style.backgroundColor = '';
                }, 3000);
            }
        }
        
        function showTitleSuggestion(suggestedTitle, action) {
            // Create or update title suggestion notification
            let suggestionDiv = document.getElementById('titleSuggestion');
            if (!suggestionDiv) {
                suggestionDiv = document.createElement('div');
                suggestionDiv.id = 'titleSuggestion';
                suggestionDiv.className = 'alert alert-info border-0 rounded-3 mt-2';
                
                // Insert after title input
                const titleInput = document.getElementById('title');
                if (titleInput && titleInput.parentNode) {
                    titleInput.parentNode.insertBefore(suggestionDiv, titleInput.nextSibling);
                }
            }
            
            const message = action === 'updated' 
                ? `Title auto-filled from filename: "${suggestedTitle}"`
                : `Suggested title from filename: "${suggestedTitle}"`;
            
            suggestionDiv.innerHTML = `
                <div class="d-flex align-items-center justify-content-between">
                    <span>
                        <i class="fas fa-lightbulb text-info me-2"></i>
                        ${message}
                    </span>
                    ${action === 'suggested' ? `
                        <button type="button" class="btn btn-sm btn-outline-info ms-2" onclick="document.getElementById('title').value='${suggestedTitle}'; document.getElementById('titleSuggestion').remove();">
                            Use This Title
                        </button>
                    ` : ''}
                </div>
            `;
            
            // Auto-hide after 8 seconds
            setTimeout(() => {
                if (suggestionDiv && suggestionDiv.parentNode) {
                    suggestionDiv.remove();
                }
            }, 8000);
        }
        
        // Form submission with loading spinner
        $('form').on('submit', function(e) {
            const saveButton = $('#saveChangesBtn');
            const originalContent = saveButton.html();
            
            // Show loading state
            saveButton.prop('disabled', true);
            saveButton.html(`
                <span class="spinner-border spinner-border-sm me-2" role="status" aria-hidden="true"></span>
                Saving Changes...
            `);
            
            // Optional: Add visual feedback with button color change
            saveButton.removeClass('btn-save').addClass('btn-save-loading');
            
            console.log('💾 Form submission started with loading state');
            
            // Note: The form will submit normally, and the page will redirect/refresh
            // If the submission fails or returns to the same page, the original state will be restored automatically
        });
        
        // Enhanced video detection (same system as upload-video.php)
        $('#detectVideoBtn').on('click', function() {
            const videoUrl = $('#videoUrl').val().trim();
            if (!videoUrl) {
                alert('Please enter a video URL first.');
                return;
            }
            
            const btn = $(this);
            const originalText = btn.html();
            btn.html('<i class="fas fa-spinner fa-spin me-2"></i>Detecting...').prop('disabled', true);
            
            // Show duration detection status
            $('#durationDetectionCard').show();
            showDurationDetection('Detecting video information...', 'Initial Detection');
            
            $.ajax({
                url: 'ajax/detect_video.php',
                method: 'POST',
                dataType: 'json',
                data: { video_url: videoUrl },
                success: function(response) {
                    if (response.success && response.data && response.data.is_valid) {
                        // Update hidden fields with detected data
                        $('#detectedVideoUrl').val(response.data.url || videoUrl);
                        $('#detectedPlatformInput').val(response.data.platform || '');
                        $('#videoIdExternal').val(response.data.video_id || '');
                        $('#embedCodeInput').val(response.data.embed_code || '');
                        
                        // Update video preview dynamically
                        updateVideoPreview(response.data);
                        
                        // Update title if detected (auto-fill for teacher convenience)
                        if (response.data.title) {
                            $('#title').val(response.data.title);
                            console.log('✅ Title auto-filled:', response.data.title);
                        }
                        
                        // Enhanced duration detection for URL videos
                        const supportedPlatforms = ['youtube', 'vimeo', 'tiktok', 'dailymotion', 'twitch', 'streamable'];
                        if (supportedPlatforms.includes(response.data.platform.toLowerCase())) {
                            detectVideoDuration(response.data.video_id, response.data.platform);
                        } else {
                            showDurationError(`Duration detection not yet supported for ${response.data.platform}`);
                        }
                        
                        alert('✅ Video detected successfully! Title and duration auto-filled.');
                    } else {
                        alert('❌ ' + (response.message || 'Video detection failed'));
                        hideDurationDetection();
                    }
                },
                error: function() {
                    alert('❌ Error detecting video. Please try again.');
                    hideDurationDetection();
                },
                complete: function() {
                    btn.html(originalText).prop('disabled', false);
                }
            });
        });
        
        // Enhanced duration detection functions (same as upload-video.php)
        function detectVideoDuration(videoId, platform) {
            console.log('🎬 Starting duration detection for:', platform, videoId);
            
            if (platform.toLowerCase() === 'youtube') {
                detectYouTubeDuration(videoId);
            } else {
                showDurationError(`Duration detection for ${platform} is not yet implemented`);
            }
        }
        
        function detectYouTubeDuration(videoId) {
            console.log('🎬 Starting YouTube duration detection for video ID:', videoId);
            showDurationDetection('Fetching YouTube video info...', 'YouTube Detection');
            
            // Method 1: Try client-side iframe embedding method (most reliable)
            tryYouTubeIframeDuration(videoId)
                .then(duration => {
                    if (duration) {
                        showDurationResult(duration, 'YouTube Iframe API', 'Success');
                        console.log('✅ YouTube iframe duration detected:', duration);
                    } else {
                        throw new Error('Iframe method failed');
                    }
                })
                .catch(() => {
                    // Method 2: Fallback to server-side detection
                    console.log('🔄 Iframe method failed, trying server-side detection...');
                    
                    $.ajax({
                        url: 'ajax/detect_youtube_duration.php',
                        method: 'POST',
                        dataType: 'json',
                        data: { video_id: videoId },
                        success: function(response) {
                            if (response.success && response.duration) {
                                showDurationResult(response.duration, response.method || 'YouTube API', 'Success');
                                console.log('✅ YouTube server-side duration detected:', response.duration);
                            } else {
                                showDurationError(response.message || 'Failed to get YouTube video duration');
                                console.log('❌ YouTube duration detection failed:', response.message);
                            }
                        },
                        error: function(xhr, status, error) {
                            console.error('❌ YouTube duration AJAX error:', error);
                            showDurationError('Duration detection failed. Please enter duration manually.');
                        },
                        timeout: 10000 // 10 second timeout
                    });
                });
        }
        
        function tryYouTubeIframeDuration(videoId) {
            return new Promise((resolve, reject) => {
                console.log('🎬 Using YouTube iframe API for duration detection...');
                
                // Load YouTube iframe API if not already loaded
                if (typeof YT === 'undefined' || typeof YT.Player === 'undefined') {
                    console.log('🔄 Loading YouTube iframe API...');
                    
                    // Create script tag to load YouTube iframe API
                    const script = document.createElement('script');
                    script.src = 'https://www.youtube.com/iframe_api';
                    script.onload = () => {
                        console.log('✅ YouTube iframe API loaded');
                        // Wait for API to be ready
                        window.onYouTubeIframeAPIReady = () => {
                            createPlayerAndGetDuration(videoId, resolve, reject);
                        };
                    };
                    script.onerror = () => {
                        console.log('❌ Failed to load YouTube iframe API');
                        reject('API load failed');
                    };
                    document.head.appendChild(script);
                } else {
                    // API already loaded
                    createPlayerAndGetDuration(videoId, resolve, reject);
                }
            });
        }
        
        function createPlayerAndGetDuration(videoId, resolve, reject) {
            console.log('🎬 Creating YouTube player to get duration...');
            
            // Create a temporary container for the player
            const tempContainer = document.createElement('div');
            tempContainer.id = 'temp-youtube-player-' + Math.random().toString(36).substr(2, 9);
            tempContainer.style.position = 'absolute';
            tempContainer.style.left = '-9999px';
            tempContainer.style.width = '1px';
            tempContainer.style.height = '1px';
            document.body.appendChild(tempContainer);
            
            let player;
            let timeout = setTimeout(() => {
                console.log('⏰ YouTube duration detection timeout');
                cleanup();
                reject('Timeout');
            }, 15000);
            
            function cleanup() {
                if (player && typeof player.destroy === 'function') {
                    try {
                        player.destroy();
                    } catch (e) {
                        console.log('Player destroy error:', e);
                    }
                }
                if (tempContainer && tempContainer.parentNode) {
                    tempContainer.parentNode.removeChild(tempContainer);
                }
                clearTimeout(timeout);
            }
            
            try {
                player = new YT.Player(tempContainer.id, {
                    height: '1',
                    width: '1',
                    videoId: videoId,
                    playerVars: {
                        'autoplay': 0,
                        'controls': 0,
                        'modestbranding': 1,
                        'rel': 0,
                        'showinfo': 0
                    },
                    events: {
                        'onReady': function(event) {
                            console.log('🎬 YouTube player ready, getting duration...');
                            
                            try {
                                const duration = player.getDuration();
                                console.log('🔍 Raw duration from getDuration():', duration);
                                
                                if (duration && duration > 0) {
                                    const formattedDuration = formatDurationFromSeconds(Math.round(duration));
                                    console.log('✅ YouTube duration detected:', formattedDuration);
                                    cleanup();
                                    resolve(formattedDuration);
                                } else {
                                    console.log('❌ Invalid duration from YouTube API:', duration);
                                    cleanup();
                                    reject('Invalid duration');
                                }
                            } catch (e) {
                                console.log('❌ Error getting duration:', e);
                                cleanup();
                                reject('getDuration() failed');
                            }
                        },
                        'onError': function(event) {
                            console.log('❌ YouTube player error:', event.data);
                            cleanup();
                            reject('Player error: ' + event.data);
                        },
                        'onStateChange': function(event) {
                            // If player is ready but onReady didn't fire, try getting duration here
                            if (event.data === YT.PlayerState.CUED || event.data === YT.PlayerState.PAUSED) {
                                try {
                                    const duration = player.getDuration();
                                    if (duration && duration > 0) {
                                        const formattedDuration = formatDurationFromSeconds(Math.round(duration));
                                        console.log('✅ YouTube duration from state change:', formattedDuration);
                                        cleanup();
                                        resolve(formattedDuration);
                                    }
                                } catch (e) {
                                    // Continue waiting
                                }
                            }
                        }
                    }
                });
            } catch (e) {
                console.log('❌ Error creating YouTube player:', e);
                cleanup();
                reject('Player creation failed');
            }
        }
        
        // Duration detection UI functions
        function showDurationDetection(message, method) {
            $('#durationSpinner').show();
            $('#durationText').text(message);
            $('#durationResult').hide();
            $('#detectionStatus').removeClass('bg-success bg-danger').addClass('bg-info').text('Processing');
        }
        
        function showDurationResult(duration, method, status) {
            $('#durationSpinner').hide();
            $('#durationText').text('Duration detection completed');
            $('#detectedDuration').text(duration);
            $('#detectionMethod').text(method);
            $('#detectionStatus').removeClass('bg-info bg-danger').addClass('bg-success').text(status);
            $('#durationResult').show();
            
            // Auto-fill the duration field
            $('#duration').val(duration);
            $('#detectedDurationInput').val(duration);
            
            console.log('✅ Duration auto-filled:', duration);
        }
        
        function showDurationError(message) {
            $('#durationSpinner').hide();
            $('#durationText').text(message);
            $('#detectionStatus').removeClass('bg-info bg-success').addClass('bg-danger').text('Failed');
            $('#durationResult').hide();
        }
        
        function hideDurationDetection() {
            $('#durationDetectionCard').hide();
        }
        
        function formatDurationFromSeconds(seconds) {
            const hours = Math.floor(seconds / 3600);
            const minutes = Math.floor((seconds % 3600) / 60);
            const secs = seconds % 60;
            
            if (hours > 0) {
                return `${hours}:${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
            } else {
                return `${minutes}:${secs.toString().padStart(2, '0')}`;
            }
        }
        
        // Function to update video preview for file mode
        function updateVideoPreviewForFileMode() {
            const previewContainer = $('.ratio.ratio-16x9');
            
            <?php if ($video['video_type'] == 'file' && !empty($video['video_url'])): ?>
            // Show existing file video if available
            const filePreviewHtml = `
                <video controls class="w-100 h-100">
                    <source src="../<?php echo htmlspecialchars($video['video_url']); ?>" type="video/mp4">
                    Your browser does not support the video tag.
                </video>
            `;
            previewContainer.html(filePreviewHtml);
            <?php else: ?>
            // Show file upload placeholder
            const filePlaceholderHtml = `
                <div class="d-flex align-items-center justify-content-center h-100 bg-light border border-dashed rounded">
                    <div class="text-center">
                        <i class="fas fa-cloud-upload-alt fa-4x text-muted mb-3"></i>
                        <h5 class="text-muted mb-2">File Upload Mode</h5>
                        <p class="text-muted mb-0">Upload a video file to see preview here</p>
                    </div>
                </div>
            `;
            previewContainer.html(filePlaceholderHtml);
            <?php endif; ?>
        }
        
        // Function to update video preview for URL mode
        function updateVideoPreviewForUrlMode() {
            const previewContainer = $('.ratio.ratio-16x9');
            
            <?php if ($video['video_type'] == 'url' && $video['platform'] == 'youtube' && !empty($video['video_id_external'])): ?>
            // Show existing YouTube video if available
            const youtubePreviewHtml = `
                <iframe class="w-100 h-100" 
                        src="https://www.youtube.com/embed/<?php echo htmlspecialchars($video['video_id_external']); ?>?rel=0&showinfo=0" 
                        frameborder="0" 
                        allowfullscreen>
                </iframe>
            `;
            previewContainer.html(youtubePreviewHtml);
            <?php elseif ($video['video_type'] == 'url' && $video['platform'] == 'vimeo' && !empty($video['video_id_external'])): ?>
            // Show existing Vimeo video if available
            const vimeoPreviewHtml = `
                <iframe class="w-100 h-100" 
                        src="https://player.vimeo.com/video/<?php echo htmlspecialchars($video['video_id_external']); ?>" 
                        frameborder="0" 
                        allowfullscreen>
                </iframe>
            `;
            previewContainer.html(vimeoPreviewHtml);
            <?php else: ?>
            // Show URL embed placeholder
            const urlPlaceholderHtml = `
                <div class="d-flex align-items-center justify-content-center h-100 bg-light border border-dashed rounded">
                    <div class="text-center">
                        <i class="fas fa-globe fa-4x text-muted mb-3"></i>
                        <h5 class="text-muted mb-2">URL Embed Mode</h5>
                        <p class="text-muted mb-0">Enter a video URL and click 'Update Video' to see preview</p>
                    </div>
                </div>
            `;
            previewContainer.html(urlPlaceholderHtml);
            <?php endif; ?>
        }
        
        // Function to update video preview dynamically (for detection results)
        function updateVideoPreview(videoData) {
            const previewContainer = $('.ratio.ratio-16x9');
            let previewHtml = '';
            
            if (videoData.platform === 'youtube' && videoData.video_id) {
                previewHtml = `<iframe class="w-100 h-100" 
                                 src="https://www.youtube.com/embed/${videoData.video_id}?rel=0&showinfo=0" 
                                 frameborder="0" 
                                 allowfullscreen>
                               </iframe>`;
            } 
            else if (videoData.platform === 'vimeo' && videoData.video_id) {
                previewHtml = `<iframe class="w-100 h-100" 
                                 src="https://player.vimeo.com/video/${videoData.video_id}" 
                                 frameborder="0" 
                                 allowfullscreen>
                               </iframe>`;
            }
            else {
                previewHtml = `<div class="d-flex align-items-center justify-content-center h-100 bg-light">
                                 <div class="text-center">
                                     <i class="fas fa-video fa-3x text-muted mb-2"></i>
                                     <p class="text-muted">Video preview updated</p>
                                 </div>
                               </div>`;
            }
            
            previewContainer.html(previewHtml);
        }
        
        // Initialize form state on page load
        const initialVideoType = $('input[name="video_type"]:checked').val();
        if (initialVideoType === 'file') {
            $('#fileUploadSection').show();
            $('#urlEmbedSection').hide();
            $('#thumbnailSection').show();
            $('#durationDetectionCard').hide(); // Hide duration detection for file uploads
            updateVideoPreviewForFileMode(); // Set correct preview for file mode
        } else if (initialVideoType === 'url') {
            $('#fileUploadSection').hide();
            $('#urlEmbedSection').show();
            $('#thumbnailSection').hide(); // Hide thumbnail for URL videos
            updateVideoPreviewForUrlMode(); // Set correct preview for URL mode
            // Duration detection card will show when detect button is clicked
        }
    });
    </script>

<?php
// Include the footer component
require_once 'components/footer.php';
?>
