<?php
require_once '../config/db_connect.php';

// Turn on all error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Create a simple HTML page
echo '<!DOCTYPE html>
<html>
<head>
    <title>Fix All Video Thumbnails</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .success { color: green; font-weight: bold; }
        .error { color: red; }
        .info { color: blue; }
        table { border-collapse: collapse; width: 100%; margin-top: 20px; }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
        th { background-color: #f2f2f2; }
        img { max-width: 150px; border: 1px solid #ddd; }
        .btn { 
            display: inline-block; 
            padding: 10px 20px; 
            background-color: #4CAF50; 
            color: white; 
            text-decoration: none; 
            border-radius: 4px; 
            cursor: pointer;
            border: none;
            font-size: 16px;
        }
    </style>
</head>
<body>
    <h1>Fix All Video Thumbnails</h1>';

// Check thumbnail directory
$thumbnail_dir = realpath(__DIR__ . '/../uploads/thumbnails/');
if (!$thumbnail_dir || !is_dir($thumbnail_dir)) {
    echo "<p class='error'>Thumbnail directory does not exist: ../uploads/thumbnails/</p>";
    echo "<p>Creating directory...</p>";
    mkdir(__DIR__ . '/../uploads/thumbnails/', 0777, true);
}

// Get all available thumbnails
$available_thumbnails = [];
if (is_dir($thumbnail_dir)) {
    $files = scandir($thumbnail_dir);
    foreach ($files as $file) {
        if ($file != '.' && $file != '..' && is_file($thumbnail_dir . '/' . $file)) {
            $available_thumbnails[] = $file;
        }
    }
}

echo "<h2>Available Thumbnails</h2>";
echo "<p>Found " . count($available_thumbnails) . " thumbnail files in uploads/thumbnails/</p>";

// Process form submission
if (isset($_POST['fix_thumbnails'])) {
    $updated = 0;
    $errors = [];
    
    // Get all videos
    $videos_query = "SELECT video_id, title, thumbnail, video_url FROM videos";
    $videos_result = $conn->query($videos_query);
    
    if ($videos_result && $videos_result->num_rows > 0) {
        while ($video = $videos_result->fetch_assoc()) {
            $video_id = $video['video_id'];
            $current_thumbnail = $video['thumbnail'];
            
            // Skip videos that already have working thumbnails
            $thumbnail_file_exists = false;
            if (!empty($current_thumbnail)) {
                // Check if the file exists
                $full_path = __DIR__ . '/../' . $current_thumbnail;
                $thumbnail_file_exists = file_exists($full_path);
            }
            
            // Only update videos without thumbnails or with missing files
            if (empty($current_thumbnail) || !$thumbnail_file_exists) {
                // Try to find a matching thumbnail
                $potential_thumbnail = null;
                
                // Try to extract ID from video URL
                $video_file = basename($video['video_url']);
                $video_id_part = '';
                
                if (preg_match('/video_([a-f0-9]+)\./', $video_file, $matches)) {
                    $video_id_part = $matches[1];
                }
                
                // Search for matching thumbnails
                foreach ($available_thumbnails as $thumbnail) {
                    // Check for matching ID parts
                    if (!empty($video_id_part) && strpos($thumbnail, $video_id_part) !== false) {
                        $potential_thumbnail = $thumbnail;
                        break;
                    }
                }
                
                // If no matching thumbnail, use the first one from available thumbnails
                if (empty($potential_thumbnail) && !empty($available_thumbnails)) {
                    // Pick one based on video_id (for consistent selection)
                    $index = $video_id % count($available_thumbnails);
                    $potential_thumbnail = $available_thumbnails[$index];
                }
                
                // Update the database
                if ($potential_thumbnail) {
                    $new_thumbnail_path = 'uploads/thumbnails/' . $potential_thumbnail;
                    $update_query = "UPDATE videos SET thumbnail = ? WHERE video_id = ?";
                    $stmt = $conn->prepare($update_query);
                    $stmt->bind_param("si", $new_thumbnail_path, $video_id);
                    
                    if ($stmt->execute()) {
                        $updated++;
                    } else {
                        $errors[] = "Failed to update video ID {$video_id}: " . $stmt->error;
                    }
                }
            }
        }
        
        if ($updated > 0) {
            echo "<p class='success'>Successfully updated {$updated} videos with thumbnails!</p>";
        } else {
            echo "<p class='info'>No videos needed updating.</p>";
        }
        
        foreach ($errors as $error) {
            echo "<p class='error'>{$error}</p>";
        }
    }
    
    // Force refresh page to show updated results
    echo "<meta http-equiv='refresh' content='2;url=fix_all_thumbnails.php'>";
}

// Display current videos and their thumbnails
echo "<h2>Current Video Data</h2>";

$videos_query = "SELECT video_id, title, thumbnail, video_url FROM videos ORDER BY video_id DESC";
$videos_result = $conn->query($videos_query);

if ($videos_result && $videos_result->num_rows > 0) {
    echo "<table>";
    echo "<tr><th>ID</th><th>Title</th><th>Thumbnail Path</th><th>Status</th><th>Preview</th></tr>";
    
    while ($video = $videos_result->fetch_assoc()) {
        $status = "";
        $preview_html = "";
        
        echo "<tr>";
        echo "<td>{$video['video_id']}</td>";
        echo "<td>" . htmlspecialchars($video['title']) . "</td>";
        echo "<td>" . ($video['thumbnail'] ? htmlspecialchars($video['thumbnail']) : "<span class='error'>NULL</span>") . "</td>";
        
        if (empty($video['thumbnail'])) {
            $status = "<span class='error'>Missing thumbnail</span>";
        } else {
            $thumbnail_path = __DIR__ . '/../' . $video['thumbnail'];
            if (file_exists($thumbnail_path)) {
                $status = "<span class='success'>OK</span>";
                $preview_html = "<img src='../{$video['thumbnail']}' alt='Thumbnail'>";
            } else {
                $status = "<span class='error'>File not found</span>";
            }
        }
        
        echo "<td>{$status}</td>";
        echo "<td>{$preview_html}</td>";
        echo "</tr>";
    }
    
    echo "</table>";
    
    // Add form for fixing thumbnails
    echo "<h2>Fix Missing Thumbnails</h2>";
    echo "<form method='post'>";
    echo "<p>Click the button below to assign thumbnails to all videos without valid thumbnails:</p>";
    echo "<button type='submit' name='fix_thumbnails' class='btn'>Fix All Thumbnails</button>";
    echo "</form>";
} else {
    echo "<p>No videos found in the database.</p>";
}

// Path tests
echo "<h2>Path Tests</h2>";
$test_dirs = [
    ['path' => realpath(__DIR__), 'label' => 'Current script directory'],
    ['path' => realpath(__DIR__ . '/../uploads/thumbnails/'), 'label' => 'Thumbnails directory'],
    ['path' => realpath(__DIR__ . '/../uploads/videos/'), 'label' => 'Videos directory'],
    ['path' => realpath(__DIR__ . '/../assets/images/'), 'label' => 'Assets images directory']
];

foreach ($test_dirs as $dir) {
    echo "<p><strong>{$dir['label']}:</strong> ";
    if ($dir['path']) {
        echo "{$dir['path']} - <span class='success'>Exists</span>";
    } else {
        echo "<span class='error'>Not found</span>";
    }
    echo "</p>";
}

echo '</body></html>';
?>
