// Form validation function
function validateForm() {
    let isValid = true;
    const errors = [];
    
    // Required fields validation
    const requiredFields = [
        { id: 'courseTitle', name: 'Course title' },
        { id: 'courseCategory', name: 'Category' },
        { id: 'courseDescription', name: 'Description' },
        { id: 'coursePrice', name: 'Price' },
        { id: 'courseDuration', name: 'Duration' }
    ];
    
    // Validate required fields
    requiredFields.forEach(field => {
        const $field = $('#' + field.id);
        if ($field.val() === '') {
            isValid = false;
            errors.push(`Please enter ${field.name}`);
            $field.addClass('is-invalid');
        } else {
            $field.removeClass('is-invalid');
        }
    });
    
    // Thumbnail validation
    const thumbnailFile = $('#courseThumbnail')[0].files[0];
    // Skip thumbnail validation if we have base64 data already
    const hasBase64 = $('#thumbnail_base64').length > 0 && $('#thumbnail_base64').val() !== '';
    
    if (!thumbnailFile && !hasBase64) {
        isValid = false;
        errors.push('Course thumbnail is required');
        $('#courseThumbnail').addClass('is-invalid');
    } else if (thumbnailFile) {
        // Check file type
        if (!['image/jpeg', 'image/png', 'image/jpg'].includes(thumbnailFile.type)) {
            isValid = false;
            errors.push('Invalid thumbnail format. Please use JPG, JPEG or PNG.');
            $('#courseThumbnail').addClass('is-invalid');
        } else {
            // Also check file size - limit to 1MB
            const maxSizeBytes = 1 * 1024 * 1024; // 1MB
            if (thumbnailFile.size > maxSizeBytes) {
                isValid = false;
                errors.push('Thumbnail must be smaller than 1MB. Please resize your image.');
                $('#courseThumbnail').addClass('is-invalid');
            } else {
                $('#courseThumbnail').removeClass('is-invalid');
            }
        }
    }
    
    // Price validation
    const price = $('#coursePrice').val();
    if (price && (isNaN(price) || parseFloat(price) < 0)) {
        isValid = false;
        errors.push('Price must be a positive number');
        $('#coursePrice').addClass('is-invalid');
    }
    
    // Duration validation
    const duration = $('#courseDuration').val();
    if (duration && (isNaN(duration) || parseInt(duration) < 1)) {
        isValid = false;
        errors.push('Duration must be a positive number');
        $('#courseDuration').addClass('is-invalid');
    }
    
    return { isValid, errors };
}

// Form submission handler
$('#createCourseForm').on('submit', function(e) {
    e.preventDefault();
    
    // Prevent double submission
    if ($(this).data('submitting')) {
        return false;
    }
    $(this).data('submitting', true);
    
    const $form = $(this);
    const $submitBtn = $('#createCourseBtn');
    const $spinner = $('#submitSpinner');
    const $messages = $('#formMessages');
    const $loadingOverlay = $('#loadingOverlay');
    
    // Show loading overlay and disable submit button
    $loadingOverlay.removeClass('d-none');
    $submitBtn.prop('disabled', true);
    $spinner.show();
    
    // Clear previous messages
    $messages.empty();
    
    // Validate form
    const validation = validateForm();
    if (!validation.isValid) {
        // Show validation errors
        let errorHtml = `
            <div class="alert alert-danger">
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                <ul class="mb-0">
        `;
        
        validation.errors.forEach(err => {
            errorHtml += `<li>${err}</li>`;
        });
        
        errorHtml += `
                </ul>
            </div>
        `;
        
        $messages.html(errorHtml);
        
        // Reset form submission state
        $submitBtn.prop('disabled', false);
        $spinner.hide();
        $loadingOverlay.addClass('d-none');
        $form.data('submitting', false);
        
        // Scroll to messages
        $('html, body').animate({
            scrollTop: $messages.offset().top - 100
        }, 300);
        
        return false;
    }
    
    // Process form submission
    processFormSubmission();
    
    // Local function to handle form submission
    function processFormSubmission() {
        // Convert thumbnail to base64 if present
        const thumbnailFile = $('#courseThumbnail')[0].files[0];
        if (thumbnailFile) {
            console.log('Processing thumbnail: ' + thumbnailFile.name + ' (' + thumbnailFile.size + ' bytes)');
            
            // Show a processing message
            $messages.html(`
                <div class="alert alert-info">
                    <i class="fas fa-spinner fa-spin"></i> Processing image, please wait...
                </div>
            `);
            
            // Process the thumbnail file
            const reader = new FileReader();
            reader.readAsDataURL(thumbnailFile);
            
            reader.onload = function() {
                console.log('Image converted to base64');
                
                // Create a hidden input with the base64 data
                if ($('#thumbnail_base64').length === 0) {
                    $('<input>').attr({
                        type: 'hidden',
                        id: 'thumbnail_base64',
                        name: 'thumbnail_base64',
                        value: reader.result
                    }).appendTo($form);
                } else {
                    $('#thumbnail_base64').val(reader.result);
                }
                
                // Also save filename
                if ($('#thumbnail_filename').length === 0) {
                    $('<input>').attr({
                        type: 'hidden',
                        id: 'thumbnail_filename',
                        name: 'thumbnail_filename',
                        value: thumbnailFile.name
                    }).appendTo($form);
                } else {
                    $('#thumbnail_filename').val(thumbnailFile.name);
                }
                
                // Continue with AJAX form submission
                submitFormViaAjax();
            };
            
            reader.onerror = function(error) {
                console.error('Error reading file:', error);
                $messages.html(`
                    <div class="alert alert-danger">
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        <strong>Error!</strong> Could not process the image. Please try another image.
                    </div>
                `);
                $submitBtn.prop('disabled', false);
                $spinner.hide();
                $loadingOverlay.addClass('d-none');
                $form.data('submitting', false);
            };
        } else {
            // No new file selected but we may have base64 already
            submitFormViaAjax();
        }
    }
    
    // Function to submit the form via AJAX
    function submitFormViaAjax() {
        const formData = new FormData($form[0]);
        
        // Send form data to server
        $.ajax({
            url: 'process-create-course.php',
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            dataType: 'json',
            success: function(response) {
                // Handle successful response
                if (response.success) {
                    // Show success message
                    Toastify({
                        text: "Course created successfully!",
                        duration: 4000,
                        close: true,
                        gravity: "top",
                        position: "center",
                        style: {
                            background: "#1cc88a"
                        },
                        stopOnFocus: true
                    }).showToast();
                    
                    // Redirect to courses page after a delay
                    setTimeout(function() {
                        window.location.href = 'courses.php';
                    }, 2000);
                } else {
                    // Handle errors
                    let errorMessage = "Failed to create course. Please try again.";
                    let detailedErrors = [];
                    
                    // Try to parse the error response
                    if (response.errors && response.errors.length) {
                        detailedErrors = response.errors;
                        errorMessage = response.errors[0]; // Use the first error as the toast message
                    }
                    if (response.debug_info) {
                        console.log('Debug info:', response.debug_info);
                    }
                    
                    // Show error notification
                    Toastify({
                        text: errorMessage,
                        duration: 4000,
                        close: true,
                        gravity: "top",
                        position: "center",
                        style: {
                            background: "#dc3545"
                        },
                        stopOnFocus: true
                    }).showToast();
                    
                    // Generate error HTML
                    let errorHtml = `
                        <div class="alert alert-danger animate__animated animate__fadeIn">
                            <i class="fas fa-exclamation-triangle me-2"></i>`;
                    
                    if (detailedErrors.length > 0) {
                        errorHtml += `<ul class="mb-0">`;
                        detailedErrors.forEach(err => {
                            errorHtml += `<li>${err}</li>`;
                        });
                        errorHtml += `</ul>`;
                    } else {
                        errorHtml += errorMessage;
                    }
                    
                    errorHtml += '</div>';
                    
                    // Show error in form messages
                    $messages.html(errorHtml).show();
                    
                    // Scroll to messages
                    $('html, body').animate({
                        scrollTop: $messages.offset().top - 100
                    }, 300);
                }
            },
            error: function(xhr, status, error) {
                // Handle AJAX errors
                console.error('AJAX error:', status, error);
                
                let errorMessage = "Server error. Please try again later.";
                if (xhr.status === 400) {
                    errorMessage = "Invalid request. Please check form data.";
                } else if (xhr.status === 500) {
                    errorMessage = "Server error. Please try again later.";
                }
                
                // Show error notification
                Toastify({
                    text: errorMessage,
                    duration: 4000,
                    close: true,
                    gravity: "top",
                    position: "center",
                    style: {
                        background: "#dc3545"
                    },
                    stopOnFocus: true
                }).showToast();
                
                // Show error message
                $messages.html(`
                    <div class="alert alert-danger">
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        <strong>Error!</strong> ${errorMessage}
                        <div class="mt-2 small">Status: ${xhr.status} ${error}</div>
                    </div>
                `);
            },
            complete: function() {
                // Re-enable form submission
                $submitBtn.prop('disabled', false);
                $spinner.hide();
                $loadingOverlay.addClass('d-none');
                $form.data('submitting', false);
            }
        });
    }
});

// Attach a direct event handler to the form submit button
$('#createCourseBtn').on('click', function(e) {
    e.preventDefault();
    $('#createCourseForm').submit();
    return false;
});

// Initialize thumbnail preview
$('#courseThumbnail').on('change', function() {
    const file = this.files[0];
    const $thumbnailPreview = $('#thumbnailPreview');
    
    if (file) {
        // Check file type
        if (!['image/jpeg', 'image/png', 'image/jpg'].includes(file.type)) {
            $thumbnailPreview.html(`
                <div class="mt-2 text-danger">
                    <i class="fas fa-exclamation-circle"></i> 
                    Invalid file type. Please select a JPG, JPEG or PNG file.
                </div>
            `);
            return;
        }
        
        // Show loading indicator
        $thumbnailPreview.html(`
            <div class="mt-2">
                <div class="spinner-border spinner-border-sm text-primary" role="status">
                    <span class="visually-hidden">Loading...</span>
                </div>
                <span class="ms-2">Loading preview...</span>
            </div>
        `);
        
        // Read and display image
        const reader = new FileReader();
        reader.onload = function(e) {
            $thumbnailPreview.html(`
                <div class="mt-2">
                    <img src="${e.target.result}" class="img-thumbnail" style="max-width: 200px; max-height: 150px;">
                    <div class="text-muted small mt-1">${file.name} (${(file.size / 1024).toFixed(1)} KB)</div>
                </div>
            `);
        }
        reader.onerror = function() {
            $thumbnailPreview.html(`
                <div class="mt-2 text-danger">
                    <i class="fas fa-exclamation-circle"></i> 
                    Error loading preview.
                </div>
            `);
        }
        reader.readAsDataURL(file);
    } else {
        $thumbnailPreview.empty();
    }
});

// Add input validation highlighting
$('.form-control, .form-select').on('input change', function() {
    if ($(this).val() && $(this).val().trim() !== '') {
        $(this).removeClass('is-invalid');
    }
});
