/**
 * Clean Video Upload System for Panadite Academy
 * Simplified, reliable video upload functionality
 */

// Global variables
let currentDetectionRequest = null;
let urlDetectionTimeout = null;

// Initialize when DOM is ready
$(document).ready(function() {
    console.log('🎥 Clean Video Upload System initialized');
    
    // Initialize form
    initializeForm();
    
    // Setup video type switching
    setupVideoTypeSwitch();
    
    // Setup video detection
    setupVideoDetection();
    
    // Setup file upload
    setupFileUpload();
    
    // Setup form submission
    setupFormSubmission();
});

/**
 * Initialize the form with default settings
 */
function initializeForm() {
    // Set default to file upload
    showFileUpload();
    
    // Clear any previous data
    clearVideoPreview();
    
    console.log('✅ Form initialized');
}

/**
 * Setup video type switching (File vs URL)
 */
function setupVideoTypeSwitch() {
    // Handle radio button changes
    $('input[name="video_type"]').on('change', function() {
        const value = $(this).val();
        console.log('📻 Video type switched to:', value);
        
        if (value === 'url') {
            showUrlEmbed();
        } else {
            showFileUpload();
        }
    });
    
    // Handle label clicks
    $('.btn-group label').on('click', function() {
        const forValue = $(this).attr('for');
        console.log('🖱️ Label clicked for:', forValue);
        
        setTimeout(() => {
            const checkedValue = $('input[name="video_type"]:checked').val();
            if (checkedValue === 'url') {
                showUrlEmbed();
            } else {
                showFileUpload();
            }
        }, 50);
    });
}

/**
 * Show file upload section
 */
function showFileUpload() {
    $('#fileUploadSection').show();
    $('#urlEmbedSection').hide();
    
    // Update form requirements
    $('#videoFile').prop('required', true);
    $('#videoUrl').prop('required', false);
    
    // Clear URL data
    clearVideoPreview();
    
    console.log('✅ File upload mode activated');
}

/**
 * Show URL embed section
 */
function showUrlEmbed() {
    $('#fileUploadSection').hide();
    $('#urlEmbedSection').show();
    
    // Update form requirements
    $('#videoFile').prop('required', false).val('');
    $('#videoUrl').prop('required', true);
    
    // Hide file preview
    $('#previewContainer').hide();
    $('#progressContainer').hide();
    
    console.log('✅ URL embed mode activated');
}

/**
 * Setup video detection functionality
 */
function setupVideoDetection() {
    // Manual detect button
    $('#detectVideoBtn').on('click', function() {
        const videoUrl = $('#videoUrl').val().trim();
        
        if (!videoUrl) {
            showAlert('Please enter a video URL first.', 'warning');
            return;
        }
        
        detectVideo(videoUrl);
    });
    
    // Auto-detection on URL input
    $('#videoUrl').on('input paste', function() {
        clearTimeout(urlDetectionTimeout);
        
        // Cancel any ongoing request
        if (currentDetectionRequest) {
            currentDetectionRequest.abort();
            currentDetectionRequest = null;
        }
        
        const url = $(this).val().trim();
        
        // Clear previous results
        clearVideoPreview();
        
        // If URL looks valid, auto-detect after delay
        if (url.length > 10 && isValidVideoUrl(url)) {
            urlDetectionTimeout = setTimeout(() => {
                detectVideo(url);
            }, 1000);
        }
    });
}

/**
 * Check if URL looks like a valid video URL
 */
function isValidVideoUrl(url) {
    const validDomains = [
        'youtube.com', 'youtu.be', 'vimeo.com', 
        'facebook.com', 'tiktok.com', 'dailymotion.com', 'wistia.com'
    ];
    
    return validDomains.some(domain => url.includes(domain));
}

/**
 * Detect video from URL
 */
function detectVideo(videoUrl) {
    console.log('🚀 Detecting video:', videoUrl);
    
    // Update button state
    const detectBtn = $('#detectVideoBtn');
    const originalText = detectBtn.html();
    detectBtn.html('<i class="fas fa-spinner fa-spin me-2"></i>Detecting...').prop('disabled', true);
    
    // Make AJAX request
    currentDetectionRequest = $.ajax({
        url: 'ajax/detect_video.php',
        method: 'POST',
        dataType: 'json',
        data: { video_url: videoUrl },
        timeout: 15000,
        success: function(response) {
            console.log('✅ Detection response:', response);
            
            if (response.success && response.data && response.data.is_valid) {
                showVideoPreview(response.data);
                populateFormFields(response.data);
                showAlert(`✅ ${response.data.platform_name} video detected successfully!`, 'success');
            } else {
                showAlert(`❌ ${response.message || 'Video detection failed'}`, 'danger');
            }
        },
        error: function(xhr, status, error) {
            if (status !== 'abort') {
                console.error('❌ Detection error:', error);
                showAlert('❌ Error detecting video. Please try again.', 'danger');
            }
        },
        complete: function() {
            // Reset button
            detectBtn.html(originalText).prop('disabled', false);
            currentDetectionRequest = null;
        }
    });
}

/**
 * Show video preview
 */
function showVideoPreview(videoData) {
    console.log('Showing video preview:', videoData);
    
    // Create preview HTML
    const previewHtml = `
        <div class="card border-success mt-3">
            <div class="card-header bg-success text-white">
                <h6 class="mb-0">
                    <i class="fas fa-video me-2"></i>
                    ${videoData.platform_name} Video Detected
                </h6>
            </div>
            <div class="card-body">
                <div class="row">
                    ${videoData.thumbnail_url ? `
                        <div class="col-md-4">
                            <img src="${videoData.thumbnail_url}" 
                                 class="img-fluid rounded" 
                                 alt="Video thumbnail"
                                 style="max-height: 120px; width: 100%; object-fit: cover;">
                        </div>
                    ` : ''}
                    <div class="col-md-${videoData.thumbnail_url ? '8' : '12'}">
                        ${videoData.title ? `<h6 class="text-primary mb-2">${videoData.title}</h6>` : ''}
                        ${videoData.description ? `<p class="text-muted small mb-2">${videoData.description.substring(0, 100)}...</p>` : ''}
                        
                        <div class="d-flex gap-2 mb-2">
                            <span class="badge bg-primary">${videoData.platform_name}</span>
                            ${videoData.duration ? `<span class="badge bg-secondary">${formatDuration(videoData.duration)}</span>` : ''}
                        </div>
                        
                        <small class="text-muted">Video ID: <code>${videoData.video_id}</code></small>
                    </div>
                </div>
                
                <button type="button" class="btn btn-sm btn-outline-danger mt-2" onclick="clearVideoPreview()">
                    <i class="fas fa-times me-1"></i>Clear
                </button>
            </div>
        </div>
    `;
    
    $('#videoPreviewArea').html(previewHtml).show();
}

/**
 * Populate form fields with video data
 */
function populateFormFields(videoData) {
    // Store in hidden fields
    $('#detectedVideoUrl').val(videoData.original_url || '');
    $('#detectedPlatformInput').val(videoData.platform || '');
    $('#videoIdExternal').val(videoData.video_id || '');
    $('#embedCodeInput').val(videoData.embed_code || '');
    
    // Auto-fill title if empty
    if (videoData.title && !$('#videoTitle').val()) {
        $('#videoTitle').val(videoData.title);
    }
    
    console.log('✅ Form fields populated');
}

/**
 * Clear video preview
 */
function clearVideoPreview() {
    $('#videoPreviewArea').hide().empty();
    
    // Clear hidden fields
    $('#detectedVideoUrl').val('');
    $('#detectedPlatformInput').val('');
    $('#videoIdExternal').val('');
    $('#embedCodeInput').val('');
    
    console.log('✅ Video preview cleared');
}

/**
 * Setup file upload functionality
 */
function setupFileUpload() {
    // File input change
    $('#videoFile').on('change', function() {
        const file = this.files[0];
        if (file) {
            validateAndPreviewFile(file);
        } else {
            hideFilePreview();
        }
    });
    
    // Browse button
    $('#browseBtn').on('click', function() {
        $('#videoFile').click();
    });
    
    // Drag and drop
    setupDragAndDrop();
    
    // Thumbnail upload
    $('#thumbnailFile').on('change', function() {
        const file = this.files[0];
        if (file) {
            previewThumbnail(file);
        }
    });
}

/**
 * Validate and preview selected file
 */
function validateAndPreviewFile(file) {
    console.log('📁 File selected:', file.name);
    
    // Validate file type
    const validTypes = ['video/mp4', 'video/webm', 'video/quicktime', 'video/x-msvideo'];
    if (!validTypes.includes(file.type)) {
        showAlert('Invalid file type. Please select MP4, WebM, MOV, or AVI.', 'danger');
        $('#videoFile').val('');
        return;
    }
    
    // Update file info
    $('#fileName').text(file.name);
    $('#fileSize').text((file.size / (1024*1024)).toFixed(2) + ' MB');
    $('#uploadStatus').removeClass().addClass('badge bg-warning').text('Processing...');
    
    // Show progress container
    $('#progressContainer').slideDown(300);
    
    // Create video preview
    const videoURL = URL.createObjectURL(file);
    const videoPreview = $('#videoPreview');
    
    videoPreview.attr('src', videoURL).on('loadedmetadata', function() {
        // Get duration
        const duration = Math.round(this.duration);
        const minutes = Math.floor(duration / 60);
        const seconds = duration % 60;
        $('#videoDuration').text(`${minutes}m ${seconds}s`);
        
        // Update status
        $('#uploadStatus').removeClass().addClass('badge bg-success').text('Ready');
        $('#uploadProgress').css('width', '100%');
        
        // Show preview
        $('#previewContainer').slideDown(300);
        
        console.log('✅ File preview ready');
        
    }).on('error', function() {
        $('#uploadStatus').removeClass().addClass('badge bg-danger').text('Error');
        showAlert('Error loading video preview. File may be corrupted.', 'danger');
    });
}

/**
 * Hide file preview
 */
function hideFilePreview() {
    $('#progressContainer').slideUp(300);
    $('#previewContainer').slideUp(300);
}

/**
 * Setup drag and drop functionality
 */
function setupDragAndDrop() {
    const uploadArea = document.getElementById('uploadArea');
    
    if (!uploadArea) return;
    
    // Prevent default drag behaviors
    ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
        uploadArea.addEventListener(eventName, preventDefaults, false);
    });
    
    function preventDefaults(e) {
        e.preventDefault();
        e.stopPropagation();
    }
    
    // Highlight drop area
    ['dragenter', 'dragover'].forEach(eventName => {
        uploadArea.addEventListener(eventName, () => {
            uploadArea.classList.add('highlight');
        }, false);
    });
    
    ['dragleave', 'drop'].forEach(eventName => {
        uploadArea.addEventListener(eventName, () => {
            uploadArea.classList.remove('highlight');
        }, false);
    });
    
    // Handle dropped files
    uploadArea.addEventListener('drop', function(e) {
        const files = e.dataTransfer.files;
        if (files.length > 0) {
            document.getElementById('videoFile').files = files;
            $('#videoFile').trigger('change');
        }
    }, false);
}

/**
 * Preview thumbnail image
 */
function previewThumbnail(file) {
    // Validate image type
    const validTypes = ['image/jpeg', 'image/png', 'image/jpg'];
    if (!validTypes.includes(file.type)) {
        showAlert('Invalid file type. Please select JPEG or PNG.', 'danger');
        $('#thumbnailFile').val('');
        return;
    }
    
    // Show preview
    const thumbnailURL = URL.createObjectURL(file);
    $('#thumbnailImg').attr('src', thumbnailURL);
    $('#thumbnailPreview').show();
}

/**
 * Setup form submission
 */
function setupFormSubmission() {
    $('#videoUploadForm').on('submit', function(e) {
        e.preventDefault();
        
        console.log('🔄 Form submitted via AJAX');
        
        // Validate form
        if (!this.checkValidity()) {
            this.classList.add('was-validated');
            showAlert('Please fix the errors in the form before submitting.', 'danger');
            return false;
        }
        
        // Show loading state
        const submitBtn = $('#submitBtn');
        const originalText = submitBtn.html();
        submitBtn.html('<i class="fas fa-spinner fa-spin me-2"></i>Uploading...').prop('disabled', true);
        
        // Prepare form data
        const formData = new FormData(this);
        formData.append('ajax', '1'); // Signal for AJAX response
        
        // Submit via AJAX
        $.ajax({
            url: 'process-video-with-content.php',
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            dataType: 'json',
            timeout: 300000, // 5 minutes for large files
            success: function(response) {
                console.log('✅ Upload response:', response);
                
                if (response.success) {
                    showAlert('✅ Video uploaded successfully!', 'success');
                    
                    // Redirect after delay
                    setTimeout(() => {
                        window.location.href = response.redirect || 'videos.php';
                    }, 2000);
                } else {
                    showAlert('❌ Upload failed: ' + (response.message || 'Unknown error'), 'danger');
                }
            },
            error: function(xhr, status, error) {
                console.error('❌ Upload error:', error);
                
                let errorMessage = 'Upload failed';
                try {
                    const response = JSON.parse(xhr.responseText);
                    errorMessage = response.message || errorMessage;
                } catch(e) {
                    errorMessage = `${xhr.status}: ${error}`;
                }
                
                showAlert('❌ ' + errorMessage, 'danger');
            },
            complete: function() {
                // Reset button
                submitBtn.html(originalText).prop('disabled', false);
            }
        });
        
        this.classList.add('was-validated');
        return false;
    });
    
    // Reset form
    $('#resetBtn').on('click', function() {
        if (confirm('Are you sure you want to reset the form? All data will be lost.')) {
            resetForm();
        }
    });
}

/**
 * Reset the entire form
 */
function resetForm() {
    const form = $('#videoUploadForm')[0];
    form.reset();
    form.classList.remove('was-validated');
    
    // Hide previews
    hideFilePreview();
    clearVideoPreview();
    $('#thumbnailPreview').hide();
    
    // Reset to file mode
    showFileUpload();
    
    showAlert('Form has been reset.', 'info');
    console.log('✅ Form reset');
}

/**
 * Utility Functions
 */

// Format duration from seconds
function formatDuration(seconds) {
    if (!seconds) return 'Unknown';
    
    const hours = Math.floor(seconds / 3600);
    const minutes = Math.floor((seconds % 3600) / 60);
    const secs = seconds % 60;
    
    if (hours > 0) {
        return `${hours}:${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
    } else {
        return `${minutes}:${secs.toString().padStart(2, '0')}`;
    }
}

// Show alert message
function showAlert(message, type = 'info') {
    // Remove existing alerts
    $('.video-upload-alert').remove();
    
    // Create new alert
    const alertHtml = `
        <div class="alert alert-${type} alert-dismissible fade show video-upload-alert" role="alert">
            <i class="fas fa-${getAlertIcon(type)} me-2"></i>
            ${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    `;
    
    // Add to form
    $('#videoUploadForm').prepend(alertHtml);
    
    // Auto-dismiss success alerts
    if (type === 'success' || type === 'info') {
        setTimeout(() => {
            $('.video-upload-alert').fadeOut();
        }, 5000);
    }
}

// Get icon for alert type
function getAlertIcon(type) {
    const icons = {
        'success': 'check-circle',
        'danger': 'exclamation-triangle',
        'warning': 'exclamation-circle',
        'info': 'info-circle'
    };
    return icons[type] || 'info-circle';
}

// Make functions globally available
window.clearVideoPreview = clearVideoPreview;
window.showFileUpload = showFileUpload;
window.showUrlEmbed = showUrlEmbed;
