/**
 * Clean Video Detection System
 * Isolated JavaScript for video URL detection and preview
 * No conflicts, no complex debugging - just clean functionality
 */

// Main video detection function
function detectVideo() {
    const videoUrl = $('#videoUrl').val().trim();
    
    if (!videoUrl) {
        showAlert('⚠️ Please enter a video URL', 'warning');
        return;
    }
    
    // Set button to loading state
    const detectBtn = $('#detectVideoBtn');
    detectBtn.prop('disabled', true);
    detectBtn.html('<i class="fas fa-spinner fa-spin me-2"></i>Detecting...');
    
    // Clear any existing preview
    clearVideoPreview();

    $.ajax({
        url: 'ajax/detect_video.php',
        method: 'POST',
        dataType: 'json',
        data: { video_url: videoUrl },
        success: function(response) {
            if (response.success && response.data) {
                showAlert('✅ ' + response.message, 'success');
                showVideoPreview(response.data);
            } else {
                showAlert('❌ ' + (response.message || 'Video detection failed'), 'danger');
            }
            
            // Reset button state with robust approach
            resetDetectButton();
        },
        error: function(xhr, status, error) {
            showAlert('❌ Error detecting video. Please try again.', 'danger');
            console.error('AJAX error:', error);
            
            // Reset button state with robust approach
            resetDetectButton();
        }
    });
}

// Robust button reset function
function resetDetectButton() {
    console.log('🔄 Resetting detect button...');
    
    const detectBtn = $('#detectVideoBtn');
    if (detectBtn.length) {
        detectBtn.prop('disabled', false);
        detectBtn.html('<i class="fas fa-search me-2"></i>Detect');
        console.log('✅ Button reset successfully');
    } else {
        console.error('❌ Detect button not found!');
    }
    
    // Alternative approach using vanilla JavaScript as backup
    const detectBtnVanilla = document.getElementById('detectVideoBtn');
    if (detectBtnVanilla) {
        detectBtnVanilla.disabled = false;
        detectBtnVanilla.innerHTML = '<i class="fas fa-search me-2"></i>Detect';
    }
}

// Show video preview
function showVideoPreview(videoData) {
    console.log('Showing video preview:', videoData);
    
    // Check if thumbnail is available, otherwise use video embed
    const hasValidThumbnail = videoData.thumbnail_url && 
                              videoData.thumbnail_url !== '' && 
                              !videoData.thumbnail_url.includes('undefined') &&
                              videoData.thumbnail_url.startsWith('http');
    
    let mediaContent;
    if (hasValidThumbnail) {
        // Show thumbnail image
        mediaContent = `
            <img src="${videoData.thumbnail_url}" 
                 class="img-fluid rounded shadow-sm" 
                 alt="Video thumbnail"
                 style="max-height: 150px; width: 100%; object-fit: cover;"
                 onerror="this.style.display='none'; this.nextElementSibling.style.display='block';">
            <div style="display: none;">
                <div class="embed-responsive embed-responsive-16by9" style="max-height: 150px;">
                    ${videoData.embed_code}
                </div>
            </div>
        `;
    } else {
        // Show smart fallback for videos without thumbnails
        if (videoData.platform === 'facebook') {
            // Facebook videos often have embedding restrictions
            mediaContent = `
                <div class="d-flex align-items-center justify-content-center bg-primary text-white rounded" style="height: 150px;">
                    <div class="text-center">
                        <i class="fab fa-facebook fa-3x mb-2"></i>
                        <h6 class="mb-1">Facebook Video</h6>
                        <small>Privacy restricted</small>
                    </div>
                </div>
                <div class="mt-2">
                    <a href="${videoData.original_url}" target="_blank" class="btn btn-sm btn-primary w-100">
                        <i class="fas fa-external-link-alt me-1"></i>View on Facebook
                    </a>
                </div>
            `;
        } else {
            // Show video embed with proper sizing based on platform
            let containerStyle = '';
            let embedWrapper = '';
            
            if (videoData.platform === 'tiktok') {
                // TikTok needs taller container for vertical videos
                containerStyle = 'width: 100%; max-width: 300px; height: 400px; margin: 0 auto;';
                embedWrapper = `<div style="${containerStyle} border-radius: 8px; overflow: hidden; background: #000;">${videoData.embed_code}</div>`;
            } else {
                // Other platforms (Vimeo, etc.) - standard 16:9 aspect ratio
                containerStyle = 'position: relative; width: 100%; padding-bottom: 56.25%; height: 0; border-radius: 8px; overflow: hidden; background: #f8f9fa;';
                embedWrapper = `
                    <div style="${containerStyle}">
                        <div style="position: absolute; top: 0; left: 0; width: 100%; height: 100%;">
                            ${videoData.embed_code}
                        </div>
                    </div>
                `;
            }
            
            mediaContent = `
                ${embedWrapper}
                <small class="text-muted d-block mt-2 text-center">
                    <i class="fas fa-play-circle me-1"></i>Live video preview
                </small>
            `;
        }
    }
    
    // Create clean preview HTML
    const previewHtml = `
        <div class="card border-success mt-3" id="videoPreviewCard">
            <div class="card-header bg-success text-white">
                <h6 class="mb-0">
                    <i class="fas fa-video me-2"></i>
                    ${videoData.platform_name} Video Detected
                </h6>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-4">
                        ${mediaContent}
                    </div>
                    <div class="col-md-8">
                        ${videoData.title ? `<h5 class="card-title mb-2 text-primary">${videoData.title}</h5>` : ''}
                        ${videoData.description ? `<p class="text-muted small mb-3" style="max-height: 60px; overflow: hidden;">${videoData.description.substring(0, 150)}${videoData.description.length > 150 ? '...' : ''}</p>` : ''}
                        
                        <div class="row g-2 mb-3">
                            ${videoData.author_name ? `
                                <div class="col-auto">
                                    <span class="badge bg-info">
                                        <i class="fas fa-user me-1"></i>${videoData.author_name}
                                    </span>
                                </div>
                            ` : ''}
                            ${videoData.duration ? `
                                <div class="col-auto">
                                    <span class="badge bg-secondary">
                                        <i class="fas fa-clock me-1"></i>${formatDuration(videoData.duration)}
                                    </span>
                                </div>
                            ` : ''}
                            ${videoData.view_count ? `
                                <div class="col-auto">
                                    <span class="badge bg-success">
                                        <i class="fas fa-eye me-1"></i>${formatViews(videoData.view_count)}
                                    </span>
                                </div>
                            ` : ''}
                        </div>
                        
                        <table class="table table-sm table-borderless small">
                            <tr>
                                <td width="25%"><strong>Platform:</strong></td>
                                <td><span class="badge bg-primary">${videoData.platform_name}</span></td>
                            </tr>
                            <tr>
                                <td><strong>Video ID:</strong></td>
                                <td><code class="small">${videoData.video_id}</code></td>
                            </tr>
                            <tr>
                                <td><strong>Status:</strong></td>
                                <td><span class="badge bg-success">Ready for upload</span></td>
                            </tr>
                        </table>
                    </div>
                </div>
                <button type="button" class="btn btn-sm btn-outline-danger" onclick="clearVideoPreview()">
                    <i class="fas fa-times me-1"></i>Clear
                </button>
            </div>
        </div>
    `;
    
    // Show preview
    $('#videoPreviewArea').html(previewHtml).show();
    
    // Auto-fill form fields with video metadata
    populateFormFields(videoData);
    
    console.log('✅ Video preview displayed successfully');
}

// Utility function to format duration (seconds to readable format)
function formatDuration(seconds) {
    if (!seconds) return 'Unknown';
    
    const hours = Math.floor(seconds / 3600);
    const minutes = Math.floor((seconds % 3600) / 60);
    const secs = seconds % 60;
    
    if (hours > 0) {
        return `${hours}:${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
    } else {
        return `${minutes}:${secs.toString().padStart(2, '0')}`;
    }
}

// Utility function to format view counts
function formatViews(views) {
    if (!views) return 'Unknown';
    
    if (views >= 1000000) {
        return (views / 1000000).toFixed(1) + 'M views';
    } else if (views >= 1000) {
        return (views / 1000).toFixed(1) + 'K views';
    } else {
        return views + ' views';
    }
}

// Populate form fields with detected video data and auto-fill title/description
function populateFormFields(videoData) {
    console.log('🔄 Populating form fields with video data:', videoData);
    
    // Store video data in correct hidden input fields that match backend expectations
    if (videoData.platform) {
        // Populate hidden fields with correct IDs/names from form
        $('#detectedVideoUrl').val(videoData.original_url || '');
        $('#detectedPlatformInput').val(videoData.platform || '');
        $('#videoIdExternal').val(videoData.video_id || '');
        $('#embedCodeInput').val(videoData.embed_code || '');
        
        console.log('✅ Hidden fields populated:');
        console.log('  - Video URL:', videoData.original_url);
        console.log('  - Platform:', videoData.platform);
        console.log('  - Video ID:', videoData.video_id);
        console.log('  - Embed Code length:', (videoData.embed_code || '').length);
    }
    
    // Auto-fill video title if available
    if (videoData.title && videoData.title !== 'null' && videoData.title !== null) {
        const titleField = $('#videoTitle');
        if (titleField.length > 0) {
            titleField.val(videoData.title);
            console.log('✅ Video title auto-filled:', videoData.title);
        }
    }
    
    // Smart thumbnail handling
    handleThumbnailDisplay(videoData);
}

function handleThumbnailDisplay(videoData) {
    const thumbnailSection = $('.thumbnail-section');
    
    // Check if video has a valid thumbnail
    const hasValidThumbnail = videoData.thumbnail_url && 
                             videoData.thumbnail_url !== 'null' && 
                             videoData.thumbnail_url !== null &&
                             videoData.thumbnail_url.trim() !== '';
    
    if (hasValidThumbnail) {
        // Video has thumbnail - hide the upload input
        thumbnailSection.hide();
        console.log('✅ Video has thumbnail - hiding upload input');
    } else {
        // Video has no thumbnail - show the upload input
        thumbnailSection.show();
        console.log('⚠️ Video has no thumbnail - showing upload input');
    }
}

// Clear video preview
function clearVideoPreview() {
    $('#videoPreviewArea').hide().empty();
    
    // Clear hidden fields
    $('#detectedPlatformInput').val('');
    $('#videoIdExternal').val('');
    $('#embedCodeInput').val('');
    
    console.log('✅ Video preview cleared');
}

// Auto-detection on URL paste (optional)
function setupAutoDetection() {
    $('#videoUrl').on('paste input', function() {
        const url = $(this).val().trim();
        if (url && (url.includes('youtube.com') || url.includes('youtu.be') || url.includes('vimeo.com'))) {
            // Optional: Auto-detect after a short delay
            setTimeout(() => {
                if ($(this).val().trim() === url && url.length > 10) {
                    console.log('Auto-detecting video...');
                    detectVideo();
                }
            }, 1000);
        }
    });
}

// Initialize video detection system when DOM is ready
$(document).ready(function() {
    console.log('🎥 Video Detection System initialized');
    
    // Clear any existing preview on page load
    clearVideoPreview();
    
    // Bind detect button  
    $('#detectVideoBtn').off('click').on('click', detectVideo);
    
    // Setup auto-detection
    setupAutoDetection();
            
    console.log('✅ Video detection ready');
});


// Clean, professional alert system
function showAlert(message, type = 'info') {
    // Remove any existing alerts first
    const existingAlerts = document.querySelectorAll('.video-detection-alert');
    existingAlerts.forEach(alert => alert.remove());
            
    // Icon mapping for different alert types
    const icons = {
        'success': 'fas fa-check-circle',
        'danger': 'fas fa-exclamation-triangle', 
        'warning': 'fas fa-exclamation-circle',
        'info': 'fas fa-info-circle'
    };
            
    // Create clean, modern alert
    const alertDiv = document.createElement('div');
    alertDiv.className = `alert alert-${type} alert-dismissible fade show video-detection-alert`;
    alertDiv.style.cssText = `
        margin-top: 15px;
        border: none;
        border-radius: 8px;
        padding: 16px 20px;
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        animation: slideIn 0.3s ease-out;
    `;
            
    alertDiv.innerHTML = `
        <i class="${icons[type] || icons.info} me-2"></i>
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert" style="padding: 0.75rem;"></button>
    `;
            
    // Insert in the video URL section
    const videoSection = document.querySelector('.video-url-section');
    if (videoSection) {
        videoSection.appendChild(alertDiv);
    }
            
    // Auto-dismiss success alerts after 4 seconds
    if (type === 'success') {
        setTimeout(() => {
            if (alertDiv && alertDiv.parentNode) {
                alertDiv.remove();
            }
        }, 4000);
    }
}
