<?php
// Start session first
require_once __DIR__ . '/../includes/session_start.php';
require_once __DIR__ . '/components/teacher_auth.php';
require_once __DIR__ . '/../includes/db_connection.php';

// Get the teacher's ID from session
$teacherId = $_SESSION['user_id'];

// Check if this is an AJAX request (support both header and URL parameter)
$isAjaxRequest = (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && 
                 strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest') ||
                 isset($_GET['ajax']);

// Function to format message time (reusable function)
function formatMessageTime($datetime) {
    $now = new DateTime();
    $time = new DateTime($datetime);
    $diff = $now->diff($time);
    
    if ($diff->y > 0) {
        return $time->format('M j, Y');
    } elseif ($diff->m > 0 || $diff->d > 6) {
        return $time->format('M j');
    } elseif ($diff->d > 0) {
        return $diff->d . ' day' . ($diff->d > 1 ? 's' : '') . ' ago';
    } elseif ($diff->h > 0) {
        return $diff->h . ' hour' . ($diff->h > 1 ? 's' : '') . ' ago';
    } elseif ($diff->i > 0) {
        return $diff->i . ' min' . ($diff->i > 1 ? 's' : '') . ' ago';
    } else {
        return 'Just now';
    }
}

// AJAX Endpoints
if ($isAjaxRequest) {
    // Load conversation between teacher and student
    if (isset($_GET['ajax']) && $_GET['ajax'] === 'load_conversation') {
        $studentId = isset($_GET['contact']) ? intval($_GET['contact']) : 0;
        
        if ($studentId > 0) {
            // Verify that student is enrolled in one of the teacher's courses
            $enrollmentQuery = "SELECT u.user_id, u.first_name, u.last_name, u.profile_picture, 
                              GROUP_CONCAT(DISTINCT c.title SEPARATOR ', ') as course_names
                              FROM enrollments e
                              JOIN users u ON e.student_id = u.user_id
                              JOIN courses c ON e.course_id = c.course_id
                              WHERE c.teacher_id = ? AND e.student_id = ?
                              GROUP BY u.user_id, u.first_name, u.last_name, u.profile_picture
                              LIMIT 1";
            
            $stmt = $conn->prepare($enrollmentQuery);
            $stmt->bind_param("ii", $teacherId, $studentId);
            $stmt->execute();
            $student = $stmt->get_result()->fetch_assoc();
            
            if ($student) {
                // Get messages for this conversation
                $messagesQuery = "SELECT m.*, u.first_name, u.last_name
                                 FROM messages m
                                 JOIN users u ON m.sender_id = u.user_id
                                 WHERE (m.sender_id = ? AND m.recipient_id = ?) 
                                    OR (m.sender_id = ? AND m.recipient_id = ?)
                                 ORDER BY m.sent_at ASC";
                
                $stmt = $conn->prepare($messagesQuery);
                $stmt->bind_param("iiii", $teacherId, $studentId, $studentId, $teacherId);
                $stmt->execute();
                $messages = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
                
                // Mark messages as read
                $markReadQuery = "UPDATE messages SET is_read = 1 
                                 WHERE sender_id = ? AND recipient_id = ? AND is_read = 0";
                $stmt = $conn->prepare($markReadQuery);
                $stmt->bind_param("ii", $studentId, $teacherId);
                $stmt->execute();
                
                // Output the conversation HTML
                ?>
                <div class="message-header">
                    <div class="d-flex align-items-center">
                        <button class="btn btn-sm btn-outline-secondary d-md-none me-3" onclick="showContactsList()">
                            <i class="fas fa-arrow-left"></i>
                        </button>
                        <div class="contact-avatar me-3">
                            <?php if ($student['profile_picture']): ?>
                                <img src="<?= htmlspecialchars($student['profile_picture']) ?>" 
                                     alt="<?= htmlspecialchars($student['first_name']) ?>" class="rounded-circle">
                            <?php else: ?>
                                <div class="avatar-placeholder">
                                    <?= strtoupper(substr($student['first_name'], 0, 1)) ?>
                                </div>
                            <?php endif; ?>
                        </div>
                        <div>
                            <h5 class="mb-1"><?= htmlspecialchars($student['first_name'] . ' ' . $student['last_name']) ?></h5>
                            <small class="text-muted">
                                <i class="bx bx-book-alt me-1"></i>
                                <?= htmlspecialchars($student['course_names']) ?>
                            </small>
                        </div>
                    </div>
                </div>
                
                <div class="messages-container">
                    <div class="messages-list" id="messages-list">
                        <?php if (!empty($messages)): ?>
                            <?php
                            $currentDate = '';
                            foreach ($messages as $message):
                                $messageDate = date('Y-m-d', strtotime($message['sent_at']));
                                if ($messageDate !== $currentDate):
                                    $currentDate = $messageDate;
                            ?>
                                <div class="date-separator">
                                    <span><?= date('F j, Y', strtotime($message['sent_at'])) ?></span>
                                </div>
                            <?php endif; ?>
                            
                            <div class="message-bubble <?= $message['sender_id'] == $teacherId ? 'message-sent' : 'message-received' ?>">
                                <?= nl2br(htmlspecialchars($message['message_text'])) ?>
                                <div class="message-time">
                                    <?= date('g:i A', strtotime($message['sent_at'])) ?>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <div class="empty-messages">
                                <div class="empty-icon">
                                    <i class="fas fa-comment-dots"></i>
                                </div>
                                <p>No messages yet. Start a conversation!</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="message-input-area">
                    <form id="message-form" class="d-flex w-100">
                        <input type="hidden" id="recipient-id" value="<?= $studentId ?>">
                        <div class="message-input-wrapper flex-grow-1 me-2">
                            <textarea class="form-control" id="message-input" 
                                placeholder="Type your message..." rows="1"></textarea>
                        </div>
                        <button type="submit" id="send-message" class="btn btn-primary message-send-btn">
                            <i class="fas fa-paper-plane"></i>
                        </button>
                    </form>
                </div>
                <?php
            } else {
                // Student not enrolled in any of teacher's courses
                echo '<div class="alert alert-warning m-3">This student is not enrolled in any of your courses.</div>';
            }
        }
        exit;
    }
    
    // Handle sending a new message
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'send_message') {
        $recipientId = isset($_POST['recipient_id']) ? intval($_POST['recipient_id']) : 0;
        $messageText = isset($_POST['message']) ? trim($_POST['message']) : '';
        $response = ['success' => false];
        
        if ($recipientId > 0 && !empty($messageText)) {
            // Verify that student is enrolled in one of the teacher's courses
            $enrollmentQuery = "SELECT 1 FROM enrollments e 
                               JOIN courses c ON e.course_id = c.course_id 
                               WHERE c.teacher_id = ? AND e.student_id = ?
                               LIMIT 1";
            
            $stmt = $conn->prepare($enrollmentQuery);
            $stmt->bind_param("ii", $teacherId, $recipientId);
            $stmt->execute();
            $isEnrolled = $stmt->get_result()->num_rows > 0;
            
            if ($isEnrolled) {
                try {
                    // Insert the message
                    $stmt = $conn->prepare("INSERT INTO messages (sender_id, recipient_id, message_text, sent_at, is_read) 
                                           VALUES (?, ?, ?, NOW(), 0)");
                    $stmt->bind_param("iis", $teacherId, $recipientId, $messageText);
                    
                    if ($stmt->execute()) {
                        $messageId = $conn->insert_id;
                        $now = date('Y-m-d H:i:s');
                        
                        $response = [
                            'success' => true,
                            'message_id' => $messageId,
                            'message' => $messageText,
                            'sent_at' => $now,
                            'formatted_time' => date('g:i A', strtotime($now))
                        ];
                    }
                } catch (Exception $e) {
                    $response['error'] = 'Database error: ' . $e->getMessage();
                }
            } else {
                $response['error'] = 'Student not enrolled in any of your courses';
            }
        } else {
            $response['error'] = 'Invalid recipient or empty message';
        }
        
        header('Content-Type: application/json');
        echo json_encode($response);
        exit;
    }
    
    exit; // End all AJAX requests here
}

// For regular page loads (not AJAX), include the teacher layout
$current_page = 'messages.php';
require_once __DIR__ . '/components/teacher_layout.php';

// Get list of students enrolled in teacher's courses (for regular page load)
$students = [];
$selectedStudentId = isset($_GET['contact']) ? intval($_GET['contact']) : null;

try {
    // Get unique students enrolled in teacher's courses (avoiding duplicates)
    $query = "SELECT u.user_id, u.first_name, u.last_name, u.profile_picture, 
              GROUP_CONCAT(DISTINCT c.title SEPARATOR ', ') as course_names,
              (SELECT COUNT(*) FROM messages WHERE sender_id = u.user_id AND recipient_id = ? AND is_read = 0) as unread_count,
              (SELECT sent_at FROM messages 
               WHERE (sender_id = u.user_id AND recipient_id = ?) OR (sender_id = ? AND recipient_id = u.user_id) 
               ORDER BY sent_at DESC LIMIT 1) as last_message_time
              FROM enrollments e
              JOIN users u ON e.student_id = u.user_id
              JOIN courses c ON e.course_id = c.course_id
              WHERE c.teacher_id = ?
              GROUP BY u.user_id, u.first_name, u.last_name, u.profile_picture
              ORDER BY last_message_time DESC, u.first_name ASC, u.last_name ASC";
    
    $stmt = $conn->prepare($query);
    $stmt->bind_param("iiii", $teacherId, $teacherId, $teacherId, $teacherId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    while ($row = $result->fetch_assoc()) {
        $students[] = $row;
    }
} catch (Exception $e) {
    $error = "Database error: " . $e->getMessage();
}

// HTML/CSS for the messaging interface
?>

<style>
    /* Ultra-Modern Messaging Interface - Complete Redesign */
    .messaging-container {
        height: calc(100vh - 140px);
        background: linear-gradient(145deg, #f8fafc 0%, #ffffff 100%);
        border-radius: 20px;
        overflow: hidden;
        box-shadow: 0 8px 32px rgba(0, 0, 0, 0.08), 0 2px 8px rgba(0, 0, 0, 0.04);
        border: 1px solid rgba(255, 255, 255, 0.8);
        position: relative;
        backdrop-filter: blur(10px);
    }
    
    .messaging-container .row {
        height: 100%;
        margin: 0;
        display: flex;
        flex-wrap: nowrap;
    }
    
    /* Modern Contacts Sidebar */
    .contacts-sidebar {
        background: rgba(248, 250, 252, 0.8);
        border-right: 1px solid rgba(226, 232, 240, 0.5);
        height: 100%;
        overflow: hidden;
        display: flex;
        flex-direction: column;
        backdrop-filter: blur(10px);
        max-width: 350px;
        min-width: 300px;
        flex: 0 0 auto;
    }
    
    .contacts-header {
        padding: 1.75rem 1.5rem 1.25rem;
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: #ffffff;
        font-weight: 700;
        position: relative;
        box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
    }
    
    .contacts-header::after {
        content: '';
        position: absolute;
        bottom: 0;
        left: 0;
        right: 0;
        height: 1px;
        background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
    }
    
    .contacts-header h5 {
        margin-bottom: 0;
        display: flex;
        align-items: center;
        font-weight: 600;
        text-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
    }
    
    .contacts-search {
        padding: 1.25rem 1.5rem;
        background: rgba(255, 255, 255, 0.7);
        backdrop-filter: blur(10px);
        border-bottom: 1px solid rgba(241, 245, 249, 0.8);
    }
    
    .search-input {
        background: rgba(255, 255, 255, 0.9);
        border: 2px solid rgba(226, 232, 240, 0.6);
        border-radius: 16px;
        padding: 0.75rem 1rem 0.75rem 2.5rem;
        transition: all 0.3s ease;
        font-size: 14px;
        font-weight: 500;
        backdrop-filter: blur(5px);
    }
    
    .search-input:focus {
        outline: none;
        border-color: #4f46e5;
        box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
    }
    
    #message-input:focus {
        outline: none;
        border-color: #4f46e5;
        box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
    }
    
    .search-icon {
        position: absolute;
        left: 2rem;
        top: 50%;
        transform: translateY(-50%);
        color: #6b7280;
        opacity: 0.7;
        z-index: 1;
    }
    
    .contacts-list {
        overflow-y: auto;
        height: calc(100% - 140px);
        padding: 0.5rem 0;
    }
    
    /* Custom scrollbar */
    .contacts-list::-webkit-scrollbar {
        width: 6px;
    }
    
    .contacts-list::-webkit-scrollbar-track {
        background: rgba(240, 242, 245, 0.5);
    }
    
    .contacts-list::-webkit-scrollbar-thumb {
        background: rgba(100, 116, 139, 0.3);
        border-radius: 10px;
    }
    
    .contacts-list::-webkit-scrollbar-thumb:hover {
        background: rgba(100, 116, 139, 0.5);
    }
    
    /* Modern Contact Items */
    .contact-item {
        padding: 1rem 1.25rem;
        border-bottom: 1px solid rgba(241, 245, 249, 0.6);
        cursor: pointer;
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        background: rgba(255, 255, 255, 0.5);
        position: relative;
        backdrop-filter: blur(10px);
        margin: 0 0.75rem;
        border-radius: 12px;
        margin-bottom: 0.5rem;
    }
    
    .contact-item:hover {
        background: rgba(255, 255, 255, 0.8);
        transform: translateY(-1px);
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08);
    }
    
    .contact-item.active {
        background: linear-gradient(135deg, rgba(79, 70, 229, 0.1), rgba(124, 58, 237, 0.1));
        border: 2px solid rgba(79, 70, 229, 0.3);
        transform: scale(1.02);
        box-shadow: 0 6px 20px rgba(79, 70, 229, 0.15);
    }
    
    .contact-avatar {
        width: 48px;
        height: 48px;
        border-radius: 50%;
        overflow: hidden;
        background: linear-gradient(135deg, #e2e8f0 0%, #cbd5e1 100%);
        display: flex;
        align-items: center;
        justify-content: center;
        flex-shrink: 0;
        box-shadow: 0 3px 10px rgba(0, 0, 0, 0.08);
    }
    
    .contact-avatar img {
        width: 100%;
        height: 100%;
        object-fit: cover;
    }
    
    .avatar-placeholder {
        color: #64748b;
        font-weight: 600;
        font-size: 1.2rem;
    }
    
    .contact-info {
        margin-left: 1rem;
        overflow: hidden;
    }
    
    .contact-name {
        font-weight: 600;
        margin-bottom: 0.2rem;
        color: #1e293b;
    }
    
    .contact-course {
        color: #64748b;
        font-size: 0.85rem;
        white-space: nowrap;
        overflow: hidden;
        text-overflow: ellipsis;
        max-width: 180px;
    }
    
    .contact-meta {
        display: flex;
        flex-direction: column;
        align-items: flex-end;
    }
    
    .contact-time {
        font-size: 0.8rem;
        color: #94a3b8;
        margin-bottom: 0.3rem;
    }
    
    .unread-badge {
        background: #dc3545;
        color: white;
        border-radius: 50%;
        width: 20px;
        height: 20px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 0.75rem;
        font-weight: 600;
    }
    
    /* Ultra-Modern Messages Area */
    .messages-area {
        height: 100%;
        display: flex;
        flex-direction: column;
        background: linear-gradient(180deg, #ffffff 0%, #fafbfc 100%);
        position: relative;
        flex: 1;
        min-width: 0;
    }
    
    .message-header {
        padding: 1.5rem 2rem;
        border-bottom: 1px solid rgba(241, 245, 249, 0.8);
        background: rgba(255, 255, 255, 0.95);
        backdrop-filter: blur(20px);
        box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
        position: sticky;
        top: 0;
        z-index: 10;
    }
    
    .message-header h5 {
        color: #1e293b;
        font-weight: 600;
        margin-bottom: 0.25rem;
        font-size: 1.1rem;
    }
    
    .message-header small {
        color: #64748b;
        font-size: 0.875rem;
        font-weight: 500;
    }
    
    .messages-container {
        flex: 1;
        overflow-y: auto;
        padding: 2rem;
        display: flex;
        flex-direction: column;
        position: relative;
    }
    
    .messages-list {
        display: flex;
        flex-direction: column;
        gap: 1rem;
        flex: 1;
        overflow-y: auto;
        padding: 1rem 0;
        scroll-behavior: smooth;
    }
    
    .messages-list::-webkit-scrollbar {
        width: 6px;
    }
    
    .messages-list::-webkit-scrollbar-track {
        background: rgba(240, 242, 245, 0.5);
    }
    
    .messages-list::-webkit-scrollbar-thumb {
        background: rgba(100, 116, 139, 0.3);
        border-radius: 10px;
    }
    
    .messages-list::-webkit-scrollbar-thumb:hover {
        background: rgba(100, 116, 139, 0.5);
    }
    
    /* Ultra-Modern Message Bubbles */
    .message-bubble {
        max-width: 75%;
        padding: 1rem 1.25rem;
        margin-bottom: 0.25rem;
        position: relative;
        word-wrap: break-word;
        font-size: 15px;
        line-height: 1.6;
        font-weight: 500;
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    }
    
    .message-sent {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: #ffffff;
        margin-left: auto;
        border-radius: 24px 24px 8px 24px;
        box-shadow: 0 4px 20px rgba(102, 126, 234, 0.3);
    }
    
    .message-received {
        background: rgba(255, 255, 255, 0.9);
        color: #374151;
        margin-right: auto;
        border-radius: 8px 24px 24px 24px;
        border: 1px solid rgba(226, 232, 240, 0.6);
        box-shadow: 0 2px 12px rgba(0, 0, 0, 0.06);
        backdrop-filter: blur(10px);
    }
    
    .message-bubble:hover {
        transform: translateY(-2px) scale(1.01);
    }
    
    .message-sent:hover {
        box-shadow: 0 8px 30px rgba(102, 126, 234, 0.4);
        background: linear-gradient(135deg, #5a67d8 0%, #6b46c1 100%);
    }
    
    .message-received:hover {
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
        background: rgba(255, 255, 255, 0.95);
    }
    
    .message-time {
        font-size: 0.75rem;
        margin-top: 0.5rem;
        text-align: right;
        font-weight: 500;
        opacity: 0.8;
    }
    
    .message-sent .message-time {
        color: rgba(255, 255, 255, 0.85);
    }
    
    .message-received .message-time {
        color: #64748b;
    }
    
    .date-separator {
        text-align: center;
        margin: 1.5rem 0;
        position: relative;
    }
    
    .date-separator span {
        background: #f1f5f9;
        color: #64748b;
        padding: 0.5rem 1rem;
        border-radius: 16px;
        font-size: 0.8rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        border: 1px solid #e2e8f0;
    }
    
    /* Ultra-Modern Message Input Area */
    .message-input-area {
        padding: 1.75rem 2rem;
        border-top: 1px solid rgba(241, 245, 249, 0.8);
        background: rgba(255, 255, 255, 0.95);
        backdrop-filter: blur(20px);
        box-shadow: 0 -4px 20px rgba(0, 0, 0, 0.06);
        position: sticky;
        bottom: 0;
        z-index: 10;
    }
    
    .message-input-wrapper {
        position: relative;
    }
    
    #message-input {
        border: 2px solid rgba(226, 232, 240, 0.8);
        border-radius: 28px;
        padding: 1rem 1.5rem;
        font-size: 15px;
        resize: none;
        overflow-y: auto;
        min-height: 52px;
        max-height: 140px;
        background: rgba(255, 255, 255, 0.9);
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        font-family: inherit;
        font-weight: 500;
        line-height: 1.6;
        box-shadow: 0 2px 10px rgba(0, 0, 0, 0.06);
        backdrop-filter: blur(10px);
    }
    
    #message-input:focus {
        outline: none;
        border-color: rgba(102, 126, 234, 0.8);
        box-shadow: 0 0 0 6px rgba(102, 126, 234, 0.15), 0 4px 20px rgba(0, 0, 0, 0.1);
        transform: translateY(-2px);
        background: rgba(255, 255, 255, 0.95);
    }
    
    .message-send-btn {
        width: 52px;
        height: 52px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        border: none;
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        color: white;
        font-size: 20px;
        box-shadow: 0 4px 16px rgba(102, 126, 234, 0.3);
    }
    
    .message-send-btn:hover {
        background: linear-gradient(135deg, #4338ca 0%, #6d28d9 100%);
        transform: scale(1.08);
        box-shadow: 0 4px 16px rgba(79, 70, 229, 0.35);
    }
    
    .message-send-btn:active {
        transform: scale(0.95);
        box-shadow: 0 1px 4px rgba(79, 70, 229, 0.3);
    }
    
    /* Empty state styling */
    .empty-messages, .welcome-message {
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        height: 100%;
        text-align: center;
        padding: 2rem;
        color: #64748b;
    }
    
    .empty-icon, .welcome-icon {
        font-size: 3.5rem;
        margin-bottom: 1.5rem;
        color: #94a3b8;
        opacity: 0.7;
    }
    
    .welcome-message h3 {
        background: linear-gradient(135deg, #6a11cb 0%, #2575fc 100%);
        -webkit-background-clip: text;
        background-clip: text;
        color: transparent;
        font-weight: 600;
        margin-bottom: 1rem;
    }
    
    /* Mobile responsiveness */
    @media (max-width: 767.98px) {
        .contacts-sidebar {
            position: absolute;
            width: 100%;
            z-index: 10;
            transition: transform 0.3s ease;
            transform: translateX(0);
        }
        
        .hide-contacts .contacts-sidebar {
            transform: translateX(-100%);
        }
        
        .message-bubble {
            max-width: 85%;
        }
    }
</style>

<!-- Teacher Messages Interface - Properly integrated with layout -->
<div class="container-fluid py-4">
    <div class="row">
        <div class="col-12">
            <div class="page-header mb-4">
                <h2><i class="bx bx-message-square-detail me-2"></i>Teacher Messages</h2>
                <p class="text-muted">Connect with your enrolled students</p>
            </div>
            
            <?php if (isset($error)): ?>
                <div class="alert alert-danger"><?= $error ?></div>
            <?php endif; ?>
        </div>
    </div>
    
    <div class="messaging-container">
        <div class="row g-0 h-100">
            <!-- Contacts Sidebar for Messages (NOT the main navigation) -->
            <div class="contacts-sidebar" id="contacts-sidebar">
                <div class="contacts-header">
                    <h5><i class="bx bx-user-voice me-2"></i>My Students</h5>
                </div>
                
                <div class="contacts-search">
                    <div class="position-relative">
                        <i class="fas fa-search search-icon"></i>
                        <input type="text" id="search-contacts" class="form-control search-input" 
                               placeholder="Search students...">
                    </div>
                </div>
                
                <div class="contacts-list">
                    <?php if (empty($students)): ?>
                        <div class="p-4 text-center text-muted">
                            <i class="fas fa-user-graduate mb-3" style="font-size: 2rem;"></i>
                            <p>No students enrolled in your courses yet.</p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($students as $student): ?>
                            <div class="contact-item d-flex justify-content-between <?= $selectedStudentId == $student['user_id'] ? 'active' : '' ?>" 
                                 data-contact-id="<?= $student['user_id'] ?>">
                                <div class="d-flex">
                                    <div class="contact-avatar">
                                        <?php if ($student['profile_picture']): ?>
                                            <img src="<?= htmlspecialchars($student['profile_picture']) ?>" 
                                                 alt="<?= htmlspecialchars($student['first_name']) ?>">
                                        <?php else: ?>
                                            <div class="avatar-placeholder">
                                                <?= strtoupper(substr($student['first_name'], 0, 1)) ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                    <div class="contact-info">
                                        <div class="contact-name">
                                            <?= htmlspecialchars($student['first_name'] . ' ' . $student['last_name']) ?>
                                        </div>
                                        <div class="contact-course">
                                            <i class="bx bx-book-alt me-1"></i>
                                            <?= htmlspecialchars($student['course_names']) ?>
                                        </div>
                                    </div>
                                </div>
                                <div class="contact-meta">
                                    <?php if ($student['last_message_time']): ?>
                                        <div class="contact-time">
                                            <?= formatMessageTime($student['last_message_time']) ?>
                                        </div>
                                    <?php endif; ?>
                                    <?php if ($student['unread_count'] > 0): ?>
                                        <div class="unread-badge"><?= $student['unread_count'] ?></div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Messages Area -->
            <div class="messages-area" id="messages-area">
                <?php if ($selectedStudentId): ?>
                    <!-- Conversation will be loaded via AJAX -->
                    <div class="loading-messages">
                        <div class="d-flex justify-content-center align-items-center h-100">
                            <div class="spinner-border text-primary" role="status">
                                <span class="visually-hidden">Loading...</span>
                            </div>
                        </div>
                    </div>
                <?php else: ?>
                    <!-- Welcome message when no conversation is selected -->
                    <div class="welcome-message">
                        <div class="welcome-icon">
                            <i class="fas fa-comments"></i>
                        </div>
                        <h3>Welcome to Messages</h3>
                        <p class="mb-0">Select a student from the left to start a conversation.</p>
                        <p class="text-muted mt-2">You can only message students who are enrolled in your courses.</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    let currentContactId = <?= $selectedStudentId ? $selectedStudentId : 'null' ?>;
    
    // Load conversation if contact is selected
    if (currentContactId) {
        loadConversation(currentContactId);
    }
    
    // Setup contact click handlers
    setupContactClicks();
    
    // Setup search functionality
    setupSearch();
    
    // Function to load a conversation
    function loadConversation(contactId) {
        currentContactId = contactId;
        
        // Update URL without reloading the page
        const newUrl = window.location.pathname + '?contact=' + contactId;
        window.history.pushState({path: newUrl}, '', newUrl);
        
        // Update active contact
        document.querySelectorAll('.contact-item').forEach(item => {
            item.classList.remove('active');
        });
        
        const contactItem = document.querySelector(`.contact-item[data-contact-id="${contactId}"]`);
        if (contactItem) {
            contactItem.classList.add('active');
            
            // Clear unread badge
            const unreadBadge = contactItem.querySelector('.unread-badge');
            if (unreadBadge) unreadBadge.remove();
        }
        
        // Show loading state
        document.getElementById('messages-area').innerHTML = `
            <div class="loading-messages">
                <div class="d-flex justify-content-center align-items-center h-100">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                </div>
            </div>
        `;
        
        // On mobile, hide the contacts sidebar
        if (window.innerWidth < 768) {
            document.body.classList.add('hide-contacts');
        }
        
        // Load conversation via AJAX
        fetch(`messages.php?ajax=load_conversation&contact=${contactId}`, {
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
            .then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                }
                return response.text();
            })
            .then(html => {
                console.log('AJAX Response received:', html.substring(0, 200)); // Debug log
                document.getElementById('messages-area').innerHTML = html;
                
                // Scroll to bottom of messages
                const messagesList = document.getElementById('messages-list');
                if (messagesList) {
                    messagesList.scrollTop = messagesList.scrollHeight;
                }
                
                // Setup message form
                setupMessageForm();
                
                // Setup textarea auto-resize
                setupTextareaAutoResize();
            })
            .catch(error => {
                console.error('Error loading conversation:', error);
                document.getElementById('messages-area').innerHTML = `
                    <div class="alert alert-danger m-3">Failed to load the conversation. Please try again.</div>
                `;
            });
    }
    
    // Function to setup contact clicks
    function setupContactClicks() {
        document.querySelectorAll('.contact-item').forEach(item => {
            item.addEventListener('click', function() {
                const contactId = this.getAttribute('data-contact-id');
                loadConversation(contactId);
            });
        });
    }
    
    // Function to setup search functionality
    function setupSearch() {
        const searchInput = document.getElementById('search-contacts');
        if (!searchInput) return;
        
        searchInput.addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            
            document.querySelectorAll('.contact-item').forEach(item => {
                const contactName = item.querySelector('.contact-name').textContent.toLowerCase();
                const contactCourse = item.querySelector('.contact-course').textContent.toLowerCase();
                
                if (contactName.includes(searchTerm) || contactCourse.includes(searchTerm)) {
                    item.style.display = 'flex';
                } else {
                    item.style.display = 'none';
                }
            });
        });
    }
    
    // Function to setup message form
    function setupMessageForm() {
        const form = document.getElementById('message-form');
        if (!form) return;
        
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const input = document.getElementById('message-input');
            const message = input.value.trim();
            const recipientId = document.getElementById('recipient-id').value;
            
            if (!message) return;
            
            // Clear the input
            input.value = '';
            input.style.height = 'auto';
            
            // Send the message via AJAX
            const formData = new FormData();
            formData.append('action', 'send_message');
            formData.append('recipient_id', recipientId);
            formData.append('message', message);
            
            fetch('messages.php', {
                method: 'POST',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                },
                body: formData
            })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        // Add the message to the UI
                        const messagesList = document.getElementById('messages-list');
                        if (!messagesList) return;
                        
                        // Check if we need to add a new date separator
                        const today = new Date().toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric' });
                        const lastDateSeparator = messagesList.querySelector('.date-separator:last-child');
                        let needsDateSeparator = true;
                        
                        if (lastDateSeparator) {
                            const lastDate = lastDateSeparator.querySelector('span').textContent;
                            needsDateSeparator = lastDate !== today;
                        }
                        
                        // Add date separator if needed
                        if (needsDateSeparator) {
                            const dateSeparator = document.createElement('div');
                            dateSeparator.className = 'date-separator';
                            dateSeparator.innerHTML = `<span>${today}</span>`;
                            messagesList.appendChild(dateSeparator);
                        }
                        
                        // Create and add the message bubble
                        const messageBubble = document.createElement('div');
                        messageBubble.className = 'message-bubble message-sent';
                        messageBubble.innerHTML = `
                            ${message.replace(/\n/g, '<br>')}
                            <div class="message-time">${data.formatted_time}</div>
                        `;
                        messagesList.appendChild(messageBubble);
                        
                        // Scroll to the bottom
                        messagesList.scrollTop = messagesList.scrollHeight;
                        
                        // If this is the first message, remove the empty state
                        const emptyMessages = messagesList.querySelector('.empty-messages');
                        if (emptyMessages) {
                            emptyMessages.remove();
                        }
                    } else {
                        console.error('Failed to send message:', data.error);
                    }
                })
                .catch(error => {
                    console.error('Error sending message:', error);
                });
        });
    }
    
    // Function to setup textarea auto-resize
    function setupTextareaAutoResize() {
        const textarea = document.getElementById('message-input');
        if (!textarea) return;
        
        textarea.addEventListener('input', function() {
            this.style.height = 'auto';
            this.style.height = (this.scrollHeight) + 'px';
        });
        
        // Also resize on focus, in case there's already content
        textarea.addEventListener('focus', function() {
            this.style.height = 'auto';
            this.style.height = (this.scrollHeight) + 'px';
        });
    }
    
    // Function to show the contacts list on mobile
    window.showContactsList = function() {
        document.body.classList.remove('hide-contacts');
    };
});
</script>

<?php
// Include teacher layout end - this will add the footer
require_once __DIR__ . '/components/teacher_layout_end.php';
?>

