<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Centralized session management
require_once "../includes/session_start.php";
require_once "../config/db_connect.php";
require_once "../includes/functions.php";
require_once "../auth/functions.php";

// Check if user is logged in and is a teacher
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'teacher') {
    header("Location: ../login.php");
    exit;
}

$teacher_id = $_SESSION['user_id'];
$teacher = get_teacher($conn, $teacher_id);

// Check if quiz ID is provided
if (!isset($_GET['id']) || empty($_GET['id'])) {
    header("Location: quizzes.php");
    exit;
}

$quiz_id = intval($_GET['id']);

// Fetch quiz details
$stmt = $conn->prepare("SELECT q.*, c.title as course_title FROM quizzes q JOIN courses c ON q.course_id = c.course_id WHERE q.quiz_id = ? AND q.teacher_id = ?");
$stmt->bind_param("ii", $quiz_id, $teacher_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    // Quiz not found or not owned by this teacher
    header("Location: quizzes.php");
    exit;
}

$quiz = $result->fetch_assoc();

// Fetch quiz questions
$questions = [];
$stmt = $conn->prepare("SELECT * FROM quiz_questions WHERE quiz_id = ? ORDER BY question_id ASC");
$stmt->bind_param("i", $quiz_id);
$stmt->execute();
$questions_result = $stmt->get_result();

while ($question = $questions_result->fetch_assoc()) {
    // Fetch options for this question if it's multiple choice or true/false
    $question['options'] = [];
    
    if ($question['question_type'] !== 'short_answer') {
        $options_stmt = $conn->prepare("SELECT * FROM quiz_question_options WHERE question_id = ?");
        $options_stmt->bind_param("i", $question['question_id']);
        $options_stmt->execute();
        $options_result = $options_stmt->get_result();
        
        while ($option = $options_result->fetch_assoc()) {
            $question['options'][] = $option;
        }
    }
    
    $questions[] = $question;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Preview Quiz - <?= htmlspecialchars($quiz['title']) ?></title>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
   :root {
            --primary: #4e73df;
            --secondary: #858796;
            --success: #1cc88a;
            --info: #36b9cc;
            --warning: #f6c23e;
            --danger: #e74a3b;
            --light: #f8f9fc;
            --dark: #5a5c69;
            --sidebar-width: 250px;
            --sidebar-bg: whitesmoke;
            --navbar-height: 60px;
            --border-radius: 8px;
            --sidebar-text: #343a40;
            --sidebar-hover: rgba(78, 115, 223, 0.1);
            --sidebar-active: var(--primary);
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            background-color: var(--content-bg);
            margin: 0;
            padding: 0;
            overflow-x: hidden;
        }
        
        /* Layout structure */
        .dashboard-container {
            display: flex;
            min-height: 100vh;
            width: 100%;
            position: relative;
        }
        
        /* Sidebar */
        .sidebar {
            width: var(--sidebar-width);
            background-color: var(--sidebar-bg);
            color: var(--sidebar-text);
            position: fixed;
            height: 100vh;
            left: 0;
            top: 0;
            z-index: 1000;
            overflow-y: auto;
            transition: all 0.3s ease;
            box-shadow: 0 0 15px rgba(0, 0, 0, 0.1);
        }
        
        .sidebar-header {
            padding: 20px 15px;
            display: flex;
            align-items: center;
            border-bottom: 1px solid rgba(0,0,0,0.05);
            margin-bottom: 10px;
        }
        
        .sidebar-brand {
            color: var(--primary);
            text-decoration: none;
            font-weight: 700;
            font-size: 1.2rem;
            display: flex;
            align-items: center;
            transition: all 0.2s ease;
        }
        
        .sidebar-brand:hover {
            color: var(--primary-dark, #3a57c4);
            transform: translateX(3px);
        }
        
        .sidebar-brand img {
            max-width: 40px;
            margin-right: 10px;
        }
        
        .sidebar-nav {
            list-style: none;
            padding: 0;
            margin: 0;
        }
        
        .sidebar-heading {
            padding: 15px 20px 8px;
            font-size: 0.75rem;
            text-transform: uppercase;
            letter-spacing: 1.5px;
            color: var(--secondary);
            font-weight: 700;
        }
        
        .nav-item {
            margin: 3px 10px;
            border-radius: 6px;
            overflow: hidden;
        }
        
        .nav-link {
            display: flex;
            align-items: center;
            padding: 12px 15px;
            color: var(--sidebar-text);
            text-decoration: none;
            transition: all 0.2s ease;
            border-radius: 6px;
            position: relative;
            font-weight: 500;
        }
        
        .nav-link i {
            margin-right: 12px;
            font-size: 1rem;
            width: 20px;
            text-align: center;
            color: var(--secondary);
            transition: all 0.2s ease;
        }
        
        .nav-link:hover {
            background-color: var(--sidebar-hover);
            color: var(--primary);
        }
        
        .nav-link:hover i {
            color: var(--primary);
        }
        
        .nav-link.active {
            background-color: var(--primary);
            color: white;
            border-radius: 6px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.1);
        }
        
        .nav-link.active i {
            color: white;
        }
        
        /* Sidebar item badge */
        .nav-badge {
            position: absolute;
            right: 15px;
            background: var(--primary);
            color: white;
            font-size: 0.7rem;
            padding: 2px 6px;
            border-radius: 10px;
        }
        
        /* Main content */
        .content-wrapper {
            flex: 1;
            margin-left: var(--sidebar-width);
            padding: 20px;
            transition: margin-left 0.3s ease;
            overflow-y: auto;
            width: calc(100% - var(--sidebar-width));
            max-width: calc(100% - var(--sidebar-width));
        }
        
        /* Navbar */
        .dashboard-navbar {
            background-color: var(--card-bg);
            border-radius: var(--border-radius);
            padding: 15px 20px;
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }
        
        /* Welcome banner with night sky */
        .welcome-banner {
            background: linear-gradient(135deg, #0c1624, #12223d, #0e1b38);
            color: #fff;
            border-radius: var(--border-radius);
            padding: 25px;
            margin-bottom: 20px;
            position: relative;
            overflow: hidden;
            box-shadow: 0 4px 25px rgba(0,0,0,0.4);
        }
        
        /* Stars animation - multiple layers for more density */
        .welcome-banner::before {
            content: '';
            position: absolute;
            width: 100%;
            height: 100%;
            top: 0;
            left: 0;
            background-image: 
                radial-gradient(white, rgba(255,255,255,.2) 1px, transparent 1px),
                radial-gradient(white, rgba(255,255,255,.15) 1px, transparent 1px),
                radial-gradient(white, rgba(255,255,255,.1) 1px, transparent 1px);
            background-size: 350px 350px, 200px 200px, 150px 150px;
            background-position: 0 0, 40px 60px, 130px 270px;
            animation: twinkle 15s ease-in-out infinite alternate;
        }
        
        /* Additional star layers for more stars */
        .welcome-banner::after {
            content: '';
            position: absolute;
            width: 100%;
            height: 100%;
            top: 0;
            left: 0;
            background-image: 
                radial-gradient(white, rgba(255,255,255,.2) 0.5px, transparent 0.5px),
                radial-gradient(white, rgba(255,255,255,.1) 0.5px, transparent 0.5px),
                radial-gradient(white, rgba(255,255,255,.05) 0.5px, transparent 0.5px);
            background-size: 120px 120px, 170px 170px, 200px 200px;
            background-position: 20px 30px, 90px 110px, 60px 40px;
            animation: twinkle 10s ease-in-out infinite alternate;
        }
        
        /* Extra distant stars */
        .star-field {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            z-index: 0;
            background-image: 
                radial-gradient(white, rgba(255,255,255,.2) 0.3px, transparent 0.3px),
                radial-gradient(white, rgba(255,255,255,.1) 0.2px, transparent 0.2px);
            background-size: 80px 80px, 60px 60px;
            background-position: 10px 15px, 30px 35px;
            animation: twinkle 12s ease-in-out infinite alternate-reverse;
        }
        
        /* Shooting stars */
        .shooting-star {
            position: absolute;
            width: 150px;
            height: 2px;
            background: linear-gradient(to right, rgba(255,255,255,0) 0%, rgba(255,255,255,1) 50%, rgba(255,255,255,0) 100%);
            opacity: 0;
            transform: rotate(-45deg);
            animation: shooting 6s infinite ease-out;
            filter: drop-shadow(0 0 6px rgba(255, 255, 255, 0.7));
            z-index: 2;
        }
        
        /* Add glow to shooting stars */
        .shooting-star::after {
            content: '';
            position: absolute;
            top: -1px;
            right: 0;
            width: 5px;
            height: 5px;
            border-radius: 50%;
            background: white;
            box-shadow: 0 0 10px 4px rgba(255, 255, 255, 0.7);
        }
        
        .shooting-star:nth-child(3) {
            top: 10%;
            left: 80%;
            animation-delay: 0s;
            width: 120px;
        }
        
        .shooting-star:nth-child(4) {
            top: 30%;
            left: 40%;
            animation-delay: 3s;
            width: 150px;
        }
        
        .shooting-star:nth-child(5) {
            top: 60%;
            left: 70%;
            animation-delay: 5s;
            width: 100px;
        }
        
        .shooting-star:nth-child(6) {
            top: 20%;
            left: 30%;
            animation-delay: 7s;
            width: 130px;
        }
        
        .shooting-star:nth-child(7) {
            top: 50%;
            left: 20%;
            animation-delay: 9s;
            width: 80px;
        }
        
        .shooting-star:nth-child(8) {
            top: 40%;
            left: 60%;
            animation-delay: 11s;
            width: 140px;
        }
        
        /* Ultra-fine distant stars */
        .welcome-banner::before,
        .welcome-banner::after,
        .star-field {
            pointer-events: none; /* Ensure stars don't interfere with clicking */
        }
        
        .micro-stars {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            z-index: 1;
            background-image: 
                radial-gradient(white, rgba(255,255,255,.15) 0.1px, transparent 0.1px),
                radial-gradient(white, rgba(255,255,255,.1) 0.1px, transparent 0.1px);
            background-size: 30px 30px, 40px 40px;
            background-position: 5px 7px, 15px 20px;
            animation: micro-twinkle 20s ease-in-out infinite alternate;
        }
        
        /* Nebula-like glow effects */
        .nebula {
            position: absolute;
            border-radius: 50%;
            filter: blur(25px);
            opacity: 0.15;
            z-index: 0;
        }
        
        .nebula:nth-child(1) {
            top: 20%;
            left: 80%;
            width: 120px;
            height: 120px;
            background: radial-gradient(circle, rgba(66,134,244,0.8), transparent 70%);
            animation: pulse 15s infinite alternate ease-in-out;
        }
        
        .nebula:nth-child(2) {
            top: 60%;
            left: 15%;
            width: 150px;
            height: 150px;
            background: radial-gradient(circle, rgba(111,66,244,0.6), transparent 70%);
            animation: pulse 18s infinite alternate-reverse ease-in-out;
        }
        
        @keyframes twinkle {
            0%, 100% { opacity: 0.6; }
            25% { opacity: 0.8; }
            50% { opacity: 1; }
            75% { opacity: 0.7; }
        }
        
        @keyframes micro-twinkle {
            0%, 100% { opacity: 0.3; }
            30% { opacity: 0.5; }
            60% { opacity: 0.4; }
        }
        
        @keyframes pulse {
            0%, 100% { transform: scale(1); opacity: 0.1; }
            50% { transform: scale(1.2); opacity: 0.2; }
        }
        
        @keyframes shooting {
            0% { transform: translateX(0) translateY(0) rotate(-45deg); opacity: 0; }
            2% { opacity: 1; }
            10% { transform: translateX(-300px) translateY(300px) rotate(-45deg); opacity: 0; }
            100% { transform: translateX(-300px) translateY(300px) rotate(-45deg); opacity: 0; }
        }
        
        .welcome-banner h2 {
            font-weight: 700;
            margin-bottom: 10px;
            text-shadow: 0 2px 10px rgba(0,0,0,0.3);
            position: relative;
            z-index: 1;
        }
        
        .welcome-banner p {
            position: relative;
            z-index: 1;
            text-shadow: 0 2px 8px rgba(0,0,0,0.2);
        }
        
        .welcome-banner .btn {
            margin-right: 10px;
            position: relative;
            z-index: 1;
            transition: all 0.3s ease;
            border: 2px solid transparent;
        }
        
        .welcome-banner .btn-light {
            background: rgba(255,255,255,0.9);
            color: #203a43;
            font-weight: 500;
        }
        
        .welcome-banner .btn-light:hover {
            background: #fff;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }
        
        .welcome-banner .btn-outline-light {
            border: 2px solid rgba(255,255,255,0.7);
        }
        
        .welcome-banner .btn-outline-light:hover {
            background: rgba(255,255,255,0.1);
            border-color: #fff;
            transform: translateY(-2px);
        }
        
        /* Stat cards */
        .stat-card {
            background-color: var(--card-bg);
            border-radius: var(--border-radius);
            padding: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
            margin-bottom: 20px;
            height: 100%;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        .stat-icon {
            width: 48px;
            height: 48px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            margin-right: 15px;
            background-color: var(--primary-light);
        }
        
        .stat-number {
            font-size: 1.8rem;
            font-weight: 700;
            margin: 10px 0 5px;
        }
        
        /* Tables */
        .data-card {
            background-color: var(--card-bg);
            border-radius: var(--border-radius);
            margin-bottom: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
            overflow: hidden;
        }
        
        .data-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 20px;
            border-bottom: 1px solid rgba(0,0,0,0.05);
        }
        
        .data-title {
            font-weight: 600;
            margin: 0;
        }
        
        .table-responsive {
            padding: 0;
        }
        
        .table {
            margin-bottom: 0;
        }
        
        .table th {
            border-top: none;
            font-weight: 600;
            padding: 12px 15px;
            color: #555;
            font-size: 0.9rem;
            text-transform: uppercase;
        }
        
        .table td {
            padding: 15px;
            vertical-align: middle;
        }
        
        /* Chart containers */
        .chart-container {
            position: relative;
            height: 300px;
            margin-bottom: 20px;
        }
        
        /* Status badges */
        .status-badge {
            padding: 5px 10px;
            border-radius: 50px;
            font-size: 0.75rem;
            font-weight: 600;
        }
        
        .status-published {
            background-color: rgba(40, 167, 69, 0.1);
            color: #28a745;
        }
        
        .status-draft {
            background-color: rgba(108, 117, 125, 0.1);
            color: #6c757d;
        }
        
        /* Media queries for responsiveness */
        @media (max-width: 991px) {
            .sidebar {
                transform: translateX(-100%);
                z-index: 1050;
            }
            
            .content-wrapper {
                margin-left: 0;
                width: 100%;
                max-width: 100%;
            }
            
            .sidebar.show {
                transform: translateX(0);
            }
        }
    </style>
</head>
<body>
    <div class="d-flex">
        <!-- Sidebar -->
        <aside class="sidebar">
            <div class="sidebar-header">
                <a href="../index.php" class="sidebar-brand">
                    <img src="../assets/images/logo.jpeg" alt="Panadite Academy" onerror="this.src='../assets/images/favicon.ico';">
                    <span>Panadite Academy</span>
                </a>
            </div>
            
            <ul class="sidebar-nav">
                <li class="sidebar-heading">MAIN</li>
                
                <li class="nav-item">
                    <a href="dashboard.php" class="nav-link active">
                        <i class="fas fa-tachometer-alt"></i>
                        <span>Dashboard</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="courses.php" class="nav-link">
                        <i class="fas fa-graduation-cap"></i>
                        <span>Courses</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="create-course.php" class="nav-link">
                        <i class="fas fa-plus-circle"></i>
                        <span>Create Course</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="videos.php" class="nav-link">
                        <i class="fas fa-video"></i>
                        <span>Videos</span>
                        <span class="nav-badge">New</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="upload-video.php" class="nav-link">
                        <i class="fas fa-cloud-upload-alt"></i>
                        <span>Upload Video</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="students.php" class="nav-link">
                        <i class="fas fa-user-graduate"></i>
                        <span>Students</span>
                    </a>
                </li>
                
                <li class="sidebar-heading">TOOLS</li>
                
                <li class="nav-item">
                    <a href="analytics.php" class="nav-link">
                        <i class="fas fa-chart-line"></i>
                        <span>Analytics</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="quizzes.php" class="nav-link">
                        <i class="fas fa-question-circle"></i>
                        <span>Quizzes</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="assignments.php" class="nav-link">
                        <i class="fas fa-tasks"></i>
                        <span>Assignments</span>
                    </a>
                </li>
                
                <li class="sidebar-heading">ACCOUNT</li>
                
                <li class="nav-item">
                    <a href="profile.php" class="nav-link">
                        <i class="fas fa-user-circle"></i>
                        <span>Profile</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="settings.php" class="nav-link">
                        <i class="fas fa-cog"></i>
                        <span>Settings</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="../logout.php" class="nav-link">
                        <i class="fas fa-sign-out-alt"></i>
                        <span>Logout</span>
                    </a>
                </li>
            </ul>
        </aside>
        
        <!-- Content Wrapper -->
        <div class="content-wrapper w-100">
            <!-- Navbar -->
           <!-- Navbar -->
           <nav class="dashboard-navbar">
                <div class="d-flex align-items-center">
                    <button id="sidebarToggle" class="btn btn-sm btn-light me-2">
                        <i class="fas fa-bars"></i>
                    </button>
                    <h4 class="mb-0">Teacher Dashboard</h4>
                </div>
                
                <div class="d-flex align-items-center">
                    <!-- Search -->
                    <div class="me-3 d-none d-md-block">
                        <form class="d-flex">
                            <input type="text" class="form-control form-control-sm" placeholder="Search...">
                        </form>
                    </div>
                    
                    <!-- Notifications -->
                    <div class="dropdown me-3">
                        <button class="btn btn-light position-relative" data-bs-toggle="dropdown">
                            <i class="fas fa-bell"></i>
                            <span class="position-absolute top-0 start-100 translate-middle badge rounded-pill bg-danger">
                                3
                            </span>
                        </button>
                        <div class="dropdown-menu dropdown-menu-end">
                            <div class="p-2 border-bottom">
                                <h6 class="mb-0">Notifications</h6>
                            </div>
                            <div class="p-2 border-bottom">
                                <div class="d-flex">
                                    <div class="flex-shrink-0 me-2">
                                        <div class="rounded-circle bg-light p-2">
                                            <i class="fas fa-user-graduate text-primary"></i>
                                        </div>
                                    </div>
                                    <div>
                                        <p class="mb-0">New student enrolled</p>
                                        <small class="text-muted">2 hours ago</small>
                                    </div>
                                </div>
                            </div>
                            <div class="p-2 text-center">
                                <a href="#" class="text-decoration-none">View all notifications</a>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Profile -->
                    <div class="dropdown">
                        <button class="btn btn-light d-flex align-items-center" data-bs-toggle="dropdown">
                            <img src="<?php echo !empty($teacher['profile_picture']) ? '../uploads/profile/' . htmlspecialchars($teacher['profile_picture']) : '../assets/images/avatar-placeholder.jpg'; ?>" 
                                 class="rounded-circle me-2" height="32" width="32" alt="<?php echo htmlspecialchars($teacher['first_name']); ?>">
                            <span class="d-none d-md-inline"><?php echo htmlspecialchars($teacher['first_name']); ?></span>
                        </button>
                        <div class="dropdown-menu dropdown-menu-end">
                            <a href="profile.php" class="dropdown-item">
                                <i class="fas fa-user me-2"></i> Profile
                            </a>
                            <a href="settings.php" class="dropdown-item">
                                <i class="fas fa-cog me-2"></i> Settings
                            </a>
                            <div class="dropdown-divider"></div>
                            <a href="../logout.php" class="dropdown-item">
                                <i class="fas fa-sign-out-alt me-2"></i> Logout
                            </a>
                        </div>
                    </div>
                </div>
            </nav>
 
            
            <!-- Main Content -->
            <div class="container-fluid px-4 py-4">
                <div class="d-flex flex-column flex-md-row justify-content-between align-items-md-center mb-4 gap-3">
                    <div>
                        <h2 class="h4 mb-1">Quiz Preview</h2>
                        <p class="text-muted mb-0"><i class="fas fa-eye me-1"></i> Viewing how the quiz will appear to students</p>
                    </div>
                    <div class="d-flex gap-2">
                        <a href="quizzes.php" class="btn btn-outline-secondary d-flex align-items-center">
                            <i class="fas fa-arrow-left me-2"></i> Back to Quizzes
                        </a>
                        <a href="edit_quiz.php?id=<?= $quiz_id ?>" class="btn btn-primary d-flex align-items-center">
                            <i class="fas fa-edit me-2"></i> Edit Quiz
                        </a>
                    </div>
                </div>
                
                <div class="card shadow-sm border-0 mb-4 overflow-hidden">
                    <div class="card-body p-0">
                        <div class="p-4 pb-3 d-flex justify-content-between align-items-center border-bottom">
                            <h5 class="m-0 fw-bold">Quiz Details</h5>
                            <span class="badge status-badge <?= $quiz['is_published'] ? 'status-published' : 'status-draft' ?>">
                                <?= $quiz['is_published'] ? 'Published' : 'Draft' ?>
                            </span>
                        </div>
                        
                        <div class="p-4">
                            <h3 class="h2 fw-bold"><?= htmlspecialchars($quiz['title']) ?></h3>
                            <div class="d-inline-flex align-items-center bg-light rounded-pill px-3 py-1 mb-3">
                                <i class="fas fa-book-open text-primary me-2"></i>
                                <span><?= htmlspecialchars($quiz['course_title']) ?></span>
                            </div>
                            
                            <div class="row g-4 mt-2">
                                <div class="col-md-7">
                                    <div class="mb-3">
                                        <h6 class="fw-bold mb-2">Description</h6>
                                        <div class="bg-light rounded p-3">
                                            <?= nl2br(htmlspecialchars($quiz['description'])) ?: '<span class="text-muted fst-italic">No description provided</span>' ?>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="col-md-5">
                                    <div class="row g-3">
                                        <!-- Stats cards -->
                                        <div class="col-6">
                                            <div class="border rounded p-3 h-100">
                                                <div class="d-flex align-items-center mb-2">
                                                    <div class="rounded-circle bg-primary bg-opacity-10 p-2 me-2">
                                                        <i class="fas fa-clock text-primary"></i>
                                                    </div>
                                                    <span class="text-muted small">Time Limit</span>
                                                </div>
                                                <h5 class="fw-bold mb-0"><?= htmlspecialchars($quiz['time_limit']) ?> <small>min</small></h5>
                                            </div>
                                        </div>
                                        
                                        <div class="col-6">
                                            <div class="border rounded p-3 h-100">
                                                <div class="d-flex align-items-center mb-2">
                                                    <div class="rounded-circle bg-success bg-opacity-10 p-2 me-2">
                                                        <i class="fas fa-award text-success"></i>
                                                    </div>
                                                    <span class="text-muted small">Passing Score</span>
                                                </div>
                                                <h5 class="fw-bold mb-0"><?= htmlspecialchars($quiz['passing_score']) ?>%</h5>
                                            </div>
                                        </div>
                                        
                                        <div class="col-6">
                                            <div class="border rounded p-3 h-100">
                                                <div class="d-flex align-items-center mb-2">
                                                    <div class="rounded-circle bg-info bg-opacity-10 p-2 me-2">
                                                        <i class="fas fa-question-circle text-info"></i>
                                                    </div>
                                                    <span class="text-muted small">Questions</span>
                                                </div>
                                                <h5 class="fw-bold mb-0"><?= count($questions) ?></h5>
                                            </div>
                                        </div>
                                        
                                        <div class="col-6">
                                            <div class="border rounded p-3 h-100">
                                                <div class="d-flex align-items-center mb-2">
                                                    <div class="rounded-circle bg-secondary bg-opacity-10 p-2 me-2">
                                                        <i class="fas fa-calendar-alt text-secondary"></i>
                                                    </div>
                                                    <span class="text-muted small">Created</span>
                                                </div>
                                                <h5 class="fw-bold mb-0"><?= date('M j, Y', strtotime($quiz['created_at'])) ?></h5>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="card shadow-sm border-0 mb-4 overflow-hidden">
                    <div class="card-body p-0">
                        <div class="p-4 pb-3 border-bottom d-flex justify-content-between align-items-center">
                            <div class="d-flex align-items-center">
                                <div class="rounded-circle bg-primary bg-opacity-10 p-2 me-3">
                                    <i class="fas fa-question text-primary"></i>
                                </div>
                                <h5 class="m-0 fw-bold">Questions</h5>
                            </div>
                            <span class="badge bg-primary rounded-pill"><?= count($questions) ?> Questions</span>
                        </div>
                        
                        <div class="p-4">
                            <?php if (empty($questions)): ?>
                                <div class="alert alert-info border-0 shadow-sm d-flex align-items-center">
                                    <i class="fas fa-info-circle me-3 fs-4"></i>
                                    <div>
                                        <p class="mb-1">This quiz doesn't have any questions yet.</p>
                                        <a href="edit_quiz.php?id=<?= $quiz_id ?>" class="btn btn-sm btn-primary mt-2">
                                            <i class="fas fa-plus me-1"></i> Add Questions
                                        </a>
                                    </div>
                                </div>
                            <?php else: ?>
                                <div class="question-list">
                                    <?php $questionNumber = 1; ?>
                                    <?php foreach ($questions as $question): ?>
                                        <div class="question-item mb-4 border rounded-3 overflow-hidden">
                                            <!-- Question header -->
                                            <div class="p-3 bg-light d-flex justify-content-between align-items-center">
                                                <div class="d-flex align-items-center">
                                                    <div class="question-number me-3 bg-primary text-white rounded-circle d-flex align-items-center justify-content-center" style="width: 28px; height: 28px; font-size: 0.85rem; font-weight: 600;">
                                                        <?= $questionNumber ?>
                                                    </div>
                                                    <h6 class="m-0 fw-bold">
                                                        <?php 
                                                            $typeName = ucfirst(str_replace('_', ' ', $question['question_type']));
                                                            $typeIcon = '';
                                                            
                                                            switch($question['question_type']) {
                                                                case 'multiple_choice':
                                                                    $typeIcon = 'list';
                                                                    break;
                                                                case 'true_false':
                                                                    $typeIcon = 'check-circle';
                                                                    break;
                                                                case 'short_answer':
                                                                    $typeIcon = 'pencil-alt';
                                                                    break;
                                                            }
                                                        ?>
                                                        <i class="fas fa-<?= $typeIcon ?> me-1 text-primary small"></i>
                                                        <?= $typeName ?> Question
                                                    </h6>
                                                </div>
                                                <div>
                                                    <span class="badge bg-primary bg-opacity-10 text-primary fw-normal"><?= $question['points'] ?> points</span>
                                                </div>
                                            </div>
                                            
                                            <!-- Question content -->
                                            <div class="p-3">
                                                <div class="question-text mb-4 p-3 bg-light bg-opacity-50 rounded">
                                                    <?= nl2br(htmlspecialchars($question['question_text'])) ?>
                                                </div>
                                                
                                                <?php if ($question['question_type'] == 'multiple_choice'): ?>
                                                    <div class="options-list">
                                                        <?php $optionIndex = 0; ?>
                                                        <?php foreach ($question['options'] as $option): ?>
                                                            <?php $optionIndex++; ?>
                                                            <div class="option-item d-flex p-2 ps-3 mb-2 rounded-3 border <?= $option['is_correct'] ? 'border-success bg-success bg-opacity-10' : '' ?>">
                                                                <div class="option-marker me-2 d-flex align-items-center justify-content-center">
                                                                    <?php if ($option['is_correct']): ?>
                                                                        <span class="d-flex align-items-center justify-content-center border border-success rounded-circle text-success" style="width: 24px; height: 24px;">
                                                                            <i class="fas fa-check"></i>
                                                                        </span>
                                                                    <?php else: ?>
                                                                        <span class="d-flex align-items-center justify-content-center border rounded-circle" style="width: 24px; height: 24px;">
                                                                            <?= chr(64 + $optionIndex) ?>
                                                                        </span>
                                                                    <?php endif; ?>
                                                                </div>
                                                                <div class="option-text d-flex align-items-center <?= $option['is_correct'] ? 'fw-semibold' : '' ?>">
                                                                    <?= htmlspecialchars($option['option_text']) ?>
                                                                    <?php if ($option['is_correct']): ?>
                                                                        <span class="ms-2 badge bg-success bg-opacity-10 text-success">Correct Answer</span>
                                                                    <?php endif; ?>
                                                                </div>
                                                            </div>
                                                        <?php endforeach; ?>
                                                    </div>
                                                <?php elseif ($question['question_type'] == 'true_false'): ?>
                                                    <div class="options-list">
                                                        <?php $trueFalseTexts = ['True', 'False']; ?>
                                                        <?php foreach ($question['options'] as $index => $option): ?>
                                                            <div class="option-item d-flex p-2 ps-3 mb-2 rounded-3 border <?= $option['is_correct'] ? 'border-success bg-success bg-opacity-10' : '' ?>">
                                                                <div class="option-marker me-2 d-flex align-items-center justify-content-center">
                                                                    <?php if ($option['is_correct']): ?>
                                                                        <span class="d-flex align-items-center justify-content-center border border-success rounded-circle text-success" style="width: 24px; height: 24px;">
                                                                            <i class="fas fa-check"></i>
                                                                        </span>
                                                                    <?php else: ?>
                                                                        <span class="d-flex align-items-center justify-content-center border rounded-circle" style="width: 24px; height: 24px;">
                                                                            <?= $index ? 'F' : 'T' ?>
                                                                        </span>
                                                                    <?php endif; ?>
                                                                </div>
                                                                <div class="option-text d-flex align-items-center <?= $option['is_correct'] ? 'fw-semibold' : '' ?>">
                                                                    <?= $trueFalseTexts[$index] ?>
                                                                    <?php if ($option['is_correct']): ?>
                                                                        <span class="ms-2 badge bg-success bg-opacity-10 text-success">Correct Answer</span>
                                                                    <?php endif; ?>
                                                                </div>
                                                            </div>
                                                        <?php endforeach; ?>
                                                    </div>
                                                <?php elseif ($question['question_type'] == 'short_answer'): ?>
                                                    <div class="alert border-success bg-success bg-opacity-10 d-flex align-items-center p-3">
                                                        <span class="d-flex align-items-center justify-content-center border border-success rounded-circle text-success me-3" style="width: 24px; height: 24px;">
                                                            <i class="fas fa-check"></i>
                                                        </span>
                                                        <div>
                                                            <div class="text-muted small mb-1">Correct Answer:</div>
                                                            <div class="fw-semibold"><?= htmlspecialchars($question['answer']) ?></div>
                                                        </div>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                        <?php $questionNumber++; ?>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                
                <!-- Action Footer -->
                <div class="action-footer py-3 px-4 border-top mt-4">
                    <div class="d-flex flex-column flex-md-row justify-content-between align-items-center gap-3">
                        <div>
                            <a href="edit_quiz.php?id=<?= $quiz_id ?>" class="btn btn-primary">
                                <i class="fas fa-edit me-2"></i> Edit Quiz
                            </a>
                            <a href="quizzes.php" class="btn btn-outline-secondary ms-2">
                                <i class="fas fa-arrow-left me-2"></i> Back to Quizzes
                            </a>
                        </div>
                        <?php if ($quiz['is_published']): ?>
                            <div class="d-flex align-items-center">
                                <i class="fas fa-link text-primary me-2"></i>
                                <span>Quiz Link: </span>
                                <div class="input-group ms-2" style="max-width: 300px;">
                                    <input type="text" class="form-control form-control-sm" id="quizLink" value="<?= htmlspecialchars($_SERVER['HTTP_HOST'] . '/student/take_quiz.php?id=' . $quiz_id) ?>" readonly>
                                    <button class="btn btn-sm btn-outline-primary" type="button" onclick="copyQuizLink()"><i class="fas fa-copy"></i></button>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Bootstrap Bundle with Popper -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- jQuery -->
    <script src="https://code.jquery.com/jquery-3.6.4.min.js"></script>
    
    <script>
    $(document).ready(function() {
        // Toggle sidebar
        $('#sidebarToggle').on('click', function() {
            $('.sidebar').toggleClass('show');
        });
        
        // Add animation to question items
        $('.question-item').each(function(index) {
            $(this).css({
                'opacity': '0',
                'transform': 'translateY(20px)'
            });
            
            setTimeout(() => {
                $(this).css({
                    'transition': 'all 0.5s ease',
                    'opacity': '1',
                    'transform': 'translateY(0)'
                });
            }, 100 * index);
        });
    });
    
    // Copy quiz link to clipboard
    function copyQuizLink() {
        const copyText = document.getElementById("quizLink");
        copyText.select();
        copyText.setSelectionRange(0, 99999); // For mobile devices
        navigator.clipboard.writeText(copyText.value);
        
        // Show feedback
        const button = copyText.nextElementSibling;
        const originalIcon = button.innerHTML;
        button.innerHTML = '<i class="fas fa-check"></i>';
        button.classList.add('btn-success');
        button.classList.remove('btn-outline-primary');
        
        setTimeout(() => {
            button.innerHTML = originalIcon;
            button.classList.remove('btn-success');
            button.classList.add('btn-outline-primary');
        }, 2000);
    }
    </script>
</body>
</html>
