<?php
// Enable full error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// CRITICAL: These runtime settings MUST be the very first thing in the script
// though they won't affect upload_max_filesize or post_max_size at runtime
// These must be set in php.ini and server restarted
ini_set('memory_limit', '2048M');
ini_set('max_execution_time', '600');
ini_set('max_input_time', '600');
ini_set('max_file_uploads', '20');

// Increase limits for large videos
//ini_set('upload_max_filesize', '2000M'); 
//ini_set('post_max_size', '2000M');      

// Set timeout to avoid interrupted uploads
set_time_limit(600);

// Start session AFTER setting limits
session_start();

// Check if user is logged in and is a teacher
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'teacher') {
    // Redirect to login page if not logged in or not a teacher
    header("Location: ../login.php");
    exit;
}

// Define variables to store form data and potential errors
$title = $description = $duration = $module_order = $error = "";
$teacher_id = $_SESSION['user_id'];
$upload_success = false;
$upload_error = false;
$error_message = "";

// Process the form submission
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Get form data
    $title = trim($_POST['title']);
    $description = trim($_POST['description']);
    $duration = isset($_POST['duration']) ? trim($_POST['duration']) : '';
    $duration_seconds = isset($_POST['duration_seconds']) ? intval($_POST['duration_seconds']) : 0;
    $module_order = isset($_POST['module_order']) ? trim($_POST['module_order']) : '';
    $visibility = isset($_POST['visibility']) ? trim($_POST['visibility']) : 'public';
    $upload_date = date('Y-m-d H:i:s');
    $course_id = isset($_POST['course_id']) ? intval($_POST['course_id']) : 1; // Default to course 1 if not specified

    // Debug information
    $debug_info = "\nPOST data: " . print_r($_POST, true) . "\nFILES data: " . print_r($_FILES, true);
    error_log($debug_info);
    
    // Fix for duration detection issues
    if (empty($duration)) {
        // If duration is empty but duration_seconds has a value, use it to generate a formatted duration
        if (!empty($duration_seconds) && $duration_seconds > 0) {
            $minutes = floor($duration_seconds / 60);
            $seconds = $duration_seconds % 60;
            $duration = sprintf("%d:%02d", $minutes, $seconds);
        }
    } else if (empty($duration_seconds) || $duration_seconds == 0) {
        // If duration has a value but duration_seconds is empty, parse the duration
        if (strpos($duration, ':') !== false) {
            $parts = explode(':', $duration);
            if (count($parts) === 2) {
                $duration_seconds = (intval($parts[0]) * 60) + intval($parts[1]);
            } else if (count($parts) === 3) {
                $duration_seconds = (intval($parts[0]) * 3600) + (intval($parts[1]) * 60) + intval($parts[2]);
            }
        } else {
            // If it's just a number, assume it's in minutes
            $duration_seconds = intval($duration) * 60;
        }
    }
    
    // Validate required fields
    if (empty($title)) {
        $error = "Video title is required";
    } elseif (empty($description)) {
        $error = "Video description is required";
    } elseif (empty($duration)) {
        $error = "Video duration is required - please enter it manually if it can't be detected";
        error_log("Duration validation failed. Duration: $duration, Duration seconds: $duration_seconds");
    } elseif (!isset($_FILES['video']) || $_FILES['video']['error'] === UPLOAD_ERR_NO_FILE) {
        $error = "Please select a video file to upload";
    } else {
        // File validation logic
        $allowed_exts = array("mp4", "mov", "avi", "wmv", "flv", "webm", "mkv");
        
        // Make sure video file exists in the upload
        if (!isset($_FILES["video"]) || $_FILES["video"]["error"] === UPLOAD_ERR_INI_SIZE) {
            // Special handling for file too large error
            $error = "The file is too large. Maximum allowed size is " . ini_get('upload_max_filesize');
            error_log("File too large error: " . $error);
            $_SESSION['error'] = $error;
            $_SESSION['debug_info'] = array(
                'error' => $error,
                'upload_max_filesize' => ini_get('upload_max_filesize'),
                'post_max_size' => ini_get('post_max_size'),
                'error_code' => isset($_FILES["video"]) ? $_FILES["video"]["error"] : 'No file'
            );
            header("Location: upload-video.php");
            exit();
        }
        
        $video_file = $_FILES["video"];
        $file_name = $video_file["name"];
        $file_tmp = $video_file["tmp_name"];
        $file_size = $video_file["size"];
        $file_ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
        
        // Calculate file size in MB for debugging and validation
        $file_size_mb = round($file_size / (1024 * 1024), 2);
        
        // Debug log
        error_log("Processing video upload: " . $file_name . " (" . $file_size_mb . "MB)");
        error_log("Temp file location: " . $file_tmp);

        // Check if the file extension is allowed
        $error = "";
        $upload_error = false;
        $upload_success = false;
        
        if (!in_array($file_ext, $allowed_exts)) {
            $error = "File extension not allowed. Please upload video files in formats: " . implode(', ', $allowed_exts);
        }
        
        // Check file size (500MB limit)
        $max_size = 500 * 1024 * 1024; // 500MB in bytes
        if ($file_size > $max_size) {
            $error = "File size too large. Maximum file size is 500MB.";
        }
        
        // If no errors, proceed with upload
        if (empty($error)) {
            // Debug: log upload attempt
            error_log("Video upload attempt: " . $file_name . " (" . $file_size . " bytes)");
            
            // Create a unique file name to prevent overwriting
            $new_file_name = uniqid('video_') . '.' . $file_ext;
            
            // Set upload directory - use absolute path instead of relative
            $upload_dir = __DIR__ . "/../uploads/videos/";
            
            // Log the actual path for debugging
            error_log("Using absolute upload directory: " . $upload_dir);
            
            // Create directory if it doesn't exist with better error handling
            if (!is_dir($upload_dir)) {
                $mkdir_result = @mkdir($upload_dir, 0777, true);
                if (!$mkdir_result) {
                    error_log("Failed to create directory: " . $upload_dir);
                    // Try using a fallback directory
                    $upload_dir = dirname(__DIR__) . "/uploads/videos/";
                    if (!is_dir($upload_dir)) {
                        @mkdir($upload_dir, 0777, true);
                    }
                    error_log("Using fallback directory: " . $upload_dir);
                }
            }
            
            // Set proper permissions for the directory
            @chmod($upload_dir, 0777);
            error_log("Directory permissions set for: " . $upload_dir);
            
            // Check if directory is writable
            if (!is_writable($upload_dir)) {
                $error = "Upload directory is not writable. Please contact administrator.";
                error_log("Directory not writable: $upload_dir");
                $upload_error = true;
            }
            
            // Force an absolute path with clear separators
            $base_dir = dirname(dirname(__FILE__)); // Two levels up to get project root
            $upload_dir = $base_dir . "/uploads/videos/";
            
            // Make sure directory exists with proper permissions
            if (!file_exists($upload_dir)) {
                mkdir($upload_dir, 0777, true);
            }
            chmod($upload_dir, 0777); // Force permissions
            
            // Use a simple path
            $upload_path = $upload_dir . $new_file_name;
            
            error_log("Using upload path: $upload_path");
            
            // Debug upload information
            error_log("Attempting to upload from temp: $file_tmp to: $upload_path");
            
            // Check if temp file exists and is readable
            if (!file_exists($file_tmp)) {
                $error = "Temporary file not found. Upload failed.";
                error_log("Temporary file not found: $file_tmp");
                $upload_error = true;
            } else {
                // For short videos, don't log excessive debug info
                $file_size_mb = $file_size / (1024 * 1024);
                
                // Move the uploaded file to the designated folder - use direct file operations for small files
                if ($file_size_mb < 10) { // Under 10MB, use simpler approach for speed
                    // Check if upload directory exists and is writable
                    if (!file_exists(dirname($upload_path))) {
                        mkdir(dirname($upload_path), 0777, true);
                        error_log("Created directory: " . dirname($upload_path));
                    }
                    
                    // Check permissions
                    $dir_writable = is_writable(dirname($upload_path));
                    error_log("Upload directory writable: " . ($dir_writable ? 'Yes' : 'No'));
                    
                    // Ultra simple upload approach - just use one reliable method at a time
                    $upload_success = false;
                    
                    error_log("Attempting file upload from $file_tmp to $upload_path");
                    
                    // Try move_uploaded_file first (most reliable for web uploads)
                    if (move_uploaded_file($file_tmp, $upload_path)) {
                        error_log("SUCCESS: File uploaded with move_uploaded_file");
                        $upload_success = true;
                    } 
                    // Then try copy as fallback
                    else if (copy($file_tmp, $upload_path)) {
                        error_log("SUCCESS: File uploaded with copy");
                        $upload_success = true;
                    }
                    // Log the error if both methods fail
                    else {
                        $error = "Failed to upload file using multiple methods";
                        error_log("UPLOAD ERROR: " . $error);
                        error_log("Temp file exists: " . (file_exists($file_tmp) ? "Yes" : "No"));
                        error_log("Temp file readable: " . (is_readable($file_tmp) ? "Yes" : "No"));
                        error_log("Dest dir writable: " . (is_writable(dirname($upload_path)) ? "Yes" : "No"));
                        $upload_error = true;
                    }
                } else {
                    // For larger files, use the normal approach with detailed logging
                    if (move_uploaded_file($file_tmp, $upload_path)) {
                        error_log("File successfully moved to: $upload_path");
                        $thumbnail_path = "";
                    } else {
                        $php_error = error_get_last();
                        $error_msg = isset($php_error['message']) ? $php_error['message'] : 'Unknown error';
                        $error = "Failed to move uploaded file. Error: $error_msg";
                        error_log("Failed to move uploaded file: $error_msg");
                        error_log("PHP upload error code: " . $video_file['error']);
                        $upload_error = true;
                    }
                }
                
                // Check if thumbnail was uploaded
                if (isset($_FILES['thumbnail']) && $_FILES['thumbnail']['error'] !== UPLOAD_ERR_NO_FILE) {
                    $thumb_file = $_FILES['thumbnail'];
                    $thumb_ext = strtolower(pathinfo($thumb_file['name'], PATHINFO_EXTENSION));
                    
                    // Check thumbnail extension
                    $allowed_thumb_exts = array("jpg", "jpeg", "png", "gif");
                    
                    if (in_array($thumb_ext, $allowed_thumb_exts)) {
                        $thumb_name = uniqid('thumb_') . '.' . $thumb_ext;
                        $thumb_path = "../uploads/thumbnails/" . $thumb_name;
                        
                        // Create thumbnail directory if it doesn't exist
                        if (!is_dir("../uploads/thumbnails/")) {
                            mkdir("../uploads/thumbnails/", 0755, true);
                        }
                        
                        if (move_uploaded_file($thumb_file['tmp_name'], $thumb_path)) {
                            $thumbnail_path = $thumb_path;
                        }
                    }
                }
            }
            
            // Check if thumbnail was uploaded
            if (isset($_FILES['thumbnail']) && $_FILES['thumbnail']['error'] !== UPLOAD_ERR_NO_FILE) {
                $thumb_file = $_FILES['thumbnail'];
                $thumb_ext = strtolower(pathinfo($thumb_file['name'], PATHINFO_EXTENSION));
                
                // Check thumbnail extension
                $allowed_thumb_exts = array("jpg", "jpeg", "png", "gif");
                
                if (in_array($thumb_ext, $allowed_thumb_exts)) {
                    $thumb_name = uniqid('thumb_') . '.' . $thumb_ext;
                    $thumb_path = "../uploads/thumbnails/" . $thumb_name;
                    
                    // Create thumbnail directory if it doesn't exist
                    if (!is_dir("../uploads/thumbnails/")) {
                        mkdir("../uploads/thumbnails/", 0755, true);
                    }
                    
                    if (move_uploaded_file($thumb_file['tmp_name'], $thumb_path)) {
                        $thumbnail_path = $thumb_path;
                    }
                }
            }
            
            // Store video information in the database
            // Fix the URL for database storage - use relative path from web root
            $db_video_url = str_replace('\\', '/', $upload_path); // Normalize path separators
            
            // Extract the relative path portion for storage in database
            $path_parts = explode('/uploads/', $db_video_url);
            if (count($path_parts) > 1) {
                $db_video_url = 'uploads/' . $path_parts[1];
            }
            
            // Process database insertion
            try {
                // Ensure we have a valid connection
                if (!isset($conn) || !$conn) {
                    $conn = new mysqli("localhost", "root", "", "panadite_academy");
                    error_log("Created new database connection");
                }
                
                // Use a web-relative path for the database URL
                $video_url = "uploads/videos/" . basename($upload_path);
                
                // Make sure we have a teacher_id
                if (!isset($teacher_id)) {
                    $teacher_id = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 1;
                }
                
                // Make sure course_id is valid
                if (empty($course_id)) {
                    $course_id = 1; // Default to first course if none specified
                }

                // Try prepared statement first
                $stmt = $conn->prepare("INSERT INTO videos (title, video_url, course_id, teacher_id) VALUES (?, ?, ?, ?)");
                if ($stmt) {
                    $stmt->bind_param("ssii", $title, $db_video_url, $course_id, $teacher_id);
                    
                    error_log("Executing SQL with: title=$title, url=$db_video_url, course_id=$course_id, teacher_id=$teacher_id");
                    
                    if ($stmt->execute()) {
                        $insert_success = true;
                    } else {
                        error_log("SQL execute failed: " . $stmt->error);
                    }
                    $stmt->close();
                } else {
                    error_log("Failed to prepare statement: " . $conn->error);
                }
            } catch (Exception $e) {
                error_log("Database exception: " . $e->getMessage());
                $error = "Database error: " . $e->getMessage();
                $upload_error = true;
            }

            // Check if thumbnail was uploaded
            if (isset($_FILES['thumbnail']) && $_FILES['thumbnail']['error'] !== UPLOAD_ERR_NO_FILE) {
                $thumb_file = $_FILES['thumbnail'];
                $thumb_ext = strtolower(pathinfo($thumb_file['name'], PATHINFO_EXTENSION));

                // Check thumbnail extension
                $allowed_thumb_exts = array("jpg", "jpeg", "png", "gif");

                if (in_array($thumb_ext, $allowed_thumb_exts)) {
                    $thumb_name = uniqid('thumb_') . '.' . $thumb_ext;
                    $thumb_path = "../uploads/thumbnails/" . $thumb_name;

                    // Create thumbnail directory if it doesn't exist
                    if (!is_dir("../uploads/thumbnails/")) {
                        mkdir("../uploads/thumbnails/", 0755, true);
                    }

                    if (move_uploaded_file($thumb_file['tmp_name'], $thumb_path)) {
                        $thumbnail_path = $thumb_path;
                }
            }
        }
    }

    // Store video information in the database
    // Fix the URL for database storage - use relative path from web root
    $db_video_url = str_replace('\\', '/', $upload_path); // Normalize path separators

    // Extract the relative path portion for storage in database
    $path_parts = explode('/uploads/', $db_video_url);
    if (count($path_parts) > 1) {
        $db_video_url = 'uploads/' . $path_parts[1];
    }

    // Process database insertion
    try {
        // Ensure we have a valid connection
        if (!isset($conn) || !$conn) {
            $conn = new mysqli("localhost", "root", "", "panadite_academy");
            error_log("Created new database connection");
        }

        // Use a web-relative path for the database URL
        $video_url = "uploads/videos/" . basename($upload_path);

        // Make sure we have a teacher_id
        if (!isset($teacher_id)) {
            $teacher_id = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 1;
        }

        // Make sure course_id is valid
        if (empty($course_id)) {
            $course_id = 1; // Default to first course if none specified
        }

        // Try prepared statement first
        $stmt = $conn->prepare("INSERT INTO videos (title, video_url, course_id, teacher_id) VALUES (?, ?, ?, ?)");
        if ($stmt) {
            $stmt->bind_param("ssii", $title, $db_video_url, $course_id, $teacher_id);

            error_log("Executing SQL with: title=$title, url=$db_video_url, course_id=$course_id, teacher_id=$teacher_id");

            if ($stmt->execute()) {
                $insert_success = true;
            } else {
                error_log("SQL execute failed: " . $stmt->error);
            }
            $stmt->close();
        } else {
            error_log("Failed to prepare statement: " . $conn->error);
        }

        // If insert was successful, redirect
        if ($insert_success) {                
            // Success - redirect to the videos page
            $_SESSION['success'] = "Video uploaded successfully.";
            header("Location: videos.php");
            exit();
        } else {
            $error = "Failed to insert video into database";
            $upload_error = true;
        }

    } catch (Exception $e) {
        error_log("Database exception: " . $e->getMessage());
        $error = "Database error: " . $e->getMessage();
        $upload_error = true;

        // Delete uploaded files if database insertion fails
        if (file_exists($upload_path)) {
            @unlink($upload_path);
        }
    }

    // Check for upload errors
    if (!isset($upload_success) || !$upload_success) {
        $error = "Failed to upload the video. Please try again.";
        $upload_error = true;
    }
} // End of the main if(empty($error)) block from line 136
else {
    // This else belongs to the if(empty($error)) check from line 136
    $upload_error = true;
    $error = "There was an issue with your file. Please try again."; // Set a default error message
}

// Debug information
$debug_info = array(
    'error' => $error,
    'file_name' => isset($file_name) ? $file_name : '',
    'file_size' => isset($file_size) ? $file_size : '',
    'file_tmp' => isset($file_tmp) ? $file_tmp : '',
    'upload_path' => isset($upload_path) ? $upload_path : '',
    'upload_error_code' => isset($_FILES["video"]["error"]) ? $_FILES["video"]["error"] : ''
);

// Add environment info to debug data
$debug_info['environment'] = sprintf("PHP %s | Upload max filesize: %s | Post max size: %s",
    phpversion(),
    ini_get('upload_max_filesize'),
    ini_get('post_max_size')
);

// Set session error message if there was an upload error
if ($upload_error) {
    $_SESSION['error'] = "Failed to upload the video. Please try again.";
    if ($debug_mode) {
        $_SESSION['debug_info'] = $debug_info;
    }
    header("Location: upload-video.php");
    exit();
} else if ($insert_success) {
    // Success - redirect to the videos page
    $_SESSION['success'] = "Video uploaded successfully.";
    header("Location: videos.php");
    exit();
} else {
    // Fallback - something unexpected happened
    $_SESSION['error'] = "An unexpected error occurred during upload.";
    header("Location: upload-video.php");
    exit();
}

// Close database connection
$conn->close();
?>
