<?php
// Enable full error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// CRITICAL: These runtime settings MUST be set before anything else
ini_set('memory_limit', '2048M');
ini_set('max_execution_time', '600');
ini_set('max_input_time', '600');
ini_set('max_file_uploads', '20');

// Set timeout to avoid interrupted uploads
set_time_limit(600);

// Start session AFTER setting limits
session_start();

// Check if user is logged in and is a teacher
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'teacher') {
    // Redirect to login page if not logged in or not a teacher
    header("Location: ../login.php");
    exit;
}

// Initialize variables
$error = "";
$upload_error = false;
$upload_success = false;
$insert_success = false;
$debug_info = array();
$debug_mode = true; // Set to false in production
$teacher_id = $_SESSION['user_id'];
$thumbnail_path = "";

// Process form submission
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Get form data
    $title = trim($_POST['title']);
    $description = isset($_POST['description']) ? trim($_POST['description']) : "";
    $course_id = isset($_POST['course_id']) ? (int)$_POST['course_id'] : 1;
    $duration = isset($_POST['duration']) ? trim($_POST['duration']) : '';
    $duration_seconds = isset($_POST['duration_seconds']) ? (int)$_POST['duration_seconds'] : 0;
    
    // Basic validation
    if (empty($title)) {
        $error = "Title is required";
    } 
    elseif (empty($duration)) {
        $error = "Video duration is required - please enter it manually if it can't be detected";
        error_log("Duration validation failed. Duration: $duration, Duration seconds: $duration_seconds");
    } 
    elseif (!isset($_FILES['video']) || $_FILES['video']['error'] === UPLOAD_ERR_NO_FILE) {
        $error = "Please select a video file to upload";
    } 
    else {
        // File validation logic
        $allowed_exts = array("mp4", "mov", "avi", "wmv", "flv", "webm", "mkv");
        
        // Check for file upload errors, especially size limits
        if ($_FILES["video"]["error"] === UPLOAD_ERR_INI_SIZE) {
            $error = "The file is too large. Maximum allowed size is " . ini_get('upload_max_filesize');
            error_log("File too large error: " . $error);
            $_SESSION['error'] = $error;
            $_SESSION['debug_info'] = array(
                'error' => $error,
                'upload_max_filesize' => ini_get('upload_max_filesize'),
                'post_max_size' => ini_get('post_max_size'),
                'error_code' => $_FILES["video"]["error"]
            );
            header("Location: upload-video.php");
            exit();
        }
        
        // Process the valid file
        $video_file = $_FILES["video"];
        $file_name = $video_file["name"];
        $file_tmp = $video_file["tmp_name"];
        $file_size = $video_file["size"];
        $file_ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
        
        // Calculate file size in MB for debugging and validation
        $file_size_mb = round($file_size / (1024 * 1024), 2);
        
        // Debug log
        error_log("Processing video upload: " . $file_name . " (" . $file_size_mb . "MB)");
        error_log("Temp file location: " . $file_tmp);

        // Check if the file extension is allowed
        if (!in_array($file_ext, $allowed_exts)) {
            $error = "File extension not allowed. Please upload video files in formats: " . implode(', ', $allowed_exts);
        }
        
        // If no errors, proceed with upload
        if (empty($error)) {
            // Debug: log upload attempt
            error_log("Video upload attempt: " . $file_name . " (" . $file_size . " bytes)");
            
            // Create a unique file name to prevent overwriting
            $new_file_name = uniqid('video_') . '.' . $file_ext;
            
            // Set upload directory - use absolute path instead of relative
            $upload_dir = __DIR__ . "/../uploads/videos/";
            
            // Log the actual path for debugging
            error_log("Using absolute upload directory: " . $upload_dir);
            
            // Create directory if it doesn't exist with better error handling
            if (!is_dir($upload_dir)) {
                error_log("Creating upload directory: " . $upload_dir);
                if (!mkdir($upload_dir, 0777, true)) {
                    $error_msg = error_get_last()['message'] ?? "Unknown error";
                    error_log("Failed to create directory: " . $error_msg);
                    $error = "Failed to create upload directory. Please contact the administrator.";
                    $upload_error = true;
                } else {
                    // Ensure proper permissions on the created directory
                    chmod($upload_dir, 0777);
                    error_log("Created directory with permissions: " . substr(sprintf('%o', fileperms($upload_dir)), -4));
                }
            }
            
            // Proceed if directory exists or was created successfully
            if (!$upload_error) {
                // Full path to where the file will be saved
                $upload_path = $upload_dir . $new_file_name;
                
                error_log("About to upload to: " . $upload_path);
                error_log("File size: " . $file_size . " bytes (" . $file_size_mb . " MB)");
                
                // CRITICAL: Try different upload methods based on file size
                $upload_success = false;
                
                // Method 1: Try PHP move_uploaded_file first (most reliable)
                if (move_uploaded_file($file_tmp, $upload_path)) {
                    error_log("File successfully moved to: $upload_path");
                    $upload_success = true;
                } 
                // Method 2: Try copy() as fallback
                elseif (copy($file_tmp, $upload_path)) {
                    error_log("File successfully copied to: $upload_path");
                    $upload_success = true;
                }
                // Method 3: Last resort for small files
                elseif ($file_size < 10 * 1024 * 1024) { // < 10MB
                    $content = @file_get_contents($file_tmp);
                    if ($content && file_put_contents($upload_path, $content)) {
                        error_log("File successfully written using file_get_contents/file_put_contents");
                        $upload_success = true;
                    }
                }
                
                // Check if upload succeeded
                if (!$upload_success) {
                    $php_error = error_get_last();
                    $error_msg = isset($php_error['message']) ? $php_error['message'] : 'Unknown error';
                    $error = "Failed to upload file. Error: $error_msg";
                    error_log("All file upload methods failed: $error_msg");
                    $upload_error = true;
                }
                
                // Check if thumbnail was uploaded
                if ($upload_success && isset($_FILES['thumbnail']) && $_FILES['thumbnail']['error'] !== UPLOAD_ERR_NO_FILE) {
                    $thumb_file = $_FILES['thumbnail'];
                    $thumb_ext = strtolower(pathinfo($thumb_file['name'], PATHINFO_EXTENSION));
                    
                    // Check thumbnail extension
                    $allowed_thumb_exts = array("jpg", "jpeg", "png", "gif");
                    
                    if (in_array($thumb_ext, $allowed_thumb_exts)) {
                        $thumb_name = uniqid('thumb_') . '.' . $thumb_ext;
                        $thumb_upload_dir = __DIR__ . "/../uploads/thumbnails/";
                        
                        // Create thumbnail directory if it doesn't exist
                        if (!is_dir($thumb_upload_dir)) {
                            mkdir($thumb_upload_dir, 0777, true);
                        }
                        
                        $thumb_path = $thumb_upload_dir . $thumb_name;
                        if (move_uploaded_file($thumb_file['tmp_name'], $thumb_path)) {
                            $thumbnail_path = "uploads/thumbnails/" . $thumb_name; // Web path for DB
                            error_log("Thumbnail uploaded to: $thumb_path");
                        } else {
                            error_log("Failed to upload thumbnail: " . error_get_last()['message'] ?? "Unknown error");
                        }
                    }
                }
                
                // Process database insertion if file upload succeeded
                if ($upload_success) {
                    // Store video information in the database
                    // Normalize path for database storage - use web-relative path
                    $db_video_url = "uploads/videos/" . $new_file_name;
                    
                    try {
                        // Ensure we have a valid connection
                        if (!isset($conn) || !$conn) {
                            $conn = new mysqli("localhost", "root", "", "panadite_academy");
                            error_log("Created new database connection");
                        }
                        
                        if ($conn->connect_error) {
                            throw new Exception("Database connection failed: " . $conn->connect_error);
                        }
                        
                        // Try prepared statement for database insertion
                        $stmt = $conn->prepare("INSERT INTO videos (title, video_url, course_id, teacher_id, description, duration) VALUES (?, ?, ?, ?, ?, ?)");
                        if ($stmt) {
                            $stmt->bind_param("ssiiss", $title, $db_video_url, $course_id, $teacher_id, $description, $duration);
                            
                            error_log("Executing SQL with: title=$title, url=$db_video_url, course_id=$course_id, teacher_id=$teacher_id");
                            
                            if ($stmt->execute()) {
                                $insert_success = true;
                                error_log("Video record inserted successfully");
                            } else {
                                error_log("SQL execute failed: " . $stmt->error);
                                $error = "Database error: " . $stmt->error;
                                $upload_error = true;
                            }
                            $stmt->close();
                        } else {
                            error_log("Failed to prepare statement: " . $conn->error);
                            $error = "Database error: " . $conn->error;
                            $upload_error = true;
                        }
                    } catch (Exception $e) {
                        error_log("Database exception: " . $e->getMessage());
                        $error = "Database error: " . $e->getMessage();
                        $upload_error = true;
                    }
                    
                    // If database insertion failed, delete the uploaded file
                    if (!$insert_success && file_exists($upload_path)) {
                        @unlink($upload_path);
                        error_log("Deleted uploaded file due to database insertion failure");
                    }
                }
            }
        }
    }
}

// Debug information for troubleshooting
$debug_info = array(
    'error' => $error,
    'file_name' => isset($file_name) ? $file_name : '',
    'file_size' => isset($file_size) ? $file_size : '',
    'file_tmp' => isset($file_tmp) ? $file_tmp : '',
    'php_version' => phpversion(),
    'upload_max_filesize' => ini_get('upload_max_filesize'),
    'post_max_size' => ini_get('post_max_size'),
    'memory_limit' => ini_get('memory_limit'),
    'max_execution_time' => ini_get('max_execution_time')
);

// Set appropriate session variables and redirect
if ($upload_error) {
    $_SESSION['error'] = "Failed to upload the video: " . $error;
    if ($debug_mode) {
        $_SESSION['debug_info'] = $debug_info;
    }
    header("Location: upload-video.php");
    exit();
} else if ($insert_success) {
    // Success - redirect to the videos page
    $_SESSION['success'] = "Video uploaded successfully.";
    header("Location: videos.php");
    exit();
} else if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // If we got here and it was a POST request, something went wrong
    $_SESSION['error'] = "An unexpected error occurred during upload.";
    if ($debug_mode) {
        $_SESSION['debug_info'] = $debug_info;
    }
    header("Location: upload-video.php");
    exit();
}
?>
