<?php
session_start();

// Configure PHP for unlimited file uploads
ini_set('upload_max_filesize', '0'); // 0 means no limit
ini_set('post_max_size', '0');       // 0 means no limit
ini_set('memory_limit', '-1');       // -1 means no limit
ini_set('max_execution_time', 0);    // 0 means no time limit
ini_set('max_input_time', -1);       // -1 means no limit
ini_set('max_file_uploads', 50);

// Alternative method: Set via .htaccess style for built-in server
@ini_set('upload_max_filesize', '999999M');
@ini_set('post_max_size', '999999M');
@ini_set('memory_limit', '8192M');

// Enable error reporting for debugging
ini_set('display_errors', 1);
error_reporting(E_ALL);

// Set content type for JSON response
header('Content-Type: application/json');

// Log the start of the request
error_log("VIDEO UPLOAD REQUEST STARTED at " . date('Y-m-d H:i:s'));
error_log("REQUEST METHOD: " . $_SERVER['REQUEST_METHOD']);
error_log("SESSION USER ID: " . ($_SESSION['user_id'] ?? 'not set'));
error_log("SESSION ROLE: " . ($_SESSION['user_role'] ?? 'not set'));

// Include database connection
require_once '../includes/config.php';

// Function to send JSON response
function sendResponse($success, $message, $data = []) {
    echo json_encode([
        'success' => $success,
        'message' => $message,
        'data' => $data
    ]);
    exit();
}

// Function to log errors
function logError($message, $data = null) {
    error_log("VIDEO UPLOAD ERROR: $message");
    if ($data) {
        error_log("ERROR DATA: " . print_r($data, true));
    }
}

// Check if user is authenticated and is a teacher
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'teacher') {
    sendResponse(false, 'Unauthorized access');
}

// Check if this is a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse(false, 'Invalid request method');
}

// Get teacher ID from session
$teacher_id = (int)$_SESSION['user_id'];

try {
    // Log request data for debugging
    error_log("POST data keys: " . implode(', ', array_keys($_POST)));
    error_log("FILES data keys: " . implode(', ', array_keys($_FILES)));
    
    if (isset($_FILES['video'])) {
        error_log("Video file info: name={$_FILES['video']['name']}, size={$_FILES['video']['size']}, error={$_FILES['video']['error']}");
    }
    
    // Validate required fields
    $title = trim($_POST['title'] ?? '');
    $course_id = (int)($_POST['course_id'] ?? 0);
    $description = trim($_POST['description'] ?? '');
    $module_order = (int)($_POST['module_order'] ?? 1);
    $price = (float)($_POST['price'] ?? 0);
    $category_name = trim($_POST['category_name'] ?? '');
    $is_published = (int)($_POST['is_published'] ?? 1);
    $is_free = (int)($_POST['is_free'] ?? 0);
    $duration = (float)($_POST['duration'] ?? 0);
    
    error_log("Parsed data: title='$title', course_id=$course_id, duration=$duration");

    // Validation
    if (empty($title)) {
        sendResponse(false, 'Video title is required');
    }

    if ($course_id <= 0) {
        sendResponse(false, 'Please select a valid course');
    }

    // Verify course belongs to teacher
    $course_check = $conn->prepare("SELECT course_id FROM courses WHERE course_id = ? AND teacher_id = ?");
    $course_check->bind_param("ii", $course_id, $teacher_id);
    $course_check->execute();
    if ($course_check->get_result()->num_rows === 0) {
        sendResponse(false, 'Course not found or access denied');
    }
    $course_check->close();

    // Check if video file was uploaded
    if (!isset($_FILES['video'])) {
        logError('No video file in $_FILES array', $_FILES);
        sendResponse(false, 'No video file uploaded. Please select a video file.');
    }
    
    if ($_FILES['video']['error'] !== UPLOAD_ERR_OK) {
        $error_code = $_FILES['video']['error'];
        
        // For size-related errors, try to continue anyway if file exists
        if ($error_code === UPLOAD_ERR_INI_SIZE || $error_code === UPLOAD_ERR_FORM_SIZE) {
            logError("Size limit error detected but attempting to continue", $_FILES['video']);
            // Don't return error - try to continue with the upload
        } else {
            $error_messages = [
                UPLOAD_ERR_PARTIAL => 'File was only partially uploaded',
                UPLOAD_ERR_NO_FILE => 'No file was uploaded',
                UPLOAD_ERR_NO_TMP_DIR => 'Missing temporary folder',
                UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk',
                UPLOAD_ERR_EXTENSION => 'A PHP extension stopped the upload'
            ];
            
            $error_msg = $error_messages[$error_code] ?? 'Unknown upload error: ' . $error_code;
            logError("Upload error code $error_code: $error_msg", $_FILES['video']);
            sendResponse(false, "Upload failed: $error_msg");
        }
    }

    $video_file = $_FILES['video'];
    $video_name = $video_file['name'];
    $video_tmp = $video_file['tmp_name'];
    $video_size = $video_file['size'];
    $video_type = $video_file['type'];

    // Validate file type
    $allowed_types = ['video/mp4', 'video/webm', 'video/quicktime', 'video/avi', 'video/x-msvideo'];
    if (!in_array($video_type, $allowed_types)) {
        sendResponse(false, 'Invalid video file type. Supported: MP4, WebM, MOV, AVI');
    }

    // Create upload directories
    $uploads_dir = '../uploads';
    $videos_dir = $uploads_dir . '/videos';
    $thumbnails_dir = $uploads_dir . '/thumbnails';

    if (!file_exists($uploads_dir)) mkdir($uploads_dir, 0755, true);
    if (!file_exists($videos_dir)) mkdir($videos_dir, 0755, true);
    if (!file_exists($thumbnails_dir)) mkdir($thumbnails_dir, 0755, true);

    // Generate unique filename for video
    $file_extension = pathinfo($video_name, PATHINFO_EXTENSION);
    $video_filename = 'video_' . uniqid() . '_' . time() . '.' . $file_extension;
    $video_path = $videos_dir . '/' . $video_filename;

    // Move uploaded video file
    if (!move_uploaded_file($video_tmp, $video_path)) {
        logError('Failed to move video file', ['from' => $video_tmp, 'to' => $video_path]);
        sendResponse(false, 'Failed to save video file');
    }

    // Handle thumbnail upload (optional)
    $thumbnail_path = null;
    if (isset($_FILES['thumbnail']) && $_FILES['thumbnail']['error'] === UPLOAD_ERR_OK) {
        $thumbnail_file = $_FILES['thumbnail'];
        $thumbnail_name = $thumbnail_file['name'];
        $thumbnail_tmp = $thumbnail_file['tmp_name'];
        $thumbnail_type = $thumbnail_file['type'];

        // Validate thumbnail type
        $allowed_thumb_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        if (in_array($thumbnail_type, $allowed_thumb_types)) {
            $thumb_extension = pathinfo($thumbnail_name, PATHINFO_EXTENSION);
            $thumbnail_filename = 'thumb_' . uniqid() . '_' . time() . '.' . $thumb_extension;
            $thumbnail_full_path = $thumbnails_dir . '/' . $thumbnail_filename;

            if (move_uploaded_file($thumbnail_tmp, $thumbnail_full_path)) {
                $thumbnail_path = 'uploads/thumbnails/' . $thumbnail_filename;
            }
        }
    }

    // Format duration for database storage
    $duration_formatted = '';
    if ($duration > 0) {
        $hours = floor($duration / 3600);
        $minutes = floor(($duration % 3600) / 60);
        $seconds = floor($duration % 60);
        
        if ($hours > 0) {
            $duration_formatted = sprintf('%d:%02d:%02d', $hours, $minutes, $seconds);
        } else {
            $duration_formatted = sprintf('%d:%02d', $minutes, $seconds);
        }
    }

    // Insert video record into database
    $video_url = 'uploads/videos/' . $video_filename;
    
    $sql = "INSERT INTO videos (title, video_url, course_id, teacher_id, description, duration, thumbnail, price, category_name, is_free, is_published, module_order) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    
    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        logError('Failed to prepare statement', $conn->error);
        sendResponse(false, 'Database error occurred');
    }

    $stmt->bind_param(
        'ssiiissdssii',
        $title,
        $video_url,
        $course_id,
        $teacher_id,
        $description,
        $duration_formatted,
        $thumbnail_path,
        $price,
        $category_name,
        $is_free,
        $is_published,
        $module_order
    );

    if (!$stmt->execute()) {
        logError('Failed to insert video record', $stmt->error);
        // Clean up uploaded files
        if (file_exists($video_path)) unlink($video_path);
        if ($thumbnail_path && file_exists('../' . $thumbnail_path)) unlink('../' . $thumbnail_path);
        sendResponse(false, 'Failed to save video information');
    }

    $video_id = $conn->insert_id;
    $stmt->close();

    // Log successful upload
    error_log("VIDEO UPLOAD SUCCESS: Video ID $video_id uploaded by teacher $teacher_id");

    // Send success response
    sendResponse(true, 'Video uploaded successfully', [
        'video_id' => $video_id,
        'video_url' => $video_url,
        'thumbnail' => $thumbnail_path,
        'title' => $title,
        'duration' => $duration_formatted,
        'file_size' => $video_size
    ]);

} catch (Exception $e) {
    logError('Exception occurred', $e->getMessage());
    sendResponse(false, 'An error occurred: ' . $e->getMessage());
}
?>
