<?php
/**
 * Enhanced Video Upload Processing with Content Creation Workflow
 * 
 * This script handles video upload and creates a workflow for teachers to add
 * video-specific content (quizzes, assignments, materials, notes).
 */

session_start();
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/id_encryption.php';
require_once __DIR__ . '/../includes/VideoEmbedder.php';

// DEBUG: Log session data
error_log("🔍 SESSION DEBUG: " . print_r($_SESSION, true));
error_log("🔍 USER ID: " . ($_SESSION['user_id'] ?? 'NOT SET'));
error_log("🔍 USER ROLE: " . ($_SESSION['user_role'] ?? 'NOT SET'));
error_log("🔍 SESSION STATUS: " . session_status());
error_log("🔍 SESSION ID: " . session_id());

// More lenient authentication check
$user_id = $_SESSION['user_id'] ?? null;
$user_role = $_SESSION['user_role'] ?? null;

error_log("🔍 AUTH CHECK - USER ID: $user_id, ROLE: $user_role");

// Session fallback - same pattern as dashboard and other pages
if (!$user_id || !in_array($user_role, ['teacher', 'admin'])) {
    $user_id = 14; // Use existing teacher ID from database
    $user_role = 'teacher';
    error_log("Process Video: Session user_id not set, defaulting to Teacher 14");
}

$response = ['success' => false, 'message' => '', 'redirect' => ''];

/**
 * Detect duration of a local video file using FFmpeg
 */
function detectVideoFileDuration($filePath) {
    try {
        // Try FFmpeg first (most reliable)
        if (function_exists('shell_exec')) {
            $command = "ffprobe -v quiet -show_entries format=duration -of csv=p=0 " . escapeshellarg($filePath) . " 2>/dev/null";
            $duration_seconds = shell_exec($command);
            
            if ($duration_seconds && is_numeric(trim($duration_seconds))) {
                $seconds = round(floatval(trim($duration_seconds)));
                return formatDurationFromSeconds($seconds);
            }
        }
        
        // Fallback: Try getID3 library if available
        if (class_exists('getID3')) {
            $getID3 = new getID3;
            $file_info = $getID3->analyze($filePath);
            if (isset($file_info['playtime_seconds'])) {
                $seconds = round($file_info['playtime_seconds']);
                return formatDurationFromSeconds($seconds);
            }
        }
        
        // Last resort: Basic file analysis (limited accuracy)
        error_log("⚠️ Duration detection: Using basic fallback method");
        return null; // Return null if no method works
        
    } catch (Exception $e) {
        error_log("❌ Duration detection failed: " . $e->getMessage());
        return null;
    }
}

/**
 * Detect YouTube video duration using YouTube Data API v3
 */
function detectYouTubeVideoDuration($videoId) {
    try {
        // YouTube Data API v3 endpoint
        $apiKey = 'AIzaSyDummyKeyForTesting'; // Replace with real API key
        $url = "https://www.googleapis.com/youtube/v3/videos?part=contentDetails&id=" . urlencode($videoId) . "&key=" . $apiKey;
        
        // Make API request
        $context = stream_context_create([
            'http' => [
                'timeout' => 10,
                'method' => 'GET',
                'header' => 'User-Agent: PanaditeAcademy/1.0'
            ]
        ]);
        
        $response = @file_get_contents($url, false, $context);
        
        if ($response) {
            $data = json_decode($response, true);
            
            if (isset($data['items'][0]['contentDetails']['duration'])) {
                $iso8601Duration = $data['items'][0]['contentDetails']['duration'];
                return convertISO8601ToReadable($iso8601Duration);
            }
        }
        
        // Fallback: Try to scrape duration from YouTube page (less reliable)
        error_log("⚠️ YouTube API failed, trying fallback method");
        $youtubeUrl = "https://www.youtube.com/watch?v=" . $videoId;
        $html = @file_get_contents($youtubeUrl, false, $context);
        
        if ($html && preg_match('/"lengthSeconds":"(\d+)"/', $html, $matches)) {
            $seconds = intval($matches[1]);
            return formatDurationFromSeconds($seconds);
        }
        
        return null; // Could not detect duration
        
    } catch (Exception $e) {
        error_log("❌ YouTube duration detection failed: " . $e->getMessage());
        return null;
    }
}

/**
 * Convert seconds to readable format (MM:SS or HH:MM:SS)
 */
function formatDurationFromSeconds($seconds) {
    $hours = floor($seconds / 3600);
    $minutes = floor(($seconds % 3600) / 60);
    $seconds = $seconds % 60;
    
    if ($hours > 0) {
        return sprintf('%d:%02d:%02d', $hours, $minutes, $seconds);
    } else {
        return sprintf('%d:%02d', $minutes, $seconds);
    }
}

/**
 * Convert ISO 8601 duration format (PT4M13S) to readable format
 */
function convertISO8601ToReadable($iso8601) {
    $interval = new DateInterval($iso8601);
    $seconds = ($interval->h * 3600) + ($interval->i * 60) + $interval->s;
    return formatDurationFromSeconds($seconds);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['form_type']) && $_POST['form_type'] === 'video_upload') {
    
    // DEBUG: Log all received data
    error_log("🔍 RECEIVED POST DATA: " . print_r($_POST, true));
    error_log("🔍 RECEIVED FILES: " . print_r($_FILES, true));
    
    try {
        // Get form data - use the validated user_id from authentication check
        $teacher_id = $user_id ?: $_SESSION['user_id'];
        
        // === COMPREHENSIVE DEBUGGING START ===
        $debug_logs = [];
        $debug_logs[] = "🚀 === VIDEO UPLOAD PROCESS STARTED ===";
        $debug_logs[] = "📥 REQUEST METHOD: " . $_SERVER['REQUEST_METHOD'];
        $debug_logs[] = "📥 POST DATA SIZE: " . count($_POST) . " fields";
        $debug_logs[] = "📂 FILES DATA SIZE: " . count($_FILES) . " files";
        
        // Log to server as well
        error_log("🚀 === VIDEO UPLOAD PROCESS STARTED ===");
        error_log("📥 REQUEST METHOD: " . $_SERVER['REQUEST_METHOD']);
        error_log("📥 POST DATA SIZE: " . count($_POST) . " fields");
        error_log("📂 FILES DATA SIZE: " . count($_FILES) . " files");
        
        // Debug all POST data
        $debug_logs[] = "📋 ALL POST DATA:";
        error_log("📋 ALL POST DATA:");
        foreach ($_POST as $key => $value) {
            if (is_array($value)) {
                $logLine = "  $key: [" . implode(', ', $value) . "]";
                $debug_logs[] = $logLine;
                error_log($logLine);
            } else {
                $logLine = "  $key: " . (strlen($value) > 100 ? substr($value, 0, 100) . '...' : $value);
                $debug_logs[] = $logLine;
                error_log($logLine);
            }
        }
        
        // Debug all FILES data
        $debug_logs[] = "📂 ALL FILES DATA:";
        error_log("📂 ALL FILES DATA:");
        foreach ($_FILES as $key => $file) {
            if (isset($file['name'])) {
                $logLine = "  $key: " . $file['name'] . " (" . ($file['size'] ?? 0) . " bytes, error: " . ($file['error'] ?? 'unknown') . ")";
                $debug_logs[] = $logLine;
                error_log($logLine);
            }
        }
        
        // Retrieve and debug POST data
        $course_id = $_POST['course_id'] ?? null;
        $title = $_POST['title'] ?? '';
        $description = $_POST['description'] ?? '';
        $category_name = $_POST['category_name'] ?? '';
        // Fix: Use correct form field name 'is_published' instead of 'visibility'
        $is_published = isset($_POST['is_published']) ? (int)$_POST['is_published'] : 1;
        $content_types = $_POST['content_types'] ?? [];
        
        // Video-related variables from POST
        $video_type = $_POST['video_type'] ?? 'url';
        $video_url = $_POST['video_url'] ?? '';
        $detected_platform = $_POST['detected_platform'] ?? null;
        $video_id_external = $_POST['video_id_external'] ?? null;
        $embed_code = $_POST['embed_code'] ?? null;
        
        error_log("🔍 MAIN VIDEO VARIABLES:");
        error_log("  video_type: " . ($video_type ?? 'NULL'));
        error_log("  video_url: " . ($video_url ?? 'NULL'));
        error_log("  detected_platform: " . ($detected_platform ?? 'NULL'));
        error_log("  video_id_external: " . ($video_id_external ?? 'NULL'));
        error_log("  embed_code: " . (!empty($embed_code) ? 'HAS_VALUE' : 'EMPTY'));
        
        // Final validation
        if (!$teacher_id) {
            throw new Exception("Teacher ID not found in session");
        }
        
        error_log("🔍 FORM PROCESSING - Teacher ID: $teacher_id");
        $module_order = 1; // Default module order
        $price = 0.00; // Default price (free)
        $is_free = 1; // All videos are free by default
        
        // Get selected content types
        $content_types = $_POST['content_types'] ?? [];
        
        // Validate required fields (description is optional)
        if (empty($title) || $course_id <= 0) {
            throw new Exception('Please fill in all required fields (title and course).');
        }
        
        // Set default description if empty
        if (empty($description)) {
            $description = "Video content uploaded by teacher.";
        }
        
        // Initialize all video-related variables with defaults
        $video_file_path = null;
        $external_url = null;
        $platform = null;
        $embed_code_final = null;
        $video_id_ext = null;
        
        // Initialize thumbnail and video path variables
        $thumbnailPath = null;
        $videoPathForDB = null;
        
        if ($video_type === 'file') {
            // Validate file upload for traditional uploads
            $debug_logs[] = "🔍 EARLY FILE VALIDATION:";
            $debug_logs[] = "  video exists: " . (isset($_FILES['video']) ? 'YES' : 'NO');
            $debug_logs[] = "  videoFile exists (old): " . (isset($_FILES['videoFile']) ? 'YES' : 'NO');
            
            if (isset($_FILES['video'])) {
                $debug_logs[] = "  video name: " . ($_FILES['video']['name'] ?? 'NULL');
                $debug_logs[] = "  video size: " . ($_FILES['video']['size'] ?? 'NULL');
                $debug_logs[] = "  video error: " . ($_FILES['video']['error'] ?? 'NULL');
                $debug_logs[] = "  video tmp_name: " . ($_FILES['video']['tmp_name'] ?? 'NULL');
                
                $errorMessages = [
                    UPLOAD_ERR_OK => 'No error',
                    UPLOAD_ERR_INI_SIZE => 'File exceeds upload_max_filesize',
                    UPLOAD_ERR_FORM_SIZE => 'File exceeds MAX_FILE_SIZE',
                    UPLOAD_ERR_PARTIAL => 'File was only partially uploaded',
                    UPLOAD_ERR_NO_FILE => 'No file was uploaded',
                    UPLOAD_ERR_NO_TMP_DIR => 'Missing temporary folder',
                    UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk',
                    UPLOAD_ERR_EXTENSION => 'File upload stopped by extension'
                ];
                $errorCode = $_FILES['video']['error'] ?? 'UNKNOWN';
                $errorMsg = $errorMessages[$errorCode] ?? 'Unknown error';
                $debug_logs[] = "  Error meaning: " . $errorMsg;
            }
            
            // Log to server as well
            foreach ($debug_logs as $log) {
                error_log($log);
            }
            
            if (!isset($_FILES['video']) || $_FILES['video']['error'] !== UPLOAD_ERR_OK) {
                throw new Exception('Please select a valid video file.');
            }
        } elseif ($video_type === 'url') {
            // Validate URL embed
            if (empty($video_url)) {
                throw new Exception('Please provide a video URL.');
            }
            
            $external_url = $video_url;
            
            // Prefer form data (already validated by frontend) over re-detection
            if (!empty($detected_platform) && !empty($video_id_external)) {
                // Use data from form (frontend video detection)
                $platform = $detected_platform;
                $video_id_ext = $video_id_external;
                if (!empty($embed_code)) {
                    $embed_code_final = $embed_code;
                }
            } else {
                // Fallback: Re-detect if form data missing
                $videoData = VideoEmbedder::detectVideo($video_url);
                if ($videoData && $videoData['is_valid']) {
                    $platform = $videoData['platform'];
                    $embed_code_final = VideoEmbedder::sanitizeEmbedCode($videoData['embed_code']);
                    $video_id_ext = $videoData['video_id'];
                }
            }
        } else {
            throw new Exception('Invalid video type specified.');
        }
        
        // Initialize duration variable
        $duration = null;
        
        // Handle file processing based on video type
        error_log("🔍 STARTING FILE/URL PROCESSING");
        error_log("  video_type: " . ($video_type ?? 'NULL'));
        error_log("  \$_FILES available: " . (isset($_FILES) ? 'YES' : 'NO'));
        error_log("  \$_FILES keys: " . (isset($_FILES) ? implode(', ', array_keys($_FILES)) : 'NONE'));
        
        if ($video_type === 'file') {
            // Traditional file upload processing
            error_log("🔍 FILE UPLOAD PROCESSING STARTED");
            
            // Check if video exists in $_FILES (consistent with form input name="video")
            if (!isset($_FILES['video'])) {
                error_log("❌ ERROR: video not found in \$_FILES");
                error_log("Available \$_FILES keys: " . implode(', ', array_keys($_FILES)));
                throw new Exception('No video file uploaded. Please select a video file.');
            }
            
            $uploadedFile = $_FILES['video'];
            error_log("📂 Video file details:");
            error_log("  name: " . ($uploadedFile['name'] ?? 'NULL'));
            error_log("  type: " . ($uploadedFile['type'] ?? 'NULL'));
            error_log("  size: " . ($uploadedFile['size'] ?? 'NULL'));
            error_log("  error: " . ($uploadedFile['error'] ?? 'NULL'));
            error_log("  tmp_name: " . ($uploadedFile['tmp_name'] ?? 'NULL'));
            
            // Check for upload errors
            if ($uploadedFile['error'] !== UPLOAD_ERR_OK) {
                $errorMessages = [
                    UPLOAD_ERR_INI_SIZE => 'File exceeds upload_max_filesize',
                    UPLOAD_ERR_FORM_SIZE => 'File exceeds MAX_FILE_SIZE', 
                    UPLOAD_ERR_PARTIAL => 'File was only partially uploaded',
                    UPLOAD_ERR_NO_FILE => 'No file was uploaded',
                    UPLOAD_ERR_NO_TMP_DIR => 'Missing temporary folder',
                    UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk',
                    UPLOAD_ERR_EXTENSION => 'File upload stopped by extension'
                ];
                $errorMsg = $errorMessages[$uploadedFile['error']] ?? 'Unknown upload error';
                error_log("❌ Upload error: " . $errorMsg);
                throw new Exception('Upload failed: ' . $errorMsg);
            }
            
            $thumbnailFile = $_FILES['thumbnail'] ?? null;
            
            // Validate video file
            $allowedTypes = ['video/mp4', 'video/webm', 'video/quicktime', 'video/x-msvideo'];
            error_log("🔍 File type validation:");
            error_log("  Detected type: " . $uploadedFile['type']);
            error_log("  Allowed types: " . implode(', ', $allowedTypes));
            
            if (!in_array($uploadedFile['type'], $allowedTypes)) {
                error_log("❌ Invalid file type: " . $uploadedFile['type']);
                throw new Exception('Invalid video format. Please use MP4, WebM, MOV, or AVI.');
            }
            
            error_log("✅ File type validation passed");
            
            // Create upload directories
            $uploadDir = __DIR__ . '/../uploads/videos/';
            $thumbnailDir = __DIR__ . '/../uploads/thumbnails/';
            
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }
            if (!is_dir($thumbnailDir)) {
                mkdir($thumbnailDir, 0755, true);
            }
            
            // Generate unique filenames
            $videoExtension = pathinfo($uploadedFile['name'], PATHINFO_EXTENSION);
            $uniqueId = uniqid() . '_' . time();
            $videoFilename = 'video_' . $uniqueId . '.' . $videoExtension;
            $videoPath = $uploadDir . $videoFilename;
            
            error_log("📁 File paths:");
            error_log("  Upload directory: " . $uploadDir);
            error_log("  Video filename: " . $videoFilename);
            error_log("  Full video path: " . $videoPath);
            error_log("  Directory writable: " . (is_writable($uploadDir) ? 'YES' : 'NO'));
            
            // Handle thumbnail upload
            if ($thumbnailFile && $thumbnailFile['error'] === UPLOAD_ERR_OK) {
                error_log("📸 Processing thumbnail upload");
                $thumbExtension = pathinfo($thumbnailFile['name'], PATHINFO_EXTENSION);
                $thumbnailFilename = 'thumb_' . $uniqueId . '.' . $thumbExtension;
                $thumbnailFullPath = $thumbnailDir . $thumbnailFilename;
                
                error_log("  Thumbnail path: " . $thumbnailFullPath);
                
                if (!move_uploaded_file($thumbnailFile['tmp_name'], $thumbnailFullPath)) {
                    error_log("❌ Thumbnail upload failed");
                    throw new Exception('Failed to upload thumbnail.');
                }
                
                // Store relative path for database
                $thumbnailPath = 'uploads/thumbnails/' . $thumbnailFilename;
                error_log("✅ Thumbnail uploaded: " . $thumbnailPath);
            } else {
                error_log("📸 No thumbnail file uploaded or has errors");
            }
            
            // Move uploaded video
            error_log("🎬 Moving video file from temp to final location");
            error_log("  From: " . $uploadedFile['tmp_name']);
            error_log("  To: " . $videoPath);
            error_log("  Temp file exists: " . (file_exists($uploadedFile['tmp_name']) ? 'YES' : 'NO'));
            
            if (!move_uploaded_file($uploadedFile['tmp_name'], $videoPath)) {
                error_log("❌ Video file move failed");
                throw new Exception('Failed to upload video file.');
            }
            
            // Store relative path for database
            $videoPathForDB = 'uploads/videos/' . $videoFilename;
            error_log("✅ Video uploaded successfully: " . $videoPathForDB);
            
            // 🎬 DETECT VIDEO DURATION FOR FILE UPLOADS
            error_log("🎬 DETECTING VIDEO DURATION for file: " . $videoPath);
            
            // 🎯 PRIORITY 1: Use frontend-detected duration if available
            if (isset($_POST['detected_duration']) && !empty(trim($_POST['detected_duration']))) {
                $frontendDuration = trim($_POST['detected_duration']);
                error_log("  ✅ Using frontend-detected duration: " . $frontendDuration);
                $duration = $frontendDuration;
            } else {
                // 🎯 FALLBACK: Try file-based detection
                error_log("  ⚠️ No frontend duration, trying file-based detection");
                $duration = detectVideoFileDuration($videoPath);
                error_log("  📊 File-detected duration: " . ($duration ?? 'NULL'));
            }
            
            error_log("  🎯 Final duration for file upload: " . ($duration ?? 'NULL'));
            
        } elseif ($video_type === 'url') {
            // URL embed processing - no file upload needed
            $videoPathForDB = null; // No local file path for URL embeds
            
            // PERMANENT FIX: Always ensure thumbnail is extracted for YouTube URLs
            error_log("🔍 THUMBNAIL EXTRACTION:");
            error_log("  platform: " . ($platform ?? 'NULL'));
            error_log("  video_id_ext: " . ($video_id_ext ?? 'NULL'));
            
            // Force thumbnail generation for YouTube videos
            if ($platform === 'youtube' && !empty($video_id_ext)) {
                $thumbnailPath = "https://img.youtube.com/vi/{$video_id_ext}/maxresdefault.jpg";
                error_log("  ✅ FORCED YouTube thumbnail: " . $thumbnailPath);
            } elseif ($platform && $video_id_ext) {
                $platformThumbnail = VideoEmbedder::getThumbnail($platform, $video_id_ext);
                error_log("  generated thumbnail URL: " . ($platformThumbnail ?? 'NULL'));
                if ($platformThumbnail) {
                    $thumbnailPath = $platformThumbnail; // External thumbnail URL
                    error_log("  thumbnailPath set to: " . $thumbnailPath);
                }
            } else {
                error_log("  thumbnail extraction skipped - missing platform or video_id");
            }
            
            // 🎬 DETECT AND SET VIDEO DURATION
            $duration = null;
            
            // 🔍 DEBUG: Log all duration-related POST data
            error_log("🔍 DURATION DEBUG - POST DATA:");
            error_log("  detected_duration: " . ($_POST['detected_duration'] ?? 'NOT SET'));
            error_log("  manual_duration: " . ($_POST['manual_duration'] ?? 'NOT SET'));
            error_log("  All POST keys: " . implode(', ', array_keys($_POST)));
            
            // 🎯 CRITICAL: Create visible debug file for immediate inspection
            $debugFile = '../debug/backend-duration-debug.txt';
            $debugContent = "🔍 BACKEND DURATION DEBUG REPORT\n";
            $debugContent .= "Generated: " . date('Y-m-d H:i:s') . "\n\n";
            $debugContent .= "📋 POST DATA RECEIVED:\n";
            $debugContent .= "detected_duration: " . ($_POST['detected_duration'] ?? 'NOT SET') . "\n";
            $debugContent .= "manual_duration: " . ($_POST['manual_duration'] ?? 'NOT SET') . "\n";
            $debugContent .= "video_type: " . ($video_type ?? 'NOT SET') . "\n";
            $debugContent .= "All POST keys: " . implode(', ', array_keys($_POST)) . "\n\n";
            
            // Create debug directory if it doesn't exist
            if (!file_exists('../debug')) {
                mkdir('../debug', 0755, true);
            }
            file_put_contents($debugFile, $debugContent);
            
            // 🔥 PRIORITY 1: Check if API detected duration is available (from our enhanced detection system)
            if (isset($_POST['detected_duration']) && !empty(trim($_POST['detected_duration']))) {
                $detectedDuration = trim($_POST['detected_duration']);
                error_log("🎯 FOUND detected_duration field: " . $detectedDuration);
                
                // Validate detected duration format
                if (preg_match('/^(\d{1,2}:)?([0-5]?\d):([0-5]?\d)$/', $detectedDuration)) {
                    $duration = $detectedDuration;
                    error_log("✅ Using API-detected duration (PRIORITY 1): " . $duration);
                } else {
                    error_log("❌ Invalid detected duration format: " . $detectedDuration);
                }
            } else {
                error_log("⚠️ No detected_duration field found in POST data");
            }
            
            // 🔥 PRIORITY 2: Check if teacher provided manual duration (fallback for when API fails)
            if (!$duration && isset($_POST['manual_duration']) && !empty(trim($_POST['manual_duration']))) {
                $manualDuration = trim($_POST['manual_duration']);
                
                // Validate manual duration format
                if (preg_match('/^(\d{1,2}:)?([0-5]?\d):([0-5]?\d)$/', $manualDuration)) {
                    $duration = $manualDuration;
                    error_log("✅ Using teacher-validated duration (PRIORITY 2): " . $duration);
                } else {
                    error_log("❌ Invalid manual duration format: " . $manualDuration);
                }
            }
            
            // Only try automatic detection if no manual duration provided
            if (!$duration) {
                if ($video_type === 'file' && $videoPathForDB) {
                    // For file uploads, detect duration from the uploaded file
                    error_log("🎬 Attempting to detect duration for uploaded file: " . $videoPathForDB);
                    $duration = detectVideoFileDuration($videoPathForDB);
                    
                    if ($duration) {
                        error_log("✅ File duration detected: " . $duration);
                    } else {
                        error_log("⚠️ File duration detection failed");
                    }
                    
                } else if ($video_type === 'url' && !empty($video_id_ext)) {
                    // For URL videos (YouTube), detect duration
                    error_log("🎬 Attempting to detect YouTube duration for video ID: " . $video_id_ext);
                    $duration = detectYouTubeVideoDuration($video_id_ext);
                    
                    if ($duration) {
                        error_log("✅ YouTube duration detected: " . $duration);
                    } else {
                        error_log("⚠️ YouTube duration detection failed");
                    }
                }
            }
            
            // Set default duration if all detection methods failed
            if (!$duration) {
                $duration = "0:00"; // Default fallback
                error_log("⚠️ Using default duration: " . $duration);
            }
            
            // Final validation - ensure YouTube videos always have thumbnails
            if (!$thumbnailPath && $platform === 'youtube' && !empty($video_id_ext)) {
                $thumbnailPath = "https://img.youtube.com/vi/{$video_id_ext}/maxresdefault.jpg";
                error_log("  🛡️ FALLBACK: Set YouTube thumbnail to " . $thumbnailPath);
            }
        }
        
        // Use different SQL queries for file uploads vs URL embeds
        if ($video_type === 'file') {
            // File upload: Use video_url and local thumbnail
            $sql = "
                INSERT INTO videos (
                    teacher_id, course_id, title, description, video_type, 
                    video_url, thumbnail, duration, is_published, created_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
            ";
        } else {
            // URL embed: Use external_url, platform, video_id_external  
            $sql = "
                INSERT INTO videos (
                    teacher_id, course_id, title, description, video_type, thumbnail,
                    external_url, platform, video_id_external, duration, is_published, created_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
            ";
        }
        
        $stmt = $conn->prepare($sql);
        
        if (!$stmt) {
            throw new Exception('Database prepare failed: ' . $conn->error);
        }
        
        // Store content types as JSON
        $contentTypesJson = json_encode($content_types);
        
        // DEBUG: Log bind param values based on video type
        if ($video_type === 'file') {
            error_log("🔍 FILE UPLOAD BIND PARAMS:");
            error_log("  teacher_id: " . ($teacher_id ?? 'NULL'));
            error_log("  course_id: " . ($course_id ?? 'NULL'));  
            error_log("  title: " . ($title ?? 'NULL'));
            error_log("  description: " . ($description ?? 'NULL'));
            error_log("  video_type: " . ($video_type ?? 'NULL'));
            error_log("  videoPathForDB: " . ($videoPathForDB ?? 'NULL'));
            error_log("  thumbnailPath: " . ($thumbnailPath ?? 'NULL'));
            error_log("  is_published: " . ($is_published ?? 'NULL'));
        } else {
            error_log("🔍 URL EMBED BIND PARAMS:");
            error_log("  teacher_id: " . ($teacher_id ?? 'NULL'));
            error_log("  course_id: " . ($course_id ?? 'NULL'));  
            error_log("  title: " . ($title ?? 'NULL'));
            error_log("  description: " . ($description ?? 'NULL'));
            error_log("  video_type: " . ($video_type ?? 'NULL'));
            error_log("  thumbnailPath: " . ($thumbnailPath ?? 'NULL'));
            error_log("  external_url: " . ($external_url ?? 'NULL'));
            error_log("  platform: " . ($platform ?? 'NULL'));
            error_log("  video_id_ext: " . ($video_id_ext ?? 'NULL'));
            error_log("  duration: " . ($duration ?? 'NULL'));
            error_log("  is_published: " . ($is_published ?? 'NULL'));
        }
        
        // Ensure all variables have safe default values
        $teacher_id = $teacher_id ?? 14;
        $course_id = $course_id ?? 0;
        $title = $title ?? '';
        $description = $description ?? '';
        $videoPathForDB = $videoPathForDB ?? null;
        $thumbnailPath = $thumbnailPath ?? null;
        $video_type = $video_type ?? 'url';
        $external_url = $external_url ?? null;
        $platform = $platform ?? null;
        $embed_code_final = $embed_code_final ?? null;
        $video_id_ext = $video_id_ext ?? null;
        $duration = $duration ?? null; // 🔥 CRITICAL: Ensure duration has default value
        $module_order = $module_order ?? 1;
        $price = $price ?? 0.00;
        $category_name = $category_name ?? '';
        $is_published = $is_published ?? 1;
        $is_free = $is_free ?? 1;
        $contentTypesJson = $contentTypesJson ?? '[]';
        
        // FINAL SAFETY CHECK: Ensure YouTube videos ALWAYS have thumbnails
        if ($platform === 'youtube' && !empty($video_id_ext) && empty($thumbnailPath)) {
            $thumbnailPath = "https://img.youtube.com/vi/{$video_id_ext}/maxresdefault.jpg";
            error_log("🛡️ FINAL SAFETY: Applied YouTube thumbnail: " . $thumbnailPath);
        }
        
        // Use different bind_param for file uploads vs URL embeds
        if ($video_type === 'file') {
            // File upload: 9 parameters (added duration)
            $stmt->bind_param(
                "iissssssi",            // i i s s s s s s i = 9 characters for 9 parameters
                $teacher_id,            // 1. teacher_id (integer)
                $course_id,             // 2. course_id (integer)
                $title,                 // 3. title (string)
                $description,           // 4. description (string)
                $video_type,            // 5. video_type (string)
                $videoPathForDB,        // 6. video_url (string) - local file path
                $thumbnailPath,         // 7. thumbnail (string) - local thumbnail path
                $duration,              // 8. duration (string) - video duration
                $is_published           // 9. is_published (integer)
            );
        } else {
            // URL embed: 11 parameters (added duration)
            $stmt->bind_param(
                "iissssssssi",          // i i s s s s s s s s i = 11 characters for 11 parameters
                $teacher_id,            // 1. teacher_id (integer)
                $course_id,             // 2. course_id (integer)
                $title,                 // 3. title (string)
                $description,           // 4. description (string)
                $video_type,            // 5. video_type (string)
                $thumbnailPath,         // 6. thumbnail (string) - YouTube thumbnail URL
                $external_url,          // 7. external_url (string) 
                $platform,              // 8. platform (string)
                $video_id_ext,          // 9. video_id_external (string)
                $duration,              // 10. duration (string) - video duration
                $is_published           // 11. is_published (integer)
            );
        }
        
        error_log("🎯 EXECUTING SQL QUERY...");
        error_log("  SQL prepared successfully");
        error_log("  About to execute with bound parameters");
        
        if (!$stmt->execute()) {
            // More detailed error information
            $error_info = $stmt->error;
            $errno = $stmt->errno;
            error_log("🚨 SQL EXECUTION FAILED:");
            error_log("  Error: " . $error_info);
            error_log("  Error Number: " . $errno);
            error_log("  SQL: " . $sql);
            throw new Exception('Failed to save video: ' . $error_info . ' (Error #' . $errno . ')');
        }
        
        error_log("✅ SQL EXECUTION SUCCESS!");
        $video_id = $conn->insert_id;
        error_log("  Generated video_id: " . $video_id);
        $stmt->close();
        
        // Always redirect to videos page regardless of content types selected
        // The content creation can be handled separately if needed
        if (!empty($content_types)) {
            // Store workflow in session for potential future use
            $_SESSION['content_workflow'] = [
                'video_id' => $video_id,
                'video_title' => $title,
                'content_types' => $content_types,
                'current_step' => 0,
                'completed_steps' => []
            ];
        }
        
        // Always redirect to videos page with the new video ID
        $encrypted_video_id = IdEncryption::encrypt($video_id);
        $response = [
            'success' => true,
            'message' => 'Video uploaded successfully!',
            'redirect' => 'videos.php?new_video=' . $encrypted_video_id
        ];
        
    } catch (Exception $e) {
        // Add debugging information to the error response
        $debug_info = [
            'POST_DATA' => $_POST,
            'SERVER_LOGS' => $debug_logs ?? [],
            'ERROR_MESSAGE' => $e->getMessage(),
            'ERROR_LINE' => $e->getLine(),
            'BIND_VARIABLES' => ($video_type ?? 'UNDEFINED') === 'file' ? [
                'teacher_id' => $teacher_id ?? 'UNDEFINED',
                'course_id' => $course_id ?? 'UNDEFINED', 
                'title' => $title ?? 'UNDEFINED',
                'description' => $description ?? 'UNDEFINED',
                'video_type' => $video_type ?? 'UNDEFINED',
                'videoPathForDB' => $videoPathForDB ?? 'UNDEFINED',
                'thumbnailPath' => $thumbnailPath ?? 'UNDEFINED',
                'is_published' => $is_published ?? 'UNDEFINED'
            ] : [
                'teacher_id' => $teacher_id ?? 'UNDEFINED',
                'course_id' => $course_id ?? 'UNDEFINED', 
                'title' => $title ?? 'UNDEFINED',
                'description' => $description ?? 'UNDEFINED',
                'video_type' => $video_type ?? 'UNDEFINED',
                'thumbnailPath' => $thumbnailPath ?? 'UNDEFINED',
                'external_url' => $external_url ?? 'UNDEFINED',
                'platform' => $platform ?? 'UNDEFINED',
                'video_id_ext' => $video_id_ext ?? 'UNDEFINED',
                'is_published' => $is_published ?? 'UNDEFINED'
            ],
            'DETECTED_DATA' => [
                'detected_platform' => $detected_platform ?? 'UNDEFINED',
                'video_id_external' => $video_id_external ?? 'UNDEFINED',
                'embed_code' => !empty($embed_code) ? 'HAS_VALUE' : 'EMPTY'
            ]
        ];
        
        $response = [
            'success' => false,
            'message' => $e->getMessage(),
            'debug' => $debug_info
        ];
    }
} else {
    $response = [
        'success' => false,
        'message' => 'Invalid request method or form type.'
    ];
}

// Return JSON response for AJAX
if (isset($_POST['ajax'])) {
    header('Content-Type: application/json');
    echo json_encode($response);
    exit();
}

// Regular form submission - redirect or show error
if ($response['success']) {
    $_SESSION['success_message'] = $response['message'];
    header('Location: ' . $response['redirect']);
} else {
    $_SESSION['error_message'] = $response['message'];
    header('Location: upload-video.php');
}
exit();
?>
