<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Centralized session management
require_once "../includes/session_start.php";
require_once "../config/db_connect.php";
require_once "../includes/functions.php";
require_once "../auth/functions.php";

// Check if user is logged in and is a teacher
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'teacher') {
    header("Location: ../login.php");
    exit;
}

$teacher_id = $_SESSION['user_id'];

// Check if the form was submitted
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Get basic quiz info
    $quiz_title = mysqli_real_escape_string($conn, $_POST['quiz_title']);
    $course_id = mysqli_real_escape_string($conn, $_POST['course_id']);
    $quiz_description = isset($_POST['quiz_description']) ? mysqli_real_escape_string($conn, $_POST['quiz_description']) : '';
    $time_limit = isset($_POST['time_limit']) ? (int)$_POST['time_limit'] : 30;
    $passing_score = isset($_POST['passing_score']) ? (int)$_POST['passing_score'] : 60;
    
    // AI Grading Configuration
    $ai_grading_enabled = isset($_POST['ai_grading_enabled']) ? 1 : 0;
    $pass_mark = isset($_POST['pass_mark']) ? (float)$_POST['pass_mark'] : 50.0;
    $max_attempts = isset($_POST['max_attempts']) ? (int)$_POST['max_attempts'] : 3;
    $auto_feedback = isset($_POST['auto_feedback']) ? 1 : 0;
    $essay_grading = isset($_POST['essay_grading']) ? 1 : 0;
    $answerkey_content = isset($_POST['answerkey_content']) ? mysqli_real_escape_string($conn, $_POST['answerkey_content']) : '';
    
    // Determine if it's published or draft
    $is_published = isset($_POST['publish']) ? 1 : 0;
    
    // Debug information
    error_log('Quiz submission received: Title=' . $quiz_title . ', Course ID=' . $course_id . ', Published=' . $is_published);
    
    // Begin transaction
    $conn->begin_transaction();
    
    try {
        // Insert quiz into database with AI grading fields
        $stmt = $conn->prepare(
            "INSERT INTO quizzes (title, description, course_id, teacher_id, time_limit, passing_score, is_published, 
             ai_grading_enabled, answer_key, essay_questions, auto_grade_mcq, feedback_enabled, created_at) 
             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())"
        );
        
        // Create essay questions JSON data
        $essay_questions_json = json_encode([
            'pass_mark' => $pass_mark,
            'max_attempts' => $max_attempts,
            'essay_grading_enabled' => $essay_grading
        ]);
        
        $stmt->bind_param(
            "ssiiiiiissii", 
            $quiz_title, $quiz_description, $course_id, $teacher_id, $time_limit, $passing_score, $is_published,
            $ai_grading_enabled, $answerkey_content, $essay_questions_json, $essay_grading, $auto_feedback
        );
        $stmt->execute();
        
        // Get the quiz ID
        $quiz_id = $conn->insert_id;
        
        // Process questions
        if (isset($_POST['questions']) && is_array($_POST['questions'])) {
            foreach ($_POST['questions'] as $question_key => $question_data) {
                // Skip if no question text
                if (!isset($question_data['text']) || empty($question_data['text'])) {
                    continue;
                }
                
                $question_text = mysqli_real_escape_string($conn, $question_data['text']);
                $question_type = mysqli_real_escape_string($conn, $question_data['type']);
                $question_points = isset($question_data['points']) ? (int)$question_data['points'] : 1;
                $correct_option = isset($question_data['correct_option']) ? (int)$question_data['correct_option'] : 1;
                
                // Insert question
                $stmt = $conn->prepare("INSERT INTO quiz_questions (quiz_id, question_text, question_type, points, created_at) VALUES (?, ?, ?, ?, NOW())");
                $stmt->bind_param("issi", $quiz_id, $question_text, $question_type, $question_points);
                $stmt->execute();
                
                $question_id = $conn->insert_id;
                
                // Process options for this question
                if (isset($question_data['options']) && is_array($question_data['options'])) {
                    foreach ($question_data['options'] as $option_number => $option_text) {
                        if (empty($option_text)) {
                            continue;
                        }
                        
                        $option_text = mysqli_real_escape_string($conn, $option_text);
                        $is_correct = ($option_number == $correct_option) ? 1 : 0;
                        
                        // Insert option
                        $stmt = $conn->prepare("INSERT INTO quiz_question_options (question_id, option_text, is_correct) VALUES (?, ?, ?)");
                        $stmt->bind_param("isi", $question_id, $option_text, $is_correct);
                        $stmt->execute();
                    }
                }
            }
        }
        
        // Commit transaction
        $conn->commit();
        
        // Log successful creation
        error_log("Quiz created successfully: ID $quiz_id, Title: $quiz_title, AI Enabled: $ai_grading_enabled");
        
        // Create enhanced success message
        $ai_status = $ai_grading_enabled ? 'with AI Grading enabled' : 'without AI Grading';
        $publish_status = $is_published ? 'published' : 'saved as draft';
        
        $_SESSION['success_message'] = "<div class='alert alert-success'>";
        $_SESSION['success_message'] .= "<i class='fas fa-check-circle me-2'></i>";
        $_SESSION['success_message'] .= "<strong>🎉 Success!</strong> Your quiz has been $publish_status successfully $ai_status!";
        
        if ($ai_grading_enabled && !empty($answerkey_content)) {
            $_SESSION['success_message'] .= "<br><small class='text-muted'><i class='fas fa-key me-1'></i>Answer key processed and ready for AI grading.</small>";
        }
        
        $_SESSION['success_message'] .= "<p class='mb-0 mt-2'>";
        $_SESSION['success_message'] .= "<a href='quizzes.php' class='btn btn-primary me-2'>View All Quizzes</a>";
        $_SESSION['success_message'] .= "<a href='create_quiz.php' class='btn btn-outline-primary'>Create Another</a></p>";
        $_SESSION['success_message'] .= "</div>";
        
        header("Location: create_quiz.php?success=1");
        exit;
        
    } catch (Exception $e) {
        // An error occurred, rollback the transaction
        $conn->rollback();
        $_SESSION['error_message'] = "Error creating quiz: " . $e->getMessage();
        header("Location: create_quiz.php");
        exit;
    }
}

// If not POST request, redirect to create quiz page
header("Location: create_quiz.php");
exit;
?>
