<?php
// Set current page for active menu highlighting
$current_page = 'profile.php';

// Include the centralized teacher layout
require_once __DIR__ . '/components/teacher_layout.php';

// Enable error reporting during development
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Ensure mysqli connection is available for legacy code
if (!isset($conn) && isset($pdo)) {
    $conn = new mysqli($host, $username, $password, $dbname);
    if ($conn->connect_error) {
        die("Connection failed: " . $conn->connect_error);
    }
}

/**
 * Get teacher data using real user information from users table
 */
function get_teacher($conn, $teacher_id) {
    // Check if we have a valid connection
    if (!$conn || $conn->connect_error) {
        error_log("Database connection failed in get_teacher");
        return null;
    }
    
    try {
        // Get ALL user data + teacher_profiles for missing fields - PERMANENT COMPLETE SOLUTION
        $stmt = $conn->prepare("SELECT u.user_id, u.first_name, u.last_name, u.email, u.bio, u.profile_picture,
                               u.cell_number as phone,
                               u.title, u.id_number, u.suburb, u.province, u.communication_method,
                               u.owns_laptop, u.education_status, u.employment_status,
                               u.heard_about_us, u.referral_person_name, u.job_field_other as experience,
                               t.website, t.social_linkedin, t.social_twitter, t.social_facebook, t.social_youtube,
                               t.expertise as course_title,
                               tp.bio as course_description, tp.qualifications
                               FROM users u 
                               LEFT JOIN teachers t ON u.user_id = t.user_id
                               LEFT JOIN teacher_profiles tp ON t.teacher_id = tp.teacher_id
                               WHERE u.user_id = ? AND u.user_role = 'teacher'");
        
        if (!$stmt) {
            error_log("Failed to prepare statement in get_teacher: " . $conn->error);
            return null;
        }
        
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result && $result->num_rows > 0) {
            $teacher = $result->fetch_assoc();
            
            // Set default bio if null
            if (empty($teacher['bio'])) {
                $teacher['bio'] = 'Welcome to my teaching profile! I am passionate about education and helping students achieve their learning goals.';
            }
            
            // Set default profile picture path if null
            if (empty($teacher['profile_picture'])) {
                $teacher['profile_picture'] = null; // Will show initials instead
            }
            
            return $teacher;
        } else {
            error_log("No teacher found with ID $teacher_id in users table");
            return null;
        }
    } catch (Exception $e) {
        error_log("Exception in get_teacher: " . $e->getMessage());
        return null;
    }
}

$teacher_id = $_SESSION['user_id'];
$teacher = get_teacher($conn, $teacher_id);

// Get user profile data - first we need to check the column name in teacher_profiles
$columns_query = $conn->query("SHOW COLUMNS FROM teacher_profiles");
$profile_id_column = "teacher_id"; // default to teacher_id as most likely column name

if ($columns_query) {
    while ($column = $columns_query->fetch_assoc()) {
        // Look for id columns referencing teachers
        if ($column['Field'] === 'user_id' || $column['Field'] === 'teacher_id' || 
            (stripos($column['Field'], 'teacher') !== false && stripos($column['Field'], 'id') !== false)) {
            $profile_id_column = $column['Field'];
            break;
        }
    }
}

$profile_query = "SELECT * FROM teacher_profiles WHERE $profile_id_column = ?";
$stmt = $conn->prepare($profile_query);
$stmt->bind_param("i", $teacher_id);
$stmt->execute();
$profile_result = $stmt->get_result();
$profile = $profile_result->fetch_assoc() ?: [];

// Process profile update if form is submitted
$message = '';
if (isset($_POST['remove_picture'])) {
    if (!empty($teacher['profile_picture'])) {
        // Delete file from server
        $file_path = "../" . $teacher['profile_picture'];
        if (file_exists($file_path)) {
            unlink($file_path);
        }
        
        // Remove from database
        $remove_pic = "UPDATE users SET profile_picture = NULL WHERE user_id = ?";
        $remove_stmt = $conn->prepare($remove_pic);
        $remove_stmt->bind_param("i", $teacher_id);
        
        if ($remove_stmt->execute()) {
            $message = '<div class="alert alert-success">Profile picture removed successfully!</div>';
            // Refresh teacher data
            $teacher = get_teacher($conn, $teacher_id);
        } else {
            $message = '<div class="alert alert-danger">Failed to remove profile picture from database: ' . $remove_stmt->error . '</div>';
        }
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_profile'])) {
    // Basic profile fields
    $first_name = trim($_POST['first_name']);
    $last_name = trim($_POST['last_name']);
    $email = trim($_POST['email']);
    $bio = trim($_POST['bio']);
    $phone = trim($_POST['phone']);
    $website = trim($_POST['website']);
    $social_linkedin = trim($_POST['social_linkedin']);
    $social_twitter = trim($_POST['social_twitter']);
    
    // Personal details fields
    $title = trim($_POST['title'] ?? '');
    $cell_number = trim($_POST['cell_number'] ?? '');
    $id_number = trim($_POST['id_number'] ?? '');
    $suburb = trim($_POST['suburb'] ?? '');
    $province = trim($_POST['province'] ?? '');
    $communication_method = trim($_POST['communication_method'] ?? '');
    
    // Learning & background fields
    $owns_laptop = trim($_POST['owns_laptop'] ?? '');
    $education_status = trim($_POST['education_status'] ?? '');
    $employment_status = trim($_POST['employment_status'] ?? '');
    $heard_about_us = trim($_POST['heard_about_us'] ?? '');
    $referral_person_name = trim($_POST['referral_person_name'] ?? '');
    
    // Facilitator fields
    $course_title = trim($_POST['course_title'] ?? '');
    $experience = trim($_POST['experience'] ?? '');
    $course_description = trim($_POST['course_description'] ?? '');
    $qualifications = trim($_POST['qualifications'] ?? '');
    
    // REMOVED FAILING QUERY - Using only the working query below that properly uses job_field_other
    
    // Proceed directly to teacher record management
    // CRITICAL FIX: Ensure teacher record exists before updating social links
    $check_teacher = "SELECT teacher_id FROM teachers WHERE user_id = ?";
    $check_stmt = $conn->prepare($check_teacher);
    $check_stmt->bind_param("i", $teacher_id);
    $check_stmt->execute();
    $teacher_exists = $check_stmt->get_result()->num_rows > 0;
    
    if (!$teacher_exists) {
        // Create teacher record with CORRECT database columns - FIXED
        $create_teacher = "INSERT INTO teachers (user_id, first_name, last_name, bio, expertise, website, social_linkedin, social_twitter) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
        $create_stmt = $conn->prepare($create_teacher);
        $teacher_bio = $course_description ?: ($bio ?: 'Welcome to my teaching profile! I am passionate about education and helping students achieve their learning goals.');
        $teacher_expertise = $course_title ?: 'Education Specialist';
        
        if ($create_stmt) {
            $create_stmt->bind_param("isssssss", $teacher_id, $first_name, $last_name, $teacher_bio, $teacher_expertise, $website, $social_linkedin, $social_twitter);
            $teacher_created = $create_stmt->execute();
        } else {
            $teacher_created = false;
            $message = "<div class='alert alert-danger'>Database error preparing teacher insert: " . $conn->error . "</div>";
        }
        
        if ($teacher_created) {
            // Also create/update teacher_profiles record with qualifications - FIXED
            $create_profile = "INSERT INTO teacher_profiles (teacher_id, expertise, bio, qualifications) VALUES (?, ?, ?, ?) ON DUPLICATE KEY UPDATE expertise = VALUES(expertise), bio = VALUES(bio), qualifications = VALUES(qualifications)";
            $profile_stmt = $conn->prepare($create_profile);
            
            // Get the newly created teacher_id
            $new_teacher_id = $conn->insert_id;
            if ($profile_stmt) {
                $profile_stmt->bind_param("isss", $new_teacher_id, $teacher_expertise, $teacher_bio, $qualifications);
                $profile_stmt->execute();
                $message = "<div class='alert alert-success'>Profile updated and teacher record created with all facilitator information!</div>";
            } else {
                $message = "<div class='alert alert-warning'>Teacher record created but failed to create profile: " . $conn->error . "</div>";
            }
        } else {
            $message = "<div class='alert alert-danger'>Profile updated, but failed to create teacher record.</div>";
        }
    } else {
        // Update existing teacher record - FIXED to use correct fields
        $update_teacher = "UPDATE teachers SET first_name = ?, last_name = ?, bio = ?, expertise = ?, website = ?, social_linkedin = ?, social_twitter = ? WHERE user_id = ?";
        $social_stmt = $conn->prepare($update_teacher);
        
        // Define variables for teacher update
        $teacher_bio = $course_description ?: ($bio ?: 'Welcome to my teaching profile! I am passionate about education and helping students achieve their learning goals.');
        $teacher_expertise = $course_title ?: 'Education Specialist';
        
        // Store experience in job_field_other as permanent solution
        $experience_value = trim($_POST['experience'] ?? '');
        
        // Get existing teacher_id
        $get_teacher_id = "SELECT teacher_id FROM teachers WHERE user_id = ?";
        $get_id_stmt = $conn->prepare($get_teacher_id);
        $get_id_stmt->bind_param("i", $teacher_id);
        $get_id_stmt->execute();
        $result = $get_id_stmt->get_result();
        $existing_teacher_data = $result->fetch_assoc();
        $existing_teacher_id = $existing_teacher_data ? $existing_teacher_data['teacher_id'] : null;
        
        // Update teacher_profiles with qualifications AND experience - FIXED COLUMNS  
        $update_profile = "INSERT INTO teacher_profiles (teacher_id, expertise, bio, qualifications) VALUES (?, ?, ?, ?) ON DUPLICATE KEY UPDATE expertise = VALUES(expertise), bio = VALUES(bio), qualifications = VALUES(qualifications)";
        $profile_stmt = $conn->prepare($update_profile);
        
        if ($social_stmt) {
            $social_stmt->bind_param("sssssssi", $first_name, $last_name, $teacher_bio, $teacher_expertise, $website, $social_linkedin, $social_twitter, $teacher_id);
        } else {
            $message = "<div class='alert alert-danger'>Database error preparing teacher update: " . $conn->error . "</div>";
        }
        
        if ($profile_stmt && $existing_teacher_id) {
            $profile_stmt->bind_param("isss", $existing_teacher_id, $teacher_expertise, $teacher_bio, $qualifications);
        } else {
            $message = "<div class='alert alert-danger'>Database error preparing teacher update or missing teacher ID: " . $conn->error . "</div>";
        }
        
        if ($social_stmt && $profile_stmt) {
            $social_updated = $social_stmt->execute();
            $profile_updated = $profile_stmt->execute();
            
            if ($social_updated && $profile_updated) {
                $message = "<div class='alert alert-success'>Complete profile updated successfully with all registration information!</div>";
                
                // CRITICAL: Refresh teacher data after successful update
                $refresh_stmt = $conn->prepare("SELECT u.user_id, u.first_name, u.last_name, u.email, u.bio, u.profile_picture,
                               u.cell_number as phone,
                               u.title, u.id_number, u.suburb, u.province, u.communication_method,
                               u.owns_laptop, u.education_status, u.employment_status,
                               u.heard_about_us, u.referral_person_name, u.job_field_other as experience,
                               t.website, t.social_linkedin, t.social_twitter, t.social_facebook, t.social_youtube,
                               t.expertise as course_title,
                               tp.bio as course_description, tp.qualifications
                               FROM users u 
                               LEFT JOIN teachers t ON u.user_id = t.user_id
                               LEFT JOIN teacher_profiles tp ON t.teacher_id = tp.teacher_id
                               WHERE u.user_id = ? AND u.user_role = 'teacher'");
                $refresh_stmt->bind_param("i", $teacher_id);
                $refresh_stmt->execute();
                $refresh_result = $refresh_stmt->get_result();
                if ($refresh_result->num_rows > 0) {
                    $teacher = $refresh_result->fetch_assoc();
                }
                
            } else {
                $message = "<div class='alert alert-warning'>Profile updated, but social links failed to save: " . $social_stmt->error . "</div>";
            }
        } else {
            $message = "<div class='alert alert-warning'>Profile updated, but social links update failed: " . $conn->error . "</div>";
        }
    }
    
    // Refresh teacher data after profile update
    $teacher = get_teacher($conn, $teacher_id);
}

// Handle profile picture upload (separate POST handler)
if (isset($_FILES['profile_picture'])) {
        error_log("Profile picture upload triggered for user $teacher_id");
        error_log("File error: " . $_FILES['profile_picture']['error']);
        error_log("File type: " . $_FILES['profile_picture']['type']);
        error_log("File size: " . $_FILES['profile_picture']['size']);
        
        if ($_FILES['profile_picture']['error'] === 0) {
        $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
        $max_size = 2 * 1024 * 1024; // 2MB
        
        if (in_array($_FILES['profile_picture']['type'], $allowed_types) && $_FILES['profile_picture']['size'] <= $max_size) {
            $upload_dir = "../uploads/profile_pictures/";
            
            // Create directory if it doesn't exist
            if (!is_dir($upload_dir)) {
                mkdir($upload_dir, 0755, true);
            }
            
            $filename = $teacher_id . '_' . time() . '_' . basename($_FILES['profile_picture']['name']);
            $target_path = $upload_dir . $filename;
            
            if (move_uploaded_file($_FILES['profile_picture']['tmp_name'], $target_path)) {
                $profile_pic_path = "uploads/profile_pictures/" . $filename;
                
                // FIXED: Update profile picture in USERS table (not teacher_profiles)
                $update_pic = "UPDATE users SET profile_picture = ? WHERE user_id = ?";
                $pic_stmt = $conn->prepare($update_pic);
                
                if ($pic_stmt) {
                    $pic_stmt->bind_param("si", $profile_pic_path, $teacher_id);
                    $pic_updated = $pic_stmt->execute();
                    
                    if ($pic_updated) {
                        $message = '<div class="alert alert-success">Profile picture uploaded successfully!</div>';
                        // CRITICAL: Refresh teacher data to show new image
                        $teacher = get_teacher($conn, $teacher_id);
                    } else {
                        $message = '<div class="alert alert-warning">Profile picture uploaded but failed to save to database: ' . $pic_stmt->error . '</div>';
                    }
                } else {
                    $message = '<div class="alert alert-danger">Failed to prepare profile picture update: ' . $conn->error . '</div>';
                }
            } else {
                $message = '<div class="alert alert-danger">Failed to upload profile picture to server.</div>';
            }
        } else {
            $message = '<div class="alert alert-danger">Invalid file. Please upload a JPG, PNG or GIF file under 2MB.</div>';
        }
    }
}

// Handle identity document upload
if (isset($_FILES['identity_document']) && $_FILES['identity_document']['error'] === 0) {
    $allowed_types = ['application/pdf', 'image/jpeg', 'image/png', 'image/jpg'];
    $max_size = 5 * 1024 * 1024; // 5MB
    
    if (in_array($_FILES['identity_document']['type'], $allowed_types) && $_FILES['identity_document']['size'] <= $max_size) {
        $upload_dir = "../uploads/identity_documents/teachers/";
        
        // Create directory if it doesn't exist
        if (!file_exists($upload_dir)) {
            mkdir($upload_dir, 0755, true);
        }
        
        // Generate secure filename
        $file_extension = pathinfo($_FILES['identity_document']['name'], PATHINFO_EXTENSION);
        $filename = 'identity_' . $teacher_id . '_' . time() . '.' . $file_extension;
        $target_path = $upload_dir . $filename;
        
        if (move_uploaded_file($_FILES['identity_document']['tmp_name'], $target_path)) {
            $identity_doc_path = "uploads/identity_documents/teachers/" . $filename;
            
            // Delete old document if exists
            $old_doc_query = "SELECT identity_document_path FROM users WHERE user_id = ?";
            $old_stmt = $conn->prepare($old_doc_query);
            $old_stmt->bind_param("i", $teacher_id);
            $old_stmt->execute();
            $old_result = $old_stmt->get_result();
            $old_doc = $old_result->fetch_assoc();
            
            if ($old_doc && !empty($old_doc['identity_document_path'])) {
                $old_file_path = "../" . $old_doc['identity_document_path'];
                if (file_exists($old_file_path)) {
                    unlink($old_file_path);
                }
            }
            
            // Update database with new document
            $update_identity = "UPDATE users SET 
                              identity_document_path = ?, 
                              identity_verification_status = 'pending',
                              identity_uploaded_at = NOW(),
                              identity_verified_at = NULL,
                              identity_verified_by = NULL,
                              identity_rejection_reason = NULL
                              WHERE user_id = ?";
            
            $update_stmt = $conn->prepare($update_identity);
            if ($update_stmt) {
                $update_stmt->bind_param("si", $identity_doc_path, $teacher_id);
                if ($update_stmt->execute()) {
                    // Log the upload action
                    $log_action = "INSERT INTO identity_verification_logs (teacher_id, action, new_status, document_path) VALUES (?, 'uploaded', 'pending', ?)";
                    $log_stmt = $conn->prepare($log_action);
                    if ($log_stmt) {
                        $log_stmt->bind_param("is", $teacher_id, $identity_doc_path);
                        $log_stmt->execute();
                    }
                    
                    $message = '<div class="alert alert-success"><svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-check-circle me-2" viewBox="0 0 16 16"><path d="M8 15A7 7 0 1 1 8 1a7 7 0 0 1 0 14zm0 1A8 8 0 1 0 8 0a8 8 0 0 0 0 16z"/><path d="M10.97 4.97a.235.235 0 0 0-.02.022L7.477 9.417 5.384 7.323a.75.75 0 0 0-1.06 1.06L6.97 11.03a.75.75 0 0 0 1.079-.02l3.992-4.99a.75.75 0 0 0-1.071-1.05z"/></svg>Identity document uploaded successfully! Our team will review it within 1-3 business days.</div>';
                    
                    // Refresh teacher data
                    $teacher = get_teacher($conn, $teacher_id);
                } else {
                    $message = '<div class="alert alert-danger">Failed to update identity document in database: ' . $update_stmt->error . '</div>';
                }
            } else {
                $message = '<div class="alert alert-danger">Database error: ' . $conn->error . '</div>';
            }
        } else {
            $message = '<div class="alert alert-danger">Failed to upload identity document to server.</div>';
        }
    } else {
        $message = '<div class="alert alert-danger">Invalid file. Please upload a PDF, JPG, or PNG file under 5MB.</div>';
    }
}
?>

<!-- Additional profile page specific CSS -->
<style>
    /* Modern Profile Page Styles */
    .profile-card {
        border: none;
        border-radius: 16px;
        overflow: hidden;
        box-shadow: 0 6px 15px rgba(0,0,0,.05);
        transition: all 0.3s ease;
    }
    
    .profile-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 10px 25px rgba(0,0,0,.08);
    }
    
    .profile-cover {
        height: 120px;
        background: linear-gradient(135deg, #6610f2 0%, #7952b3 100%);
    }
    
    .profile-image-container {
        display: flex;
        justify-content: center;
        margin-top: -50px;
        margin-bottom: 15px;
    }
    
    .profile-image {
        width: 100px;
        height: 100px;
        border-radius: 50%;
        object-fit: cover;
        border: 4px solid white;
        box-shadow: 0 5px 15px rgba(0,0,0,.1);
    }
    
    .profile-details {
        padding: 0 20px 20px;
        text-align: center;
    }
    
    .profile-name {
        font-weight: 600;
        margin-bottom: 5px;
        color: #212529;
    }
    
    .profile-role {
        color: #6c757d;
        font-size: 0.95rem;
        margin-bottom: 15px;
    }
    
    .profile-stats {
        display: flex;
        justify-content: space-around;
        padding: 15px 0;
        border-top: 1px solid rgba(0,0,0,.1);
        border-bottom: 1px solid rgba(0,0,0,.1);
    }
    
    .stat {
        text-align: center;
    }
    
    .stat-number {
        font-size: 1.5rem;
        font-weight: 600;
        color: #6610f2;
    }
    
    .stat-label {
        font-size: 0.8rem;
        color: #6c757d;
    }
    
    .social-links {
        display: flex;
        justify-content: center;
        gap: 15px;
        margin-top: 20px;
    }
    
    .social-links a {
        width: 36px;
        height: 36px;
        border-radius: 50%;
        background-color: rgba(102, 16, 242, 0.1);
        color: #6610f2;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: all 0.3s ease;
    }
    
    .social-links a:hover {
        background-color: #6610f2;
        color: white;
    }
    
    /* Form styling */
    .form-card {
        border: none;
        border-radius: 16px;
        box-shadow: 0 6px 15px rgba(0,0,0,.05);
    }
    
    .form-card .card-header {
        background-color: transparent;
        border-bottom: 1px solid rgba(0,0,0,.1);
        padding: 16px 20px;
    }
    
    .form-card .card-title {
        font-weight: 600;
        margin: 0;
        color: #212529;
    }
    
    .form-control {
        border-radius: 8px;
        border: 1px solid rgba(0,0,0,.1);
        padding: 10px 15px;
        transition: all 0.2s ease;
    }
    
    .form-control:focus {
        border-color: #6610f2;
        box-shadow: 0 0 0 0.25rem rgba(102, 16, 242, 0.25);
    }
    
    .btn-panadite {
        background-color: #6610f2;
        border: none;
        border-radius: 50px;
        padding: 10px 25px;
        color: white;
        font-weight: 500;
        transition: all 0.3s ease;
    }
    
    .btn-panadite:hover {
        background-color: #5a0db6;
        transform: translateY(-2px);
        box-shadow: 0 5px 15px rgba(102, 16, 242, 0.3);
        color: white;
    }
    
    .contact-info {
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 8px;
        color: #6c757d;
    }
    
    /* Background utilities */
    .bg-primary-light {
        background-color: rgba(13, 110, 253, 0.1);
    }
    
    /* Profile image upload controls */
    .profile-image-wrapper:hover .profile-image-controls {
        opacity: 1 !important;
    }
    
    .profile-image-controls button {
        font-size: 12px;
        font-weight: 500;
        border: none;
        box-shadow: 0 2px 5px rgba(0,0,0,0.2);
    }
    
    .profile-image-controls .btn-light:hover {
        background-color: #fff;
        transform: translateY(-1px);
    }
    
    .profile-image-controls .btn-danger:hover {
        background-color: #dc3545;
        transform: translateY(-1px);
    }
</style>

<!-- Main Content -->
    
    <!-- Alert Message -->
    <?php if(!empty($message)): ?>
        <?= $message ?>
    <?php endif; ?>
    
    <!-- Main Profile Content -->
    <div class="row">
                    <!-- Profile Overview -->
                    <div class="col-lg-4">
                        <div class="card profile-card">
                            <div class="profile-cover"></div>
                            <div class="profile-image-container position-relative">
                                <div class="profile-image-wrapper position-relative">
                                    <?php if(!empty($teacher['profile_picture'])): ?>
                                        <img src="../<?= htmlspecialchars($teacher['profile_picture']) ?>" class="profile-image" alt="Profile Image" id="profileImageDisplay">
                                    <?php else: ?>
                                        <div class="profile-image profile-image-placeholder d-flex align-items-center justify-content-center" style="background: linear-gradient(135deg, #6610f2 0%, #7952b3 100%); color: white; font-size: 32px; font-weight: bold;" id="profileImageDisplay">
                                            <?= strtoupper(substr($teacher['first_name'], 0, 1) . substr($teacher['last_name'], 0, 1)) ?>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <!-- Upload controls overlay -->
                                    <div class="profile-image-controls position-absolute w-100 h-100 d-flex align-items-center justify-content-center" style="top: 0; left: 0; background: rgba(0,0,0,0.7); border-radius: 50%; opacity: 0; transition: opacity 0.3s ease; cursor: pointer;" id="imageControls">
                                        <div class="text-center">
                                            <?php if(!empty($teacher['profile_picture'])): ?>
                                                <div class="d-flex gap-2 flex-column">
                                                    <button type="button" class="btn btn-sm btn-light rounded-pill px-3" onclick="document.getElementById('profilePictureInput').click()">
                                                        <i class="fas fa-camera"></i> Change
                                                    </button>
                                                    <form method="post" style="display: inline;">
                                                        <button type="submit" name="remove_picture" class="btn btn-sm btn-danger rounded-pill px-3" onclick="return confirm('Are you sure you want to remove your profile picture?')">
                                                            <i class="fas fa-trash"></i> Remove
                                                        </button>
                                                    </form>
                                                </div>
                                            <?php else: ?>
                                                <button type="button" class="btn btn-sm btn-light rounded-pill px-3" onclick="document.getElementById('profilePictureInput').click()">
                                                    <i class="fas fa-camera"></i> Upload Photo
                                                </button>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Hidden file input -->
                                <form method="post" enctype="multipart/form-data" id="profilePictureForm">
                                    <input type="file" id="profilePictureInput" name="profile_picture" accept="image/*" style="display: none;" onchange="handleFileUpload(this)">
                                </form>
                                
                                <script>
                                function handleFileUpload(input) {
                                    if (input.files && input.files[0]) {
                                        console.log('File selected:', input.files[0].name);
                                        console.log('File size:', input.files[0].size);
                                        console.log('File type:', input.files[0].type);
                                        
                                        // Show loading message
                                        const imageDisplay = document.getElementById('profileImageDisplay');
                                        const controls = document.getElementById('imageControls');
                                        controls.innerHTML = '<div class="text-center text-white"><i class="fas fa-spinner fa-spin"></i> Uploading...</div>';
                                        
                                        // Submit form
                                        document.getElementById('profilePictureForm').submit();
                                    }
                                }
                                </script>
                            </div>
                            <div class="profile-details">
                                <h4 class="profile-name"><?= htmlspecialchars($teacher['first_name'] . ' ' . $teacher['last_name']) ?></h4>
                                <div class="d-flex align-items-center justify-content-center mb-3">
                                    <span class="profile-role">Teacher</span>
                                    <span class="badge bg-primary-light text-primary ms-2 px-3 py-1 rounded-pill">Verified</span>
                                </div>
                                
                                <?php if(!empty($profile['bio'])): ?>
                                    <p class="mb-3 text-muted"><?= htmlspecialchars($profile['bio']) ?></p>
                                <?php endif; ?>
                                
                                <?php
                                // Get teacher statistics from the database
                                $stats_query = "SELECT 
                                    (SELECT COUNT(*) FROM courses WHERE teacher_id = ?) as course_count,
                                    (SELECT COUNT(*) FROM enrollments e JOIN courses c ON e.course_id = c.course_id WHERE c.teacher_id = ?) as student_count,
                                    (SELECT COUNT(*) FROM quizzes q JOIN courses c ON q.course_id = c.course_id WHERE c.teacher_id = ?) as quiz_count";
                                $stats_stmt = $conn->prepare($stats_query);
                                $stats_stmt->bind_param("iii", $teacher_id, $teacher_id, $teacher_id);
                                $stats_stmt->execute();
                                $stats_result = $stats_stmt->get_result();
                                $stats = $stats_result->fetch_assoc();
                                ?>
                                
                                <div class="profile-stats">
                                    <div class="stat">
                                        <div class="stat-number"><?= $stats['course_count'] ?? 0 ?></div>
                                        <div class="stat-label">Courses</div>
                                    </div>
                                    <div class="stat">
                                        <div class="stat-number"><?= $stats['student_count'] ?? 0 ?></div>
                                        <div class="stat-label">Students</div>
                                    </div>
                                    <div class="stat">
                                        <div class="stat-number"><?= $stats['quiz_count'] ?? 0 ?></div>
                                        <div class="stat-label">Quizzes</div>
                                    </div>
                                </div>
                                
                                <div class="social-links mt-3">
                                    <?php if(!empty($profile['social_linkedin'])): ?>
                                        <a href="<?= htmlspecialchars($profile['social_linkedin']) ?>" target="_blank" title="LinkedIn">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-linkedin" viewBox="0 0 16 16">
                                                <path d="M0 1.146C0 .513.526 0 1.175 0h13.65C15.474 0 16 .513 16 1.146v13.708c0 .633-.526 1.146-1.175 1.146H1.175C.526 16 0 15.487 0 14.854V1.146zm4.943 12.248V6.169H2.542v7.225h2.401zm-1.2-8.212c.837 0 1.358-.554 1.358-1.248-.015-.709-.52-1.248-1.342-1.248-.822 0-1.359.54-1.359 1.248 0 .694.521 1.248 1.327 1.248h.016zm4.908 8.212V9.359c0-.216.016-.432.08-.586.173-.431.568-.878 1.232-.878.869 0 1.216.662 1.216 1.634v3.865h2.401V9.25c0-2.22-1.184-3.252-2.764-3.252-1.274 0-1.845.7-2.165 1.193v.025h-.016a5.54 5.54 0 0 1 .016-.025V6.169h-2.4c.03.678 0 7.225 0 7.225h2.4z"/>
                                            </svg>
                                        </a>
                                    <?php endif; ?>
                                    <?php if(!empty($profile['social_twitter'])): ?>
                                        <a href="<?= htmlspecialchars($profile['social_twitter']) ?>" target="_blank" title="Twitter">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-twitter-x" viewBox="0 0 16 16">
                                                <path d="M12.6.75h2.454l-5.36 6.142L16 15.25h-4.937l-3.867-5.07-4.425 5.07H.316l5.733-6.57L0 .75h5.063l3.495 4.633L12.601.75Zm-.86 13.028h1.36L4.323 2.145H2.865l8.875 11.633Z"/>
                                            </svg>
                                        </a>
                                    <?php endif; ?>
                                    <?php if(!empty($profile['website'])): ?>
                                        <a href="<?= htmlspecialchars($profile['website']) ?>" target="_blank" title="Website">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-globe" viewBox="0 0 16 16">
                                                <path d="M0 8a8 8 0 1 1 16 0A8 8 0 0 1 0 8zm7.5-6.923c-.67.204-1.335.82-1.887 1.855A7.97 7.97 0 0 0 5.145 4H7.5V1.077zM4.09 4a9.267 9.267 0 0 1 .64-1.539 6.7 6.7 0 0 1 .597-.933A7.025 7.025 0 0 0 2.255 4H4.09zm-.582 3.5c.03-.877.138-1.718.312-2.5H1.674a6.958 6.958 0 0 0-.656 2.5h2.49zM4.847 5a12.5 12.5 0 0 0-.338 2.5H7.5V5H4.847zM8.5 5v2.5h2.99a12.495 12.495 0 0 0-.337-2.5H8.5zM4.51 8.5a12.5 12.5 0 0 0 .337 2.5H7.5V8.5H4.51zm3.99 0V11h2.653c.187-.765.306-1.608.338-2.5H8.5zM5.145 12c.138.386.295.744.468 1.068.552 1.035 1.218 1.65 1.887 1.855V12H5.145zm.182 2.472a6.696 6.696 0 0 1-.597-.933A9.268 9.268 0 0 1 4.09 12H2.255a7.024 7.024 0 0 0 3.072 2.472zM3.82 11a13.652 13.652 0 0 1-.312-2.5h-2.49c.062.89.291 1.733.656 2.5H3.82zm6.853 3.472A7.024 7.024 0 0 0 13.745 12H11.91a9.27 9.27 0 0 1-.64 1.539 6.688 6.688 0 0 1-.597.933zM8.5 12v2.923c.67-.204 1.335-.82 1.887-1.855.173-.324.33-.682.468-1.068H8.5zm3.68-1h2.146c.365-.767.594-1.61.656-2.5h-2.49a13.65 13.65 0 0 1-.312 2.5zm2.802-3.5a6.959 6.959 0 0 0-.656-2.5H12.18c.174.782.282 1.623.312 2.5h2.49zM11.27 2.461c.247.464.462.98.64 1.539h1.835a7.024 7.024 0 0 0-3.072-2.472c.218.284.418.598.597.933zM10.855 4a7.966 7.966 0 0 0-.468-1.068C9.835 1.897 9.17 1.282 8.5 1.077V4h2.355z"/>
                                            </svg>
                                        </a>
                                    <?php endif; ?>
                                </div>
                                
                                <?php if(!empty($profile['phone'])): ?>
                                    <div class="mt-3 contact-info">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-telephone" viewBox="0 0 16 16">
                                            <path d="M3.654 1.328a.678.678 0 0 0-1.015-.063L1.605 2.3c-.483.484-.661 1.169-.45 1.77a17.568 17.568 0 0 0 4.168 6.608 17.569 17.569 0 0 0 6.608 4.168c.601.211 1.286.033 1.77-.45l1.034-1.034a.678.678 0 0 0-.063-1.015l-2.307-1.794a.678.678 0 0 0-.58-.122l-2.19.547a1.745 1.745 0 0 1-1.657-.459L5.482 8.062a1.745 1.745 0 0 1-.46-1.657l.548-2.19a.678.678 0 0 0-.122-.58L3.654 1.328zM1.884.511a1.745 1.745 0 0 1 2.612.163L6.29 2.98c.329.423.445.974.315 1.494l-.547 2.19a.678.678 0 0 0 .178.643l2.457 2.457a.678.678 0 0 0 .644.178l2.189-.547a1.745 1.745 0 0 1 1.494.315l2.306 1.794c.829.645.905 1.87.163 2.611l-1.034 1.034c-.74.74-1.846 1.065-2.877.702a18.634 18.634 0 0 1-7.01-4.42 18.634 18.634 0 0 1-4.42-7.009c-.362-1.03-.037-2.137.703-2.877L1.885.511z"/>
                                        </svg>
                                        <span><?= htmlspecialchars($profile['phone']) ?></span>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Edit Profile Form -->
                    <div class="col-lg-8">
                        <div class="card form-card">
                            <div class="card-header d-flex align-items-center">
                                <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" fill="currentColor" class="bi bi-pencil-square me-2 text-primary" viewBox="0 0 16 16">
                                    <path d="M15.502 1.94a.5.5 0 0 1 0 .706L14.459 3.69l-2-2L13.502.646a.5.5 0 0 1 .707 0l1.293 1.293zm-1.75 2.456-2-2L4.939 9.21a.5.5 0 0 0-.121.196l-.805 2.414a.25.25 0 0 0 .316.316l2.414-.805a.5.5 0 0 0 .196-.12l6.813-6.814z"/>
                                    <path fill-rule="evenodd" d="M1 13.5A1.5 1.5 0 0 0 2.5 15h11a1.5 1.5 0 0 0 1.5-1.5v-6a.5.5 0 0 0-1 0v6a.5.5 0 0 1-.5.5h-11a.5.5 0 0 1-.5-.5v-11a.5.5 0 0 1 .5-.5H9a.5.5 0 0 0 0-1H2.5A1.5 1.5 0 0 0 1 2.5v11z"/>
                                </svg>
                                <h5 class="card-title">Edit Profile</h5>
                            </div>
                            <div class="card-body p-4">
                                
                                <!-- VISIBLE DEBUGGING SECTION -->
                                <?php if (!empty($message)): ?>
                                    <?= $message ?>
                                <?php endif; ?>
                                
                                <?php if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_profile'])): ?>
                                <div class="alert alert-info">
                                    <h5>🔍 LIVE DEBUG - Form Submission Analysis</h5>
                                    <strong>Form Data Received:</strong><br>
                                    <?php 
                                    foreach ($_POST as $key => $value) {
                                        if ($key !== 'update_profile' && !is_array($value)) {
                                            $display_value = empty($value) ? '<em style="color: #999;">EMPTY</em>' : htmlspecialchars($value);
                                            echo "<strong>{$key}:</strong> {$display_value}<br>";
                                        }
                                    }
                                    ?>
                                    <hr>
                                    <strong>Current User ID:</strong> <?= $teacher_id ?><br>
                                    <strong>Session Data:</strong> <?= json_encode($_SESSION['user_id'] ?? 'No session') ?><br>
                                    <strong>Database Connection:</strong> <?= $conn ? 'Connected' : 'Failed' ?><br>
                                </div>
                                
                                <!-- DIRECT USERS TABLE DEBUG -->
                                <div class="alert alert-warning">
                                    <h5>🔍 DIRECT DEBUG - Users Table Update Process</h5>
                                    <?php
                                    // Show the exact SQL being prepared
                                     $debug_update_user = "UPDATE users SET 
                                                        first_name = ?, last_name = ?, email = ?, bio = ?, cell_number = ?,
                                                        title = ?, id_number = ?, suburb = ?, province = ?, communication_method = ?,
                                                        owns_laptop = ?, education_status = ?, employment_status = ?, 
                                                        heard_about_us = ?, referral_person_name = ?, job_field_other = ?
                                                        WHERE user_id = ?";
                                    echo "<strong>SQL Query:</strong><br><code style='display:block; background:white; padding:10px;'>" . htmlspecialchars($debug_update_user) . "</code><br>";
                                    
                                    // Try to prepare the statement and show results
                                    $debug_stmt = $conn->prepare($debug_update_user);
                                    if (!$debug_stmt) {
                                        echo "<span style='color: red;'>🚨 <strong>PREPARE FAILED!</strong></span><br>";
                                        echo "<strong>MySQL Error:</strong> " . $conn->error . "<br>";
                                        echo "<strong>MySQL Error Number:</strong> " . $conn->errno . "<br>";
                                    } else {
                                        echo "<span style='color: green;'>✅ <strong>PREPARE SUCCESS!</strong></span><br>";
                                        
                                        // Now try to bind and execute
                                        $teacher_bio = trim($_POST['bio']);
                                        $teacher_expertise = trim($_POST['course_title']);
                                        $teacher_experience = trim($_POST['experience']);
                                         $debug_values = [
                                             trim($_POST['first_name']),
                                             trim($_POST['last_name']),
                                             trim($_POST['email']),
                                             trim($_POST['bio']),
                                             trim($_POST['cell_number'] ?? ''),
                                             trim($_POST['title'] ?? ''),
                                             trim($_POST['id_number'] ?? ''),
                                             trim($_POST['suburb'] ?? ''),
                                             trim($_POST['province'] ?? ''),
                                             in_array(trim($_POST['communication_method'] ?? ''), ['Email', 'WhatsApp']) ? trim($_POST['communication_method']) : 'Email',
                                             trim($_POST['owns_laptop'] ?? ''),
                                             trim($_POST['education_status'] ?? ''),
                                             trim($_POST['employment_status'] ?? ''),
                                             trim($_POST['heard_about_us'] ?? ''),
                                             trim($_POST['referral_person_name'] ?? ''),
                                             trim($_POST['experience'] ?? ''),
                                             $teacher_id
                                         ];
                                        
                                        echo "<strong>Values being bound:</strong><br>";
                                        foreach ($debug_values as $i => $val) {
                                            $display_val = empty($val) ? '<em style="color: #999;">EMPTY</em>' : htmlspecialchars($val);
                                            echo "Param " . ($i+1) . ": {$display_val}<br>";
                                        }
                                        
                                        $debug_bind_result = $debug_stmt->bind_param("ssssssssssssssssi", ...$debug_values);
                                        if (!$debug_bind_result) {
                                            echo "<span style='color: red;'>🚨 <strong>BIND FAILED!</strong></span><br>";
                                            echo "<strong>Bind Error:</strong> " . $debug_stmt->error . "<br>";
                                        } else {
                                            echo "<span style='color: green;'>✅ <strong>BIND SUCCESS!</strong></span><br>";
                                            
                                            $debug_execute_result = $debug_stmt->execute();
                                            if (!$debug_execute_result) {
                                                echo "<span style='color: red;'>🚨 <strong>EXECUTE FAILED!</strong></span><br>";
                                                echo "<strong>Execute Error:</strong> " . $debug_stmt->error . "<br>";
                                            } else {
                                                echo "<span style='color: green;'>✅ <strong>EXECUTE SUCCESS!</strong></span><br>";
                                                echo "<strong>Affected Rows:</strong> " . $debug_stmt->affected_rows . "<br>";
                                                
                                                if ($debug_stmt->affected_rows == 0) {
                                                    echo "<hr><span style='color: red;'>🚨 <strong>ZERO ROWS AFFECTED - INVESTIGATING...</strong></span><br>";
                                                    
                                                    // Check if user ID 44 exists at all and compare ALL values
                                                    $check_user = $conn->query("SELECT * FROM users WHERE user_id = 44");
                                                    if ($check_user && $check_user->num_rows > 0) {
                                                        echo "<strong>✅ User ID 44 EXISTS in database</strong><br>";
                                                        echo "<strong>🔍 COMPARING CURRENT VS NEW VALUES:</strong><br>";
                                                        $current_data = $check_user->fetch_assoc();
                                                        
                                                        // Compare the values we're trying to update
                                                        $comparison_fields = [
                                                            'first_name' => trim($_POST['first_name']),
                                                            'last_name' => trim($_POST['last_name']), 
                                                            'email' => trim($_POST['email']),
                                                            'bio' => trim($_POST['bio']),
                                                            'cell_number' => trim($_POST['cell_number'] ?? ''),
                                                            'title' => trim($_POST['title'] ?? ''),
                                                            'id_number' => trim($_POST['id_number'] ?? ''),
                                                            'suburb' => trim($_POST['suburb'] ?? ''),
                                                            'province' => trim($_POST['province'] ?? ''),
                                                            'communication_method' => trim($_POST['communication_method'] ?? ''),
                                                            'owns_laptop' => trim($_POST['owns_laptop'] ?? ''),
                                                            'education_status' => trim($_POST['education_status'] ?? ''),
                                                            'employment_status' => trim($_POST['employment_status'] ?? ''),
                                                            'heard_about_us' => trim($_POST['heard_about_us'] ?? ''),
                                                            'referral_person_name' => trim($_POST['referral_person_name'] ?? '')
                                                        ];
                                                        
                                                        $changes_count = 0;
                                                        foreach ($comparison_fields as $field => $new_value) {
                                                            $current_value = $current_data[$field] ?? '';
                                                            if ($current_value != $new_value) {
                                                                echo "<span style='color: orange;'>📝 <strong>{$field}:</strong></span><br>";
                                                                echo "&nbsp;&nbsp;Current: '" . htmlspecialchars($current_value) . "'<br>";
                                                                echo "&nbsp;&nbsp;New: '" . htmlspecialchars($new_value) . "'<br>";
                                                                $changes_count++;
                                                            } else {
                                                                echo "<span style='color: green;'>✅ <strong>{$field}:</strong> No change ('" . htmlspecialchars($current_value) . "')</span><br>";
                                                            }
                                                        }
                                                        
                                                        echo "<hr><strong>🎯 SUMMARY:</strong><br>";
                                                        echo "<strong>Fields to change:</strong> {$changes_count}<br>";
                                                        if ($changes_count == 0) {
                                                            echo "<span style='color: red;'>🚨 <strong>NO CHANGES DETECTED!</strong></span><br>";
                                                            echo "<strong>Explanation:</strong> MySQL didn't update because all values are identical to current database values.<br>";
                                                        } else {
                                                            echo "<span style='color: green;'>✅ <strong>CHANGES DETECTED!</strong></span><br>";
                                                            echo "<strong>Something else is preventing the update...</strong><br>";
                                                        }
                                                    } else {
                                                        echo "<strong>❌ User ID 44 does NOT exist in users table!</strong><br>";
                                                        
                                                        // Find users with similar email
                                                        $find_user = $conn->prepare("SELECT user_id, username, email, first_name, last_name FROM users WHERE email = ?");
                                                        $find_user->bind_param("s", trim($_POST['email']));
                                                        $find_user->execute();
                                                        $result = $find_user->get_result();
                                                        
                                                        if ($result && $result->num_rows > 0) {
                                                            echo "<strong>🔍 Found user with your email:</strong><br>";
                                                            while ($found_user = $result->fetch_assoc()) {
                                                                foreach ($found_user as $field => $value) {
                                                                    echo "&nbsp;&nbsp;<strong>{$field}:</strong> " . htmlspecialchars($value) . "<br>";
                                                                }
                                                            }
                                                        } else {
                                                            echo "<strong>❌ No user found with email: " . htmlspecialchars(trim($_POST['email'])) . "</strong><br>";
                                                        }
                                                        $find_user->close();
                                                    }
                                                }
                                            }
                                        }
                                        $debug_stmt->close();
                                    }
                                    ?>
                                 </div>
                                 <?php endif; ?>
                
                <!-- Document Upload Tip -->
                <div class="alert alert-info d-flex align-items-start">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-lightbulb me-2 mt-1 text-primary" viewBox="0 0 16 16">
                        <path d="M2 6a6 6 0 1 1 10.174 4.31c-.203.196-.359.4-.453.619l-.762 1.769A.5.5 0 0 1 10.5 13a.5.5 0 0 1 0 1 .5.5 0 0 1 0 1l-.224.447a1 1 0 0 1-.894.553H6.618a1 1 0 0 1-.894-.553L5.5 15a.5.5 0 0 1 0-1 .5.5 0 0 1 0-1 .5.5 0 0 1-.459-.31l-.762-1.77A4.926 4.926 0 0 1 3.69 11.06c-.161-.925-.621-1.569-1.023-1.928A.5.5 0 0 1 2.5 9V6c0-.621.504-1.125 1.125-1.125h9.75c.621 0 1.125.504 1.125 1.125v3a.5.5 0 0 1-.168.374c-.402.359-.862 1.003-1.023 1.928-.073.42-.156.745-.31 1.06A5.926 5.926 0 0 1 11.31 10.69c.203-.196.359-.4.453-.619l.762-1.769A.5.5 0 0 0 12.5 8a.5.5 0 0 0 0-1 .5.5 0 0 0 0-1l.224-.447a1 1 0 0 0 .894-.553H9.382a1 1 0 0 0 .894.553L10.5 6a.5.5 0 0 0 0 1 .5.5 0 0 0 0 1 .5.5 0 0 0 .459.31l.762 1.77c.16.373.346.678.453.91.107.23.134.484.134.709v.5a.5.5 0 0 0 .5.5h1a.5.5 0 0 0 .5-.5v-.5c0-.225-.027-.479-.134-.709a4.926 4.926 0 0 0-.453-.91l-.762-1.77A.5.5 0 0 0 12.5 8a.5.5 0 0 0 0-1 .5.5 0 0 0 0-1 .5.5 0 0 0-.459-.31l-.762-1.77A4.926 4.926 0 0 0 10.826 3.06c.161-.925.621-1.569 1.023-1.928A.5.5 0 0 0 12.5 1V0c0-.621-.504-1.125-1.125-1.125h-9.75C.504-1.125 0-.621 0 0v1a.5.5 0 0 0 .668.468c.402-.359.862-1.003 1.023-1.928A5.926 5.926 0 0 0 2.174 0z"/>
                    </svg>
                    <div>
                        <strong>📄 Document Upload Tip:</strong><br>
                        <small class="text-muted">To upload your identity document, scroll to the bottom of the form. After uploading your document, remember to click the "Save Changes" button to complete the verification process.</small>
                    </div>
                </div>
                
                <form action="profile.php" method="post" enctype="multipart/form-data">
                                    <!-- Personal Info -->
                                    <div class="mb-4">

                                        <div class="d-flex align-items-center mb-3">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-person me-2 text-primary" viewBox="0 0 16 16">
                                                <path d="M8 8a3 3 0 1 0 0-6 3 3 0 0 0 0 6Zm2-3a2 2 0 1 1-4 0 2 2 0 0 1 4 0Zm4 8c0 1-1 1-1 1H3s-1 0-1-1 1-4 6-4 6 3 6 4Zm-1-.004c-.001-.246-.154-.986-.832-1.664C11.516 10.68 10.289 10 8 10c-2.29 0-3.516.68-4.168 1.332-.678.678-.83 1.418-.832 1.664h10Z"/>
                                            </svg>
                                            <h6 class="fw-bold mb-0">Personal Information</h6>
                                        </div>
                                        <div class="row g-3">
                                            <div class="col-md-6">
                                                <label for="first_name" class="form-label">First Name</label>
                                                <input type="text" class="form-control" id="first_name" name="first_name" value="<?= htmlspecialchars($teacher['first_name']) ?>" required>
                                            </div>
                                            <div class="col-md-6">
                                                <label for="last_name" class="form-label">Last Name</label>
                                                <input type="text" class="form-control" id="last_name" name="last_name" value="<?= htmlspecialchars($teacher['last_name']) ?>" required>
                                            </div>
                                            <div class="col-md-12">
                                                <label for="bio" class="form-label">Bio</label>
                                                <textarea class="form-control" id="bio" name="bio" rows="3" placeholder="Tell students about yourself, your expertise and teaching style"><?= htmlspecialchars($teacher['bio'] ?? '') ?></textarea>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <!-- Contact Info -->
                                    <div class="mb-4">
                                        <div class="d-flex align-items-center mb-3">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-envelope me-2 text-primary" viewBox="0 0 16 16">
                                                <path d="M0 4a2 2 0 0 1 2-2h12a2 2 0 0 1 2 2v8a2 2 0 0 1-2 2H2a2 2 0 0 1-2-2V4Zm2-1a1 1 0 0 0-1 1v.217l7 4.2 7-4.2V4a1 1 0 0 0-1-1H2Zm13 2.383-4.708 2.825L15 11.105V5.383Zm-.034 6.876-5.64-3.471L8 9.583l-1.326-.795-5.64 3.47A1 1 0 0 0 2 13h12a1 1 0 0 0 .966-.741ZM1 11.105l4.708-2.897L1 5.383v5.722Z"/>
                                            </svg>
                                            <h6 class="fw-bold mb-0">Contact Information</h6>
                                        </div>
                                        <div class="row g-3">
                                            <div class="col-md-6">
                                                <label for="email" class="form-label">Email</label>
                                                <div class="input-group">
                                                    <span class="input-group-text bg-light">
                                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="currentColor" class="bi bi-at" viewBox="0 0 16 16">
                                                            <path d="M13.106 7.222c0-2.967-2.249-5.032-5.482-5.032-3.35 0-5.646 2.318-5.646 5.702 0 3.493 2.235 5.708 5.762 5.708.862 0 1.689-.123 2.304-.335v-.862c-.43.199-1.354.328-2.29.328-2.926 0-4.813-1.88-4.813-4.798 0-2.844 1.921-4.881 4.594-4.881 2.735 0 4.608 1.688 4.608 4.156 0 1.682-.554 2.769-1.416 2.769-.492 0-.772-.28-.772-.76V5.206h-.774v.834h-.126c-.252-.414-.752-.613-1.272-.613-1.203 0-2.073.923-2.073 2.374 0 1.372.762 2.277 1.873 2.277.602 0 1.054-.247 1.377-.702h.087c.088.553.732.959 1.454.959 1.478 0 2.383-1.327 2.383-3.103zm-5.988 1.096c-.66 0-1.086-.57-1.086-1.49 0-.928.505-1.55 1.178-1.55.65 0 1.096.532 1.096 1.507 0 .892-.501 1.533-1.188 1.533z"/>
                                                        </svg>
                                                    </span>
                                                    <input type="email" class="form-control" id="email" name="email" value="<?= htmlspecialchars($teacher['email']) ?>" required>
                                                </div>
                                                <small class="text-muted">Enter your email address</small>
                                            </div>
                                            <div class="col-md-6">
                                                <label for="phone" class="form-label">Phone</label>
                                                <div class="input-group">
                                                    <span class="input-group-text bg-light">
                                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="currentColor" class="bi bi-telephone" viewBox="0 0 16 16">
                                                            <path d="M3.654 1.328a.678.678 0 0 0-1.015-.063L1.605 2.3c-.483.484-.661 1.169-.45 1.77a17.568 17.568 0 0 0 4.168 6.608 17.569 17.569 0 0 0 6.608 4.168c.601.211 1.286.033 1.77-.45l1.034-1.034a.678.678 0 0 0-.063-1.015l-2.307-1.794a.678.678 0 0 0-.58-.122l-2.19.547a1.745 1.745 0 0 1-1.657-.459L5.482 8.062a1.745 1.745 0 0 1-.46-1.657l.548-2.19a.678.678 0 0 0-.122-.58L3.654 1.328zM1.884.511a1.745 1.745 0 0 1 2.612.163L6.29 2.98c.329.423.445.974.315 1.494l-.547 2.19a.678.678 0 0 0 .178.643l2.457 2.457a.678.678 0 0 0 .644.178l2.189-.547a1.745 1.745 0 0 1 1.494.315l2.306 1.794c.829.645.905 1.87.163 2.611l-1.034 1.034c-.74.74-1.846 1.065-2.877.702a18.634 18.634 0 0 1-7.01-4.42 18.634 18.634 0 0 1-4.42-7.009c-.362-1.03-.037-2.137.703-2.877L1.885.511z"/>
                                                        </svg>
                                                    </span>
                                                    <input type="tel" class="form-control" id="phone" name="phone" value="<?= htmlspecialchars($teacher['phone'] ?? '') ?>" placeholder="+1 (234) 567-8900">
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <!-- Personal Details Section -->
                                    <div class="mb-4">
                                        <div class="d-flex align-items-center mb-3">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-person-badge me-2 text-primary" viewBox="0 0 16 16">
                                                <path d="M6.5 2a.5.5 0 0 0 0 1h3a.5.5 0 0 0 0-1h-3zM11 8a3 3 0 1 1-6 0 3 3 0 0 1 6 0z"/>
                                                <path d="M4.5 0A2.5 2.5 0 0 0 2 2.5V14a2 2 0 0 0 2 2h8a2 2 0 0 0 2-2V2.5A2.5 2.5 0 0 0 11.5 0h-7zM3 2.5A1.5 1.5 0 0 1 4.5 1h7A1.5 1.5 0 0 1 13 2.5v10.795a4.2 4.2 0 0 0-.776-.492C11.392 12.387 10.063 12 8 12s-3.392.387-4.224.803a4.2 4.2 0 0 0-.776.492V2.5z"/>
                                            </svg>
                                            <h6 class="fw-bold mb-0">Personal Details</h6>
                                        </div>
                                        <div class="row g-3">
                                            <div class="col-md-4">
                                                <label for="title" class="form-label">Title</label>
                                                <select class="form-control" id="title" name="title">
                                                    <option value="">Select Title</option>
                                                    <option value="Mr" <?= ($teacher['title'] ?? '') === 'Mr' ? 'selected' : '' ?>>Mr</option>
                                                    <option value="Mrs" <?= ($teacher['title'] ?? '') === 'Mrs' ? 'selected' : '' ?>>Mrs</option>
                                                    <option value="Ms" <?= ($teacher['title'] ?? '') === 'Ms' ? 'selected' : '' ?>>Ms</option>
                                                    <option value="Dr" <?= ($teacher['title'] ?? '') === 'Dr' ? 'selected' : '' ?>>Dr</option>
                                                    <option value="Prof" <?= ($teacher['title'] ?? '') === 'Prof' ? 'selected' : '' ?>>Prof</option>
                                                </select>
                                            </div>
                                            <div class="col-md-4">
                                                <label for="cell_number" class="form-label">Cell Number</label>
                                                <input type="tel" class="form-control" id="cell_number" name="cell_number" 
                                                       value="<?= htmlspecialchars($teacher['cell_number'] ?? '') ?>" 
                                                       placeholder="e.g., 0823456789">
                                            </div>
                                            <div class="col-md-4">
                                                <label for="id_number" class="form-label">ID Number</label>
                                                <input type="text" class="form-control" id="id_number" name="id_number" 
                                                       value="<?= htmlspecialchars($teacher['id_number'] ?? '') ?>" 
                                                       placeholder="e.g., 9001015800001">
                                            </div>
                                        </div>
                                        <div class="row g-3 mt-1">
                                            <div class="col-md-4">
                                                <label for="suburb" class="form-label">Suburb</label>
                                                <input type="text" class="form-control" id="suburb" name="suburb" 
                                                       value="<?= htmlspecialchars($teacher['suburb'] ?? '') ?>" 
                                                       placeholder="e.g., Pretoria Central">
                                            </div>
                                            <div class="col-md-4">
                                                <label for="province" class="form-label">Province</label>
                                                <select class="form-control" id="province" name="province">
                                                    <option value="">Select Province</option>
                                                    <option value="Gauteng" <?= ($teacher['province'] ?? '') === 'Gauteng' ? 'selected' : '' ?>>Gauteng</option>
                                                    <option value="Western Cape" <?= ($teacher['province'] ?? '') === 'Western Cape' ? 'selected' : '' ?>>Western Cape</option>
                                                    <option value="KwaZulu-Natal" <?= ($teacher['province'] ?? '') === 'KwaZulu-Natal' ? 'selected' : '' ?>>KwaZulu-Natal</option>
                                                    <option value="Eastern Cape" <?= ($teacher['province'] ?? '') === 'Eastern Cape' ? 'selected' : '' ?>>Eastern Cape</option>
                                                    <option value="Free State" <?= ($teacher['province'] ?? '') === 'Free State' ? 'selected' : '' ?>>Free State</option>
                                                    <option value="Limpopo" <?= ($teacher['province'] ?? '') === 'Limpopo' ? 'selected' : '' ?>>Limpopo</option>
                                                    <option value="Mpumalanga" <?= ($teacher['province'] ?? '') === 'Mpumalanga' ? 'selected' : '' ?>>Mpumalanga</option>
                                                    <option value="North West" <?= ($teacher['province'] ?? '') === 'North West' ? 'selected' : '' ?>>North West</option>
                                                    <option value="Northern Cape" <?= ($teacher['province'] ?? '') === 'Northern Cape' ? 'selected' : '' ?>>Northern Cape</option>
                                                </select>
                                            </div>
                                            <div class="col-md-4">
                                                <label for="communication_method" class="form-label">Preferred Communication</label>
                                                <select class="form-control" id="communication_method" name="communication_method">
                                                     <option value="">Select Method</option>
                                                    <option value="Email" <?= ($teacher['communication_method'] ?? '') === 'Email' ? 'selected' : '' ?>>Email</option>
                                                    <option value="WhatsApp" <?= ($teacher['communication_method'] ?? '') === 'WhatsApp' ? 'selected' : '' ?>>WhatsApp</option>
                                                </select>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <!-- Learning & Background Information -->
                                    <div class="mb-4">
                                        <div class="d-flex align-items-center mb-3">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-book me-2 text-primary" viewBox="0 0 16 16">
                                                <path d="M1 2.828c.885-.37 2.154-.769 3.388-.893 1.33-.134 2.458.063 3.112.752v9.746c-.935-.53-2.12-.603-3.213-.493-1.18.12-2.37.461-3.287.811V2.828zm7.5-.141c.654-.689 1.782-.886 3.112-.752 1.234.124 2.503.523 3.388.893v9.923c-.918-.35-2.107-.692-3.287-.81-1.094-.111-2.278-.039-3.213.492V2.687zM8 1.783C7.015.936 5.587.81 4.287.94c-1.514.153-3.042.672-3.994 1.105A.5.5 0 0 0 0 2.5v11a.5.5 0 0 0 .707.455c.882-.4 2.303-.881 3.68-1.02 1.409-.142 2.59.087 3.223.877a.5.5 0 0 0 .78 0c.633-.79 1.814-1.019 3.222-.877 1.378.139 2.8.62 3.681 1.02A.5.5 0 0 0 16 13.5v-11a.5.5 0 0 0-.293-.455c-.952-.433-2.48-.952-3.994-1.105C10.413.809 8.985.936 8 1.783z"/>
                                            </svg>
                                            <h6 class="fw-bold mb-0">Learning & Background Information</h6>
                                        </div>
                                        <div class="row g-3">
                                            <div class="col-md-6">
                                                <label for="owns_laptop" class="form-label">Do you own a Laptop/PC?</label>
                                                <select class="form-control" id="owns_laptop" name="owns_laptop">
                                                    <option value="">Select Option</option>
                                                    <option value="yes" <?= (strtolower($teacher['owns_laptop'] ?? '')) === 'yes' ? 'selected' : '' ?>>Yes, I own a laptop/PC</option>
                                                    <option value="no" <?= (strtolower($teacher['owns_laptop'] ?? '')) === 'no' ? 'selected' : '' ?>>No, I don't have one</option>
                                                    <option value="shared" <?= (strtolower($teacher['owns_laptop'] ?? '')) === 'shared' ? 'selected' : '' ?>>I have shared access</option>
                                                </select>
                                            </div>
                                            <div class="col-md-6">
                                                <label for="education_status" class="form-label">Education Status</label>
                                                <select class="form-control" id="education_status" name="education_status">
                                                    <option value="">Select Education Level</option>
                                                    <option value="matric" <?= ($teacher['education_status'] ?? '') === 'matric' ? 'selected' : '' ?>>Matric/Grade 12</option>
                                                    <option value="diploma" <?= ($teacher['education_status'] ?? '') === 'diploma' ? 'selected' : '' ?>>Diploma</option>
                                                    <option value="degree" <?= ($teacher['education_status'] ?? '') === 'degree' ? 'selected' : '' ?>>Bachelor's Degree</option>
                                                    <option value="honours" <?= ($teacher['education_status'] ?? '') === 'honours' ? 'selected' : '' ?>>Honours Degree</option>
                                                    <option value="masters" <?= ($teacher['education_status'] ?? '') === 'masters' ? 'selected' : '' ?>>Master's Degree</option>
                                                    <option value="phd" <?= ($teacher['education_status'] ?? '') === 'phd' ? 'selected' : '' ?>>PhD</option>
                                                </select>
                                            </div>
                                        </div>
                                        <div class="row g-3 mt-1">
                                            <div class="col-md-6">
                                                <label for="employment_status" class="form-label">Employment Status</label>
                                                <select class="form-control" id="employment_status" name="employment_status">
                                                    <option value="">Select Employment Status</option>
                                                    <option value="employed_fulltime" <?= ($teacher['employment_status'] ?? '') === 'employed_fulltime' ? 'selected' : '' ?>>Employed Full-time</option>
                                                    <option value="employed_parttime" <?= ($teacher['employment_status'] ?? '') === 'employed_parttime' ? 'selected' : '' ?>>Employed Part-time</option>
                                                    <option value="self_employed" <?= ($teacher['employment_status'] ?? '') === 'self_employed' ? 'selected' : '' ?>>Self-employed</option>
                                                    <option value="unemployed" <?= ($teacher['employment_status'] ?? '') === 'unemployed' ? 'selected' : '' ?>>Unemployed</option>
                                                    <option value="student" <?= ($teacher['employment_status'] ?? '') === 'student' ? 'selected' : '' ?>>Student</option>
                                                    <option value="retired" <?= ($teacher['employment_status'] ?? '') === 'retired' ? 'selected' : '' ?>>Retired</option>
                                                </select>
                                            </div>
                                            <div class="col-md-6">
                                                <label for="heard_about_us" class="form-label">How did you hear about us?</label>
                                                <select class="form-control" id="heard_about_us" name="heard_about_us">
                                                    <option value="">Select Option</option>
                                                    <option value="google" <?= ($teacher['heard_about_us'] ?? '') === 'google' ? 'selected' : '' ?>>Google Search</option>
                                                    <option value="facebook" <?= ($teacher['heard_about_us'] ?? '') === 'facebook' ? 'selected' : '' ?>>Facebook</option>
                                                    <option value="instagram" <?= ($teacher['heard_about_us'] ?? '') === 'instagram' ? 'selected' : '' ?>>Instagram</option>
                                                    <option value="linkedin" <?= ($teacher['heard_about_us'] ?? '') === 'linkedin' ? 'selected' : '' ?>>LinkedIn</option>
                                                    <option value="referral" <?= ($teacher['heard_about_us'] ?? '') === 'referral' ? 'selected' : '' ?>>Referral from friend/colleague</option>
                                                    <option value="advertisement" <?= ($teacher['heard_about_us'] ?? '') === 'advertisement' ? 'selected' : '' ?>>Online Advertisement</option>
                                                    <option value="other" <?= ($teacher['heard_about_us'] ?? '') === 'other' ? 'selected' : '' ?>>Other</option>
                                                </select>
                                            </div>
                                        </div>
                                        <div class="row g-3 mt-1">
                                            <div class="col-md-12">
                                                <label for="referral_person_name" class="form-label">Referred by (Optional)</label>
                                                <input type="text" class="form-control" id="referral_person_name" name="referral_person_name" 
                                                       value="<?= htmlspecialchars($teacher['referral_person_name'] ?? '') ?>" 
                                                       placeholder="Name of person who referred you">
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <!-- Facilitator Information -->
                                    <div class="mb-4">
                                        <div class="d-flex align-items-center mb-3">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-mortarboard me-2 text-primary" viewBox="0 0 16 16">
                                                <path d="M8.211 2.047a.5.5 0 0 0-.422 0L1.5 5.5l6.289 3.234a.5.5 0 0 0 .422 0L14.5 5.5l-6.289-3.453Z"/>
                                                <path d="M17.5 6A1.5 1.5 0 0 0 16 4.5c-1.076 0-2.597.654-4.15 1.466l-.423 3.97c.27.556.626 1.064.928 1.318l.002.002c.28.24.72.24 1 0A5.057 5.057 0 0 0 17.5 6Z"/>
                                                <path d="M2.678 11.894a1 1 0 0 1 .287.801 10.97 10.97 0 0 1-.398 2c-.395.902-1.101 1.677-1.878 2.135A.5.5 0 0 1 0 16.5V14s-2-1-2-3c0-1 1-1 1-1h2.678Z"/>
                                                <path d="M12.5 13.5a3 3 0 1 1-6 0 3 3 0 0 1 6 0Z"/>
                                                <path d="M12.5 13.5V16a.5.5 0 0 1-.5.5H4a.5.5 0 0 1-.5-.5v-2.5a3.5 3.5 0 1 1 7 0Z"/>
                                            </svg>
                                            <h6 class="fw-bold mb-0">Facilitator Information</h6>
                                        </div>
                                        <div class="row g-3">
                                            <div class="col-md-8">
                                                <label for="course_title" class="form-label">Proposed Course Title</label>
                                                <input type="text" class="form-control" id="course_title" name="course_title" 
                                                       value="<?= htmlspecialchars($teacher['course_title'] ?? '') ?>" 
                                                       placeholder="What course do you want to teach?">
                                            </div>
                                            <div class="col-md-4">
                                                <label for="experience" class="form-label">Years of Experience</label>
                                                <select class="form-control" id="experience" name="experience">
                                                    <option value="">Select experience level</option>
                                                    <option value="0-1" <?= ($teacher['experience'] ?? '') === '0-1' ? 'selected' : '' ?>>0-1 years</option>
                                                    <option value="2-5" <?= ($teacher['experience'] ?? '') === '2-5' ? 'selected' : '' ?>>2-5 years</option>
                                                    <option value="6-10" <?= ($teacher['experience'] ?? '') === '6-10' ? 'selected' : '' ?>>6-10 years</option>
                                                    <option value="11+" <?= ($teacher['experience'] ?? '') === '11+' ? 'selected' : '' ?>>11+ years</option>
                                                </select>
                                            </div>
                                        </div>
                                        <div class="row g-3 mt-1">
                                            <div class="col-md-12">
                                                <label for="course_description" class="form-label">Course Description</label>
                                                <textarea class="form-control" id="course_description" name="course_description" rows="3" 
                                                          placeholder="Brief description of your proposed course. Describe what students will learn and the course objectives."><?= htmlspecialchars($teacher['course_description'] ?? '') ?></textarea>
                                                <small class="text-muted">Minimum 50 characters</small>
                                            </div>
                                        </div>
                                        <div class="row g-3 mt-1">
                                            <div class="col-md-12">
                                                <label for="qualifications" class="form-label">Your Qualifications</label>
                                                <textarea class="form-control" id="qualifications" name="qualifications" rows="3" 
                                                          placeholder="List your relevant qualifications, certifications, and expertise. Include degrees, certifications, professional experience, etc."><?= htmlspecialchars($teacher['qualifications'] ?? '') ?></textarea>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <!-- Social Links -->
                                    <div class="mb-4">
                                        <div class="d-flex align-items-center mb-3">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-share me-2 text-primary" viewBox="0 0 16 16">
                                                <path d="M13.5 1a1.5 1.5 0 1 0 0 3 1.5 1.5 0 0 0 0-3zM11 2.5a2.5 2.5 0 1 1 .603 1.628l-6.718 3.12a2.499 2.499 0 0 1 0 1.504l6.718 3.12a2.5 2.5 0 1 1-.488.876l-6.718-3.12a2.5 2.5 0 1 1 0-3.256l6.718-3.12A2.5 2.5 0 0 1 11 2.5zm-8.5 4a1.5 1.5 0 1 0 0 3 1.5 1.5 0 0 0 0-3zm11 5.5a1.5 1.5 0 1 0 0 3 1.5 1.5 0 0 0 0-3z"/>
                                            </svg>
                                            <h6 class="fw-bold mb-0">Social Links</h6>
                                        </div>
                                        <div class="row g-3">
                                            <div class="col-md-6">
                                                <label for="website" class="form-label">Website</label>
                                                <div class="input-group">
                                                    <span class="input-group-text bg-light">
                                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="currentColor" class="bi bi-globe" viewBox="0 0 16 16">
                                                            <path d="M0 8a8 8 0 1 1 16 0A8 8 0 0 1 0 8zm7.5-6.923c-.67.204-1.335.82-1.887 1.855A7.97 7.97 0 0 0 5.145 4H7.5V1.077zM4.09 4a9.267 9.267 0 0 1 .64-1.539 6.7 6.7 0 0 1 .597-.933A7.025 7.025 0 0 0 2.255 4H4.09zm-.582 3.5c.03-.877.138-1.718.312-2.5H1.674a6.958 6.958 0 0 0-.656 2.5h2.49zM4.847 5a12.5 12.5 0 0 0-.338 2.5H7.5V5H4.847zM8.5 5v2.5h2.99a12.495 12.495 0 0 0-.337-2.5H8.5zM4.51 8.5a12.5 12.5 0 0 0 .337 2.5H7.5V8.5H4.51zm3.99 0V11h2.653c.187-.765.306-1.608.338-2.5H8.5zM5.145 12c.138.386.295.744.468 1.068.552 1.035 1.218 1.65 1.887 1.855V12H5.145zm.182 2.472a6.696 6.696 0 0 1-.597-.933A9.268 9.268 0 0 1 4.09 12H2.255a7.024 7.024 0 0 0 3.072 2.472zM3.82 11a13.652 13.652 0 0 1-.312-2.5h-2.49c.062.89.291 1.733.656 2.5H3.82zm6.853 3.472A7.024 7.024 0 0 0 13.745 12H11.91a9.27 9.27 0 0 1-.64 1.539 6.688 6.688 0 0 1-.597.933zM8.5 12v2.923c.67-.204 1.335-.82 1.887-1.855.173-.324.33-.682.468-1.068H8.5zm3.68-1h2.146c.365-.767.594-1.61.656-2.5h-2.49a13.65 13.65 0 0 1-.312 2.5zm2.802-3.5a6.959 6.959 0 0 0-.656-2.5H12.18c.174.782.282 1.623.312 2.5h2.49zM11.27 2.461c.247.464.462.98.64 1.539h1.835a7.024 7.024 0 0 0-3.072-2.472c.218.284.418.598.597.933zM10.855 4a7.966 7.966 0 0 0-.468-1.068C9.835 1.897 9.17 1.282 8.5 1.077V4h2.355z"/>
                                                        </svg>
                                                    </span>
                                                    <input type="url" class="form-control" id="website" name="website" value="<?= htmlspecialchars($teacher['website'] ?? '') ?>" placeholder="https://yourwebsite.com">
                                                </div>
                                            </div>
                                            <div class="col-md-6">
                                                <label for="social_linkedin" class="form-label">LinkedIn</label>
                                                <div class="input-group">
                                                    <span class="input-group-text bg-light">
                                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="currentColor" class="bi bi-linkedin" viewBox="0 0 16 16">
                                                            <path d="M0 1.146C0 .513.526 0 1.175 0h13.65C15.474 0 16 .513 16 1.146v13.708c0 .633-.526 1.146-1.175 1.146H1.175C.526 16 0 15.487 0 14.854V1.146zm4.943 12.248V6.169H2.542v7.225h2.401zm-1.2-8.212c.837 0 1.358-.554 1.358-1.248-.015-.709-.52-1.248-1.342-1.248-.822 0-1.359.54-1.359 1.248 0 .694.521 1.248 1.327 1.248h.016zm4.908 8.212V9.359c0-.216.016-.432.08-.586.173-.431.568-.878 1.232-.878.869 0 1.216.662 1.216 1.634v3.865h2.401V9.25c0-2.22-1.184-3.252-2.764-3.252-1.274 0-1.845.7-2.165 1.193v.025h-.016a5.54 5.54 0 0 1 .016-.025V6.169h-2.4c.03.678 0 7.225 0 7.225h2.4z"/>
                                                        </svg>
                                                    </span>
                                                    <input type="url" class="form-control" id="social_linkedin" name="social_linkedin" value="<?= htmlspecialchars($teacher['social_linkedin'] ?? '') ?>" placeholder="https://linkedin.com/in/yourprofile">
                                                </div>
                                            </div>
                                            <div class="col-md-6">
                                                <label for="social_twitter" class="form-label">Twitter</label>
                                                <div class="input-group">
                                                    <span class="input-group-text bg-light">
                                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="currentColor" class="bi bi-twitter-x" viewBox="0 0 16 16">
                                                            <path d="M12.6.75h2.454l-5.36 6.142L16 15.25h-4.937l-3.867-5.07-4.425 5.07H.316l5.733-6.57L0 .75h5.063l3.495 4.633L12.601.75Zm-.86 13.028h1.36L4.323 2.145H2.865l8.875 11.633Z"/>
                                                        </svg>
                                                    </span>
                                                    <input type="url" class="form-control" id="social_twitter" name="social_twitter" value="<?= htmlspecialchars($teacher['social_twitter'] ?? '') ?>" placeholder="https://twitter.com/yourusername">
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    

                                    <!-- Identity Verification Section -->
                                    <div class="mb-4">
                                        <div class="d-flex align-items-center mb-3">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-shield-check me-2 text-success" viewBox="0 0 16 16">
                                                <path d="M5.338 1.59a61.44 61.44 0 0 0-2.837.856.481.481 0 0 0-.328.39c-.554 4.157.726 7.19 2.253 9.188a10.725 10.725 0 0 0 2.287 2.233c.346.244.652.42.893.533.12.057.218.095.293.118a.55.55 0 0 0 .101.025.615.615 0 0 0 .1-.025c.076-.023.174-.061.294-.118.24-.113.547-.29.893-.533a10.726 10.726 0 0 0 2.287-2.233c1.527-1.997 2.807-5.031 2.253-9.188a.48.48 0 0 0-.328-.39c-.651-.213-1.75-.56-2.837-.855C9.552 1.29 8.531 1.067 8 1.067c-.53 0-1.552.223-2.662.524zM5.072.56C6.157.265 7.31 0 8 0s1.843.265 2.928.56c1.11.3 2.229.655 2.887.87a1.54 1.54 0 0 1 1.044 1.262c.596 4.477-.787 7.795-2.465 9.99a11.775 11.775 0 0 1-2.517 2.453 7.159 7.159 0 0 1-1.048.625c-.28.132-.581.24-.829.24s-.548-.108-.829-.24a7.158 7.158 0 0 1-1.048-.625 11.777 11.777 0 0 1-2.517-2.453C1.928 10.487.545 7.169 1.141 2.692A1.54 1.54 0 0 1 2.185 1.43 62.456 62.456 0 0 1 5.072.56z"/>
                                                <path d="M10.854 5.146a.5.5 0 0 1 0 .708l-3 3a.5.5 0 0 1-.708 0l-1.5-1.5a.5.5 0 1 1 .708-.708L7.5 7.793l2.646-2.647a.5.5 0 0 1 .708 0z"/>
                                            </svg>
                                            <h5 class="mb-0">Identity Verification</h5>
                                            <?php 
                                            // Get current verification status
                                            $verification_query = "SELECT identity_verification_status, identity_document_path, identity_uploaded_at, identity_verified_at, identity_rejection_reason FROM users WHERE user_id = ?";
                                            $verification_stmt = $conn->prepare($verification_query);
                                            $verification_stmt->bind_param("i", $teacher_id);
                                            $verification_stmt->execute();
                                            $verification_result = $verification_stmt->get_result();
                                            $verification_data = $verification_result->fetch_assoc() ?: [];
                                            
                                            $status = $verification_data['identity_verification_status'] ?? 'pending';
                                            $badge_class = $status === 'verified' ? 'bg-success' : ($status === 'rejected' ? 'bg-danger' : 'bg-warning');
                                            $badge_text = $status === 'verified' ? 'Verified' : ($status === 'rejected' ? 'Rejected' : 'Pending');
                                            ?>
                                            <span class="badge <?= $badge_class ?> ms-2"><?= $badge_text ?></span>
                                        </div>
                                        
                                        <?php if ($status === 'verified'): ?>
                                            <div class="alert alert-success d-flex align-items-center" role="alert">
                                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-check-circle me-2" viewBox="0 0 16 16">
                                                    <path d="M8 15A7 7 0 1 1 8 1a7 7 0 0 1 0 14zm0 1A8 8 0 1 0 8 0a8 8 0 0 0 0 16z"/>
                                                    <path d="M10.97 4.97a.235.235 0 0 0-.02.022L7.477 9.417 5.384 7.323a.75.75 0 0 0-1.06 1.06L6.97 11.03a.75.75 0 0 0 1.079-.02l3.992-4.99a.75.75 0 0 0-1.071-1.05z"/>
                                                </svg>
                                                <div>
                                                    Your identity has been verified on <?= date('M j, Y', strtotime($verification_data['identity_verified_at'])) ?>. 
                                                    You can now create and sell courses on our platform.
                                                </div>
                                            </div>
                                        <?php elseif ($status === 'rejected'): ?>
                                            <div class="alert alert-danger" role="alert">
                                                <div class="d-flex align-items-center mb-2">
                                                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-x-circle me-2" viewBox="0 0 16 16">
                                                        <path d="M8 15A7 7 0 1 1 8 1a7 7 0 0 1 0 14zm0 1A8 8 0 1 0 8 0a8 8 0 0 0 0 16z"/>
                                                        <path d="M4.646 4.646a.5.5 0 0 1 .708 0L8 7.293l2.646-2.647a.5.5 0 0 1 .708.708L8.707 8l2.647 2.646a.5.5 0 0 1-.708.708L8 8.707l-2.646 2.647a.5.5 0 0 1-.708-.708L7.293 8 4.646 5.354a.5.5 0 0 1 0-.708z"/>
                                                    </svg>
                                                    <strong>Identity verification was rejected</strong>
                                                </div>
                                                <p class="mb-2"><strong>Reason:</strong> <?= htmlspecialchars($verification_data['identity_rejection_reason'] ?? 'No reason provided') ?></p>
                                                <p class="mb-0">Please upload a new identity document below.</p>
                                            </div>
                                        <?php elseif ($status === 'pending' && !empty($verification_data['identity_document_path'])): ?>
                                            <div class="alert alert-info d-flex align-items-center" role="alert">
                                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-clock me-2" viewBox="0 0 16 16">
                                                    <path d="M8 3.5a.5.5 0 0 0-1 0V9a.5.5 0 0 0 .252.434l3.5 2a.5.5 0 0 0 .496-.868L8 8.71V3.5z"/>
                                                    <path d="M8 16A8 8 0 1 0 8 0a8 8 0 0 0 0 16zm7-8A7 7 0 1 1 1 8a7 7 0 0 1 14 0z"/>
                                                </svg>
                                                <div>
                                                    Identity document submitted on <?= date('M j, Y', strtotime($verification_data['identity_uploaded_at'])) ?>. 
                                                    Our team is reviewing your document. This usually takes 1-3 business days.
                                                </div>
                                            </div>
                                        <?php else: ?>
                                            <div class="alert alert-warning" role="alert">
                                                <div class="d-flex align-items-center mb-2">
                                                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-exclamation-triangle me-2" viewBox="0 0 16 16">
                                                        <path d="M7.938 2.016A.13.13 0 0 1 8.002 2a.13.13 0 0 1 .063.016.146.146 0 0 1 .054.057l6.857 11.667c.036.06.035.124.002.183a.163.163 0 0 1-.054.06.116.116 0 0 1-.066.017H1.146a.115.115 0 0 1-.066-.017.163.163 0 0 1-.054-.06.176.176 0 0 1 .002-.183L7.884 2.073a.147.147 0 0 1 .054-.057zm1.044-.45a1.13 1.13 0 0 0-1.96 0L.165 13.233c-.457.778.091 1.767.98 1.767h13.713c.889 0 1.438-.99.98-1.767L8.982 1.566z"/>
                                                        <path d="M7.002 12a1 1 0 1 1 2 0 1 1 0 0 1-2 0zM7.1 5.995a.905.905 0 1 1 1.8 0l-.35 3.507a.552.552 0 0 1-1.1 0L7.1 5.995z"/>
                                                    </svg>
                                                    <strong>Identity verification required</strong>
                                                </div>
                                                <p class="mb-0">To maintain trust and security on our platform, please upload a government-issued identity document (ID, passport, or driver's license).</p>
                                            </div>
                                        <?php endif; ?>
                                        
                                        <?php if ($status !== 'verified'): ?>
                                            <div class="row">
                                                <div class="col-md-8">
                                                    <label for="identity_document" class="form-label">
                                                        <strong>Upload Identity Document</strong>
                                                        <small class="text-muted d-block">Accepted formats: PDF, JPG, PNG (max 5MB)</small>
                                                    </label>
                                                    <input type="file" class="form-control" id="identity_document" name="identity_document" accept=".pdf,.jpg,.jpeg,.png" onchange="previewIdentityDocument()">
                                                    <div class="form-text">
                                                        Accepted documents: National ID, Passport, Driver's License. 
                                                        Ensure all text is clearly visible and the document is not expired.
                                                    </div>
                                                </div>
                                                <div class="col-md-4">
                                                    <?php if (!empty($verification_data['identity_document_path'])): ?>
                                                        <label class="form-label">Current Document</label>
                                                        <div class="border rounded p-2 text-center">
                                                            <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" fill="currentColor" class="bi bi-file-earmark-check text-success mb-1" viewBox="0 0 16 16">
                                                                <path d="M10.854 7.854a.5.5 0 0 0-.708-.708L7.5 9.793 6.354 8.646a.5.5 0 1 0-.708.708l1.5 1.5a.5.5 0 0 0 .708 0l3-3z"/>
                                                                <path d="M14 14V4.5L9.5 0H4a2 2 0 0 0-2 2v12a2 2 0 0 0 2 2h8a2 2 0 0 0 2-2zM9.5 3A1.5 1.5 0 0 0 11 4.5h2V14a1 1 0 0 1-1 1H4a1 1 0 0 1-1-1V2a1 1 0 0 1 1-1h5.5v2z"/>
                                                            </svg>
                                                            <div class="small text-muted">Document uploaded</div>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <div class="d-grid gap-2 d-md-flex justify-content-md-end mt-4">
                                        <button type="reset" class="btn btn-light me-md-2 px-4">Reset</button>
                                        <button type="submit" name="update_profile" class="btn btn-panadite px-4">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-check2 me-1" viewBox="0 0 16 16">
                                                <path d="M13.854 3.646a.5.5 0 0 1 0 .708l-7 7a.5.5 0 0 1-.708 0l-3.5-3.5a.5.5 0 1 1 .708-.708L6.5 10.293l6.646-6.647a.5.5 0 0 1 .708 0z"/>
                                            </svg>
                                            Save Changes
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
/* Targeted fixes for profile page only */
body {
    overflow-x: hidden;
}

.page-content {
    padding: 20px 30px;
    max-width: 100%;
    overflow-x: hidden;
}

.container-fluid {
    padding-left: 15px;
    padding-right: 15px;
}

.row {
    margin-left: 0;
    margin-right: 0;
}
</style>

<?php
// Include the centralized teacher layout end (footer, scripts, etc)
require_once __DIR__ . '/components/teacher_layout_end.php';
?>
