<?php
// This script publishes all unpublished courses in the database
require_once "../includes/session_start.php";
require_once "../config/db_connect.php";
require_once "../auth/functions.php";

// Require teacher role authentication
requireRole('teacher');

// Check if user is logged in and is a teacher
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'teacher') {
    header('Location: ../login.php');
    exit;
}

$teacher_id = $_SESSION['user_id'];
$success = false;
$message = '';
$courses = [];

try {
    // Get all courses for this teacher
    $stmt = $conn->prepare("SELECT course_id, title, is_published FROM courses WHERE teacher_id = ?");
    $stmt->bind_param("i", $teacher_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $courses[] = $row;
        }
    }
    
    // If course ID is provided in URL, publish only that course
    if (isset($_GET['course_id']) && is_numeric($_GET['course_id'])) {
        $course_id = (int)$_GET['course_id'];
        $updateStmt = $conn->prepare("UPDATE courses SET is_published = 1 WHERE course_id = ? AND teacher_id = ?");
        $updateStmt->bind_param("ii", $course_id, $teacher_id);
        
        if ($updateStmt->execute() && $updateStmt->affected_rows > 0) {
            $success = true;
            $message = "Course has been successfully published!";
        } else {
            $message = "Failed to publish course or course not found.";
        }
    } 
    // If "publish_all" is set, publish all courses for this teacher
    else if (isset($_GET['publish_all'])) {
        $updateStmt = $conn->prepare("UPDATE courses SET is_published = 1 WHERE teacher_id = ?");
        $updateStmt->bind_param("i", $teacher_id);
        
        if ($updateStmt->execute()) {
            $success = true;
            $message = "All your courses have been published!";
        } else {
            $message = "Failed to publish courses.";
        }
    }
    
} catch (Exception $e) {
    $message = "Error: " . $e->getMessage();
}

// Include header
include_once 'components/header.php';
?>

<div class="container-fluid pt-4">
    <div class="row">
        <div class="col-12">
            <div class="card shadow-sm">
                <div class="card-header">
                    <h6 class="card-title mb-0">Publish Courses</h6>
                </div>
                <div class="card-body">
                    <?php if (!empty($message)): ?>
                        <div class="alert alert-<?= $success ? 'success' : 'danger' ?> alert-dismissible fade show" role="alert">
                            <?= $message ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                        </div>
                    <?php endif; ?>
                    
                    <div class="mb-4">
                        <p>Use this page to publish your courses so they become visible to students.</p>
                        <a href="publish_courses.php?publish_all=1" class="btn btn-primary mb-3">
                            Publish All My Courses
                        </a>
                    </div>
                    
                    <?php if (empty($courses)): ?>
                        <div class="alert alert-info">
                            You don't have any courses yet. <a href="create-course.php" class="alert-link">Create a course</a> to get started.
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Title</th>
                                        <th>Status</th>
                                        <th>Action</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($courses as $course): ?>
                                        <tr>
                                            <td><?= htmlspecialchars($course['title']) ?></td>
                                            <td>
                                                <?php if ($course['is_published']): ?>
                                                    <span class="badge bg-success">Published</span>
                                                <?php else: ?>
                                                    <span class="badge bg-secondary">Draft</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php if (!$course['is_published']): ?>
                                                    <a href="publish_courses.php?course_id=<?= $course['course_id'] ?>" class="btn btn-sm btn-outline-success">
                                                        Publish
                                                    </a>
                                                <?php else: ?>
                                                    <span class="text-muted">Already published</span>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                    
                    <div class="mt-4">
                        <a href="courses.php" class="btn btn-outline-secondary">Back to Courses</a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php
// Include footer
include_once 'components/footer.php';
?>
