<?php
// Include the centralized teacher layout
require_once __DIR__ . '/components/teacher_layout.php';
require_once '../auth/functions.php';
requireRole('teacher');
require_once '../config/db_connect.php';

// Get current teacher ID
$teacher_id = $_SESSION['user_id'];

// Get quiz filter
$quiz_filter = isset($_GET['quiz_id']) ? intval($_GET['quiz_id']) : null;

// Get teacher's quizzes for filter dropdown
$teacher_quizzes_stmt = $pdo->prepare("
    SELECT q.quiz_id, q.title, c.title as course_title 
    FROM quizzes q 
    JOIN courses c ON q.course_id = c.course_id 
    WHERE q.teacher_id = ? 
    ORDER BY c.title, q.title
");
$teacher_quizzes_stmt->execute([$teacher_id]);
$teacher_quizzes = $teacher_quizzes_stmt->fetchAll(PDO::FETCH_ASSOC);

// Build query for quiz submissions
$submissions_query = "
    SELECT 
        qa.attempt_id,
        qa.quiz_id,
        qa.student_id,
        qa.start_time,
        qa.end_time,
        qa.score,
        qa.pass_status,
        q.title as quiz_title,
        q.passing_score,
        c.title as course_title,
        u.first_name,
        u.last_name,
        u.email,
        COUNT(DISTINCT qr.response_id) as answered_questions,
        COUNT(DISTINCT qq.question_id) as total_questions
    FROM quiz_attempts qa
    JOIN quizzes q ON qa.quiz_id = q.quiz_id
    JOIN courses c ON q.course_id = c.course_id
    JOIN users u ON qa.student_id = u.user_id
    LEFT JOIN quiz_responses qr ON qa.attempt_id = qr.attempt_id
    LEFT JOIN quiz_questions qq ON q.quiz_id = qq.quiz_id
    WHERE q.teacher_id = ?
";

$params = [$teacher_id];
if ($quiz_filter) {
    $submissions_query .= " AND qa.quiz_id = ?";
    $params[] = $quiz_filter;
}

$submissions_query .= " 
    GROUP BY qa.attempt_id
    ORDER BY qa.start_time DESC
";

$submissions_stmt = $pdo->prepare($submissions_query);
$submissions_stmt->execute($params);
$submissions = $submissions_stmt->fetchAll(PDO::FETCH_ASSOC);

// Calculate statistics
$total_submissions = count($submissions);
$passed_submissions = count(array_filter($submissions, function($s) { return $s['pass_status']; }));
$failed_submissions = $total_submissions - $passed_submissions;
$avg_score = $total_submissions > 0 ? array_sum(array_column($submissions, 'score')) / $total_submissions : 0;

$page_title = "Quiz Submissions";
?>

<style>
    .submissions-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        padding: 2rem 0;
        margin-bottom: 2rem;
        border-radius: 15px;
    }

    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 1.5rem;
        margin-bottom: 2rem;
    }

    .stat-card {
        background: white;
        padding: 1.5rem;
        border-radius: 12px;
        box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        text-align: center;
        border-left: 4px solid var(--primary-color);
    }

    .stat-number {
        font-size: 2rem;
        font-weight: bold;
        color: var(--primary-color);
        margin-bottom: 0.5rem;
    }

    .stat-label {
        color: #6b7280;
        font-weight: 500;
    }

    .submissions-table {
        background: white;
        border-radius: 12px;
        overflow: hidden;
        box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    }

    .submissions-table th {
        background: #f8fafc;
        color: #374151;
        font-weight: 600;
        padding: 1rem;
        border: none;
    }

    .submissions-table td {
        padding: 1rem;
        border-bottom: 1px solid #e5e7eb;
        vertical-align: middle;
    }

    .submission-row:hover {
        background-color: #f9fafb;
    }

    .score-badge {
        display: inline-block;
        padding: 0.25rem 0.75rem;
        border-radius: 50px;
        font-weight: 600;
        font-size: 0.875rem;
    }

    .score-excellent { background: #d1fae5; color: #065f46; }
    .score-good { background: #dbeafe; color: #1e40af; }
    .score-average { background: #fef3c7; color: #92400e; }
    .score-poor { background: #fee2e2; color: #991b1b; }

    .status-badge {
        display: inline-block;
        padding: 0.25rem 0.75rem;
        border-radius: 50px;
        font-weight: 600;
        font-size: 0.875rem;
    }

    .status-passed { background: #d1fae5; color: #065f46; }
    .status-failed { background: #fee2e2; color: #991b1b; }

    .action-btn {
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        padding: 0.5rem 1rem;
        border-radius: 6px;
        text-decoration: none;
        font-weight: 500;
        font-size: 0.875rem;
        transition: all 0.2s ease;
    }

    .btn-primary {
        background: #3b82f6;
        color: white;
    }

    .btn-primary:hover {
        background: #2563eb;
        color: white;
    }

    .filter-section {
        background: white;
        padding: 1.5rem;
        border-radius: 12px;
        margin-bottom: 2rem;
        box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    }

    .empty-state {
        text-align: center;
        padding: 3rem;
        background: white;
        border-radius: 12px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    }

    .empty-icon {
        font-size: 4rem;
        color: #9ca3af;
        margin-bottom: 1rem;
    }
</style>

<body>
    <?php include_once 'components/sidebar.php'; ?>
    
    <div class="submissions-header">
        <div class="container">
            <h1 class="display-6 fw-bold mb-0">
                <i class="bx bx-clipboard-check"></i>
                Quiz Submissions
            </h1>
            <p class="mb-0 opacity-75">Monitor and review student quiz attempts</p>
        </div>
    </div>

    <div class="container">
        <!-- Statistics -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-number"><?= $total_submissions ?></div>
                <div class="stat-label">Total Submissions</div>
            </div>
            <div class="stat-card">
                <div class="stat-number" style="color: #10b981;"><?= $passed_submissions ?></div>
                <div class="stat-label">Passed</div>
            </div>
            <div class="stat-card">
                <div class="stat-number" style="color: #ef4444;"><?= $failed_submissions ?></div>
                <div class="stat-label">Failed</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?= number_format($avg_score, 1) ?>%</div>
                <div class="stat-label">Average Score</div>
            </div>
        </div>

        <!-- Filter Section -->
        <div class="filter-section">
            <div class="row align-items-center">
                <div class="col-md-6">
                    <h5 class="mb-0">Filter Submissions</h5>
                </div>
                <div class="col-md-6">
                    <select class="form-select" id="quizFilter" onchange="filterByQuiz()">
                        <option value="">All Quizzes</option>
                        <?php foreach ($teacher_quizzes as $quiz): ?>
                            <option value="<?= $quiz['quiz_id'] ?>" <?= $quiz_filter == $quiz['quiz_id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($quiz['course_title'] . ' - ' . $quiz['title']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>
        </div>

        <!-- Submissions Table -->
        <?php if (empty($submissions)): ?>
            <div class="empty-state">
                <div class="empty-icon">
                    <i class="bx bx-clipboard"></i>
                </div>
                <h4>No Submissions Found</h4>
                <p class="text-muted">Students haven't submitted any quizzes yet, or no quizzes match your filter.</p>
                <a href="quizzes.php" class="action-btn btn-primary">
                    <i class="bx bx-arrow-back"></i>
                    Back to Quizzes
                </a>
            </div>
        <?php else: ?>
            <div class="submissions-table">
                <table class="table table-hover mb-0">
                    <thead>
                        <tr>
                            <th>Student</th>
                            <th>Quiz</th>
                            <th>Course</th>
                            <th>Score</th>
                            <th>Status</th>
                            <th>Questions</th>
                            <th>Submitted</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($submissions as $submission): ?>
                            <?php
                            $score_class = 'score-poor';
                            if ($submission['score'] >= 90) $score_class = 'score-excellent';
                            elseif ($submission['score'] >= 80) $score_class = 'score-good';
                            elseif ($submission['score'] >= 70) $score_class = 'score-average';
                            ?>
                            <tr class="submission-row">
                                <td>
                                    <div>
                                        <div class="fw-semibold"><?= htmlspecialchars($submission['first_name'] . ' ' . $submission['last_name']) ?></div>
                                        <div class="text-muted small"><?= htmlspecialchars($submission['email']) ?></div>
                                    </div>
                                </td>
                                <td>
                                    <div class="fw-medium"><?= htmlspecialchars($submission['quiz_title']) ?></div>
                                </td>
                                <td>
                                    <span class="badge bg-light text-dark"><?= htmlspecialchars($submission['course_title']) ?></span>
                                </td>
                                <td>
                                    <span class="score-badge <?= $score_class ?>"><?= number_format($submission['score'], 1) ?>%</span>
                                </td>
                                <td>
                                    <?php if ($submission['pass_status']): ?>
                                        <span class="status-badge status-passed">
                                            <i class="bx bx-check"></i> Passed
                                        </span>
                                    <?php else: ?>
                                        <span class="status-badge status-failed">
                                            <i class="bx bx-x"></i> Failed
                                        </span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span class="text-muted"><?= $submission['answered_questions'] ?>/<?= $submission['total_questions'] ?></span>
                                </td>
                                <td>
                                    <div class="text-muted small">
                                        <?= date('M d, Y', strtotime($submission['start_time'])) ?><br>
                                        <?= date('g:i A', strtotime($submission['start_time'])) ?>
                                    </div>
                                </td>

                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function filterByQuiz() {
            const quizId = document.getElementById('quizFilter').value;
            const currentUrl = new URL(window.location.href);
            
            if (quizId) {
                currentUrl.searchParams.set('quiz_id', quizId);
            } else {
                currentUrl.searchParams.delete('quiz_id');
            }
            
            window.location.href = currentUrl.toString();
        }

        // Animate elements on load
        document.addEventListener('DOMContentLoaded', function() {
            const elements = document.querySelectorAll('.stat-card, .submission-row');
            elements.forEach((element, index) => {
                element.style.opacity = '0';
                element.style.transform = 'translateY(20px)';
                
                setTimeout(() => {
                    element.style.transition = 'all 0.6s ease';
                    element.style.opacity = '1';
                    element.style.transform = 'translateY(0)';
                }, index * 50);
            });
        });
    </script>
</body>
</html>
