<?php
require_once '../config/db_connect.php';

echo "<h1>🔧 YouTube Thumbnail Repair Tool</h1>";

// Find YouTube videos without thumbnails
$query = "SELECT video_id, title, video_id_external, platform 
          FROM videos 
          WHERE platform = 'youtube' 
          AND video_id_external IS NOT NULL 
          AND (thumbnail IS NULL OR thumbnail = '')
          ORDER BY created_at DESC";

$result = $conn->query($query);

if (!$result) {
    echo "<p style='color: red;'>Error: " . $conn->error . "</p>";
    exit;
}

$videosToFix = [];
while ($row = $result->fetch_assoc()) {
    $videosToFix[] = $row;
}

echo "<h2>📊 YouTube Videos Missing Thumbnails</h2>";

if (empty($videosToFix)) {
    echo "<p style='color: green;'>✅ All YouTube videos already have thumbnails!</p>";
} else {
    echo "<p>Found <strong>" . count($videosToFix) . "</strong> YouTube videos without thumbnails:</p>";
    
    echo "<table border='1' cellpadding='8' style='border-collapse: collapse; width: 100%;'>";
    echo "<tr style='background: #f0f0f0;'>
            <th>Video ID</th>
            <th>Title</th>
            <th>YouTube ID</th>
            <th>Generated Thumbnail URL</th>
            <th>Preview</th>
          </tr>";
    
    foreach ($videosToFix as $video) {
        $thumbnail_url = "https://img.youtube.com/vi/{$video['video_id_external']}/maxresdefault.jpg";
        
        echo "<tr>";
        echo "<td>" . $video['video_id'] . "</td>";
        echo "<td>" . htmlspecialchars($video['title']) . "</td>";
        echo "<td><code>" . $video['video_id_external'] . "</code></td>";
        echo "<td><small>" . $thumbnail_url . "</small></td>";
        echo "<td><img src='" . $thumbnail_url . "' style='max-width: 120px; height: auto;' alt='Thumbnail'></td>";
        echo "</tr>";
    }
    echo "</table>";
    
    echo "<h2>🛠️ Repair Action</h2>";
    
    if (isset($_POST['repair_thumbnails'])) {
        echo "<div style='background: #f9f9f9; padding: 15px; border-radius: 5px; margin: 20px 0;'>";
        echo "<h3>Processing repairs...</h3>";
        
        $successCount = 0;
        $errorCount = 0;
        
        foreach ($videosToFix as $video) {
            $video_id = $video['video_id'];
            $youtube_id = $video['video_id_external'];
            $thumbnail_url = "https://img.youtube.com/vi/{$youtube_id}/maxresdefault.jpg";
            
            $stmt = $conn->prepare("UPDATE videos SET thumbnail = ? WHERE video_id = ?");
            $stmt->bind_param("si", $thumbnail_url, $video_id);
            
            if ($stmt->execute()) {
                echo "<p style='color: green;'>✅ Updated video ID {$video_id}: " . htmlspecialchars($video['title']) . "</p>";
                $successCount++;
            } else {
                echo "<p style='color: red;'>❌ Failed to update video ID {$video_id}: " . $stmt->error . "</p>";
                $errorCount++;
            }
            
            $stmt->close();
        }
        
        echo "<hr>";
        echo "<h4>Summary:</h4>";
        echo "<p><strong>Successfully updated:</strong> <span style='color: green;'>{$successCount} videos</span></p>";
        if ($errorCount > 0) {
            echo "<p><strong>Errors:</strong> <span style='color: red;'>{$errorCount} videos</span></p>";
        }
        
        if ($successCount > 0) {
            echo "<p style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
            echo "<strong>🎉 Repair completed!</strong> The YouTube thumbnails should now display properly on your videos page.";
            echo "</p>";
            
            echo "<p><a href='videos.php' style='background: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;'>View Videos Page</a></p>";
        }
        
        echo "</div>";
        
    } else {
        echo "<p>Click the button below to automatically add YouTube thumbnail URLs to all videos missing thumbnails:</p>";
        
        echo "<form method='post' style='margin: 20px 0;'>";
        echo "<input type='submit' name='repair_thumbnails' value='🔧 Repair All YouTube Thumbnails' style='background: #28a745; color: white; padding: 15px 30px; border: none; border-radius: 5px; font-size: 16px; cursor: pointer;'>";
        echo "</form>";
        
        echo "<div style='background: #fff3cd; padding: 15px; border-radius: 5px; color: #856404; margin: 20px 0;'>";
        echo "<strong>⚠️ Note:</strong> This will update " . count($videosToFix) . " videos in your database. The thumbnails will be pulled from YouTube's servers automatically.";
        echo "</div>";
    }
}

// Show verification of all videos
echo "<h2>🔍 Current Status Check</h2>";
$checkQuery = "SELECT COUNT(*) as total, 
               SUM(CASE WHEN platform = 'youtube' THEN 1 ELSE 0 END) as youtube_total,
               SUM(CASE WHEN platform = 'youtube' AND (thumbnail IS NOT NULL AND thumbnail != '') THEN 1 ELSE 0 END) as youtube_with_thumbnails
               FROM videos";

$checkResult = $conn->query($checkQuery);
if ($checkResult) {
    $stats = $checkResult->fetch_assoc();
    echo "<ul>";
    echo "<li><strong>Total videos:</strong> " . $stats['total'] . "</li>";
    echo "<li><strong>YouTube videos:</strong> " . $stats['youtube_total'] . "</li>";
    echo "<li><strong>YouTube videos with thumbnails:</strong> " . $stats['youtube_with_thumbnails'] . " / " . $stats['youtube_total'] . "</li>";
    
    $percentage = $stats['youtube_total'] > 0 ? round(($stats['youtube_with_thumbnails'] / $stats['youtube_total']) * 100, 1) : 0;
    $color = $percentage == 100 ? 'green' : ($percentage > 50 ? 'orange' : 'red');
    echo "<li><strong>YouTube thumbnail coverage:</strong> <span style='color: {$color}; font-weight: bold;'>{$percentage}%</span></li>";
    echo "</ul>";
}

echo "<hr>";
echo "<p><strong>Last updated:</strong> " . date('Y-m-d H:i:s') . "</p>";
echo "<p><a href='debug_video_table.php'>← Back to Debug Report</a></p>";
?>
