<?php
// Set current page for active menu highlighting
$current_page = 'settings.php';

// Include the centralized teacher layout
require_once "components/teacher_layout.php";

// Enable error reporting during development
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Ensure mysqli connection is available for legacy code
if (!isset($conn) && isset($pdo)) {
    $conn = new mysqli($host, $username, $password, $dbname);
    if ($conn->connect_error) {
        die("Connection failed: " . $conn->connect_error);
    }
}

/**
 * Get teacher data with robust fallback approach
 */
function get_teacher($conn, $teacher_id) {
    // Default teacher data in case of failure
    $default_teacher = [
        'user_id' => $teacher_id,
        'first_name' => 'Academy',
        'last_name' => 'Teacher',
        'email' => 'teacher@panadite.com',
        'bio' => 'Teacher profile is currently unavailable.',
        'profile_picture' => '../assets/img/default-teacher.png',
    ];
    
    // Check if we have a valid connection
    if (!$conn || $conn->connect_error) {
        error_log("Database connection failed in get_teacher");
        return $default_teacher;
    }
    
    try {
        // Check if the user table exists
        $table_check = $conn->query("SHOW TABLES LIKE 'users'");
        if (!$table_check || $table_check->num_rows == 0) {
            error_log("Users table does not exist");
            return $default_teacher;
        }
        
        // Prepare statement with left join to teacher_profiles
        $stmt = $conn->prepare("SELECT u.user_id, u.first_name, u.last_name, u.email, 
                              tp.bio, COALESCE(tp.profile_image, '../assets/img/default-teacher.png') as profile_picture
                              FROM users u 
                              LEFT JOIN teacher_profiles tp ON u.user_id = tp.teacher_id 
                              WHERE u.user_id = ? AND u.user_role = 'teacher'");
        
        if (!$stmt) {
            error_log("Failed to prepare statement in get_teacher: " . $conn->error);
            return $default_teacher;
        }
        
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result && $result->num_rows > 0) {
            return $result->fetch_assoc();
        } else {
            error_log("No teacher found with ID $teacher_id");
            return $default_teacher;
        }
    } catch (Exception $e) {
        error_log("Exception in get_teacher: " . $e->getMessage());
        return $default_teacher;
    }
}

$teacher_id = $_SESSION['user_id'];
$teacher = get_teacher($conn, $teacher_id);

// Get user settings
$settings_query = "SELECT * FROM teacher_settings WHERE user_id = ?";
$stmt = $conn->prepare($settings_query);
$stmt->bind_param("i", $teacher_id);
$stmt->execute();
$settings_result = $stmt->get_result();
$settings = $settings_result->fetch_assoc() ?: [];

// Initialize default settings if not set
if (empty($settings)) {
    $settings = [
        'email_notifications' => 1,
        'sms_notifications' => 0,
        'dashboard_theme' => 'light',
        'timezone' => 'UTC',
        'language' => 'en',
        'enable_2fa' => 0
    ];
}

// Process settings update if form is submitted
$message = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_settings'])) {
    $email_notifications = isset($_POST['email_notifications']) ? 1 : 0;
    $sms_notifications = isset($_POST['sms_notifications']) ? 1 : 0;
    $dashboard_theme = $_POST['dashboard_theme'];
    $timezone = $_POST['timezone'];
    $language = $_POST['language'];
    $enable_2fa = isset($_POST['enable_2fa']) ? 1 : 0;
    
    // Check if settings exist for this user
    if (empty($settings['user_id'])) {
        // Create settings record
        $create_settings = "INSERT INTO teacher_settings (user_id, email_notifications, sms_notifications, dashboard_theme, timezone, language, enable_2fa) 
                            VALUES (?, ?, ?, ?, ?, ?, ?)";
        $stmt = $conn->prepare($create_settings);
        $stmt->bind_param("iiisssi", $teacher_id, $email_notifications, $sms_notifications, $dashboard_theme, $timezone, $language, $enable_2fa);
        $settings_updated = $stmt->execute();
    } else {
        // Update existing settings
        $update_settings = "UPDATE teacher_settings SET 
                            email_notifications = ?, 
                            sms_notifications = ?, 
                            dashboard_theme = ?, 
                            timezone = ?, 
                            language = ?, 
                            enable_2fa = ? 
                            WHERE user_id = ?";
        $stmt = $conn->prepare($update_settings);
        $stmt->bind_param("iisssii", $email_notifications, $sms_notifications, $dashboard_theme, $timezone, $language, $enable_2fa, $teacher_id);
        $settings_updated = $stmt->execute();
    }
    
    // Handle password change if requested
    if (!empty($_POST['current_password']) && !empty($_POST['new_password']) && !empty($_POST['confirm_password'])) {
        if ($_POST['new_password'] === $_POST['confirm_password']) {
            // Verify current password
            $get_password = "SELECT password FROM users WHERE id = ?";
            $stmt = $conn->prepare($get_password);
            $stmt->bind_param("i", $teacher_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $user = $result->fetch_assoc();
            
            if (password_verify($_POST['current_password'], $user['password'])) {
                // Hash new password and update
                $new_password_hash = password_hash($_POST['new_password'], PASSWORD_DEFAULT);
                
                $update_password = "UPDATE users SET password = ? WHERE id = ?";
                $stmt = $conn->prepare($update_password);
                $stmt->bind_param("si", $new_password_hash, $teacher_id);
                $password_updated = $stmt->execute();
                
                if ($password_updated) {
                    $message .= '<div class="alert alert-success">Password updated successfully.</div>';
                } else {
                    $message .= '<div class="alert alert-danger">Failed to update password.</div>';
                }
            } else {
                $message .= '<div class="alert alert-danger">Current password is incorrect.</div>';
            }
        } else {
            $message .= '<div class="alert alert-danger">New passwords do not match.</div>';
        }
    }
    
    // Refresh settings data
    $stmt = $conn->prepare($settings_query);
    $stmt->bind_param("i", $teacher_id);
    $stmt->execute();
    $settings_result = $stmt->get_result();
    $settings = $settings_result->fetch_assoc() ?: $settings; // Use posted values if no record found
    
    if (isset($settings_updated) && $settings_updated) {
        $message .= '<div class="alert alert-success">Settings updated successfully!</div>';
    } elseif (!isset($password_updated)) {
        $message .= '<div class="alert alert-danger">Failed to update settings.</div>';
    }
}
?>

<!-- Page header with title and breadcrumb -->
<div class="page-header">
    <div>
        <h1 class="page-title">Settings</h1>
        <nav aria-label="breadcrumb">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="dashboard.php">Home</a></li>
                <li class="breadcrumb-item active" aria-current="page">Settings</li>
            </ol>
        </nav>
    </div>
</div>

<?php if (!empty($message)): ?>
<div class="alert-container">
    <?php echo $message; ?>
</div>
<?php endif; ?>

<div class="container-fluid">
    <!-- Settings Content -->
    <form method="POST" action="">
        <div class="row">
            <!-- Notification Settings -->
            <div class="col-lg-6">
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Notification Settings</h5>
                    </div>
                    <div class="card-body">
                        <div class="form-group mb-3">
                            <div class="form-switch">
                                <input type="checkbox" class="form-check-input" id="emailNotifications" name="email_notifications" <?php echo (!empty($settings['email_notifications'])) ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="emailNotifications">Email Notifications</label>
                            </div>
                            <small class="form-text text-muted">Receive notifications about assignments, quizzes, and student activity via email.</small>
                        </div>
                        
                        <div class="form-group mb-4">
                            <div class="form-switch">
                                <input type="checkbox" class="form-check-input" id="smsNotifications" name="sms_notifications" <?php echo (!empty($settings['sms_notifications'])) ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="smsNotifications">SMS Notifications</label>
                            </div>
                            <small class="form-text text-muted">Receive important alerts via SMS (requires phone number in profile).</small>
                        </div>
                    </div>
                </div>
                
                <!-- Appearance Settings -->
                <div class="card mt-4">
                    <div class="card-header">
                        <h5 class="mb-0">Appearance & Preferences</h5>
                    </div>
                    <div class="card-body">
                        <div class="form-group mb-3">
                            <label for="dashboardTheme" class="form-label">Dashboard Theme</label>
                            <select class="form-select" id="dashboardTheme" name="dashboard_theme">
                                <option value="light" <?php echo ($settings['dashboard_theme'] == 'light') ? 'selected' : ''; ?>>Light</option>
                                <option value="dark" <?php echo ($settings['dashboard_theme'] == 'dark') ? 'selected' : ''; ?>>Dark</option>
                                <option value="system" <?php echo ($settings['dashboard_theme'] == 'system') ? 'selected' : ''; ?>>System Default</option>
                            </select>
                            <small class="form-text text-muted">Select your preferred dashboard theme.</small>
                        </div>
                        
                        <div class="form-group mb-3">
                            <label for="timezone" class="form-label">Timezone</label>
                            <select class="form-select" id="timezone" name="timezone">
                                <option value="UTC" <?php echo ($settings['timezone'] == 'UTC') ? 'selected' : ''; ?>>UTC</option>
                                <option value="America/New_York" <?php echo ($settings['timezone'] == 'America/New_York') ? 'selected' : ''; ?>>Eastern Time (ET)</option>
                                <option value="America/Chicago" <?php echo ($settings['timezone'] == 'America/Chicago') ? 'selected' : ''; ?>>Central Time (CT)</option>
                                <option value="America/Denver" <?php echo ($settings['timezone'] == 'America/Denver') ? 'selected' : ''; ?>>Mountain Time (MT)</option>
                                <option value="America/Los_Angeles" <?php echo ($settings['timezone'] == 'America/Los_Angeles') ? 'selected' : ''; ?>>Pacific Time (PT)</option>
                                <option value="Europe/London" <?php echo ($settings['timezone'] == 'Europe/London') ? 'selected' : ''; ?>>London (GMT)</option>
                                <option value="Europe/Paris" <?php echo ($settings['timezone'] == 'Europe/Paris') ? 'selected' : ''; ?>>Paris (CET)</option>
                                <option value="Asia/Tokyo" <?php echo ($settings['timezone'] == 'Asia/Tokyo') ? 'selected' : ''; ?>>Tokyo (JST)</option>
                            </select>
                            <small class="form-text text-muted">Select your timezone for accurate deadlines and schedules.</small>
                        </div>
                        
                        <div class="form-group mb-4">
                            <label for="language" class="form-label">Language</label>
                            <select class="form-select" id="language" name="language">
                                <option value="en" <?php echo ($settings['language'] == 'en') ? 'selected' : ''; ?>>English</option>
                                <option value="es" <?php echo ($settings['language'] == 'es') ? 'selected' : ''; ?>>Spanish</option>
                                <option value="fr" <?php echo ($settings['language'] == 'fr') ? 'selected' : ''; ?>>French</option>
                                <option value="de" <?php echo ($settings['language'] == 'de') ? 'selected' : ''; ?>>German</option>
                                <option value="zh" <?php echo ($settings['language'] == 'zh') ? 'selected' : ''; ?>>Chinese</option>
                            </select>
                            <small class="form-text text-muted">Choose your preferred language for the dashboard.</small>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Security Settings -->
            <div class="col-lg-6">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Security Settings</h5>
                    </div>
                    <div class="card-body">
                        <div class="form-group mb-4">
                            <div class="form-switch">
                                <input type="checkbox" class="form-check-input" id="enable2fa" name="enable_2fa" <?php echo (!empty($settings['enable_2fa'])) ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="enable2fa">Two-Factor Authentication</label>
                            </div>
                            <small class="form-text text-muted">Add an extra layer of security to your account by requiring a verification code.</small>
                        </div>
                        
                        <div class="settings-divider"></div>
                        
                        <div class="settings-section-title">Change Password</div>
                        
                        <div class="form-group mb-3">
                            <label for="currentPassword" class="form-label">Current Password</label>
                            <input type="password" class="form-control" id="currentPassword" name="current_password" placeholder="Enter current password">
                        </div>
                        
                        <div class="form-group mb-3">
                            <label for="newPassword" class="form-label">New Password</label>
                            <input type="password" class="form-control" id="newPassword" name="new_password" placeholder="Enter new password" onkeyup="checkPasswordStrength()">
                            <div class="password-strength-meter mt-2">
                                <div class="password-strength" id="password-strength"></div>
                            </div>
                            <small class="form-text text-muted" id="passwordHelpText">Use at least 8 characters with a mix of letters, numbers & symbols.</small>
                        </div>
                        
                        <div class="form-group mb-4">
                            <label for="confirmPassword" class="form-label">Confirm New Password</label>
                            <input type="password" class="form-control" id="confirmPassword" name="confirm_password" placeholder="Confirm new password">
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="mt-4">
            <button type="submit" name="update_settings" class="btn btn-primary">Save Changes</button>
            <a href="dashboard.php" class="btn btn-outline-secondary ms-2">Cancel</a>
        </div>
    </form>
</div>

<!-- Password strength checker script -->
<script>
    function checkPasswordStrength() {
        const password = document.getElementById('newPassword').value;
        const strengthBar = document.getElementById('password-strength');
        const helpText = document.getElementById('passwordHelpText');
        
        // Reset
        strengthBar.className = 'password-strength';
        strengthBar.style.width = '0%';
        
        // Check length
        if (password.length === 0) {
            helpText.textContent = 'Use at least 8 characters with a mix of letters, numbers & symbols.';
            return;
        }
        
        // Calculate strength
        let strength = 0;
        
        // Length
        if (password.length >= 8) strength += 25;
        
        // Lowercase letters
        if (password.match(/[a-z]+/)) strength += 15;
        
        // Uppercase letters
        if (password.match(/[A-Z]+/)) strength += 15;
        
        // Numbers
        if (password.match(/[0-9]+/)) strength += 15;
        
        // Special characters
        if (password.match(/[\W_]+/)) strength += 30;
        
        // Set the strength bar width
        strengthBar.style.width = strength + '%';
        
        // Set color and message
        if (strength < 30) {
            strengthBar.style.backgroundColor = '#e74a3b';
            helpText.textContent = 'Very Weak Password';
        } else if (strength < 50) {
            strengthBar.style.backgroundColor = '#f6c23e';
            helpText.textContent = 'Weak Password - Add numbers and special characters';
        } else if (strength < 70) {
            strengthBar.style.backgroundColor = '#36b9cc';
            helpText.textContent = 'Moderate Password - Add uppercase letters';
        } else if (strength < 90) {
            strengthBar.style.backgroundColor = '#1cc88a';
            helpText.textContent = 'Strong Password';
        } else {
            strengthBar.style.backgroundColor = '#1cc88a';
            helpText.textContent = 'Very Strong Password';
        }
    }
</script>

<style>
/* Targeted fixes for settings page only */
.page-content {
    padding: 20px 30px;
}

.container-fluid {
    padding-left: 15px;
    padding-right: 15px;
}

.row {
    margin-left: 0;
    margin-right: 0;
}
</style>

<?php require_once "components/teacher_layout_end.php"; ?>