<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Simple debug logging function
function logMessage($message) {
    $log_file = "upload_test.log";
    $timestamp = date('[Y-m-d H:i:s]');
    file_put_contents($log_file, "$timestamp $message\n", FILE_APPEND);
}

logMessage("Starting upload test script");

// Handle both regular upload and base64 upload
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    logMessage("Received POST request");
    
    // Check for base64 data
    if (isset($_POST['image_data']) && !empty($_POST['image_data'])) {
        logMessage("Found base64 image data");
        
        // Extract the base64 data
        $image_data = $_POST['image_data'];
        $image_parts = explode(",", $image_data);
        
        if (count($image_parts) > 1) {
            // Remove the data URI scheme part
            $image_base64 = $image_parts[1];
            $image_decoded = base64_decode($image_base64);
            
            // Create directory if it doesn't exist
            $upload_dir = "../uploads/test/";
            if (!file_exists($upload_dir)) {
                mkdir($upload_dir, 0777, true);
            }
            
            // Generate unique filename
            $filename = 'test_' . time() . '.jpg';
            $filepath = $upload_dir . $filename;
            
            // Save file
            if (file_put_contents($filepath, $image_decoded)) {
                logMessage("Successfully saved base64 image to: $filepath");
                $response = [
                    'success' => true,
                    'message' => 'Image uploaded successfully',
                    'path' => $filepath
                ];
            } else {
                logMessage("Failed to save base64 image");
                $response = [
                    'success' => false,
                    'message' => 'Failed to save image'
                ];
            }
        } else {
            logMessage("Invalid base64 data format");
            $response = [
                'success' => false,
                'message' => 'Invalid image data'
            ];
        }
    } 
    // Check for regular file upload
    elseif (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
        logMessage("Found regular file upload");
        
        // Get file info
        $tmp_name = $_FILES['image']['tmp_name'];
        $name = $_FILES['image']['name'];
        
        // Create directory if it doesn't exist
        $upload_dir = "../uploads/test/";
        if (!file_exists($upload_dir)) {
            mkdir($upload_dir, 0777, true);
        }
        
        // Generate unique filename
        $filename = 'test_' . time() . '_' . $name;
        $filepath = $upload_dir . $filename;
        
        // Move uploaded file
        if (move_uploaded_file($tmp_name, $filepath)) {
            logMessage("Successfully moved uploaded file to: $filepath");
            $response = [
                'success' => true,
                'message' => 'Image uploaded successfully',
                'path' => $filepath
            ];
        } else {
            logMessage("Failed to move uploaded file");
            $response = [
                'success' => false,
                'message' => 'Failed to move uploaded file'
            ];
        }
    } 
    // Handle upload errors
    elseif (isset($_FILES['image'])) {
        $error_code = $_FILES['image']['error'];
        $error_messages = [
            UPLOAD_ERR_INI_SIZE => 'The uploaded file exceeds the upload_max_filesize directive in php.ini',
            UPLOAD_ERR_FORM_SIZE => 'The uploaded file exceeds the MAX_FILE_SIZE directive in the HTML form',
            UPLOAD_ERR_PARTIAL => 'The uploaded file was only partially uploaded',
            UPLOAD_ERR_NO_FILE => 'No file was uploaded',
            UPLOAD_ERR_NO_TMP_DIR => 'Missing a temporary folder',
            UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk',
            UPLOAD_ERR_EXTENSION => 'A PHP extension stopped the file upload'
        ];
        
        $error_message = isset($error_messages[$error_code]) ? $error_messages[$error_code] : 'Unknown upload error';
        logMessage("File upload error ($error_code): $error_message");
        
        $response = [
            'success' => false,
            'message' => "Upload error: $error_message",
            'error_code' => $error_code
        ];
    } 
    // No file or data provided
    else {
        logMessage("No image data or file provided");
        $response = [
            'success' => false,
            'message' => 'No image data or file provided'
        ];
    }
    
    // Return JSON response
    header('Content-Type: application/json');
    echo json_encode($response);
    exit;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Image Upload Test</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body>
    <div class="container mt-5">
        <h1>Image Upload Test</h1>
        <p>This is a simple test to diagnose file upload issues with Panadite Academy.</p>
        
        <div class="row mt-4">
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header">Regular File Upload</div>
                    <div class="card-body">
                        <form id="regularUploadForm" enctype="multipart/form-data">
                            <div class="mb-3">
                                <label for="regularImage" class="form-label">Select Image</label>
                                <input type="file" class="form-control" id="regularImage" name="image" accept="image/*">
                            </div>
                            <button type="button" id="regularUploadBtn" class="btn btn-primary">Upload</button>
                        </form>
                        <div id="regularUploadResult" class="mt-3"></div>
                    </div>
                </div>
            </div>
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header">Base64 Upload</div>
                    <div class="card-body">
                        <form id="base64UploadForm">
                            <div class="mb-3">
                                <label for="base64Image" class="form-label">Select Image</label>
                                <input type="file" class="form-control" id="base64Image" accept="image/*">
                            </div>
                            <button type="button" id="base64UploadBtn" class="btn btn-primary">Upload</button>
                        </form>
                        <div id="base64UploadResult" class="mt-3"></div>
                        <div id="imagePreview" class="mt-3"></div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script>
        $(document).ready(function() {
            // Regular file upload
            $('#regularUploadBtn').on('click', function() {
                const fileInput = $('#regularImage')[0];
                if (!fileInput.files.length) {
                    $('#regularUploadResult').html('<div class="alert alert-warning">Please select a file</div>');
                    return;
                }
                
                const formData = new FormData();
                formData.append('image', fileInput.files[0]);
                
                $.ajax({
                    url: 'upload-test.php',
                    type: 'POST',
                    data: formData,
                    processData: false,
                    contentType: false,
                    dataType: 'json',
                    success: function(response) {
                        if (response.success) {
                            $('#regularUploadResult').html(
                                `<div class="alert alert-success">
                                    ${response.message}<br>
                                    Path: ${response.path}
                                </div>`
                            );
                        } else {
                            $('#regularUploadResult').html(
                                `<div class="alert alert-danger">
                                    ${response.message}
                                    ${response.error_code ? '<br>Error code: ' + response.error_code : ''}
                                </div>`
                            );
                        }
                    },
                    error: function(xhr, status, error) {
                        $('#regularUploadResult').html(
                            `<div class="alert alert-danger">
                                AJAX error: ${error}<br>
                                Status: ${status}<br>
                                Response: ${xhr.responseText}
                            </div>`
                        );
                    }
                });
            });
            
            // Base64 file upload
            $('#base64Image').on('change', function() {
                const fileInput = this;
                if (!fileInput.files.length) {
                    $('#imagePreview').empty();
                    return;
                }
                
                const file = fileInput.files[0];
                const reader = new FileReader();
                
                reader.onload = function(e) {
                    $('#imagePreview').html(
                        `<img src="${e.target.result}" class="img-thumbnail" style="max-height: 200px">`
                    );
                };
                
                reader.readAsDataURL(file);
            });
            
            $('#base64UploadBtn').on('click', function() {
                const fileInput = $('#base64Image')[0];
                if (!fileInput.files.length) {
                    $('#base64UploadResult').html('<div class="alert alert-warning">Please select a file</div>');
                    return;
                }
                
                const file = fileInput.files[0];
                const reader = new FileReader();
                
                reader.onload = function(e) {
                    // Send the base64 data to the server
                    $.ajax({
                        url: 'upload-test.php',
                        type: 'POST',
                        data: {
                            image_data: e.target.result
                        },
                        dataType: 'json',
                        success: function(response) {
                            if (response.success) {
                                $('#base64UploadResult').html(
                                    `<div class="alert alert-success">
                                        ${response.message}<br>
                                        Path: ${response.path}
                                    </div>`
                                );
                            } else {
                                $('#base64UploadResult').html(
                                    `<div class="alert alert-danger">
                                        ${response.message}
                                    </div>`
                                );
                            }
                        },
                        error: function(xhr, status, error) {
                            $('#base64UploadResult').html(
                                `<div class="alert alert-danger">
                                    AJAX error: ${error}<br>
                                    Status: ${status}<br>
                                    Response: ${xhr.responseText}
                                </div>`
                            );
                        }
                    });
                };
                
                reader.onerror = function() {
                    $('#base64UploadResult').html(
                        `<div class="alert alert-danger">
                            Failed to read file
                        </div>`
                    );
                };
                
                reader.readAsDataURL(file);
            });
        });
    </script>
</body>
</html>
