<?php
// Start session to access session variables
session_start();

// Include necessary files and configurations
// Only include db_connect.php to avoid function conflicts
require_once '../config/db_connect.php';
// Skip auth/functions.php which conflicts with db_connect.php
require_once '../includes/functions.php';

// Fix for stray JavaScript showing on page
ob_start(); // Start output buffering to catch any stray output

// Check if user is authenticated and is a teacher
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'teacher') {
    // Redirect to login page if not authenticated
    header("Location: ../login.php");
    exit();
}

// Debug information for uploads
$debug_mode = true; // Set to false in production

// Increase upload limits at runtime
ini_set('upload_max_filesize', '600M');
ini_set('post_max_size', '600M');
ini_set('memory_limit', '512M');
ini_set('max_execution_time', '300');
ini_set('max_input_time', '300');

// No duplicated includes needed here
$teacher_id = $_SESSION['user_id'];
$teacher = get_teacher($conn, $teacher_id);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Upload Video - Panadite Academy</title>
    
    <!-- Favicon -->
    <link rel="shortcut icon" href="../assets/images/favicon.ico" type="image/x-icon">
    
    <!-- CSS Libraries -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/toastify-js/src/toastify.min.css">
    
    <!-- jQuery -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    
    <!-- Clean JavaScript implementation -->
    <script>
        // Global variables
        var selectedFile = null;
        var videoDuration = 0;
        
        // Initialize when DOM is loaded
        document.addEventListener('DOMContentLoaded', function() {
            console.log('DOM loaded, initializing upload form');
            
            // Get form elements
            var browseBtn = document.getElementById('browseBtn');
            var videoFile = document.getElementById('videoFile');
            var form = document.getElementById('videoUploadForm');
            
            // Browse button handler
            if (browseBtn && videoFile) {
                browseBtn.addEventListener('click', function() {
                    videoFile.click();
                });
            }
            
            // Form submit handler
            if (form) {
                form.addEventListener('submit', function(event) {
                    event.preventDefault();
                    handleFormSubmit(event);
                });
            }
            
            // File selection handler  
            if (videoFile) {
                videoFile.onchange = function() {
                    if (this.files && this.files[0]) {
                        var file = this.files[0];
                        console.log('File selected:', file.name, file.size, file.type);
                        
                        // Store selected file globally
                        selectedFile = file;
                        
                        // Update display elements if they exist
                        var fileNameElement = document.getElementById('fileName');
                        var fileSizeElement = document.getElementById('fileSize');
                        
                        if (fileNameElement) fileNameElement.textContent = file.name;
                        if (fileSizeElement) fileSizeElement.textContent = formatFileSize(file.size);
                        
                        // Store metadata in hidden fields
                        var fileTypeInput = document.getElementById('fileType');
                        var fileSizeInput = document.getElementById('fileSizeHidden');
                        
                        if (fileTypeInput) fileTypeInput.value = file.type;
                        if (fileSizeInput) fileSizeInput.value = file.size;
                        
                        // Get video duration
                        getVideoDuration(file);
                    }
                };
            }
            
            // Helper functions
            function formatFileSize(bytes) {
                if (bytes === 0) return '0 Bytes';
                var k = 1024;
                var sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];
                var i = Math.floor(Math.log(bytes) / Math.log(k));
                return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
            }
            
            function formatTime(seconds) {
                seconds = Math.round(seconds);
                var hrs = Math.floor(seconds / 3600);
                var mins = Math.floor((seconds % 3600) / 60);
                var secs = Math.floor(seconds % 60);
                
                if (hrs > 0) {
                    return hrs + ':' + (mins < 10 ? '0' : '') + mins + ':' + (secs < 10 ? '0' : '') + secs;
                } else {
                    return mins + ':' + (secs < 10 ? '0' : '') + secs;
                }
            }
            
            function getVideoDuration(file) {
                var video = document.createElement('video');
                video.preload = 'metadata';
                video.onloadedmetadata = function() {
                    var duration = video.duration;
                    console.log('Video duration:', duration);
                    videoDuration = duration;
                    
                    var durationInput = document.getElementById('videoDurationSeconds');
                    if (durationInput) {
                        durationInput.value = Math.round(duration);
                    }
                    
                    URL.revokeObjectURL(video.src);
                };
                video.src = URL.createObjectURL(file);
            }
            
            // Add reset button handler
            var resetBtn = document.getElementById('resetBtn');
            if (resetBtn) {
                resetBtn.onclick = function() {
                    console.log('Reset button clicked');
                    // Reset form elements
                    document.getElementById('videoUploadForm').reset();
                    
                    // Reset file selection display
                    document.getElementById('fileName').textContent = 'No file selected';
                    document.getElementById('fileSize').textContent = '0 KB';
                    document.getElementById('progressWrapper').style.display = 'none';
                    
                    // Clear any potential large file warning
                    var largeFileInfo = document.getElementById('largeFileInfo');
                    if (largeFileInfo) largeFileInfo.style.display = 'none';
                    
                    // Return focus to title field
                    document.getElementById('videoTitle').focus();
                };
            }
        });
        
        // Form validation and submission with base64 conversion
        function handleFormSubmit(event) {
            event.preventDefault();
            console.log('Handling form submission with base64 conversion...');
            
            // Validate form
            if (!validateForm()) {
                console.log('Form validation failed');
                return;
            }
            
            // Process the file upload
            processFileUpload();
        }
        
        function processFileUpload() {
            console.log('Processing file upload...');
            
            // Show loading state
            var submitBtn = document.getElementById('submitBtn');
            if (submitBtn) {
                submitBtn.disabled = true;
                submitBtn.innerHTML = 'Uploading...';
            }
            
            // Convert file to base64
            var reader = new FileReader();
            reader.onload = function(e) {
                var base64String = e.target.result;
                
                // Set base64 data in hidden field
                var base64Input = document.getElementById('base64data');
                if (base64Input) {
                    base64Input.value = base64String;
                }
                
                // Set file metadata
                var fileNameInput = document.getElementById('fileName');
                var fileTypeInput = document.getElementById('fileType');
                var fileSizeInput = document.getElementById('fileSizeHidden');
                var durationInput = document.getElementById('videoDurationSeconds');
                
                if (fileNameInput) fileNameInput.value = selectedFile.name;
                if (fileTypeInput) fileTypeInput.value = selectedFile.type;
                if (fileSizeInput) fileSizeInput.value = selectedFile.size;
                if (durationInput && videoDuration) durationInput.value = Math.round(videoDuration);
                
                // Submit the form
                document.getElementById('videoUploadForm').submit();
            };
            
            reader.onerror = function() {
                alert('Error reading file');
                if (submitBtn) {
                    submitBtn.disabled = false;
                    submitBtn.innerHTML = 'Upload Video';
                }
            };
            
            reader.readAsDataURL(selectedFile);
        }
        
        function validateForm() {
            var videoTitle = document.getElementById('videoTitle');
            var courseId = document.getElementById('courseId');
            
            if (!videoTitle || !videoTitle.value.trim()) {
                alert('Please enter a video title');
                if (videoTitle) videoTitle.focus();
                return false;
            }
            
            if (!courseId || courseId.value === '' || courseId.value === '0') {
                alert('Please select a course');
                if (courseId) courseId.focus();
                return false;
            }
            
            return true;
        }
    </script>
    
    <style>
        :root {
            --primary: #276BE4;
            --primary-light: #E5EDFD;
            --secondary: #6C757D;
            --success: #28A745;
            --danger: #DC3545;
            --warning: #FFC107;
            --info: #17A2B8;
            --light: #F8F9FA;
            --dark: #343A40;
            --sidebar-width: 250px;
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            background-color: #f5f7fb;
            color: #333;
            display: flex;
            min-height: 100vh;
            margin: 0;
        }
        
        /* Dashboard container */
        .dashboard-container {
            display: flex;
            width: 100%;
        }
        
        /* Sidebar styles */
        .sidebar {
            width: var(--sidebar-width);
            background-color: whitesmoke;
            height: 100vh;
            position: fixed;
            left: 0;
            top: 0;
            overflow-y: auto;
            transition: all 0.3s ease;
            z-index: 1000;
            box-shadow: 0 0 15px rgba(0, 0, 0, 0.05);
        }
        
        /* Sidebar header */
        .sidebar-header {
            padding: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-bottom: 1px solid rgba(0, 0, 0, 0.05);
        }
        
        .sidebar-brand {
            color: var(--primary);
            text-decoration: none;
            font-weight: 700;
            font-size: 1.2rem;
            display: flex;
            align-items: center;
            transition: all 0.2s ease;
        }
        
        .sidebar-brand:hover {
            opacity: 0.9;
        }
        
        .sidebar-brand img {
            height: 30px;
            margin-right: 10px;
        }
        
        /* Sidebar navigation */
        .sidebar-nav {
            padding: 0;
            list-style: none;
            margin: 0;
        }
        
        .sidebar-heading {
            font-size: 0.75rem;
            font-weight: 600;
            color: #666;
            padding: 20px 20px 8px;
            margin-bottom: 0;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .nav-item {
            margin-bottom: 5px;
        }
        
        .nav-link {
            display: flex;
            align-items: center;
            padding: 12px 20px;
            color: #495057;
            text-decoration: none;
            border-radius: 5px;
            margin: 0 10px;
            transition: all 0.2s ease;
        }
        
        .nav-link i {
            margin-right: 10px;
            font-size: 1.1rem;
            width: 20px;
            text-align: center;
            color: #555;
        }
        
        .nav-link:hover {
            background-color: rgba(39, 107, 228, 0.1);
            color: var(--primary);
        }
        
        .nav-link:hover i {
            color: var(--primary);
        }
        
        .nav-link.active {
            background-color: var(--primary-light);
            color: var(--primary);
            font-weight: 500;
            box-shadow: 0 2px 5px rgba(39, 107, 228, 0.1);
        }
        
        .nav-link.active i {
            color: var(--primary);
        }
        
        .nav-badge {
            background-color: var(--primary);
            color: white;
            font-size: 0.7rem;
            padding: 2px 6px;
            border-radius: 10px;
            margin-left: 10px;
        }
        
        /* Content wrapper */
        .content-wrapper {
            margin-left: var(--sidebar-width);
            width: calc(100% - var(--sidebar-width));
            max-width: 100%;
            transition: all 0.3s ease;
            padding: 20px;
        }
        
        /* Dashboard navbar */
        .dashboard-navbar {
            background-color: #fff;
            box-shadow: 0 2px 15px rgba(0, 0, 0, 0.05);
            border-radius: 10px;
            padding: 15px 20px;
            margin-bottom: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        /* File upload styles */
        .upload-area {
            border: 2px dashed #dee2e6;
            border-radius: 8px;
            padding: 30px;
            text-align: center;
            background-color: #f8f9fa;
            transition: all 0.3s ease;
            cursor: pointer;
            position: relative;
            margin-bottom: 15px;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.05);
        }
        
        .upload-area:hover {
            border-color: var(--primary);
            background-color: var(--primary-light);
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(39, 107, 228, 0.1);
        }
        
        .upload-area:after {
            content: 'Click to upload';
            position: absolute;
            bottom: 10px;
            left: 0;
            right: 0;
            margin: auto;
            font-size: 12px;
            color: #6c757d;
            font-weight: 500;
        }
        
        .overlay-clickable {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            cursor: pointer;
            z-index: 5;
        }
        
        /* Ensure the button remains clickable above the overlay */
        #browseBtn {
            position: relative;
            z-index: 10;
        }
        
        .upload-icon {
            font-size: 48px;
            color: var(--primary);
            margin-bottom: 15px;
        }
        
        .progress-wrapper {
            display: none;
            margin-top: 20px;
        }
           /* Media queries for responsiveness */
           @media (max-width: 991px) {
            .sidebar {
                transform: translateX(-100%);
                z-index: 1050;
            }
            
            .content-wrapper {
                margin-left: 0;
                width: 100%;
                max-width: 100%;
            }
            
            .sidebar.show {
                transform: translateX(0);
            }
        }
    </style>
</head>
<body>
    <div class="dashboard-container">
        <!-- Sidebar -->
        <aside class="sidebar">
            <div class="sidebar-header">
                <a href="../index.php" class="sidebar-brand">
                    <img src="../assets/images/logo.jpeg" alt="Panadite Academy" onerror="this.src='../assets/images/favicon.ico';">
                    <span>Panadite Academy</span>
                </a>
            </div>
            
            <ul class="sidebar-nav">
                <li class="sidebar-heading">MAIN</li>
                
                <li class="nav-item">
                    <a href="dashboard.php" class="nav-link">
                        <i class="fas fa-tachometer-alt"></i>
                        <span>Dashboard</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="courses.php" class="nav-link">
                        <i class="fas fa-graduation-cap"></i>
                        <span>Courses</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="create-course.php" class="nav-link">
                        <i class="fas fa-plus-circle"></i>
                        <span>Create Course</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="videos.php" class="nav-link">
                        <i class="fas fa-video"></i>
                        <span>Videos</span>
                        <span class="nav-badge">New</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="upload-video.php" class="nav-link active">
                        <i class="fas fa-cloud-upload-alt"></i>
                        <span>Upload Video</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="students.php" class="nav-link">
                        <i class="fas fa-user-graduate"></i>
                        <span>Students</span>
                    </a>
                </li>
                
                <li class="sidebar-heading">TOOLS</li>
                
                <li class="nav-item">
                    <a href="analytics.php" class="nav-link">
                        <i class="fas fa-chart-line"></i>
                        <span>Analytics</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="quizzes.php" class="nav-link">
                        <i class="fas fa-question-circle"></i>
                        <span>Quizzes</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="assignments.php" class="nav-link">
                        <i class="fas fa-tasks"></i>
                        <span>Assignments</span>
                    </a>
                </li>
                
                <li class="sidebar-heading">ACCOUNT</li>
                
                <li class="nav-item">
                    <a href="profile.php" class="nav-link">
                        <i class="fas fa-user-circle"></i>
                        <span>Profile</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="settings.php" class="nav-link">
                        <i class="fas fa-cog"></i>
                        <span>Settings</span>
                    </a>
                </li>
                
                <li class="nav-item">
                    <a href="../logout.php" class="nav-link">
                        <i class="fas fa-sign-out-alt"></i>
                        <span>Logout</span>
                    </a>
                </li>
            </ul>
        </aside>
        
        <!-- Main Content Wrapper -->
        <div class="content-wrapper">
            <!-- Navbar -->
            <nav class="dashboard-navbar">
                <div class="d-flex align-items-center">
                    <button id="sidebarToggle" class="btn btn-sm btn-light me-2">
                        <i class="fas fa-bars"></i>
                    </button>
                    <h4 class="mb-0">Upload Video</h4>
                </div>
                
                <div class="d-flex align-items-center">
                    <!-- Search -->
                    <div class="me-3 d-none d-md-block">
                        <form class="d-flex">
                            <input type="text" class="form-control form-control-sm" placeholder="Search...">
                        </form>
                    </div>
                    
                    <!-- Notifications -->
                    <div class="dropdown me-3">
                        <button class="btn btn-light position-relative" data-bs-toggle="dropdown">
                            <i class="fas fa-bell"></i>
                            <span class="position-absolute top-0 start-100 translate-middle badge rounded-pill bg-danger">
                                3
                            </span>
                        </button>
                        <div class="dropdown-menu dropdown-menu-end">
                            <div class="p-2 border-bottom">
                                <h6 class="mb-0">Notifications</h6>
                            </div>
                            <div class="p-2 border-bottom">
                                <div class="d-flex">
                                    <div class="flex-shrink-0 me-2">
                                        <div class="rounded-circle bg-light p-2">
                                            <i class="fas fa-user-graduate text-primary"></i>
                                        </div>
                                    </div>
                                    <div>
                                        <p class="mb-0">New student enrolled</p>
                                        <small class="text-muted">2 hours ago</small>
                                    </div>
                                </div>
                            </div>
                            <div class="p-2 text-center">
                                <a href="#" class="text-decoration-none">View all notifications</a>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Profile -->
                    <div class="dropdown">
                        <button class="btn btn-light d-flex align-items-center" data-bs-toggle="dropdown">
                            <img src="<?php echo !empty($teacher['profile_picture']) ? '../uploads/profile/' . htmlspecialchars($teacher['profile_picture']) : '../assets/images/avatar-placeholder.jpg'; ?>" 
                                 class="rounded-circle me-2" height="32" width="32" alt="<?php echo htmlspecialchars($teacher['first_name']); ?>">
                            <span class="d-none d-md-inline"><?php echo htmlspecialchars($teacher['first_name']); ?></span>
                        </button>
                        <div class="dropdown-menu dropdown-menu-end">
                            <a href="profile.php" class="dropdown-item">
                                <i class="fas fa-user me-2"></i> Profile
                            </a>
                            <a href="settings.php" class="dropdown-item">
                                <i class="fas fa-cog me-2"></i> Settings
                            </a>
                            <div class="dropdown-divider"></div>
                            <a href="../logout.php" class="dropdown-item">
                                <i class="fas fa-sign-out-alt me-2"></i> Logout
                            </a>
                        </div>
                    </div>
                </div>
            </nav>

            <!-- Main Content -->
            <div class="container-fluid p-0">
                <!-- Status Messages -->
                <?php if (isset($_GET['success']) && $_GET['success'] == 1): ?>
                <div class="row mb-4">
                    <div class="col-12">
                        <div class="alert alert-success">
                            <i class="fas fa-check-circle me-2"></i> Video uploaded successfully! You can <a href="videos.php">view all your videos here</a>.
                        </div>
                    </div>
                </div>
                <?php endif; ?>
                
                <?php if (isset($_SESSION['error_message'])): ?>
                <div class="row mb-4">
                    <div class="col-12">
                        <div class="alert alert-danger">
                            <i class="fas fa-exclamation-triangle me-2"></i> <?php echo $_SESSION['error_message']; ?>
                        </div>
                        
                        <?php if ($debug_mode && isset($_SESSION['upload_debug'])): ?>
                        <div class="card mt-3">
                            <div class="card-header bg-light">
                                <h6 class="m-0">Debug Information (Only visible in debug mode)</h6>
                            </div>
                            <div class="card-body">
                                <pre class="mb-0"><?php print_r($_SESSION['upload_debug']); ?></pre>
                            </div>
                            <div class="card-footer">
                                <p class="mb-0"><small>Environment: PHP <?php echo phpversion(); ?> | Upload max filesize: <?php echo ini_get('upload_max_filesize'); ?> | Post max size: <?php echo ini_get('post_max_size'); ?></small></p>
                            </div>
                        </div>
                        <?php endif; ?>
                        
                        <?php 
                        // Clean up session variables after displaying
                        unset($_SESSION['error_message']);
                        if (isset($_SESSION['upload_debug'])) unset($_SESSION['upload_debug']);
                        ?>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Page Header -->
                <div class="row mb-4">
                    <div class="col-12">
                        <div class="bg-white p-4 rounded shadow-sm">
                            <h5 class="mb-4">Upload New Video</h5>
                            
                            <!-- Video upload form with drag & drop functionality -->
                            <?php
// Debug information for troubleshooting
$teacher_id = $_SESSION['user_id']; // Explicitly set teacher_id
error_log("Upload form loaded for teacher ID: $teacher_id");
?>
                            <form id="videoUploadForm" action="process-upload-video.php" method="POST" enctype="application/x-www-form-urlencoded">
<!-- Hidden fields for base64 data and file metadata -->
<input type="hidden" id="base64data" name="base64data" value="" />
<input type="hidden" id="fileName" name="fileName" value="" />
<input type="hidden" id="fileType" name="fileType" value="" />
<input type="hidden" id="fileSize" name="fileSize" value="" />
<input type="hidden" id="videoDuration" name="videoDuration" value="" />
<input type="hidden" id="videoDurationSeconds" name="duration_seconds" value="" />
<!-- Hidden field for teacher ID -->
<input type="hidden" name="teacher_id" value="<?php echo $teacher_id; ?>">
                                <div class="row mb-3">
                                    <div class="col-md-6 mb-3 mb-md-0">
                                        <label for="videoTitle" class="form-label">Video Title</label>
                                        <input type="text" class="form-control" id="videoTitle" name="title" required placeholder="Enter video title">
                                    </div>
                                    <div class="col-md-6">
                                        <label for="courseId" class="form-label">Course</label>
                                        <?php
                                        // Connect to database
                                        $conn = new mysqli("localhost", "root", "", "panadite_academy");
                                        
                                        // Check connection
                                        if ($conn->connect_error) {
                                            $db_error = "Database connection failed: " . $conn->connect_error;
                                        } else {
                                            // Get courses created by this teacher
                                            $teacher_id = $_SESSION['user_id'];
                                            $sql = "SELECT course_id, title FROM courses WHERE teacher_id = ? ORDER BY title";
                                            $stmt = $conn->prepare($sql);
                                            
                                            if ($stmt) {
                                                $stmt->bind_param("i", $teacher_id);
                                                $stmt->execute();
                                                $result = $stmt->get_result();
                                                $course_count = $result->num_rows;
                                            } else {
                                                $db_error = "Error preparing statement: " . $conn->error;
                                            }
                                        }
                                        ?>
                                        
                                        <?php if (isset($db_error)): ?>
                                            <div class="alert alert-danger p-2 mb-2"><?php echo $db_error; ?></div>
                                            <select class="form-select" id="courseId" name="course_id" required>
                                                <option value="">Select a course</option>
                                                <option value="1">Default Course</option>
                                            </select>
                                        <?php elseif ($course_count == 0): ?>
                                            <div class="alert alert-warning p-2 mb-2">
                                                <i class="fas fa-exclamation-triangle me-2"></i> You need to <a href="create-course.php" class="alert-link">create a course</a> before uploading videos.
                                            </div>
                                            <select class="form-select" id="courseId" name="course_id" required disabled>
                                                <option value="">No courses available</option>
                                            </select>
                                        <?php else: ?>
                                            <select class="form-select" id="courseId" name="course_id" required>
                                                <option value="">Select a course</option>
                                                <?php while ($course = $result->fetch_assoc()): ?>
                                                    <option value="<?php echo $course['course_id']; ?>"><?php echo htmlspecialchars($course['title']); ?></option>
                                                <?php endwhile; ?>
                                            </select>
                                        <?php endif; ?>
                                        
                                        <?php if (isset($stmt)) $stmt->close(); ?>
                                        <?php if (isset($conn)) $conn->close(); ?>
                                    </div>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="videoDescription" class="form-label">Video Description</label>
                                    <textarea class="form-control" id="videoDescription" name="description" rows="3" placeholder="Enter video description"></textarea>
                                </div>
                                
                                <div class="mb-4">
                                    <label class="form-label">Video File</label>
                                    <div class="upload-area" id="dropZone">
                                        <input type="file" name="video" id="videoFile" class="d-none" accept="video/*">
                                        <i class="fas fa-cloud-upload-alt upload-icon"></i>
                                        <h5>Drag & Drop video here</h5>
                                        <p class="text-muted mb-2">or</p>
                                        <button type="button" class="btn btn-primary" id="browseBtn">Browse Files</button>
                                        <p class="text-muted mt-2">Maximum file size: 500MB</p>
                                        <div class="overlay-clickable"></div>
                                    </div>
                                    
                                    <div class="progress-wrapper" id="progressWrapper">
                                        <div class="progress mt-3" style="height: 20px;">
                                            <div id="progressBar" class="progress-bar progress-bar-striped progress-bar-animated" 
                                                 role="progressbar" style="width: 0%;" aria-valuenow="0" 
                                                 aria-valuemin="0" aria-valuemax="100">0%</div>
                                        </div>
                                        <div class="d-flex justify-content-between mt-2">
                                            <small id="fileName" class="text-muted"></small>
                                            <small id="fileSize" class="text-muted"></small>
                                        </div>
                                        <div id="largeFileWarning" class="alert alert-warning mt-3" style="display: none;">
                                            <p><strong>Your video file exceeds the 40MB upload limit.</strong></p>
                                            <p>Please use our <a href="compress_video.html" class="alert-link">video compression tool</a> to reduce the file size before uploading.</p>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-6 mb-3 mb-md-0">
                                        <label for="videoDuration" class="form-label">Duration (minutes)</label>
                                        <div class="input-group">
                                            <input type="text" class="form-control" id="videoDuration" name="duration" placeholder="E.g. 15">
                                            <input type="hidden" id="videoDurationSeconds" name="duration_seconds" value="0">
                                            <span class="input-group-text" id="durationDetectedBadge" style="display:none;background-color:#e5edfd;color:#276BE4;font-size:12px;">Auto-detected</span>
                                        </div>
                                        <small class="text-muted" id="durationHelp">Will be automatically detected when video is selected</small>
                                    </div>
                                    <div class="col-md-6">
                                        <label for="moduleOrder" class="form-label">Module Order</label>
                                        <input type="number" class="form-control" id="moduleOrder" name="module_order" min="1" placeholder="E.g. 1">
                                    </div>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-6 mb-3 mb-md-0">
                                        <label for="thumbnailImage" class="form-label">Thumbnail Image (Optional)</label>
                                        <input type="file" class="form-control" id="thumbnailImage" name="thumbnail" accept="image/*">
                                    </div>
                                    <div class="col-md-6">
                                        <label for="visibility" class="form-label">Visibility</label>
                                        <select class="form-select" id="visibility" name="visibility">
                                            <option value="public">Public</option>
                                            <option value="private">Private</option>
                                            <option value="enrolled">Enrolled Students Only</option>
                                        </select>
                                    </div>
                                </div>
                                
                                <div class="mt-4">
                                    <div class="d-flex justify-content-between mt-4">
                                        <button type="button" id="resetBtn" class="btn btn-outline-secondary">
                                            <i class="fas fa-times me-2"></i> Reset
                                        </button>
                                        <button type="submit" id="submitBtn" class="btn btn-primary">
                                            <i class="fas fa-cloud-upload-alt me-2"></i> Upload Video
                                        </button>
                                    </div>
                                    
                                    <!-- Upload Progress Indicator -->
                                    <div id="progressWrapper" style="display:none; margin-top:20px;">
                                        <div class="d-flex justify-content-between mb-1">
                                            <span>Upload Progress</span>
                                            <span id="progressText">Preparing...</span>
                                        </div>
                                        <div class="progress" style="height: 20px;">
                                            <div id="progressBar" class="progress-bar progress-bar-striped progress-bar-animated" 
                                                role="progressbar" aria-valuenow="0" aria-valuemin="0" aria-valuemax="100" style="width: 0%"></div>
                                        </div>
                                    </div>
                                    
                                    <div id="uploadingMsg" style="display:none; margin-top:15px;" class="alert alert-info text-center">
                                        <div class="spinner-border spinner-border-sm text-primary me-2" role="status">
                                            <span class="visually-hidden">Loading...</span>
                                        </div>
                                        <span><strong>Uploading video...</strong> This may take several minutes. Please do not close this page.</span>
                                    </div>
                                    
                                    <!-- Upload Information -->
                                    <div id="largeFileInfo" class="alert alert-info mt-3" style="display: none;">
                                        <p><strong>Large Video File Detected</strong></p>
                                        <p>This system supports large educational videos (up to 4GB). Larger files will take longer to upload - please be patient and don't close this page during upload.</p>
                                    </div>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div><!-- End content-wrapper -->
    </div><!-- End dashboard-container -->
    
    <!-- JavaScript Libraries -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/toastify-js"></script>
    
    <script>
        // Format file size in bytes to a readable format (KB, MB, GB)
        function formatBytes(bytes, decimals = 2) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const dm = decimals < 0 ? 0 : decimals;
            const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + ' ' + sizes[i];
        }
        
        // Alias for formatBytes to maintain compatibility with both function names
        function formatFileSize(bytes, decimals = 2) {
            return formatBytes(bytes, decimals);
        }
        
        // Helper function to show toast notifications
        function showToast(message, type = 'info') {
            let bgColor = '#0d6efd'; // default blue
            
            if (type === 'error') {
                bgColor = '#dc3545'; // red
            } else if (type === 'warning') {
                bgColor = '#ffc107'; // yellow
            } else if (type === 'success') {
                bgColor = '#198754'; // green
            }
            
            Toastify({
                text: message,
                duration: 3000,
                close: true,
                gravity: "top",
                position: "center",
                style: {
                    background: bgColor
                }
            }).showToast();
        }

        // Sidebar toggle functionality
        document.getElementById('sidebarToggle').addEventListener('click', function() {
            document.querySelector('.sidebar').classList.toggle('show');
        });
        
        // DOM Elements
        const dropZone = document.getElementById('dropZone');
        const browseBtn = document.getElementById('browseBtn');
        const fileInput = document.getElementById('videoFile');
        const progressWrapper = document.getElementById('progressWrapper');
        const progressBar = document.getElementById('progressBar');
        const fileNameDisplay = document.getElementById('fileName');
        const fileSizeDisplay = document.getElementById('fileSize');
        
        // Check if elements exist
        if (!dropZone || !browseBtn || !fileInput) {
            console.error('Required DOM elements not found');
            return;
        }
        
        console.log('DOM elements found successfully');
        
        // Initialize progress wrapper as hidden
        if (progressWrapper) {
            progressWrapper.style.display = 'none';
        }
        
        // Helper functions
        function showToast(message, type = 'info') {
            const backgroundColor = {
                'success': '#28a745',
                'error': '#dc3545',
                'warning': '#ffc107',
                'info': '#17a2b8'
            }[type] || '#17a2b8';
            
            if (typeof Toastify !== 'undefined') {
                Toastify({
                    text: message,
                    duration: 3000,
                    close: true,
                    gravity: "top",
                    position: "center",
                    backgroundColor: backgroundColor,
                }).showToast();
            } else {
                alert(message);
            }
        }
        
        function formatFileSize(bytes) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        }
        
        function formatTime(seconds) {
            const hours = Math.floor(seconds / 3600);
            const minutes = Math.floor((seconds % 3600) / 60);
            const secs = Math.floor(seconds % 60);
            
            if (hours > 0) {
                return `${hours}:${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
            } else {
                return `${minutes}:${secs.toString().padStart(2, '0')}`;
            }
        }
        
        // Add video preview container to the page
        // Function to handle form submission with base64 conversion
        document.getElementById('videoUploadForm').addEventListener('submit', function(e) {
            e.preventDefault(); // Always prevent default submission
            
            console.log('Form submission handler running');
            
            // Get form elements and values
            const videoFile = document.getElementById('videoFile').files[0];
            const title = document.getElementById('videoTitle').value;
            const description = document.getElementById('videoDescription').value;
            const courseId = document.getElementById('courseId').value;
            
            // Validate required fields
            if (!title || !description || !videoFile || !courseId || courseId === '0') {
                showToast('Please fill in all required fields and select a video file', 'error');
                console.log('Form validation failed');
                return false;
            }
            
            // Show loading state
            const submitBtn = document.getElementById('submitBtn');
            if (submitBtn) {
                submitBtn.disabled = true;
                submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i> Uploading...';
            }
            
            // Convert file to base64 and submit
            convertToBase64AndSubmit(videoFile);
        });
        
        function convertToBase64AndSubmit(file) {
            console.log('Converting file to base64:', file.name, file.size, file.type);
            
            const reader = new FileReader();
            reader.onload = function(e) {
                try {
                    const base64String = e.target.result;
                    console.log('Base64 conversion complete, length:', base64String.length);
                    
                    // Set base64 data and metadata in hidden fields
                    document.getElementById('base64data').value = base64String;
                    document.getElementById('fileName').value = file.name;
                    document.getElementById('fileType').value = file.type;
                    document.getElementById('fileSize').value = file.size;
                    
                    // Set duration if available
                    const durationSeconds = document.getElementById('videoDurationSeconds').value;
                    if (durationSeconds) {
                        document.getElementById('videoDuration').value = durationSeconds;
                    }
                    
                    console.log('Submitting form with base64 data...');
                    
                    // Submit the form normally (will now include base64 data)
                    document.getElementById('videoUploadForm').submit();
                    
                } catch (error) {
                    console.error('Error in base64 conversion:', error);
                    showToast('Error processing file: ' + error.message, 'error');
                    resetSubmitButton();
                }
            };
            
            reader.onerror = function(error) {
                console.error('FileReader error:', error);
                showToast('Error reading file', 'error');
                resetSubmitButton();
            };
            
            reader.readAsDataURL(file);
        }
        
        function resetSubmitButton() {
            const submitBtn = document.getElementById('submitBtn');
            if (submitBtn) {
                submitBtn.disabled = false;
                submitBtn.innerHTML = '<i class="fas fa-cloud-upload-alt me-2"></i> Upload Video';
            }
        }

        $(document).ready(function() {
            // Add video preview container after the progress wrapper
            $('#progressWrapper').after('<div id="videoPreviewContainer" class="mt-4" style="display:none;"><h6 class="mb-2">Video Preview</h6><div class="card"><div class="card-body p-0 rounded overflow-hidden"><video id="videoPreview" class="w-100" controls style="max-height:240px;background:#000;"></video></div><div class="card-footer p-2 bg-light d-flex justify-content-between align-items-center"><small class="text-muted">Preview your video</small><span class="badge bg-primary" id="videoDurationBadge"></span></div></div></div>');
            
            // Note: File input change handling is done by vanilla JS handleFile function
            
            // Handle reset button click
            $('#resetBtn').on('click', function() {
                // Clear the video preview
                const videoPreview = document.getElementById('videoPreview');
                const videoContainer = document.getElementById('videoPreviewContainer');
                
                if (videoPreview) {
                    videoPreview.pause();
                    videoPreview.removeAttribute('src');
                    videoPreview.load();
                }
                
                if (videoContainer) {
                    videoContainer.style.display = 'none';
                }
                
                // Reset progress indicators
                const progressWrapper = document.getElementById('progressWrapper');
                if (progressWrapper) {
                    progressWrapper.style.display = 'none';
                }
                
                // Reset duration fields
                document.getElementById('videoDuration').value = '';
                document.getElementById('videoDurationSeconds').value = '';
                document.getElementById('durationHelp').textContent = 'Will be automatically detected when video is selected';
                document.getElementById('durationDetectedBadge').style.display = 'none';
                
                // Reset other form elements if needed
                Toastify({
                    text: "Form has been reset. You can select a new video.",
                    duration: 2000,
                    close: true,
                    gravity: "top",
                    position: "center",
                    backgroundColor: "#17a2b8",
                }).showToast();
            });
        });
        
        // Browse button click event
        document.getElementById('browseBtn').addEventListener('click', function() {
            document.getElementById('videoFile').click();
        });
        
        // Make the entire drop zone clickable using the overlay
        const overlayClickable = document.querySelector('.overlay-clickable');
        if (overlayClickable) {
            overlayClickable.addEventListener('click', function(e) {
                e.stopPropagation();
                fileInput.click();
            });
        }
        
        // Alternative click handler for the drop zone
        dropZone.addEventListener('click', function(e) {
            // Prevent triggering if the actual button is clicked
            if (e.target !== browseBtn && !browseBtn.contains(e.target) && 
                !e.target.classList.contains('overlay-clickable')) {
                fileInput.click();
            }
        });
        
        // File input change event
        fileInput.addEventListener('change', function() {
            handleFile(this.files[0]);
        });
        
        // Drag and drop events
        dropZone.addEventListener('dragover', function(e) {
            e.preventDefault();
            dropZone.classList.add('bg-light');
        });
        
        dropZone.addEventListener('dragleave', function(e) {
            e.preventDefault();
            dropZone.classList.remove('bg-light');
        });
        
        dropZone.addEventListener('dragenter', function(e) {
            e.preventDefault();
            dropZone.classList.add('bg-light');
        });
        
        dropZone.addEventListener('drop', function(e) {
            e.preventDefault();
            dropZone.classList.remove('bg-light');
            
            if (e.dataTransfer.files.length) {
                fileInput.files = e.dataTransfer.files;
                handleFile(e.dataTransfer.files[0]);
            }
        });
        
        // Global variable to store selected file
        let selectedVideoFile = null;
        
        // Handle the selected file
        function handleFile(file) {
            console.log('handleFile called with:', file);
            if (!file) {
                console.log('No file provided to handleFile');
                return;
            }
            
            // Check file type
            if (!file.type.startsWith('video/')) {
                console.log('Invalid file type:', file.type);
                showToast('Please select a valid video file!', 'error');
                return;
            }
            
            console.log('File type is valid:', file.type);
            
            // Store the file globally
            selectedVideoFile = file;
            
            // IMPORTANT: Make sure the file is properly assigned to the file input element
            // This ensures the file will be included in form submissions
            const videoFileInput = document.getElementById('videoFile');
            
            // Create a DataTransfer object (modern browsers)
            try {
                const dataTransfer = new DataTransfer();
                dataTransfer.items.add(file);
                videoFileInput.files = dataTransfer.files;
                console.log('File assigned to input using DataTransfer');
            } catch (error) {
                // Fallback for older browsers
                console.log('DataTransfer failed:', error);
                // Try direct assignment (works in some browsers)
                try {
                    videoFileInput.files = [file];
                    console.log('Direct file assignment attempted');
                } catch (err) {
                    console.error('Could not assign file to input element:', err);
                }
            }
            
            // Check file size (500MB max)
            const maxSize = 500 * 1024 * 1024; // 500MB in bytes
            if (file.size > maxSize) {
                showToast('File size exceeds the 500MB limit!', 'error');
                return;
            }
            
            // Display file info
            if (fileNameDisplay) fileNameDisplay.textContent = file.name;
            if (fileSizeDisplay) fileSizeDisplay.textContent = formatFileSize(file.size);
            if (progressWrapper) progressWrapper.style.display = 'block';
            
            console.log('File selected:', file.name, 'Size:', formatFileSize(file.size));
            
            // Detect video duration using HTML5 video API
            const videoElement = document.createElement('video');
            videoElement.preload = 'metadata';
            
            videoElement.onloadedmetadata = function() {
                window.URL.revokeObjectURL(videoElement.src);
                clearTimeout(metadataTimeout); // Clear the timeout since metadata loaded successfully
                const durationInSeconds = videoElement.duration;
                
                // Store the exact duration in seconds (rounded to avoid floating point issues)
                const exactSeconds = Math.round(durationInSeconds);
                const minutes = Math.floor(exactSeconds / 60);
                const seconds = exactSeconds % 60;
                
                // Format the time for display
                const formattedTime = formatTime(durationInSeconds);
                
                // Update the visible input with the formatted time
                const durationInput = document.getElementById('videoDuration');
                durationInput.value = formattedTime; // Display as MM:SS format
                
                // Store the raw seconds in the hidden field
                const durationSecondsInput = document.getElementById('videoDurationSeconds');
                durationSecondsInput.value = exactSeconds;
                
                // Show the auto-detected badge
                const durationBadge = document.getElementById('durationDetectedBadge');
                durationBadge.style.display = 'inline-block';
                
                // Update the help text with a readable format
                const durationHelp = document.getElementById('durationHelp');
                
                if (minutes > 0) {
                    durationHelp.textContent = `${minutes}min ${seconds}sec (${exactSeconds} seconds total)`;
                } else {
                    durationHelp.textContent = `${seconds} seconds`;
                }
            };
            
            // Handle errors in video metadata loading
            videoElement.onerror = function() {
                window.URL.revokeObjectURL(videoElement.src);
                clearTimeout(metadataTimeout);
                console.error('Error loading video metadata');
                enableManualDurationEntry();
                showToast('Could not detect video duration. Please enter it manually.', 'warning');
            };
            
            // Set a timeout in case the metadata event never fires
            let metadataTimeout = setTimeout(function() {
                window.URL.revokeObjectURL(videoElement.src);
                console.warn('Video metadata loading timed out');
                enableManualDurationEntry();
                showToast('Video duration detection timed out. Please enter it manually.', 'warning');
            }, 5000); // 5 second timeout
            
            const videoURL = URL.createObjectURL(file);
            videoElement.src = videoURL;
            
            // Set progress to 100% immediately for short videos
            progressBar.style.width = '100%';
            progressBar.setAttribute('aria-valuenow', 100);
            
            // Display video preview if desired
            showVideoPreview(file);
        }
        
        // Enable manual duration entry when automatic detection fails
        function enableManualDurationEntry() {
            // Make the duration input editable
            const durationInput = document.getElementById('videoDuration');
            durationInput.readOnly = false;
            durationInput.value = '';
            durationInput.placeholder = 'Enter duration manually (e.g. 5:30)';
            durationInput.focus();
            
            // Update the badge to indicate manual entry
            const durationBadge = document.getElementById('durationDetectedBadge');
            if (durationBadge) {
                durationBadge.textContent = 'Enter manually';
                durationBadge.style.display = 'inline-block';
                durationBadge.classList.remove('bg-success');
                durationBadge.classList.add('bg-warning');
            }
            
            // Update help text
            const durationHelp = document.getElementById('durationHelp');
            if (durationHelp) {
                durationHelp.textContent = 'Please enter the duration in minutes:seconds format (e.g. 5:30)';
            }
            
            // Add an event listener to parse manual input
            durationInput.addEventListener('blur', function() {
                const manualValue = this.value.trim();
                if (manualValue) {
                    // Parse manual input
                    const parts = manualValue.split(':');
                    let seconds = 0;
                    
                    if (parts.length === 1) {
                        // Assume input is in seconds
                        seconds = parseInt(parts[0], 10);
                    } else if (parts.length === 2) {
                        // MM:SS format
                        seconds = parseInt(parts[0], 10) * 60 + parseInt(parts[1], 10);
                    } else if (parts.length === 3) {
                        // HH:MM:SS format
                        seconds = parseInt(parts[0], 10) * 3600 + parseInt(parts[1], 10) * 60 + parseInt(parts[2], 10);
                    }
                    
                    // Update the seconds input
                    const durationSecondsInput = document.getElementById('videoDurationSeconds');
                    if (durationSecondsInput && !isNaN(seconds)) {
                        durationSecondsInput.value = seconds;
                        
                        // Format the display value nicely
                        this.value = formatTime(seconds);
                        
                        // Update help text with parsed duration
                        if (durationHelp) {
                            const minutes = Math.floor(seconds / 60);
                            const remainingSeconds = seconds % 60;
                            if (minutes > 0) {
                                durationHelp.textContent = `${minutes}min ${remainingSeconds}sec (${seconds} seconds total)`;
                            } else {
                                durationHelp.textContent = `${seconds} seconds`;
                            }
                        }
                    }
                }
            });
        }
        
        // Format time in HH:MM:SS format
        function formatTime(seconds) {
            // Ensure we're working with a number and round to avoid floating point issues
            seconds = Math.round(parseFloat(seconds));
            
            const hours = Math.floor(seconds / 3600);
            const minutes = Math.floor((seconds % 3600) / 60);
            const secs = Math.floor(seconds % 60);
            
            let timeString = '';
            if (hours > 0) {
                timeString += `${hours}:${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
            } else {
                timeString += `${minutes}:${secs.toString().padStart(2, '0')}`;
            }
            
            return timeString;
        }
        
        // Show video preview
        function showVideoPreview(file) {
            const videoURL = URL.createObjectURL(file);
            const videoPreview = document.getElementById('videoPreview');
            const videoContainer = document.getElementById('videoPreviewContainer');
            const durationBadge = document.getElementById('videoDurationBadge');
            
            videoPreview.src = videoURL;
            videoContainer.style.display = 'block';
            
            // Add event listener to handle when metadata is loaded
            videoPreview.addEventListener('loadedmetadata', function() {
                // We already handle this in handleFile function, but this is a backup
                if (!document.getElementById('videoDuration').value) {
                    const durationInSeconds = videoPreview.duration;
                    const durationInMinutes = durationInSeconds / 60;
                    document.getElementById('videoDuration').value = durationInMinutes.toFixed(2);
                }
                
                // Update the duration badge in the preview
                const previewDuration = Math.round(videoPreview.duration);
                durationBadge.textContent = formatTime(previewDuration);
                
                // Make sure the form fields are also updated with this more accurate player data
                document.getElementById('videoDuration').value = formatTime(previewDuration);
                document.getElementById('videoDurationSeconds').value = previewDuration;
                
                // Automatically play the video preview (optional)
                // videoPreview.play();
            });
            
            // Handle errors in video preview
            videoPreview.addEventListener('error', function() {
                Toastify({
                    text: "Could not preview the video. The file might be corrupted.",
                    duration: 3000,
                    close: true,
                    gravity: "top",
                    position: "center",
                    backgroundColor: "#ffc107",
                }).showToast();
            });
        }
        
        // Format file size
        function formatFileSize(bytes) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        }
        
        // Show video preview
        function showVideoPreview(file) {
            const videoPreview = document.getElementById('videoPreview');
            const videoContainer = document.getElementById('videoPreviewContainer');
            const durationBadge = document.getElementById('videoDurationBadge');
            
            if (!videoPreview || !videoContainer) {
                console.log('Video preview elements not found');
                return;
            }
            
            // Create a URL for the video file
            const videoURL = URL.createObjectURL(file);
            
            // Set the video source
            videoPreview.src = videoURL;
            
            // Show the preview container
            videoContainer.style.display = 'block';
            
            // Handle video metadata loaded
            videoPreview.addEventListener('loadedmetadata', function() {
                const previewDuration = Math.round(videoPreview.duration);
                if (durationBadge) {
                    durationBadge.textContent = formatTime(previewDuration);
                }
                
                // Update duration fields if they haven't been set yet
                const durationInput = document.getElementById('videoDuration');
                const durationSecondsInput = document.getElementById('videoDurationSeconds');
                
                if (durationInput && !durationInput.value) {
                    durationInput.value = formatTime(previewDuration);
                }
                if (durationSecondsInput && !durationSecondsInput.value) {
                    durationSecondsInput.value = previewDuration;
                }
                
                console.log('Video preview loaded, duration:', previewDuration);
            });
            
            // Handle errors in video preview
            videoPreview.addEventListener('error', function() {
                showToast('Could not preview the video. The file might be corrupted.', 'warning');
                videoContainer.style.display = 'none';
            });
            
            // Clean up the URL when the video is removed
            videoPreview.addEventListener('emptied', function() {
                URL.revokeObjectURL(videoURL);
            });
        }
    
    // If we have duration but no seconds, parse the duration
    if (durationInput.value && (!durationSecondsInput.value || durationSecondsInput.value === '0')) {
        console.log('Parsing duration from input:', durationInput.value);
        const parts = durationInput.value.split(':');
        let seconds = 0;
        
        if (parts.length === 1) {
            // Assume input is in seconds
            seconds = parseInt(parts[0], 10);
        } else if (parts.length === 2) {
            // MM:SS format
            seconds = parseInt(parts[0], 10) * 60 + parseInt(parts[1], 10);
        } else if (parts.length === 3) {
            // HH:MM:SS format
            seconds = parseInt(parts[0], 10) * 3600 + parseInt(parts[1], 10) * 60 + parseInt(parts[2], 10);
        }
        
        if (seconds > 0) {
            console.log('Setting duration_seconds to', seconds);
            durationSecondsInput.value = seconds;
        } else {
            console.log('Setting default duration_seconds to 30');
            durationSecondsInput.value = '30'; // Fallback to 30 seconds
        }
    }
                // Ensure the file is in the input
                const videoFileInput = document.getElementById('videoFile');
                const courseIdSelect = document.getElementById('courseId');
                const titleInput = document.getElementById('videoTitle');
                const durationInput = document.getElementById('videoDuration');
                const durationSecondsInput = document.getElementById('videoDurationSeconds');
                
                console.log('Duration value:', durationInput.value);
                console.log('Duration seconds value:', durationSecondsInput.value);
                
                // Check if video file is selected
                if (!videoFileInput.files || videoFileInput.files.length === 0) {
                    e.preventDefault(); // Stop form submission
                    showToast('Please select a video file', 'error');
                    return false;
                }
            
                // Check if title is entered
                if (!titleInput.value.trim()) {
                    e.preventDefault();
                    showToast('Please enter a video title', 'error');
                    titleInput.focus();
                    return false;
                }
                
                // Check if a course is selected
                if (!courseIdSelect.value || courseIdSelect.value === '0' || courseIdSelect.value === '') {
                    e.preventDefault();
                    showToast('Please select a course for this video', 'error');
                    courseIdSelect.focus();
                    return false;
                }
            
                // Log the course ID to verify it's being correctly selected
                console.log('Selected course_id:', courseIdSelect.value);
                
                // Check if duration is detected or manually entered
                if (!durationInput.value) {
                    e.preventDefault();
                    showToast('Video duration is required', 'error');
                    durationInput.focus();
                    return false;
                }
            
                // Show upload notice for large files
                if (videoFileInput.files[0].size > 100 * 1024 * 1024) { // 100MB
                    $('#largeFileInfo').show();
                    showToast(`Large file detected (${formatFileSize(videoFileInput.files[0].size)}). Upload may take several minutes.`, 'info');
                }
                
                console.log('Duration value:', durationInput.value);
                console.log('Duration seconds value:', durationSecondsInput.value);
                
                // Fix duration issues - ensure we have values for both duration fields
                if (!durationInput.value && videoFileInput.files[0]) {
                    console.log('Setting default duration values');
                    // If duration wasn't detected but we have a video, set a default value
                    durationInput.value = '0:30'; // Default to 30 seconds
                    durationSecondsInput.value = '30'; // 30 seconds in total
                }
            
            // If we have duration but no seconds, parse the duration
            var parseVideoDuration = function() {
                if (durationInput && durationInput.value && durationSecondsInput && 
                    (!durationSecondsInput.value || durationSecondsInput.value === '0')) {
                    console.log('Parsing duration from input:', durationInput.value);
                    const parts = durationInput.value.split(':');
                    let seconds = 0;
                    
                    if (parts.length === 1) {
                        // Assume input is in seconds
                        seconds = parseInt(parts[0], 10);
                    } else if (parts.length === 2) {
                        // MM:SS format
                        seconds = parseInt(parts[0], 10) * 60 + parseInt(parts[1], 10);
                    } else if (parts.length === 3) {
                        // HH:MM:SS format
                        seconds = parseInt(parts[0], 10) * 3600 + parseInt(parts[1], 10) * 60 + parseInt(parts[2], 10);
                    }
                    
                    if (seconds > 0) {
                        console.log('Setting duration_seconds to', seconds);
                        durationSecondsInput.value = seconds;
                    } else {
                        console.log('Setting default duration_seconds to 30');
                        durationSecondsInput.value = '30'; // Fallback to 30 seconds
                    }
                }
            };
            
            // Call the function to parse duration
            parseVideoDuration();
            });
            
            // - File selection is handled by the videoFile.onchange handler
            // - Form submission is handled by handleFormSubmit function
            // - We're using direct form submission rather than AJAX to handle large video uploads
            // - Base64 encoding is done client-side to bypass PHP upload size limitations
        });
        </script>
</body>
</html>
