<?php
// Set current page for active menu highlighting
$current_page = 'upload-video.php';

// Include the centralized teacher layout start
require_once __DIR__ . '/components/teacher_layout.php';

// Get teacher ID from session (same pattern as dashboard page)
$teacher_id = $_SESSION['user_id'] ?? null;

// Session fallback - same as dashboard page
if (!$teacher_id || !is_numeric($teacher_id)) {
    $teacher_id = 14; // Use one of your existing teacher IDs
    error_log("Upload Video: Session user_id not set, defaulting to Teacher 14");
}

// Check teacher identity verification status (with error handling)
try {
    // First check if the columns exist in the database
    $column_check = $conn->query("SHOW COLUMNS FROM users LIKE 'identity_verification_status'");
    
    if ($column_check && $column_check->num_rows > 0) {
        // Columns exist, proceed with verification check
        $identity_verification_query = "SELECT identity_verification_status, identity_document_path, identity_uploaded_at FROM users WHERE user_id = ? AND user_role = 'teacher'";
        $identity_stmt = $conn->prepare($identity_verification_query);
        
        if ($identity_stmt) {
            $identity_stmt->bind_param("i", $teacher_id);
            $identity_stmt->execute();
            $identity_result = $identity_stmt->get_result();
            $verification_data = $identity_result->fetch_assoc();
            $identity_stmt->close();
            
            $identity_status = $verification_data['identity_verification_status'] ?? 'not_submitted';
            $is_verified = ($identity_status === 'verified');
        } else {
            // Prepare failed
            error_log("Upload Video: Failed to prepare identity verification query: " . $conn->error);
            $identity_status = 'not_submitted';
            $is_verified = false;
            $verification_data = [];
        }
    } else {
        // Identity verification columns don't exist yet - setup needed
        error_log("Upload Video: Identity verification columns not found. Database setup required.");
        $identity_status = 'not_submitted';
        $is_verified = false;
        $verification_data = [];
    }
} catch (Exception $e) {
    // Database error - fallback to requiring verification
    error_log("Upload Video: Database error during identity check: " . $e->getMessage());
    $identity_status = 'not_submitted';
    $is_verified = false;
    $verification_data = [];
}

// Get courses for dropdown
$courses = [];
$stmt = $conn->prepare("SELECT course_id, title FROM courses WHERE teacher_id = ? ORDER BY title");
if ($stmt) {
    $stmt->bind_param("i", $teacher_id);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($result) {
        $courses = $result->fetch_all(MYSQLI_ASSOC);
    }
    $stmt->close();
}

// Get categories for dropdown
$categories = [];
$category_stmt = $conn->prepare("SELECT category_id, category_name FROM categories ORDER BY category_name");
if ($category_stmt) {
    $category_stmt->execute();
    $result = $category_stmt->get_result();
    if ($result) {
        $categories = $result->fetch_all(MYSQLI_ASSOC);
    }
    $category_stmt->close();
}
?>

<style>
    /* Clean Professional Styling */
    body {
        background-color: #f5f5f5 !important;
    }
    
    .card {
        border: none !important;
        border-radius: 12px !important;
        box-shadow: 0 4px 6px rgba(0, 0, 0, 0.07) !important;
        background: white !important;
    }
    
    .form-control, .form-select {
        border-radius: 8px !important;
        border: 1px solid #e1e5e9 !important;
        padding: 12px 16px !important;
        font-size: 14px !important;
        background: white !important;
    }
    
    .btn {
        border-radius: 8px !important;
        padding: 10px 20px !important;
        font-weight: 500 !important;
        transition: all 0.2s ease !important;
    }
    
    .btn-primary {
        background: linear-gradient(135deg, #007bff 0%, #0056b3 100%) !important;
        border: none !important;
    }
    
    .upload-area {
        border: 2px dashed rgba(0, 123, 255, 0.4);
        border-radius: 16px;
        padding: 40px 20px;
        text-align: center;
        transition: all 0.3s ease;
        background-color: rgba(0, 123, 255, 0.03);
        cursor: pointer;
    }
    
    .upload-area:hover {
        border-color: #007bff;
        background-color: rgba(0, 123, 255, 0.08);
        transform: translateY(-2px);
        box-shadow: 0 10px 20px rgba(0, 0, 0, 0.05);
    }
    
    .upload-area.dragover {
        border-color: #28a745 !important;
        background-color: rgba(40, 167, 69, 0.1) !important;
        transform: translateY(-4px) !important;
        box-shadow: 0 15px 30px rgba(40, 167, 69, 0.2) !important;
    }
    
    .upload-area.file-selected {
        border-color: #28a745 !important;
        background-color: rgba(40, 167, 69, 0.05) !important;
    }
    
    /* Platform badges styling */
    .platform-badge {
        display: inline-flex;
        align-items: center;
        padding: 6px 12px;
        border-radius: 20px;
        font-size: 0.8rem;
        font-weight: 500;
        transition: all 0.3s ease;
        cursor: pointer;
        border: 2px solid transparent;
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
    }
    
    .platform-badge:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 8px rgba(0,0,0,0.15);
    }
    
    /* YouTube - Red */
    .youtube-badge {
        background: linear-gradient(135deg, #FF0000, #CC0000);
        color: white;
        border-color: #FF0000;
    }
    .youtube-badge:hover {
        background: linear-gradient(135deg, #CC0000, #AA0000);
        border-color: #CC0000;
    }
    
    /* Vimeo - Blue */
    .vimeo-badge {
        background: linear-gradient(135deg, #1AB7EA, #0E8FC7);
        color: white;
        border-color: #1AB7EA;
    }
    .vimeo-badge:hover {
        background: linear-gradient(135deg, #0E8FC7, #0A6B96);
        border-color: #0E8FC7;
    }
    
    /* Facebook - Blue */
    .facebook-badge {
        background: linear-gradient(135deg, #1877F2, #166FE5);
        color: white;
        border-color: #1877F2;
    }
    .facebook-badge:hover {
        background: linear-gradient(135deg, #166FE5, #1357D1);
        border-color: #166FE5;
    }
    
    /* TikTok - Black/Pink */
    .tiktok-badge {
        background: linear-gradient(135deg, #000000, #FF0050);
        color: white;
        border-color: #FF0050;
    }
    .tiktok-badge:hover {
        background: linear-gradient(135deg, #FF0050, #E6004A);
        border-color: #FF0050;
    }
    
    /* Instagram - Gradient */
    .instagram-badge {
        background: linear-gradient(135deg, #E4405F, #C13584, #833AB4);
        color: white;
        border-color: #E4405F;
    }
    .instagram-badge:hover {
        background: linear-gradient(135deg, #C13584, #833AB4, #5851DB);
        border-color: #C13584;
    }
    
    /* Twitter - Blue */
    .twitter-badge {
        background: linear-gradient(135deg, #1DA1F2, #0D8BD9);
        color: white;
        border-color: #1DA1F2;
    }
    .twitter-badge:hover {
        background: linear-gradient(135deg, #0D8BD9, #0A6FB8);
        border-color: #0D8BD9;
    }
    
    /* More - Gradient */
    .more-badge {
        background: linear-gradient(135deg, #6C757D, #495057);
        color: white;
        border-color: #6C757D;
    }
    .more-badge:hover {
        background: linear-gradient(135deg, #495057, #343A40);
        border-color: #495057;
    }
    
    /* Responsive design for platform badges */
    @media (max-width: 768px) {
        .platform-badge {
            font-size: 0.75rem;
            padding: 4px 8px;
        }
        
        .platform-badge span {
            display: none;
        }
        
        .platform-badge i {
            margin: 0 !important;
        }
    }
</style>

<!-- Upload Video Page Content -->
<div class="container-xl py-4">
    <div class="row">
        <div class="col-lg-8">
            <div class="card">
                <div class="card-header">
                    <h5 class="card-title mb-0"><i class="fas fa-film me-2 text-primary"></i>Video Upload</h5>
                </div>
                <div class="card-body">
                    <?php if (!$is_verified): ?>
                        <!-- IDENTITY VERIFICATION REQUIRED -->
                        <div class="text-center py-5">
                            <?php if ($identity_status === 'pending' && !empty($verification_data['identity_document_path'])): ?>
                                <!-- Document uploaded, waiting for approval -->
                                <div class="mb-4">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="64" height="64" fill="currentColor" class="bi bi-clock text-warning mb-3" viewBox="0 0 16 16">
                                        <path d="M8 3.5a.5.5 0 0 0-1 0V9a.5.5 0 0 0 .252.434l3.5 2a.5.5 0 0 0 .496-.868L8 8.71V3.5z"/>
                                        <path d="M8 16A8 8 0 1 0 8 0a8 8 0 0 0 0 16zm7-8A7 7 0 1 1 1 8a7 7 0 0 1 14 0z"/>
                                    </svg>
                                </div>
                                <h4 class="text-warning mb-3">🕐 Verification In Progress</h4>
                                <p class="text-muted mb-4">
                                    Your identity document was submitted on <strong><?php echo date('M j, Y', strtotime($verification_data['identity_uploaded_at'])); ?></strong>.<br>
                                    Our team is reviewing your documents. This usually takes 1-3 business days.
                                </p>
                                <div class="alert alert-info mb-4">
                                    <i class="bi bi-shield-check me-2"></i>
                                    Video upload will be enabled once your identity is verified.<br>
                                    You'll receive an email notification when verification is complete.
                                </div>
                            <?php elseif ($identity_status === 'rejected'): ?>
                                <!-- Document rejected, need to resubmit -->
                                <div class="mb-4">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="64" height="64" fill="currentColor" class="bi bi-exclamation-triangle text-danger mb-3" viewBox="0 0 16 16">
                                        <path d="M7.938 2.016A.13.13 0 0 1 8.002 2a.13.13 0 0 1 .063.016.146.146 0 0 1 .054.057l6.857 11.667c.036.06.035.124.002.183a.163.163 0 0 1-.054.06.116.116 0 0 1-.066.017H1.146a.115.115 0 0 1-.066-.017.163.163 0 0 1-.054-.06.176.176 0 0 1 .002-.183L7.884 2.073a.147.147 0 0 1 .054-.057zm1.044-.45a1.13 1.13 0 0 0-1.96 0L.165 13.233c-.457.778.091 1.767.98 1.767h13.713c.889 0 1.437-.99.98-1.767L8.982 1.566z"/>
                                        <path d="M7.002 12a1 1 0 1 1 2 0 1 1 0 0 1-2 0zM7.1 5.995a.905.905 0 1 1 1.8 0l-.35 3.507a.552.552 0 0 1-1.1 0L7.1 5.995z"/>
                                    </svg>
                                </div>
                                <h4 class="text-danger mb-3">❌ Verification Required</h4>
                                <p class="text-muted mb-4">
                                    Your identity verification was not approved. Please upload new documents for review.
                                </p>
                                <div class="alert alert-warning mb-4">
                                    <i class="bi bi-upload me-2"></i>
                                    <strong>Action Required:</strong> Upload clear, valid identity documents to enable video uploads.
                                </div>
                                <a href="profile.php#identity-verification" class="btn btn-warning btn-lg">
                                    <i class="bi bi-upload me-2"></i>Upload New Documents
                                </a>
                            <?php else: ?>
                                <!-- No document submitted yet -->
                                <div class="mb-4">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="64" height="64" fill="currentColor" class="bi bi-shield-exclamation text-primary mb-3" viewBox="0 0 16 16">
                                        <path d="M5.338 1.59a61.44 61.44 0 0 0-2.837.856.481.481 0 0 0-.328.39c-.554 4.157.726 7.19 2.253 9.188a10.725 10.725 0 0 0 2.287 2.233c.346.244.652.42.893.533.12.057.218.095.293.118a.55.55 0 0 0 .101.025.615.615 0 0 0 .1-.025c.076-.023.174-.061.294-.118.24-.113.547-.29.893-.533a10.726 10.726 0 0 0 2.287-2.233c1.527-1.997 2.807-5.031 2.253-9.188a.48.48 0 0 0-.328-.39c-.651-.213-1.75-.56-2.837-.855C9.552 1.29 8.531 1.067 8 1.067c-.53 0-1.552.223-2.662.524zM5.072.56C6.157.265 7.31 0 8 0s1.843.265 2.928.56c1.11.3 2.229.655 2.887.87a1.54 1.54 0 0 1 1.044 1.262c.596 4.477-.787 7.795-2.465 9.99a11.775 11.775 0 0 1-2.517 2.453 7.159 7.159 0 0 1-1.048.625c-.28.132-.581.24-.829.24s-.548-.108-.829-.24a7.158 7.158 0 0 1-1.048-.625 11.777 11.777 0 0 1-2.517-2.453C1.928 10.487.545 7.169 1.141 2.692A1.54 1.54 0 0 1 2.185 1.43 62.456 62.456 0 0 1 5.072.56z"/>
                                        <path d="M7.001 11a1 1 0 1 1 2 0 1 1 0 0 1-2 0zM7.1 4.995a.905.905 0 1 1 1.8 0l-.35 3.507a.553.553 0 0 1-1.1 0L7.1 4.995z"/>
                                    </svg>
                                </div>
                                <h4 class="text-warning mb-3">🆔 Identity Verification Required</h4>
                                <p class="text-muted mb-4">
                                    To maintain trust and security on our platform, teachers must verify their identity before uploading videos.<br>
                                    Please upload a government-issued ID document to get started.
                                </p>
                                <div class="alert alert-info mb-4">
                                    <i class="bi bi-shield-check me-2"></i>
                                    <strong>Accepted Documents:</strong> National ID, Passport, Driver's License
                                </div>
                                <a href="profile.php#identity-verification" class="btn btn-primary btn-lg">
                                    <i class="bi bi-upload me-2"></i>Upload Identity Document
                                </a>
                            <?php endif; ?>
                        </div>
                    <?php else: ?>
                        <!-- VERIFIED TEACHER - SHOW VIDEO UPLOAD FORM -->
                        <div class="alert alert-success mb-4">
                            <i class="bi bi-shield-check-fill me-2"></i>
                            <strong>Identity Verified!</strong> You can now upload videos to your courses.
                        </div>
                    
                    <form id="videoUploadForm" method="post" action="process-video-with-content.php" enctype="multipart/form-data" class="needs-validation" novalidate>
                        <!-- Hidden fields -->
                        <input type="hidden" name="form_type" value="video_upload">
                        <input type="hidden" name="teacher_id" value="<?php echo $teacher_id; ?>">
                        <input type="hidden" id="detectedVideoUrl" name="video_url" value="">
                        <input type="hidden" id="detectedPlatformInput" name="detected_platform" value="">
                        <input type="hidden" id="videoIdExternal" name="video_id_external" value="">
                        <input type="hidden" id="embedCodeInput" name="embed_code" value="">
                        <input type="hidden" id="detectedDurationInput" name="detected_duration" value="">
                        
                        <!-- Video Type Selection -->
                        <div class="mb-4">
                            <label class="form-label fw-bold mb-3">
                                <i class="fas fa-video me-2 text-primary"></i>Video Source
                            </label>
                            <div class="btn-group w-100 mb-3" role="group">
                                <input type="radio" class="btn-check" name="video_type" id="videoTypeFile" value="file" checked>
                                <label class="btn btn-outline-primary" for="videoTypeFile">
                                    <i class="fas fa-cloud-upload-alt me-2"></i>Upload File
                                </label>
                                
                                <input type="radio" class="btn-check" name="video_type" id="videoTypeUrl" value="url">
                                <label class="btn btn-outline-primary" for="videoTypeUrl">
                                    <i class="fas fa-globe me-2"></i>Embed URL
                                </label>
                            </div>
                        </div>

                        <!-- Video File Upload -->
                        <div class="mb-4" id="fileUploadSection">
                            <label class="form-label fw-bold mb-3">
                                <i class="fas fa-file-video me-2 text-primary"></i>Video File
                            </label>
                            <div class="upload-area" id="uploadArea" onclick="document.getElementById('videoFile').click();">
                                <div class="upload-content">
                                    <i class="fas fa-cloud-upload-alt fa-3x text-primary mb-3"></i>
                                    <h5 class="fw-bold mb-2">Drag & Drop your video here</h5>
                                    <p class="text-muted mb-3">or click to browse</p>
                                    <button type="button" class="btn btn-primary btn-lg px-4" id="browseBtn" onclick="event.stopPropagation(); document.getElementById('videoFile').click();">
                                        <i class="fas fa-folder-open me-2"></i>Browse Files
                                    </button>
                                </div>
                                <div class="upload-formats mt-3">
                                    <small class="text-muted">
                                        <i class="fas fa-info-circle me-1"></i>
                                        Supported: MP4, WebM, MOV, AVI • Max size: 500MB
                                    </small>
                                </div>
                            </div>
                            <!-- CRITICAL: Keep file input outside upload area so it doesn't get replaced -->
                            <input type="file" id="videoFile" name="video" accept="video/*" style="display: none;" onchange="console.log('🔍 File input changed, calling handleFileSelection...'); handleFileSelection(this)">
                            
                            <!-- Video Preview -->
                            <div class="preview-container mt-4" id="previewContainer" style="display: none;">
                                <h6 class="mb-2 fw-bold">Video Preview</h6>
                                <video id="videoPreview" class="w-100" controls style="max-height: 300px;"></video>
                            </div>
                        </div>

                        <!-- Video URL Input -->
                        <div class="mb-4" id="urlEmbedSection" style="display: none;">
                            <label class="form-label fw-bold mb-3">
                                <i class="fas fa-link me-2 text-primary"></i>Video URL
                            </label>
                            <div class="input-group input-group-lg">
                                <span class="input-group-text bg-light">
                                    <i class="fas fa-globe text-muted"></i>
                                </span>
                                <input type="url" class="form-control" id="videoUrl" name="video_url" 
                                       placeholder="Paste your video URL here (YouTube, Vimeo, TikTok, etc.)">
                                <button type="button" class="btn btn-primary btn-lg px-4" id="detectVideoBtn">
                                    <i class="fas fa-magic me-2"></i>Detect Video
                                </button>
                            </div>
                            
                            <!-- Supported platforms -->
                            <div class="mt-3">
                                <div class="d-flex align-items-center mb-2">
                                    <small class="text-muted me-3">Supported platforms:</small>
                                    <div class="d-flex flex-wrap gap-2">
                                        <!-- YouTube -->
                                        <div class="platform-badge youtube-badge">
                                            <i class="fab fa-youtube me-1"></i>
                                            <span>YouTube</span>
                                        </div>
                                        
                                        <!-- Vimeo - COMMENTED OUT -->
                                        <!-- <div class="platform-badge vimeo-badge">
                                            <i class="fab fa-vimeo-v me-1"></i>
                                            <span>Vimeo</span>
                                        </div> -->
                                        
                                        <!-- Facebook - COMMENTED OUT -->
                                        <!-- <div class="platform-badge facebook-badge">
                                            <i class="fab fa-facebook me-1"></i>
                                            <span>Facebook</span>
                                        </div> -->
                                        
                                        <!-- TikTok - COMMENTED OUT -->
                                        <!-- <div class="platform-badge tiktok-badge">
                                            <i class="fab fa-tiktok me-1"></i>
                                            <span>TikTok</span>
                                        </div> -->
                                        
                                        <!-- Instagram - COMMENTED OUT -->
                                        <!-- <div class="platform-badge instagram-badge">
                                            <i class="fab fa-instagram me-1"></i>
                                            <span>Instagram</span>
                                        </div> -->
                                        
                                        <!-- Twitter - COMMENTED OUT -->
                                        <!-- <div class="platform-badge twitter-badge">
                                            <i class="fab fa-twitter me-1"></i>
                                            <span>Twitter/X</span>
                                        </div> -->
                                        
                                        <!-- And More - COMMENTED OUT -->
                                        <!-- <div class="platform-badge more-badge">
                                            <i class="fas fa-plus me-1"></i>
                                            <span>& more</span>
                                        </div> -->
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Video Preview Area -->
                            <div id="videoPreviewArea" style="display: none;"></div>
                        </div>

                        <!-- Duration Detection Display -->
                        <div class="mb-4" id="durationDetectionSection" style="display: none;">
                            <div class="card border-info">
                                <div class="card-header bg-info text-white">
                                    <h6 class="card-title mb-0">
                                        <i class="fas fa-clock me-2"></i>Duration Detection
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <div id="durationStatus" class="d-flex align-items-center">
                                        <div class="spinner-border spinner-border-sm text-primary me-2" role="status" id="durationSpinner">
                                            <span class="visually-hidden">Loading...</span>
                                        </div>
                                        <span id="durationText">Detecting video duration...</span>
                                    </div>
                                    
                                    <div id="durationResult" class="mt-3" style="display: none;">
                                        <div class="row">
                                            <div class="col-md-6">
                                                <strong>Detected Duration:</strong>
                                                <div class="badge bg-success fs-6 ms-2" id="detectedDuration">--:--</div>
                                            </div>
                                            <div class="col-md-6">
                                                <strong>Detection Method:</strong>
                                                <div class="badge bg-secondary fs-6 ms-2" id="detectionMethod">Unknown</div>
                                            </div>
                                        </div>
                                        
                                        <div class="row mt-2">
                                            <div class="col-12">
                                                <strong>Status:</strong>
                                                <div class="badge bg-info fs-6 ms-2" id="detectionStatus">Processing</div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div id="durationError" class="mt-3 alert alert-warning" style="display: none;">
                                        <i class="fas fa-exclamation-triangle me-2"></i>
                                        <strong>Duration Detection Failed:</strong>
                                        <div id="errorMessage">Could not detect video duration.</div>
                                        <small class="d-block mt-1">Please enter the correct duration manually below.</small>
                                    </div>
                                    
                                    <!-- Manual Duration Input Section -->
                                    <div id="manualDurationSection" class="mt-3" style="display: none;">
                                        <div class="alert alert-info">
                                            <div class="row align-items-center">
                                                <div class="col-12">
                                                    <h6 class="alert-heading mb-2">
                                                        <i class="fas fa-clock me-2"></i>Manual Duration Entry Required
                                                    </h6>
                                                    <p class="mb-2"><strong>📚 Why is this important?</strong></p>
                                                    <ul class="mb-3 small">
                                                        <li><strong>Student Progress Tracking:</strong> Accurate duration helps track how much of the video students have watched</li>
                                                        <li><strong>Learning Analytics:</strong> Essential for measuring engagement and completion rates</li>
                                                        <li><strong>YouTube Privacy:</strong> Automated detection is limited due to YouTube's privacy protections</li>
                                                    </ul>
                                                </div>
                                            </div>
                                            
                                            <div class="row align-items-center">
                                                <div class="col-md-6">
                                                    <label for="manualDuration" class="form-label fw-bold">
                                                        <i class="fas fa-stopwatch me-1"></i>Enter Video Duration:
                                                    </label>
                                                    <div class="input-group">
                                                        <span class="input-group-text"><i class="fas fa-clock"></i></span>
                                                        <input type="text" 
                                                               class="form-control" 
                                                               id="manualDuration" 
                                                               name="manual_duration"
                                                               placeholder="e.g., 19:45 or 1:23:30" 
                                                               pattern="^(\d+:)?[0-5]?\d:[0-5]?\d$"
                                                               title="Format: MM:SS or HH:MM:SS (e.g., 19:45 or 1:23:30)">
                                                        <button type="button" class="btn btn-success" id="validateDurationBtn">
                                                            <i class="fas fa-check me-1"></i>Validate
                                                        </button>
                                                    </div>
                                                    <div class="form-text">
                                                        <strong>Format:</strong> MM:SS (e.g., 19:45) or HH:MM:SS (e.g., 1:23:30)
                                                    </div>
                                                    <div class="invalid-feedback">
                                                        Duration is required for YouTube videos to ensure accurate student progress tracking.
                                                    </div>
                                                </div>
                                                
                                                <div class="col-md-6">
                                                    <div class="bg-light p-3 rounded">
                                                        <h6 class="mb-2"><i class="fas fa-lightbulb text-warning me-1"></i>Quick Tips:</h6>
                                                        <ul class="mb-0 small">
                                                            <li>Check the video player's total duration</li>
                                                            <li>Round to the nearest second</li>
                                                            <li>Double-check for accuracy</li>
                                                        </ul>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <div id="manualValidationResult" class="mt-2" style="display: none;"></div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Video Details -->
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="videoTitle" class="form-label">Video Title <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="videoTitle" name="title" required>
                                    <div class="invalid-feedback">Please provide a video title.</div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="courseSelect" class="form-label">Course <span class="text-danger">*</span></label>
                                    <select class="form-select" id="courseSelect" name="course_id" required>
                                        <option value="">Select a course</option>
                                        <?php foreach ($courses as $course): ?>
                                            <option value="<?php echo $course['course_id']; ?>">
                                                <?php echo htmlspecialchars($course['title']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <div class="invalid-feedback">Please select a course for this video.</div>
                                </div>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label for="videoDescription" class="form-label">Description</label>
                            <textarea class="form-control" id="videoDescription" name="description" rows="4"></textarea>
                        </div>

                        <!-- Thumbnail Upload Section -->
                        <div id="thumbnailUploadSection" class="mb-3">
                            <label for="thumbnailFile" class="form-label">Thumbnail (Optional)</label>
                            <input type="file" class="form-control" id="thumbnailFile" name="thumbnail" accept="image/*">
                            <div class="form-text">Upload a custom thumbnail image (recommended: 1280x720)</div>
                            
                            <!-- Thumbnail Preview -->
                            <div id="thumbnailPreview" style="display: none;" class="mt-2">
                                <img id="thumbnailImg" src="" alt="Thumbnail Preview" class="img-thumbnail" style="max-width: 200px;">
                            </div>
                        </div>
                        
                        <!-- Auto Thumbnail Notice -->
                        <div id="autoThumbnailNotice" class="mb-3" style="display: none;">
                            <div class="alert alert-info">
                                <i class="fas fa-magic me-2"></i>
                                <strong>Platform Handles Thumbnails:</strong> <span id="thumbnailNoticeText">This platform provides video thumbnails automatically. No manual upload needed!</span>
                            </div>
                        </div>

                        <!-- Category Selection -->
                        <div class="mb-3">
                            <label for="categorySelect" class="form-label">Category</label>
                            <select class="form-select" id="categorySelect" name="category_name">
                                <option value="">Select a category</option>
                                <?php foreach ($categories as $category): ?>
                                    <option value="<?php echo htmlspecialchars($category['category_name']); ?>">
                                        <?php echo htmlspecialchars($category['category_name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <!-- Visibility Settings -->
                        <div class="mb-3">
                            <label class="form-label">Visibility</label>
                            <div class="d-flex gap-3">
                                <div class="form-check">
                                    <input class="form-check-input" type="radio" name="is_published" id="publishedYes" value="1" checked>
                                    <label class="form-check-label" for="publishedYes">
                                        <i class="fas fa-eye me-1"></i>Published
                                    </label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="radio" name="is_published" id="publishedNo" value="0">
                                    <label class="form-check-label" for="publishedNo">
                                        <i class="fas fa-eye-slash me-1"></i>Draft
                                    </label>
                                </div>
                            </div>
                        </div>

                        <!-- Additional Learning Materials -->
                        <div class="mb-4">
                            <label class="form-label fw-bold">Additional Learning Materials</label>
                            <div class="form-text mb-3">Select the types of learning materials you want to create for this video. After uploading, you'll be guided through creating each selected item.</div>
                            
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="card border-0 bg-light">
                                        <div class="card-body p-3">
                                            <div class="form-check">
                                                <input class="form-check-input" type="checkbox" name="content_types[]" value="quiz" id="contentQuiz">
                                                <label class="form-check-label" for="contentQuiz">
                                                    <i class="fas fa-question-circle text-primary me-2"></i>
                                                    <strong>Interactive Quiz</strong><br>
                                                    <small class="text-muted">Test student understanding</small>
                                                </label>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Form Actions -->
                        <div class="d-flex justify-content-between mt-4">
                            <button type="button" class="btn btn-outline-secondary" id="resetBtn">Reset Form</button>
                            <button type="submit" class="btn btn-primary" id="submitBtn">
                                <i class="fas fa-cloud-upload me-2"></i>Upload Video
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Upload Tips -->
        <div class="col-lg-4">
            <div class="card shadow-sm">
                <div class="card-header bg-gradient text-white" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-lightbulb me-2"></i>Upload Guidelines
                    </h5>
                    <small class="opacity-75">Tips for successful video uploads</small>
                </div>
                <div class="card-body p-0">
                    
                    <!-- File Upload Tips -->
                    <div class="p-3 border-bottom">
                        <h6 class="fw-bold text-primary mb-3">
                            <i class="fas fa-file-video me-2"></i>File Uploads
                        </h6>
                        <div class="row g-2">
                            <div class="col-12">
                                <div class="d-flex align-items-center mb-2">
                                    <div class="bg-success bg-opacity-10 rounded-circle p-2 me-3">
                                        <i class="fas fa-check text-success"></i>
                                    </div>
                                    <div>
                                        <small class="fw-semibold">Supported Formats</small><br>
                                        <small class="text-muted">MP4, WebM, MOV, AVI (Max: 500MB)</small>
                                    </div>
                                </div>
                            </div>
                            <div class="col-12">
                                <div class="d-flex align-items-center mb-2">
                                    <div class="bg-info bg-opacity-10 rounded-circle p-2 me-3">
                                        <i class="fas fa-video text-info"></i>
                                    </div>
                                    <div>
                                        <small class="fw-semibold">Quality Settings</small><br>
                                        <small class="text-muted">1080p+ recommended, 720p minimum</small>
                                    </div>
                                </div>
                            </div>
                            <div class="col-12">
                                <div class="d-flex align-items-center">
                                    <div class="bg-warning bg-opacity-10 rounded-circle p-2 me-3">
                                        <i class="fas fa-clock text-warning"></i>
                                    </div>
                                    <div>
                                        <small class="fw-semibold">Duration Detection</small><br>
                                        <small class="text-muted">Automatic for file uploads</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- URL Upload Tips -->
                    <div class="p-3 border-bottom">
                        <h6 class="fw-bold text-danger mb-3">
                            <i class="fas fa-globe me-2"></i>Video URLs
                        </h6>
                        <div class="row g-2">
                            <div class="col-12">
                                <div class="d-flex align-items-center mb-2">
                                    <div class="bg-success bg-opacity-10 rounded-circle p-2 me-3">
                                        <i class="fab fa-youtube text-danger"></i>
                                    </div>
                                    <div>
                                        <small class="fw-semibold">Supported Platforms</small><br>
                                        <small class="text-muted">YouTube only</small>
                                    </div>
                                </div>
                            </div>
                            <div class="col-12">
                                <div class="d-flex align-items-center mb-2">
                                    <div class="bg-warning bg-opacity-10 rounded-circle p-2 me-3">
                                        <i class="fas fa-stopwatch text-warning"></i>
                                    </div>
                                    <div>
                                        <small class="fw-semibold">Duration Required</small><br>
                                        <small class="text-muted">Manual entry needed (MM:SS format)</small>
                                    </div>
                                </div>
                            </div>
                            <div class="col-12">
                                <div class="d-flex align-items-center">
                                    <div class="bg-info bg-opacity-10 rounded-circle p-2 me-3">
                                        <i class="fas fa-image text-info"></i>
                                    </div>
                                    <div>
                                        <small class="fw-semibold">Thumbnails</small><br>
                                        <small class="text-muted">Auto-generated (no upload needed)</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Best Practices -->
                    <div class="p-3 border-bottom">
                        <h6 class="fw-bold text-success mb-3">
                            <i class="fas fa-star me-2"></i>Best Practices
                        </h6>
                        <div class="row g-2">
                            <div class="col-12">
                                <div class="d-flex align-items-center mb-2">
                                    <div class="bg-primary bg-opacity-10 rounded-circle p-2 me-3">
                                        <i class="fas fa-heading text-primary"></i>
                                    </div>
                                    <div>
                                        <small class="fw-semibold">Clear Titles</small><br>
                                        <small class="text-muted">Help students find content easily</small>
                                    </div>
                                </div>
                            </div>
                            <div class="col-12">
                                <div class="d-flex align-items-center mb-2">
                                    <div class="bg-secondary bg-opacity-10 rounded-circle p-2 me-3">
                                        <i class="fas fa-align-left text-secondary"></i>
                                    </div>
                                    <div>
                                        <small class="fw-semibold">Good Descriptions</small><br>
                                        <small class="text-muted">Improve searchability & engagement</small>
                                    </div>
                                </div>
                            </div>
                            <div class="col-12">
                                <div class="d-flex align-items-center">
                                    <div class="bg-success bg-opacity-10 rounded-circle p-2 me-3">
                                        <i class="fas fa-folder-open text-success"></i>
                                    </div>
                                    <div>
                                        <small class="fw-semibold">Course Organization</small><br>
                                        <small class="text-muted">Select appropriate course category</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Duration Guide -->
                    <div class="p-3 bg-light">
                        <h6 class="fw-bold text-info mb-2">
                            <i class="fas fa-question-circle me-2"></i>Duration Format Guide
                        </h6>
                        <div class="row g-1">
                            <div class="col-6">
                                <div class="text-center p-2 bg-white rounded border">
                                    <code class="text-success fw-bold">5:30</code><br>
                                    <small class="text-muted">5 min 30 sec</small>
                                </div>
                            </div>
                            <div class="col-6">
                                <div class="text-center p-2 bg-white rounded border">
                                    <code class="text-success fw-bold">1:23:45</code><br>
                                    <small class="text-muted">1 hr 23 min 45 sec</small>
                                </div>
                            </div>
                        </div>
                        <div class="alert alert-info mt-2 mb-0 py-2">
                            <small>
                                <i class="fas fa-info-circle me-1"></i>
                                <strong>Why duration matters:</strong> Enables accurate student progress tracking and completion analytics.
                            </small>
                        </div>
                    </div>
                    
                    <?php endif; ?>
                </div>
            </div>
        </div>

    </div>

</div>

<!-- Scripts -->
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

<!-- NEW: Video API Integration for TikTok and Vimeo duration detection -->
<script src="../js/video-api-integration.js"></script>

<script>
// Global function for handling file selection (inline handler)
function handleFileSelection(input) {
    const file = input.files[0];
    console.log('🔍 handleFileSelection called:', {
        input: input,
        inputId: input.id,
        hasFiles: !!input.files,
        fileCount: input.files ? input.files.length : 0,
        file: file ? file.name : 'NO FILE'
    });
    
    if (file) {
        console.log('✅ File selected:', file.name, file.size, file.type);
        
        // Validate file type
        const validTypes = ['video/mp4', 'video/webm', 'video/quicktime', 'video/x-msvideo', 'video/avi', 'video/mov'];
        if (!validTypes.includes(file.type.toLowerCase())) {
            alert('❌ Invalid file type. Please select MP4, WebM, MOV, or AVI.');
            input.value = '';
            return;
        }
        
        // Validate file size (100MB limit)
        if (file.size > 100 * 1024 * 1024) {
            alert('❌ File is too large. Maximum size is 100MB.');
            input.value = '';
            return;
        }
        
        // Show preview
        const videoURL = URL.createObjectURL(file);
        $('#videoPreview').attr('src', videoURL);
        $('#previewContainer').show();
        
        // Store file reference globally for form submission
        window.selectedVideoFile = file;
        
        // Update UI to show file is selected - DON'T replace file input!
        $('#uploadArea').addClass('file-selected').html(`
            <i class="fas fa-check-circle fa-3x text-success mb-3"></i>
            <h5 class="fw-bold mb-2 text-success">File Selected: ${file.name}</h5>
            <p class="text-muted mb-3">Size: ${(file.size / (1024*1024)).toFixed(2)} MB</p>
            <button type="button" class="btn btn-outline-primary" onclick="resetUploadArea()">Change File</button>
        `);
        
        // 🎬 TRIGGER DURATION DETECTION FOR FILE UPLOAD
        detectFileDuration(file);
        
        // CRITICAL: Verify the file is actually set in the input
        setTimeout(() => {
            const verifyElement = document.getElementById('videoFile');
            const verifyFile = verifyElement && verifyElement.files ? verifyElement.files[0] : null;
            console.log('🔍 File verification after UI update:', {
                elementFound: !!verifyElement,
                fileStillThere: !!verifyFile,
                fileName: verifyFile ? verifyFile.name : 'NO FILE'
            });
        }, 100);
        
    } else {
        console.log('❌ No file provided to handleFileSelection');
    }
}

// Global function to reset upload area
function resetUploadArea() {
    // Clear global file reference
    window.selectedVideoFile = null;
    
    // Reset the file input value - DON'T recreate the element!
    const fileInput = document.getElementById('videoFile');
    if (fileInput) {
        fileInput.value = '';
    }
    
    $('#uploadArea').removeClass('file-selected').html(`
        <i class="fas fa-cloud-upload-alt fa-3x text-primary mb-3"></i>
        <h5 class="fw-bold mb-2">Drag & Drop your video here</h5>
        <p class="text-muted mb-3">or click to browse</p>
        <button type="button" class="btn btn-primary" onclick="document.getElementById('videoFile').click();">Browse Files</button>
    `).attr('onclick', "document.getElementById('videoFile').click();");
    
    $('#previewContainer').hide();
    console.log('📄 Upload area reset');
}

// 🎯 FIX: Handle file selection and trigger duration detection
function handleFileSelection(input) {
    console.log('🎯 handleFileSelection called with input:', input);
    
    if (input.files && input.files[0]) {
        const file = input.files[0];
        console.log('📁 File selected:', file.name, 'Size:', file.size, 'Type:', file.type);
        
        // Validate file type
        if (!file.type.startsWith('video/')) {
            console.log('❌ Invalid file type:', file.type);
            alert('❌ Please select a valid video file.');
            input.value = '';
            return;
        }
        
        console.log('✅ File validation passed, proceeding with processing...');
        
        // Show video preview
        const reader = new FileReader();
        reader.onload = function(e) {
            console.log('📖 FileReader loaded video data');
            const videoPreview = document.getElementById('videoPreview');
            const previewContainer = document.getElementById('previewContainer');
            
            if (videoPreview && previewContainer) {
                videoPreview.src = e.target.result;
                previewContainer.style.display = 'block';
                console.log('🎬 Video preview set and container shown');
                
                // Also create a video element for duration detection
                videoPreview.onloadedmetadata = function() {
                    console.log('📊 Video metadata loaded for preview');
                };
            }
        };
        reader.readAsDataURL(file);
        
        // Trigger duration detection
        console.log('⏱️ Calling detectFileDuration...');
        detectFileDuration(file);
        
        // Update upload area with file information
        console.log('🔄 Calling updateUploadAreaWithFile...');
        updateUploadAreaWithFile(file);
        
        console.log('✅ handleFileSelection completed');
    } else {
        console.log('❌ No file selected or files array empty');
    }
}

// Function to update upload area when file is selected
function updateUploadAreaWithFile(file) {
    const uploadArea = document.getElementById('uploadArea');
    if (uploadArea) {
        uploadArea.innerHTML = `
            <div class="upload-content">
                <i class="fas fa-check-circle fa-3x text-success mb-3"></i>
                <h5 class="fw-bold mb-2">File Selected</h5>
                <p class="text-muted mb-3">${file.name}</p>
                <p class="text-muted mb-3">Size: ${(file.size / (1024 * 1024)).toFixed(2)} MB</p>
                <button type="button" class="btn btn-outline-primary btn-lg px-4" onclick="document.getElementById('videoFile').click();">
                    <i class="fas fa-exchange-alt me-2"></i>Change File
                </button>
            </div>
        `;
        console.log('📁 Upload area updated with file info');
    }
}

$(document).ready(function() {
    console.log('🎥 Video Upload System initialized');
    
    // Debug: Form submission testing
    console.log('Debug: Form submission debugging enabled');
    
    // Show emergency login option
    if (window.location.search.indexOf('emergency_login') === -1) {
        console.log('💡 TIP: Try adding ?emergency_login=teacher to URL for testing');
    }
    
    // Video type switching
    $('input[name="video_type"]').on('change', function() {
        if ($(this).val() === 'url') {
            $('#fileUploadSection').hide();
            $('#urlEmbedSection').show();
            $('#videoFile').prop('required', false);
            $('#videoUrl').prop('required', true);
        } else {
            $('#fileUploadSection').show();
            $('#urlEmbedSection').hide();
            $('#videoFile').prop('required', true);
            $('#videoUrl').prop('required', false);
        }
    });
    
    // Add drag and drop functionality to upload area
    $('#uploadArea').on('dragover', function(e) {
        e.preventDefault();
        e.stopPropagation();
        $(this).addClass('dragover');
    });
    
    $('#uploadArea').on('dragleave', function(e) {
        e.preventDefault();
        e.stopPropagation();
        $(this).removeClass('dragover');
    });
    
    $('#uploadArea').on('drop', function(e) {
        e.preventDefault();
        e.stopPropagation();
        $(this).removeClass('dragover');
        
        const files = e.originalEvent.dataTransfer.files;
        if (files.length > 0) {
            const file = files[0];
            
            // Set the file to the input element safely
            const videoFileElement = document.getElementById('videoFile');
            if (videoFileElement) {
                const dt = new DataTransfer();
                dt.items.add(file);
                videoFileElement.files = dt.files;
                
                // Trigger the onchange handler
                handleFileSelection(videoFileElement);
            } else {
                console.error('❌ Video file input element not found');
                alert('❌ Upload area not properly initialized. Please refresh the page.');
            }
        }
    });
    
    // Video detection
    $('#detectVideoBtn').on('click', function() {
        const videoUrl = $('#videoUrl').val().trim();
        if (!videoUrl) {
            alert('Please enter a video URL first.');
            return;
        }
        
        const btn = $(this);
        const originalText = btn.html();
        btn.html('<i class="fas fa-spinner fa-spin me-2"></i>Detecting...').prop('disabled', true);
        
        $.ajax({
            url: 'ajax/detect_video.php',
            method: 'POST',
            dataType: 'json',
            data: { video_url: videoUrl },
            success: function(response) {
                if (response.success && response.data && response.data.is_valid) {
                    showVideoPreview(response.data);
                    populateFormFields(response.data);
                    
                    // 🎬 ENHANCED DURATION DETECTION FOR URL VIDEO
                    const supportedPlatforms = ['youtube', 'vimeo', 'tiktok', 'dailymotion', 'twitch', 'streamable'];
                    if (supportedPlatforms.includes(response.data.platform)) {
                        // Use API integration for all supported platforms
                        detectDurationWithAPI(videoUrl, response.data.platform);
                    } else {
                        showDurationError(`Duration detection not yet supported for ${response.data.platform}`);
                    }
                    
                    alert('✅ Video detected successfully!');
                } else {
                    alert('❌ ' + (response.message || 'Video detection failed'));
                }
            },
            error: function() {
                alert('❌ Error detecting video. Please try again.');
            },
            complete: function() {
                btn.html(originalText).prop('disabled', false);
            }
        });
    });
    
    // Show video preview with embedded content for platforms without direct thumbnails
    function showVideoPreview(videoData) {
        const platform = videoData.platform?.toLowerCase() || '';
        const hasDirectThumbnail = platformProvidesDirectThumbnails(platform);
        const hasEmbedCode = videoData.embed_code && videoData.embed_code.trim().length > 0;
        
        // Build preview content based on what's available
        let previewContent = '';
        
        if (hasDirectThumbnail && videoData.thumbnail_url) {
            // Show thumbnail image for platforms like YouTube, Vimeo
            previewContent = `
                <img src="${videoData.thumbnail_url}" class="img-fluid rounded mb-2" style="max-height: 160px; max-width: 100%;">
            `;
        } else if (hasEmbedCode) {
            // Show embedded video for platforms like TikTok, Instagram, Facebook
            previewContent = `
                <div class="embed-responsive mb-2" style="max-height: 200px; overflow: hidden;">
                    <div class="rounded" style="transform: scale(0.8); transform-origin: top left; width: 125%; height: 125%;">
                        ${videoData.embed_code}
                    </div>
                </div>
                <div class="small text-muted mb-2">
                    <i class="fas fa-info-circle me-1"></i>
                    Preview scaled for display - actual video will embed at full size
                </div>
            `;
        } else {
            // Fallback when no visual content is available
            previewContent = `
                <div class="text-center py-3 bg-light rounded mb-2">
                    <i class="fas fa-video fa-2x text-primary mb-2"></i>
                    <div class="text-muted">Video detected successfully</div>
                    <div class="small text-muted">Preview not available for this platform</div>
                </div>
            `;
        }
        
        const previewHtml = `
            <div class="card border-success mt-3">
                <div class="card-header bg-success text-white">
                    <h6 class="mb-0">
                        <i class="fas fa-video me-2"></i>
                        ${videoData.platform_name} Video Detected
                    </h6>
                </div>
                <div class="card-body">
                    ${previewContent}
                    ${videoData.title ? `<h6 class="text-primary mb-2">${videoData.title}</h6>` : ''}
                    <div class="d-flex justify-content-between align-items-center">
                        <small class="text-muted">Video ID: <code>${videoData.video_id}</code></small>
                        ${videoData.duration ? `<small class="text-success"><i class="fas fa-clock me-1"></i>${videoData.duration}</small>` : ''}
                    </div>
                </div>
            </div>
        `;
        $('#videoPreviewArea').html(previewHtml).show();
    }
    
    // Populate form fields
    function populateFormFields(videoData) {
        $('#detectedVideoUrl').val(videoData.original_url || '');
        $('#detectedPlatformInput').val(videoData.platform || '');
        $('#videoIdExternal').val(videoData.video_id || '');
        $('#embedCodeInput').val(videoData.embed_code || '');
        
        // IMPORTANT: Also populate the videoUrl field for validation
        if (videoData.original_url) {
            $('#videoUrl').val(videoData.original_url);
        }
        
        if (videoData.title && !$('#videoTitle').val()) {
            $('#videoTitle').val(videoData.title);
        }
        
        // Update thumbnail section based on platform capabilities
        if (videoData.platform) {
            updateThumbnailSection(videoData.platform, videoData);
        }
        
        console.log('Form fields populated:', {
            videoUrl: $('#videoUrl').val(),
            videoTitle: $('#videoTitle').val(),
            courseId: $('#courseSelect').val()
        });
    }
    
    // Form submission
    $('#videoUploadForm').on('submit', function(e) {
        e.preventDefault();
        
        const videoType = $('input[name="video_type"]:checked').val();
        
        // Safely get the video file element and its files
        const videoFileElement = document.getElementById('videoFile');
        let videoFile = videoFileElement && videoFileElement.files ? videoFileElement.files[0] : null;
        
        // Fallback to globally stored file if form input doesn't have it
        if (!videoFile && window.selectedVideoFile) {
            console.log('🔄 Using globally stored file as fallback');
            videoFile = window.selectedVideoFile;
        }
        const videoUrl = $('#videoUrl').val().trim();
        
        // Debug: Log form state before validation
        console.log('🔍 FORM SUBMISSION DEBUG:', {
            videoType: videoType,
            videoUrl: videoUrl,
            videoTitle: $('#videoTitle').val(),
            courseId: $('#courseSelect').val(),
            videoFileElement: videoFileElement ? 'FOUND' : 'NOT FOUND',
            videoFileElementId: videoFileElement ? videoFileElement.id : 'NO ID',
            videoFileElementFiles: videoFileElement ? videoFileElement.files : 'NO FILES PROPERTY',
            videoFileElementFilesLength: videoFileElement && videoFileElement.files ? videoFileElement.files.length : 0,
            videoFile: videoFile ? videoFile.name : 'NO FILE',
            videoFileSize: videoFile ? videoFile.size : 0,
            uploadAreaClasses: $('#uploadArea').attr('class'),
            uploadAreaHasFileSelectedClass: $('#uploadArea').hasClass('file-selected')
        });
        
        // Custom validation for video upload
        if (videoType === 'file') {
            console.log('🔍 FILE VALIDATION CHECK:', {
                hasVideoFile: !!videoFile,
                videoFileType: videoFile ? videoFile.type : 'NO TYPE',
                videoFileName: videoFile ? videoFile.name : 'NO NAME'
            });
            
            if (!videoFile) {
                console.log('❌ VALIDATION FAILED: No video file found');
                alert('❌ Please select a video file to upload.');
                
                // Additional debugging - check if UI shows file selected but form doesn't have it
                if ($('#uploadArea').hasClass('file-selected')) {
                    console.log('🚨 BUG DETECTED: UI shows file selected but form validation fails!');
                    console.log('🔍 DOM STATE:', {
                        uploadAreaHTML: $('#uploadArea').html(),
                        fileInputValue: $('#videoFile').val(),
                        fileInputFiles: document.getElementById('videoFile').files
                    });
                }
                
                $('#videoFile').focus();
                return false;
            }
            
            // Validate file type
            const validTypes = ['video/mp4', 'video/webm', 'video/quicktime', 'video/x-msvideo', 'video/avi', 'video/mov'];
            if (!validTypes.includes(videoFile.type.toLowerCase())) {
                alert('❌ Invalid file type. Please select MP4, WebM, MOV, or AVI.');
                $('#videoFile').focus();
                return false;
            }
            
            // Validate file size (100MB limit)
            if (videoFile.size > 100 * 1024 * 1024) {
                alert('❌ File is too large. Maximum size is 100MB.');
                $('#videoFile').focus();
                return false;
            }
            
            console.log('✅ File validation passed:', videoFile.name, videoFile.size, videoFile.type);
        } else if (videoType === 'url') {
            if (!videoUrl) {
                alert('❌ Please provide a video URL.');
                $('#videoUrl').focus();
                return false;
            }
            
            // Check if this is a YouTube video and manual duration input is visible
            const isYouTubeVideo = videoUrl.includes('youtube.com') || videoUrl.includes('youtu.be');
            const manualDurationVisible = $('#manualDurationSection').is(':visible');
            
            if (isYouTubeVideo && manualDurationVisible) {
                const manualDuration = $('#manualDuration').val().trim();
                
                if (!manualDuration) {
                    alert('❌ Manual duration is required for YouTube videos to ensure accurate student progress tracking.');
                    $('#manualDuration').focus();
                    return false;
                }
                
                // Validate the manual duration format
                const validation = validateManualDuration(manualDuration);
                if (!validation.valid) {
                    alert('❌ Invalid duration format. ' + validation.message);
                    $('#manualDuration').focus();
                    return false;
                }
                
                // Ensure the duration is validated (has been clicked validate button)
                const validationResult = $('#manualValidationResult');
                if (!validationResult.is(':visible') || !validationResult.find('.alert-success').length) {
                    alert('❌ Please click the "Validate" button to confirm your manual duration entry.');
                    $('#validateDurationBtn').focus();
                    return false;
                }
                
                console.log('✅ Manual duration validation passed:', manualDuration);
            }
        }
        
        if (!this.checkValidity()) {
            this.classList.add('was-validated');
            
            // Debug: Log validation errors
            const invalidElements = this.querySelectorAll(':invalid');
            console.log('Form validation failed. Invalid elements:', invalidElements);
            invalidElements.forEach(element => {
                console.log('Invalid element:', element.name, element.value, element.validationMessage);
            });
            
            alert('Please fix the errors in the form before submitting.');
            return false;
        }
        
        const submitBtn = $('#submitBtn');
        const originalText = submitBtn.html();
        submitBtn.html('<i class="fas fa-spinner fa-spin me-2"></i>Uploading...').prop('disabled', true);
        
        const formData = new FormData(this);
        formData.append('ajax', '1');
        
        // CRITICAL FIX: Manually ensure the video file is in FormData
        if (videoType === 'file' && videoFile) {
            // Remove any existing 'video' entry and add our file
            formData.delete('video');
            formData.append('video', videoFile);
            console.log('✅ Manually added video file to FormData:', videoFile.name);
        }
        
        // DEBUG: Check duration field values before submission
        console.log('🔍 DURATION FIELD DEBUG BEFORE SUBMISSION:');
        const detectedDurationField = document.getElementById('detectedDurationInput');
        const manualDurationField = document.getElementById('manualDuration');
        
        console.log('📊 Duration field values:', {
            detectedDurationInput_exists: !!detectedDurationField,
            detectedDurationInput_value: detectedDurationField ? detectedDurationField.value : 'FIELD NOT FOUND',
            manualDuration_exists: !!manualDurationField,
            manualDuration_value: manualDurationField ? manualDurationField.value : 'FIELD NOT FOUND',
            formData_detected_duration: formData.get('detected_duration'),
            formData_manual_duration: formData.get('manual_duration')
        });
        
        // Log ALL form data for debugging
        console.log('📋 COMPLETE FORM DATA:');
        for (let [key, value] of formData.entries()) {
            if (key === 'video') {
                console.log(`${key}: [FILE] ${value.name} (${value.size} bytes)`);
            } else {
                console.log(`${key}: ${value}`);
            }
        }
        
        console.log('🚀 Starting AJAX upload request...');
        
        // Add progress tracking
        let progressInterval = setInterval(() => {
            console.log('⏱️ Upload still in progress...');
        }, 5000);
        
        $.ajax({
            url: 'process-video-with-content.php',
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            dataType: 'json',
            timeout: 60000, // Reduce timeout to 60 seconds to catch hangs faster
            
            // Add progress tracking for uploads
            xhr: function() {
                const xhr = new window.XMLHttpRequest();
                xhr.upload.addEventListener('progress', function(evt) {
                    if (evt.lengthComputable) {
                        const percentComplete = (evt.loaded / evt.total) * 100;
                        console.log('📤 Upload Progress:', percentComplete.toFixed(1) + '%');
                        submitBtn.html(`<i class="fas fa-spinner fa-spin me-2"></i>Uploading... ${percentComplete.toFixed(0)}%`);
                    }
                }, false);
                
                xhr.addEventListener('loadstart', function() {
                    console.log('🚀 Upload started');
                });
                
                xhr.addEventListener('loadend', function() {
                    console.log('🏁 Upload completed (waiting for server response...)');
                    clearInterval(progressInterval);
                });
                
                return xhr;
            },
            success: function(response) {
                if (response.success) {
                    alert('✅ Video uploaded successfully!');
                    setTimeout(() => {
                        window.location.href = response.redirect || 'videos.php';
                    }, 1000);
                } else {
                    // Check if session expired
                    if (response.message && response.message.includes('Session expired')) {
                        alert('❌ ' + response.message);
                        if (response.redirect) {
                            window.location.href = response.redirect;
                        } else {
                            window.location.reload();
                        }
                    } else {
                        let errorMsg = '❌ Upload failed: ' + (response.message || 'Unknown error');
                        
                        // Add debugging information if available
                        if (response.debug) {
                            console.log('🔍 UPLOAD DEBUG INFO:', response.debug);
                            
                            // Display server logs in live debug panel
                            if (response.debug.SERVER_LOGS && response.debug.SERVER_LOGS.length > 0) {
                                const logContainer = document.getElementById('server-logs');
                                logContainer.innerHTML = response.debug.SERVER_LOGS.join('\n') + '\n\n❌ UPLOAD FAILED\n';
                                logContainer.scrollTop = logContainer.scrollHeight;
                            }
                            
                            // Create a prominent debug panel
                            let debugHtml = `
                                <div id="debug-panel" style="
                                    position: fixed; 
                                    top: 20px; 
                                    right: 20px; 
                                    width: 400px; 
                                    max-height: 80vh; 
                                    overflow-y: auto;
                                    background: #1e1e1e; 
                                    color: #fff;
                                    padding: 20px; 
                                    border-radius: 10px; 
                                    font-family: 'Consolas', monospace; 
                                    font-size: 11px;
                                    box-shadow: 0 4px 20px rgba(0,0,0,0.3);
                                    z-index: 9999;
                                    border: 2px solid #ff6b6b;
                                ">
                                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
                                        <h4 style="margin: 0; color: #ff6b6b;">🔍 Debug Panel</h4>
                                        <button onclick="document.getElementById('debug-panel').style.display='none'" 
                                                style="background: #ff6b6b; border: none; color: white; padding: 5px 10px; border-radius: 3px; cursor: pointer;">✖</button>
                                    </div>
                                    
                                    <div style="background: #2d2d2d; padding: 10px; border-radius: 5px; margin-bottom: 10px;">
                                        <strong style="color: #4ecdc4;">📋 Bind Variables:</strong><br>`;
                            
                            Object.entries(response.debug.BIND_VARIABLES).forEach(([key, value]) => {
                                const color = value === 'UNDEFINED' ? '#ff6b6b' : '#95e1d3';
                                const icon = value === 'UNDEFINED' ? '❌' : '✅';
                                debugHtml += `<span style="color: ${color};">${icon} ${key}: ${value}</span><br>`;
                            });
                            
                            debugHtml += `</div>
                                    <div style="background: #2d2d2d; padding: 10px; border-radius: 5px; margin-bottom: 10px;">
                                        <strong style="color: #4ecdc4;">📂 Form Data:</strong><br>`;
                            
                            if (response.debug.POST_DATA) {
                                Object.entries(response.debug.POST_DATA).forEach(([key, value]) => {
                                    debugHtml += `<span style="color: #ffd93d;">${key}: ${value || 'EMPTY'}</span><br>`;
                                });
                            }
                            
                            debugHtml += `</div>
                                    <div style="background: #2d2d2d; padding: 10px; border-radius: 5px;">
                                        <strong style="color: #ff6b6b;">❌ Error Message:</strong><br>
                                        <span style="color: #ff9999;">${response.message || 'Unknown error'}</span>
                                    </div>
                                </div>`;
                            
                            // Remove any existing debug panel and add the new one
                            $('#debug-panel').remove();
                            $('body').append(debugHtml);
                        }
                        
                        alert(errorMsg);
                    }
                }
            },
            error: function(xhr, status, error) {
                clearInterval(progressInterval); // Stop progress logging
                
                console.error('❌ AJAX REQUEST FAILED:', {
                    status: xhr.status,
                    statusText: xhr.statusText,
                    readyState: xhr.readyState,
                    responseText: xhr.responseText ? xhr.responseText.substring(0, 500) : 'No response',
                    error: error,
                    ajaxStatus: status,
                    timeout: status === 'timeout'
                });
                
                let errorMessage = 'Upload failed';
                
                // Handle specific error cases
                if (status === 'timeout') {
                    errorMessage = 'Upload timed out after 60 seconds. The server may be overloaded or the file too large.';
                    console.error('🕐 Server took too long to respond - likely PHP timeout or processing issue');
                } else if (xhr.status === 0) {
                    errorMessage = 'Network connection failed. Please check your internet connection.';
                    console.error('🌐 Network connection issue - no response from server');
                } else if (xhr.status >= 500) {
                    errorMessage = `Server error (${xhr.status}). Please try again or contact support.`;
                    console.error('🚨 Server error - PHP script likely crashed or encountered fatal error');
                }
                
                // Check if response is HTML (like error pages)
                if (xhr.responseText && xhr.responseText.includes('<!DOCTYPE') || xhr.responseText.includes('<html')) {
                    errorMessage = 'Server returned an error page. Please check your session and try again.';
                    console.error('Server returned HTML instead of JSON:', xhr.responseText.substring(0, 200));
                } else {
                    try {
                        const response = JSON.parse(xhr.responseText);
                        errorMessage = response.message || errorMessage;
                    } catch(e) {
                        errorMessage = `${xhr.status}: ${error}`;
                        if (xhr.status === 0) {
                            errorMessage = 'Network error - please check your connection';
                        }
                    }
                }
                
                alert('❌ ' + errorMessage);
            },
            complete: function() {
                submitBtn.html(originalText).prop('disabled', false);
            }
        });
        
        this.classList.add('was-validated');
        return false;
    });
    
    // Thumbnail preview
    $('#thumbnailFile').on('change', function() {
        const file = this.files[0];
        if (file) {
            // Validate image type
            const validTypes = ['image/jpeg', 'image/png', 'image/jpg'];
            if (!validTypes.includes(file.type)) {
                alert('Invalid file type. Please select JPEG or PNG.');
                $(this).val('');
                return;
            }
            
            // Show preview
            const imageURL = URL.createObjectURL(file);
            $('#thumbnailImg').attr('src', imageURL);
            $('#thumbnailPreview').show();
        } else {
            $('#thumbnailPreview').hide();
        }
    });
    
    // Reset form
    $('#resetBtn').on('click', function() {
        if (confirm('Are you sure you want to reset the form? All data will be lost.')) {
            const form = $('#videoUploadForm')[0];
            form.reset();
            form.classList.remove('was-validated');
            $('#previewContainer').hide();
            $('#videoPreviewArea').hide();
            $('#thumbnailPreview').hide();
            $('#durationDetectionSection').hide();
            
            // Reset thumbnail section to default state
            resetThumbnailSection();
            hideManualDurationInput();
        }
    });
});

// 🎬 DURATION DETECTION FUNCTIONS

/**
 * Detect duration for uploaded video files
 */
function detectFileDuration(file) {
    console.log('🎬 Starting file duration detection for:', file.name, 'Type:', file.type, 'Size:', file.size);
    showDurationDetection('Analyzing video file...', 'File Analysis');
    
    // Create a video element to load the file and get duration
    const video = document.createElement('video');
    video.preload = 'metadata';
    
    console.log('📺 Created video element, setting up event handlers...');
    
    video.onloadedmetadata = function() {
        const duration = video.duration;
        console.log('📊 Video metadata loaded - Raw duration:', duration, 'seconds');
        
        if (duration && !isNaN(duration)) {
            const formattedDuration = formatDurationFromSeconds(Math.round(duration));
            console.log('⏰ Formatted duration:', formattedDuration);
            console.log('🎯 Calling showDurationResult with:', formattedDuration, 'HTML5 Video Element', 'Success');
            
            showDurationResult(formattedDuration, 'HTML5 Video Element', 'Success');
            console.log('✅ File duration detection completed successfully');
            
            // Verify hidden field was updated
            setTimeout(() => {
                const hiddenField = document.getElementById('detectedDurationInput');
                const fieldValue = hiddenField ? hiddenField.value : 'FIELD NOT FOUND';
                console.log('🔍 Hidden field verification - detectedDurationInput value:', fieldValue);
            }, 100);
            
        } else {
            console.log('❌ Invalid duration detected:', duration);
            showDurationError('Could not determine video duration from file metadata');
        }
        
        // Clean up
        URL.revokeObjectURL(video.src);
        console.log('🧹 Video element cleaned up');
    };
    
    video.onerror = function(e) {
        console.log('❌ Video element error:', e);
        showDurationError('Failed to load video file for duration analysis');
        URL.revokeObjectURL(video.src);
    };
    
    // Set video source to the file
    const videoUrl = URL.createObjectURL(file);
    console.log('🔗 Created video URL:', videoUrl.substring(0, 50) + '...');
    video.src = videoUrl;
    console.log('📂 Video source set, waiting for metadata...');
}

/**
 * Detect duration for YouTube videos
 */
function detectYouTubeDuration(videoId) {
    console.log('🎬 Starting YouTube duration detection for video ID:', videoId);
    showDurationDetection('Fetching YouTube video info...', 'YouTube Detection');
    
    // Method 1: Try client-side iframe embedding method (most reliable)
    tryYouTubeIframeDuration(videoId)
        .then(duration => {
            if (duration) {
                showDurationResult(duration, 'YouTube Iframe API', 'Success');
                console.log('✅ YouTube iframe duration detected:', duration);
            } else {
                throw new Error('Iframe method failed');
            }
        })
        .catch(() => {
            // Method 2: Fallback to server-side detection
            console.log('🔄 Iframe method failed, trying server-side detection...');
            
            $.ajax({
                url: 'ajax/detect_youtube_duration.php',
                method: 'POST',
                dataType: 'json',
                data: { video_id: videoId },
                success: function(response) {
                    if (response.success && response.duration) {
                        showDurationResult(response.duration, response.method || 'YouTube API', 'Success');
                        console.log('✅ YouTube server-side duration detected:', response.duration);
                    } else {
                        showDurationError(response.message || 'Failed to get YouTube video duration');
                        console.log('❌ YouTube duration detection failed:', response.message);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('❌ YouTube duration AJAX error:', error);
                    showDurationError('Both client and server-side detection failed. Please configure YouTube Data API key for reliable detection.');
                },
                timeout: 10000 // 10 second timeout
            });
        });
}

/**
 * Try to get YouTube video duration using iframe embedding
 */
function tryYouTubeIframeDuration(videoId) {
    return new Promise((resolve, reject) => {
        console.log('🎬 Trying YouTube iframe duration detection...');
        
        // Create a hidden iframe to load the video
        const iframe = document.createElement('iframe');
        iframe.style.display = 'none';
        iframe.style.width = '1px';
        iframe.style.height = '1px';
        iframe.src = `https://www.youtube.com/embed/${videoId}?enablejsapi=1&autoplay=0&controls=0`;
        
        let resolved = false;
        
        // Timeout after 8 seconds
        const timeout = setTimeout(() => {
            if (!resolved) {
                resolved = true;
                document.body.removeChild(iframe);
                console.log('⏰ YouTube iframe detection timeout');
                reject('Timeout');
            }
        }, 8000);
        
        // Listen for YouTube API messages
        const messageHandler = (event) => {
            if (event.origin !== 'https://www.youtube.com') return;
            
            try {
                const data = JSON.parse(event.data);
                console.log('📨 YouTube iframe message:', data);
                
                // Look for duration info in the message
                if (data.info && data.info.duration) {
                    const duration = formatDurationFromSeconds(Math.round(data.info.duration));
                    
                    if (!resolved) {
                        resolved = true;
                        clearTimeout(timeout);
                        document.body.removeChild(iframe);
                        window.removeEventListener('message', messageHandler);
                        console.log('✅ YouTube iframe duration found:', duration);
                        resolve(duration);
                    }
                }
            } catch (e) {
                // Not JSON or not relevant message
            }
        };
        
        window.addEventListener('message', messageHandler);
        document.body.appendChild(iframe);
        
        // Fallback: Try to extract from iframe after load
        iframe.onload = () => {
            setTimeout(() => {
                if (!resolved) {
                    resolved = true;
                    clearTimeout(timeout);
                    document.body.removeChild(iframe);
                    window.removeEventListener('message', messageHandler);
                    console.log('❌ YouTube iframe method failed - no duration found');
                    reject('No duration found');
                }
            }, 3000);
        };
        
        iframe.onerror = () => {
            if (!resolved) {
                resolved = true;
                clearTimeout(timeout);
                document.body.removeChild(iframe);
                window.removeEventListener('message', messageHandler);
                console.log('❌ YouTube iframe failed to load');
                reject('Iframe load error');
            }
        };
    });
}

/**
 * NEW: Detect duration using Video API Integration for TikTok and Vimeo
 */
function detectDurationWithAPI(videoUrl, platform) {
    console.log(`🎬 Using API integration for ${platform} duration detection...`);
    
    showDurationDetection(`Detecting ${platform} video duration using API...`, `${platform.charAt(0).toUpperCase() + platform.slice(1)} API`);
    
    // Use the Video API Integration module
    if (window.VideoAPIIntegration) {
        window.VideoAPIIntegration.getVideoDuration(
            videoUrl,
            function(result) {
                // Success callback
                console.log(`✅ ${platform} API duration detected:`, result);
                
                // Show result similar to existing YouTube detection
                showDurationResult(result.formattedDuration, result.method, 'Success');
                
                // Also show in the detection section
                $('#detectedDuration').text(result.formattedDuration);
                $('#detectionMethod').text(result.method).removeClass('bg-secondary bg-danger').addClass('bg-success');
                $('#detectionStatus').text('API Success').removeClass('bg-secondary bg-danger').addClass('bg-success');
            },
            function(error) {
                // Error callback - fall back to manual input like existing system
                console.log(`❌ ${platform} API detection failed:`, error);
                showDurationError(`${platform} API detection failed: ${error}`);
            }
        );
    } else {
        console.error('❌ VideoAPIIntegration not loaded');
        showDurationError('API integration not available - please refresh the page');
    }
}

/**
 * Show duration detection in progress
 */
function showDurationDetection(message, method) {
    $('#durationDetectionSection').show();
    $('#durationSpinner').show();
    $('#durationText').text(message);
    $('#durationResult').hide();
    $('#durationError').hide();
    
    if (method) {
        $('#detectionMethod').text(method).removeClass('bg-success bg-danger').addClass('bg-secondary');
    }
}

/**
 * Show successful duration detection result
 */
function showDurationResult(duration, method, status) {
    $('#durationSpinner').hide();
    
    // For YouTube videos, don't show the detected duration (can be misleading)
    // Instead, go straight to manual input
    if (method && (method.toLowerCase().includes('youtube') || method.toLowerCase().includes('iframe'))) {
        $('#durationText').text('YouTube video detected - manual duration entry required');
        $('#durationResult').hide(); // Hide the detected duration display
        $('#durationError').hide();
        showManualDurationInput('YouTube video detected - please verify duration for accurate student progress tracking');
    } else {
        // For file uploads, show the detected duration as it's more reliable
        $('#durationText').text('Duration detection completed successfully!');
        $('#detectedDuration').text(duration);
        $('#detectionMethod').text(method).removeClass('bg-secondary bg-danger').addClass('bg-success');
        $('#detectionStatus').text(status).removeClass('bg-secondary bg-danger').addClass('bg-success');
        $('#durationResult').show();
        $('#durationError').hide();
        
        // 🎯 FIX: Update hidden field so duration gets saved to database
        $('#detectedDurationInput').val(duration);
        console.log('✅ Duration stored in hidden field for database:', duration);
    }
}

/**
 * Show duration detection error (for URL videos, skip ugly error display)
 */
function showDurationError(errorMessage) {
    $('#durationSpinner').hide();
    
    // For URL videos, skip the ugly error display and go straight to clean manual input
    // Only show error details for file uploads where it's more relevant
    const videoType = $('input[name="video_type"]:checked').val();
    
    if (videoType === 'url') {
        // Clean approach for URL videos - no ugly error display
        $('#durationText').text('Manual duration entry required');
        $('#durationResult').hide();
        $('#durationError').hide();
        showManualDurationInput('Please enter the video duration manually for accurate student progress tracking');
    } else {
        // For file uploads, still show error details as they're more helpful
        $('#durationText').text('Duration detection failed');
        $('#detectedDuration').text('--:--');
        $('#detectionMethod').text('Failed').removeClass('bg-secondary bg-success').addClass('bg-danger');
        $('#detectionStatus').text('Error').removeClass('bg-secondary bg-success').addClass('bg-danger');
        $('#errorMessage').text(errorMessage);
        $('#durationResult').show();
        $('#durationError').show();
        showManualDurationInput('Duration detection failed');
    }
}

/**
 * Format duration from seconds to readable format (MM:SS or HH:MM:SS)
 */
function formatDurationFromSeconds(seconds) {
    const hours = Math.floor(seconds / 3600);
    const minutes = Math.floor((seconds % 3600) / 60);
    const secs = seconds % 60;
    
    if (hours > 0) {
        return `${hours}:${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
    } else {
        return `${minutes}:${secs.toString().padStart(2, '0')}`;
    }
}

/**
 * Show manual duration input section
 */
function showManualDurationInput(reason = 'Duration detection failed') {
    console.log('📝 Showing manual duration input:', reason);
    $('#manualDurationSection').show();
    $('#manualDuration').focus();
}

/**
 * Hide manual duration input section
 */
function hideManualDurationInput() {
    console.log('❌ Hiding manual duration input');
    $('#manualDurationSection').hide();
    $('#manualDuration').val('');
    $('#manualValidationResult').hide();
}

/**
 * Check if platform provides automatic thumbnails
 */
function platformProvidesAutoThumbnails(platform) {
    // Platforms that provide automatic thumbnails (don't need manual upload)
    const autoThumbnailPlatforms = [
        'youtube', 'vimeo', 'dailymotion', 'twitch',  // Original platforms
        'tiktok', 'instagram', 'facebook', 'twitter',  // Social media
        'linkedin', 'reddit', 'streamable', 'wistia'   // Additional platforms
    ];
    return autoThumbnailPlatforms.includes(platform?.toLowerCase());
}

/**
 * Check if platform provides video thumbnail URLs vs embedded content
 */
function platformProvidesDirectThumbnails(platform) {
    // Platforms that provide direct thumbnail image URLs
    const directThumbnailPlatforms = ['youtube', 'vimeo', 'dailymotion', 'twitch', 'wistia'];
    return directThumbnailPlatforms.includes(platform?.toLowerCase());
}

/**
 * Show/hide thumbnail upload section based on platform capabilities
 */
function updateThumbnailSection(platform, videoData = null) {
    console.log('🖼️ Updating thumbnail section for platform:', platform);
    
    if (platformProvidesAutoThumbnails(platform)) {
        // Platform provides thumbnails - hide upload section, show notice
        $('#thumbnailUploadSection').hide();
        $('#autoThumbnailNotice').show();
        
        // Clear any uploaded thumbnail since platform will provide one
        $('#thumbnailFile').val('');
        $('#thumbnailPreview').hide();
        
        // Customize message based on platform type
        const platformLower = platform.toLowerCase();
        let noticeText = '';
        
        if (platformProvidesDirectThumbnails(platformLower)) {
            // Platforms like YouTube, Vimeo that provide image thumbnails
            noticeText = `${platform} provides high-quality video thumbnails automatically. No manual upload needed!`;
        } else {
            // Platforms like TikTok, Instagram, Facebook that provide embedded content
            noticeText = `${platform} handles video thumbnails through its embedded player. No manual upload needed!`;
        }
        
        $('#thumbnailNoticeText').text(noticeText);
        console.log('✅ Platform provides auto thumbnails - hiding upload section');
    } else {
        // Platform doesn't provide thumbnails - show upload section, hide notice
        $('#thumbnailUploadSection').show();
        $('#autoThumbnailNotice').hide();
        
        console.log('📤 Platform requires manual thumbnail - showing upload section');
    }
}

/**
 * Reset thumbnail section to default state (for file uploads or no detection)
 */
function resetThumbnailSection() {
    console.log('🔄 Resetting thumbnail section to default state');
    $('#thumbnailUploadSection').show();
    $('#autoThumbnailNotice').hide();
}

/**
 * Validate manual duration input format
 */
function validateManualDuration(durationStr) {
    if (!durationStr || durationStr.trim() === '') {
        return { valid: false, message: 'Duration is required' };
    }

    // Remove any spaces
    durationStr = durationStr.trim();

    // Check format: MM:SS or HH:MM:SS
    const regex = /^(\d{1,2}:)?([0-5]?\d):([0-5]?\d)$/;
    const match = durationStr.match(regex);

    if (!match) {
        return { 
            valid: false, 
            message: 'Invalid format. Use MM:SS (e.g., 19:45) or HH:MM:SS (e.g., 1:23:30)' 
        };
    }

    // Parse components
    let hours = 0, minutes = 0, seconds = 0;
    
    if (match[1]) { // HH:MM:SS format
        hours = parseInt(match[1].replace(':', ''));
        minutes = parseInt(match[2]);
        seconds = parseInt(match[3]);
    } else { // MM:SS format
        minutes = parseInt(match[2]);
        seconds = parseInt(match[3]);
    }

    // Additional validation
    if (minutes >= 60) {
        return { valid: false, message: 'Minutes must be less than 60' };
    }
    if (seconds >= 60) {
        return { valid: false, message: 'Seconds must be less than 60' };
    }

    const totalSeconds = hours * 3600 + minutes * 60 + seconds;
    if (totalSeconds === 0) {
        return { valid: false, message: 'Duration must be greater than 00:00' };
    }

    return { 
        valid: true, 
        duration: durationStr,
        totalSeconds: totalSeconds,
        message: `Valid duration: ${durationStr}` 
    };
}

/**
 * Handle manual duration validation
 */
function handleManualDurationValidation() {
    const durationInput = $('#manualDuration').val();
    const validation = validateManualDuration(durationInput);
    const resultDiv = $('#manualValidationResult');

    if (validation.valid) {
        resultDiv.html(`
            <div class="alert alert-success">
                <i class="fas fa-check-circle me-2"></i>
                <strong>✓ Duration Validated:</strong> ${validation.duration}
                <br><small>This duration will be used for student progress tracking.</small>
            </div>
        `).show();
        
        // Update the detected duration display with manual input
        $('#detectedDuration').text(validation.duration);
        $('#detectionMethod').text('Manual Input').removeClass('bg-danger bg-secondary').addClass('bg-primary');
        $('#detectionStatus').text('Validated').removeClass('bg-danger bg-secondary').addClass('bg-success');
        
        console.log('✅ Manual duration validated:', validation.duration);
        return true;
    } else {
        resultDiv.html(`
            <div class="alert alert-danger">
                <i class="fas fa-exclamation-triangle me-2"></i>
                <strong>Validation Error:</strong> ${validation.message}
            </div>
        `).show();
        
        console.log('❌ Manual duration validation failed:', validation.message);
        return false;
    }
}

// Event handlers for manual duration input
$(document).ready(function() {
    // Manual duration validation button
    $('#validateDurationBtn').click(function() {
        handleManualDurationValidation();
    });
    
    // Enter key on manual duration input
    $('#manualDuration').keypress(function(e) {
        if (e.which === 13) { // Enter key
            e.preventDefault();
            handleManualDurationValidation();
        }
    });
    
    // Real-time validation as user types
    $('#manualDuration').on('input', function() {
        const durationInput = $(this).val();
        if (durationInput.length > 0) {
            const validation = validateManualDuration(durationInput);
            if (validation.valid) {
                $(this).removeClass('is-invalid').addClass('is-valid');
            } else {
                $(this).removeClass('is-valid').addClass('is-invalid');
            }
        } else {
            $(this).removeClass('is-valid is-invalid');
        }
    });
    
    // Handle video source switching (File Upload vs Embed URL)
    $('input[name="video_type"]').on('change', function() {
        const videoType = $(this).val();
        console.log('🔄 Video source switched to:', videoType);
        
        if (videoType === 'file') {
            // Switched to file upload - reset thumbnail section to default (show upload)
            resetThumbnailSection();
            hideManualDurationInput();
            $('#durationDetectionSection').hide();
        } else if (videoType === 'url') {
            // Switched to URL embed - thumbnail section will be updated when video is detected
            resetThumbnailSection(); // Start with default state
            hideManualDurationInput();
            $('#durationDetectionSection').hide();
        }
    });
});
</script>

<!-- 🔧 Video API Integration for Duration Detection -->
<script src="../js/video-api-integration.js"></script>

<script>
/**
 * Enhanced duration detection using API integration
 * Called from the video detection logic for TikTok and Vimeo
 */
function detectDurationWithAPI(videoUrl, platform) {
    console.log(`🎬 Starting API duration detection for ${platform}:`, videoUrl);
    
    // Show loading state
    $('#durationSpinner').show();
    $('#durationText').text('Detecting duration via API...');
    $('#durationResult').show();
    $('#durationError').hide();
    
    // Use the VideoAPIIntegration to get duration
    if (window.VideoAPIIntegration) {
        window.VideoAPIIntegration.getVideoDuration(
            videoUrl,
            function(data) {
                // ✅ Success - Show detected duration
                console.log(`✅ ${platform} API duration detected:`, data);
                
                // 🔥 CRITICAL: Save detected duration to hidden form field for database submission
                $('#detectedDurationInput').val(data.formattedDuration);
                console.log(`💾 Saved duration to form field: ${data.formattedDuration}`);
                
                // 🎯 ADD DURATION TO VIDEO PREVIEW CARD ONLY
                addDurationToVideoCard(data.formattedDuration, data.method, data.note);
                
                // Hide all bottom duration sections since we show it on the card
                $('#durationSpinner').hide();
                $('#durationResult').hide();
                $('#durationError').hide();
                hideManualDurationInput();
                
                // Show note for TikTok estimated duration
                if (data.note) {
                    $('#durationText').text(data.note);
                }
                
                console.log(`🎯 Duration detection complete: ${data.formattedDuration}`);
            },
            function(error) {
                // ❌ Error - Fall back to manual input
                console.warn(`⚠️ ${platform} API detection failed:`, error);
                showDurationError(error);
            }
        );
    } else {
        console.error('❌ VideoAPIIntegration not loaded');
        showDurationError('Video API integration not available');
    }
}

/**
 * Add duration display to video preview card
 */
function addDurationToVideoCard(duration, method, note) {
    console.log('🎯 Adding duration to video card:', duration, method);
    
    // Target the specific green "YouTube Video Detected" card
    // Look for the green card with "Video Detected" text
    let videoCard = null;
    
    // Try multiple selectors to find the green video card
    const cardSelectors = [
        '.alert-success:contains("Video Detected")',
        '.bg-success:contains("Video Detected")', 
        '.card .bg-success:contains("Video Detected")',
        '.card-header.bg-success',
        '[style*="background"]:contains("Video Detected")',
        '.text-white:contains("Video Detected")'
    ];
    
    for (const selector of cardSelectors) {
        videoCard = $(selector).closest('.card, .alert, div[class*="card"]').first();
        if (videoCard.length > 0) {
            console.log('✅ Found video card using selector:', selector);
            break;
        }
    }
    
    // If still not found, look for any container that has video info
    if (videoCard.length === 0) {
        // Look for any element containing the video title
        const videoTitle = $('*:contains("Bootstrap 5 Crash Course"), *:contains("Video ID:")').first();
        if (videoTitle.length > 0) {
            videoCard = videoTitle.closest('.card, .alert, div[class*="card"], .mb-4').first();
            console.log('✅ Found video card by video title');
        }
    }
    
    // If still not found, use fallback approach
    if (videoCard.length === 0) {
        console.warn('⚠️ Video card not found, using fallback');
        // Add after the video URL input as fallback
        const fallbackArea = $('#videoUrl').closest('.mb-4');
        if (fallbackArea.length > 0) {
            addDurationBadge(fallbackArea, duration, method, note);
            return;
        }
    }
    
    console.log('🎯 Adding duration badge to video card');
    addDurationBadge(videoCard, duration, method, note);
}

/**
 * Add duration badge to specified element
 */
function addDurationBadge(container, duration, method, note) {
    // Remove any existing duration display
    container.find('.duration-display-badge').remove();
    
    // Create duration badge
    let badgeClass = 'bg-success';
    let icon = 'fas fa-clock';
    
    // Different styling for different methods
    if (method && method.includes('Estimated')) {
        badgeClass = 'bg-warning text-dark';
        icon = 'fas fa-stopwatch';
    }
    
    const durationBadge = `
        <div class="duration-display-badge mt-2">
            <div class="d-flex align-items-center gap-2">
                <span class="badge ${badgeClass} d-flex align-items-center gap-1">
                    <i class="${icon}"></i>
                    <strong>Duration: ${duration}</strong>
                </span>
                <small class="text-muted">${method || 'API Detection'}</small>
            </div>
            ${note ? `<div class="mt-1"><small class="text-info"><i class="fas fa-info-circle me-1"></i>${note}</small></div>` : ''}
        </div>
    `;
    
    // Append the badge
    container.append(durationBadge);
    
    // Animate the badge appearance
    container.find('.duration-display-badge').hide().fadeIn(500);
    
    console.log('✅ Duration badge added to video card');
}
</script>

</body>
</html>
