// Video Upload Handler
class VideoUploader {
    constructor() {
        this.selectedVideoFile = null;
        this.selectedThumbnailFile = null;
        this.videoDuration = null;
        this.currentVideoURL = null;
        this.initializeElements();
        this.bindEvents();
    }

    initializeElements() {
        this.uploadArea = document.getElementById('uploadArea');
        this.browseBtn = document.getElementById('browseBtn');
        this.videoFile = document.getElementById('videoFile');
        this.thumbnailFile = document.getElementById('thumbnailFile');
        this.form = document.getElementById('videoUploadForm');
        this.progressContainer = document.getElementById('progressContainer');
        this.previewContainer = document.getElementById('previewContainer');
        this.videoPreview = document.getElementById('videoPreview');
        this.thumbnailPreview = document.getElementById('thumbnailPreview');
        this.thumbnailImg = document.getElementById('thumbnailImg');
        this.uploadProgress = document.getElementById('uploadProgress');
        this.submitBtn = document.getElementById('submitBtn');
        this.resetBtn = document.getElementById('resetBtn');
    }

    bindEvents() {
        // File input events
        this.browseBtn.addEventListener('click', () => this.videoFile.click());
        this.videoFile.addEventListener('change', (e) => this.handleVideoSelect(e.target.files[0]));
        this.thumbnailFile.addEventListener('change', (e) => this.handleThumbnailSelect(e.target.files[0]));

        // Drag and drop events
        this.uploadArea.addEventListener('click', () => this.videoFile.click());
        this.uploadArea.addEventListener('dragover', (e) => this.handleDragOver(e));
        this.uploadArea.addEventListener('dragleave', (e) => this.handleDragLeave(e));
        this.uploadArea.addEventListener('drop', (e) => this.handleDrop(e));

        // Form events
        this.form.addEventListener('submit', (e) => this.handleSubmit(e));
        this.resetBtn.addEventListener('click', () => this.resetForm());

        // Price checkbox event
        document.getElementById('isFree').addEventListener('change', (e) => {
            const priceInput = document.getElementById('videoPrice');
            if (e.target.checked) {
                priceInput.value = '0';
                priceInput.disabled = true;
            } else {
                priceInput.disabled = false;
            }
        });
    }

    handleDragOver(e) {
        e.preventDefault();
        this.uploadArea.classList.add('dragover');
    }

    handleDragLeave(e) {
        e.preventDefault();
        this.uploadArea.classList.remove('dragover');
    }

    handleDrop(e) {
        e.preventDefault();
        this.uploadArea.classList.remove('dragover');
        
        const files = e.dataTransfer.files;
        if (files.length > 0) {
            const file = files[0];
            if (file.type.startsWith('video/')) {
                this.handleVideoSelect(file);
            } else {
                this.showToast('Please select a valid video file', 'error');
            }
        }
    }

    handleVideoSelect(file) {
        if (!file) return;

        // Validate file type
        if (!file.type.startsWith('video/')) {
            this.showToast('Please select a valid video file', 'error');
            return;
        }

        this.selectedVideoFile = file;
        
        // Update UI
        this.progressContainer.style.display = 'block';
        this.previewContainer.style.display = 'block';
        
        // Display file info
        document.getElementById('fileName').textContent = file.name;
        document.getElementById('fileSize').textContent = this.formatFileSize(file.size);
        document.getElementById('uploadStatus').textContent = 'Ready';
        
        // Create video preview
        const videoURL = URL.createObjectURL(file);
        this.videoPreview.src = videoURL;
        
        // Store the URL to revoke it later
        this.currentVideoURL = videoURL;
        
        // Get video duration
        this.videoPreview.addEventListener('loadedmetadata', () => {
            this.videoDuration = this.videoPreview.duration;
            document.getElementById('videoDuration').textContent = this.formatDuration(this.videoDuration);
        }, { once: true });
        
        // Handle video errors
        this.videoPreview.addEventListener('error', (e) => {
            console.error('Video preview error:', e);
            document.getElementById('videoDuration').textContent = 'Error loading video';
        }, { once: true });

        this.showToast(`Video selected: ${file.name} (${this.formatFileSize(file.size)})`, 'success');
    }

    handleThumbnailSelect(file) {
        if (!file) return;

        if (!file.type.startsWith('image/')) {
            this.showToast('Please select a valid image file for thumbnail', 'error');
            return;
        }

        this.selectedThumbnailFile = file;
        
        // Show thumbnail preview
        const reader = new FileReader();
        reader.onload = (e) => {
            this.thumbnailImg.src = e.target.result;
            this.thumbnailPreview.style.display = 'block';
        };
        reader.readAsDataURL(file);

        this.showToast('Thumbnail selected', 'success');
    }

    async handleSubmit(event) {
        event.preventDefault();
        
        if (!this.selectedVideoFile) {
            this.showToast('Please select a video file', 'error');
            return;
        }
        
        // Validate form first
        if (!this.validateForm()) {
            return;
        }
        
        // Prevent double submission
        if (this.submitBtn.disabled) {
            return;
        }
        
        // Update UI
        this.submitBtn.disabled = true;
        this.submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Converting...';
        
        try {
            console.log('Starting base64 upload to process-video-upload-base64.php');
            
            // Convert video to base64
            this.showToast('Converting video to base64...', 'info');
            const videoBase64 = await this.fileToBase64(this.selectedVideoFile);
            console.log('Video converted to base64, length:', videoBase64.length);
            
            // Update UI
            this.submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Uploading...';
            
            // Prepare form data
            const form = event.target;
            const postData = {
                video_base64: videoBase64,
                video_filename: this.selectedVideoFile.name,
                title: form.querySelector('[name="title"]').value,
                course_id: form.querySelector('[name="course_id"]').value,
                description: form.querySelector('[name="description"]').value,
                module_order: form.querySelector('[name="module_order"]').value,
                price: form.querySelector('[name="price"]').value,
                category_name: form.querySelector('[name="category_name"]').value,
                is_published: form.querySelector('[name="is_published"]').checked ? '1' : '0',
                is_free: form.querySelector('[name="is_free"]').checked ? '1' : '0',
                duration: this.videoDuration || 0
            };
            
            // Add thumbnail if selected
            if (this.selectedThumbnailFile) {
                console.log('Converting thumbnail to base64...');
                const thumbnailBase64 = await this.fileToBase64(this.selectedThumbnailFile);
                postData.thumbnail_base64 = thumbnailBase64;
                postData.thumbnail_filename = this.selectedThumbnailFile.name;
            }
            
            console.log('Sending base64 data to server...');
            
            const response = await this.uploadWithProgress(postData);
            
            if (response.success) {
                this.showToast('Video uploaded successfully!', 'success');
                document.getElementById('uploadStatus').textContent = 'Completed';
                document.getElementById('uploadStatusCard').innerHTML = `
                    <div class="text-success">
                        <i class="fas fa-check-circle"></i> Upload completed successfully!
                        <br><small>Video ID: ${response.data.video_id}</small>
                    </div>
                `;
                
                // Optionally redirect to video management page
                setTimeout(() => {
                    window.location.href = 'videos.php';
                }, 2000);
            } else {
                throw new Error(response.message || 'Upload failed');
            }
            
        } catch (error) {
            console.error('Upload error:', error);
            this.showToast(error.message || 'Upload failed', 'error');
            document.getElementById('uploadStatus').textContent = 'Failed';
            document.getElementById('uploadStatusCard').innerHTML = `
                <div class="text-danger">
                    <i class="fas fa-exclamation-circle"></i> Upload failed
                    <br><small>${error.message}</small>
                </div>
            `;
        } finally {
            // Re-enable submit button
            this.submitBtn.disabled = false;
            this.submitBtn.innerHTML = '<i class="fas fa-cloud-upload-alt"></i> Upload Video';
        }
    }

    uploadWithProgress(postData) {
        return new Promise((resolve, reject) => {
            const xhr = new XMLHttpRequest();
            
            // Handle upload progress
            xhr.upload.addEventListener('progress', (e) => {
                if (e.lengthComputable) {
                    const percentComplete = Math.round((e.loaded / e.total) * 100);
                    console.log(`Upload progress: ${percentComplete}%`);
                    this.uploadProgress.style.width = percentComplete + '%';
                    this.uploadProgress.textContent = percentComplete + '%';
                }
            });
            
            // Handle upload completion
            xhr.addEventListener('loadend', () => {
                console.log('Upload completed, status:', xhr.status);
                console.log('Response text:', xhr.responseText);
                
                try {
                    const response = JSON.parse(xhr.responseText);
                    console.log('Parsed response:', response);
                    
                    if (xhr.status === 200) {
                        if (response.debug) {
                            console.log('Debug info:', response.debug);
                        }
                        resolve(response);
                    } else {
                        reject(new Error(`HTTP ${xhr.status}: ${xhr.statusText}`));
                    }
                } catch (error) {
                    console.error('JSON parse error:', error);
                    reject(new Error('Invalid server response'));
                }
            });
            
            // Handle network errors
            xhr.addEventListener('error', () => {
                console.error('Network error during upload');
                reject(new Error('Network error occurred'));
            });
            
            // Handle timeout
            xhr.addEventListener('timeout', () => {
                console.error('Upload timeout');
                reject(new Error('Upload timeout - file may be too large'));
            });
            
            console.log('Sending base64 upload request...');
            
            // Configure and send request
            xhr.open('POST', 'process-video-upload-base64.php');
            xhr.timeout = 600000; // 10 minutes timeout
            xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
            
            // Convert to URL-encoded format
            const urlEncodedData = Object.keys(postData)
                .map(key => encodeURIComponent(key) + '=' + encodeURIComponent(postData[key]))
                .join('&');
                
            console.log('Data keys being sent:', Object.keys(postData));
            xhr.send(urlEncodedData);
        });
    }
    
    // Helper method to convert file to base64
    fileToBase64(file) {
        return new Promise((resolve, reject) => {
            const reader = new FileReader();
            reader.onload = () => resolve(reader.result);
            reader.onerror = reject;
            reader.readAsDataURL(file);
        });
    }

    validateForm() {
        const title = document.getElementById('videoTitle').value.trim();
        const courseId = document.getElementById('courseSelect').value;

        if (!this.selectedVideoFile) {
            this.showToast('Please select a video file', 'error');
            return false;
        }

        if (!title) {
            this.showToast('Please enter a video title', 'error');
            document.getElementById('videoTitle').focus();
            return false;
        }

        if (!courseId) {
            this.showToast('Please select a course', 'error');
            document.getElementById('courseSelect').focus();
            return false;
        }

        return true;
    }

    resetForm() {
        // Reset form
        this.form.reset();
        
        // Clean up blob URLs
        if (this.currentVideoURL) {
            URL.revokeObjectURL(this.currentVideoURL);
            this.currentVideoURL = null;
        }
        
        // Reset file selections
        this.selectedVideoFile = null;
        this.selectedThumbnailFile = null;
        this.videoDuration = null;
        
        // Hide UI elements
        this.progressContainer.style.display = 'none';
        this.previewContainer.style.display = 'none';
        this.thumbnailPreview.style.display = 'none';
        
        // Reset progress
        this.uploadProgress.style.width = '0%';
        this.uploadProgress.textContent = '0%';
        
        // Reset status
        document.getElementById('uploadStatusCard').innerHTML = '<p class="text-muted">No upload in progress</p>';
        
        // Clear video preview
        this.videoPreview.src = '';
        this.videoPreview.load(); // Force video to clear
        
        // Re-enable price input if needed
        document.getElementById('videoPrice').disabled = false;
        
        this.showToast('Form reset', 'info');
    }

    formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }

    formatDuration(seconds) {
        const hrs = Math.floor(seconds / 3600);
        const mins = Math.floor((seconds % 3600) / 60);
        const secs = Math.floor(seconds % 60);
        
        if (hrs > 0) {
            return `${hrs}:${mins.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
        } else {
            return `${mins}:${secs.toString().padStart(2, '0')}`;
        }
    }

    showToast(message, type = 'info') {
        const colors = {
            success: '#28a745',
            error: '#dc3545',
            warning: '#ffc107',
            info: '#17a2b8'
        };

        Toastify({
            text: message,
            duration: 3000,
            close: true,
            gravity: "top",
            position: "right",
            style: {
                background: colors[type] || colors.info,
            },
        }).showToast();
    }
}

// Initialize the uploader when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    new VideoUploader();
});
