<?php
// Define current page for sidebar highlighting
$current_page = 'videos.php';

// Include the centralized teacher layout
require_once('components/teacher_layout.php');

// Include ID encryption utility
require_once "../includes/id_encryption.php";

// Get teacher ID from session
$userId = $_SESSION['user_id'];

// Check if debug mode is enabled
$debug_mode = isset($_GET['debug']) && $_GET['debug'] === '1';

// Get the current teacher details
$current_teacher_id = $userId;
$teacher_name = '';

// Get teacher information for profile display
$query = "SELECT * FROM users WHERE user_id = ? LIMIT 1";
$stmt = $conn->prepare($query);
$stmt->bind_param("i", $userId);
$stmt->execute();
$result = $stmt->get_result();
$teacher = $result->fetch_assoc();

// Get courses for dropdown filter
$query = "SELECT * FROM courses WHERE teacher_id = ?";
$stmt = $conn->prepare($query);
$stmt->bind_param("i", $userId);
$stmt->execute();
$courses_result = $stmt->get_result();
$courses = $courses_result->fetch_all(MYSQLI_ASSOC);

// Get course filter - handle both encrypted and legacy IDs
$course_filter = null;
if (isset($_GET['course_id'])) {
    if (is_numeric($_GET['course_id'])) {
        // Legacy integer ID (for backward compatibility)
        $course_filter = (int)$_GET['course_id'];
    } else {
        // Try to decrypt encrypted ID
        $course_filter = IdEncryption::decrypt($_GET['course_id']);
        if ($course_filter === false) {
            $course_filter = null; // Invalid encrypted ID, ignore filter
        }
    }
} elseif (isset($_GET['course']) && is_numeric($_GET['course'])) {
    // Legacy support for 'course' parameter
    $course_filter = intval($_GET['course']);
}

// Initialize message variable
$message = '';

// Check for newly uploaded video
$new_video_id = null;
if (isset($_GET['new_video'])) {
    $new_video_id = IdEncryption::decrypt($_GET['new_video']);
    if ($new_video_id) {
        $message = '<div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fas fa-check-circle me-2"></i>
            <strong>Success!</strong> Your video has been uploaded successfully and is now available in your video library.
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>';
    }
}

// Check if a video should be deleted
if (isset($_GET['delete']) && is_numeric($_GET['delete'])) {
    $video_id = $_GET['delete'];
    
    // Check if database connection is available
    if (!$conn) {
        $message = '<div class="alert alert-danger">Database connection error. Please try again later.</div>';
        error_log("Database connection is null in videos.php (delete video section)");
    } else {
        // Verify that this video belongs to the logged-in teacher
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM videos WHERE video_id = ? AND teacher_id = ?");
        $stmt->bind_param("ii", $video_id, $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        
        if ($row['count'] == 0) {
            $message = '<div class="alert alert-danger">You do not have permission to delete this video.</div>';
        } else {
            // Check if the video has any purchases
            $stmt = $conn->prepare("SELECT COUNT(*) as count FROM video_purchases WHERE video_id = ?");
            $stmt->bind_param("i", $video_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $row = $result->fetch_assoc();
            
            if ($row['count'] > 0) {
                $message = '<div class="alert alert-danger">Cannot delete this video because it has been purchased by students.</div>';
            } else {
                // Get video information for file deletion
                $stmt = $conn->prepare("SELECT video_url, thumbnail FROM videos WHERE video_id = ?");
                $stmt->bind_param("i", $video_id);
                $stmt->execute();
                $result = $stmt->get_result();
                $video = $result->fetch_assoc();
                
                // Delete the video
                $stmt = $conn->prepare("DELETE FROM videos WHERE video_id = ?");
                $stmt->bind_param("i", $video_id);
                
                if ($stmt->execute()) {
                    $message = '<div class="alert alert-success">Video deleted successfully.</div>';
                    
                    // Delete video file and thumbnail if they exist
                    if (!empty($video['video_url'])) {
                        // Properly handle paths - use simple path joining
                        $video_file_path = '../' . ltrim($video['video_url'], '/');
                        if (file_exists($video_file_path)) {
                            unlink($video_file_path);
                        }
                    }
                    
                    if (!empty($video['thumbnail'])) {
                        // Properly handle paths - use simple path joining
                        $thumbnail_path = '../' . ltrim($video['thumbnail'], '/');
                        if (file_exists($thumbnail_path)) {
                            unlink($thumbnail_path);
                        }
                    }
                } else {
                    $message = '<div class="alert alert-danger">Error deleting video: ' . $conn->error . '</div>';
                }
            }
        }
    }
}

// Toggle video publication status
if (isset($_GET['toggle_publish']) && is_numeric($_GET['toggle_publish'])) {
    $video_id = $_GET['toggle_publish'];
    
    // Check if database connection is available
    if (!$conn) {
        $message = '<div class="alert alert-danger">Database connection error. Please try again later.</div>';
        error_log("Database connection is null in videos.php (toggle_publish section)");
    } else {
        // Verify that this video belongs to the logged-in teacher
        $stmt = $conn->prepare("SELECT is_published FROM videos WHERE video_id = ? AND teacher_id = ?");
        $stmt->bind_param("ii", $video_id, $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
    
        if ($result->num_rows == 0) {
            $message = '<div class="alert alert-danger">You do not have permission to modify this video.</div>';
        } else {
            // Toggle status
            $new_status = $row['is_published'] ? 0 : 1;
            $status_text = $new_status ? "published" : "unpublished";
            
            $stmt = $conn->prepare("UPDATE videos SET is_published = ? WHERE video_id = ?");
            $stmt->bind_param("ii", $new_status, $video_id);
            if ($stmt->execute()) {
                $message = '<div class="alert alert-success">Video successfully ' . $status_text . '.</div>';
            } else {
                $message = '<div class="alert alert-danger">Error updating video status: ' . $conn->error . '</div>';
            }
        }
    }
}

// Get teacher's courses for filtering
$courses = [];

// Check if database connection is available
if (!$conn) {
    error_log("Database connection is null in videos.php (courses retrieval section)");
} else {
    try {
        $stmt = $conn->prepare("SELECT course_id, title FROM courses WHERE teacher_id = ? ORDER BY title");
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $courses[] = $row;
            }
        }
    } catch (Exception $e) {
        error_log("Error retrieving courses: " . $e->getMessage());
    }
}

// Build query to get courses with their videos - grouped by course
$query = "SELECT c.course_id, c.title as course_title, c.thumbnail as course_thumbnail, c.is_published as course_published,
                 v.video_id, v.title as video_title, v.thumbnail as video_thumbnail, 
                 v.is_published as video_published, v.created_at as video_created_at, v.duration
          FROM courses c 
          LEFT JOIN videos v ON v.course_id = c.course_id
          WHERE c.teacher_id = ?";

$queryParams = [$userId];

// Add course filter if specified
if ($course_filter) {
    $query .= " AND c.course_id = ?";
    $queryParams[] = $course_filter;
}
$query .= " ORDER BY c.title ASC, v.created_at DESC";

// Add debug output at the top of the content area - in a container to prevent layout disruption
if ($debug_mode) {
    echo "<div class='container mt-4 mb-4'>";
    echo "<div class='row'>";
    echo "<div class='col-12'>";
    echo "<div style='background-color: #f8f9fa; border: 1px solid #dee2e6; padding: 15px; margin-bottom: 20px;'>";
    echo "<h4>Debug Information</h4>";
    echo "<p><strong>Current Teacher ID:</strong> $current_teacher_id</p>";
    echo "<p><strong>User Role:</strong> " . ($_SESSION['user_role'] ?? 'Not set') . "</p>";
    echo "<p><strong>Session User ID:</strong> " . ($_SESSION['user_id'] ?? 'Not set') . "</p>";
    echo "<p><strong>Query being executed:</strong> " . htmlspecialchars($query) . "</p>";
    echo "<p><strong>Query Parameters:</strong> " . (empty($queryParams) ? 'None' : implode(', ', $queryParams)) . "</p>";
    echo "</div>";
    echo "</div>";
    echo "</div>";
    echo "</div>";
}

// Prepare and execute the query to get courses with videos
$courses_with_videos = [];
$error_message = null;

// Check if database connection is available
if (!$conn) {
    error_log("Database connection is null in videos.php (main video query section)");
    $error_message = "<div class='alert alert-danger'>Database connection error. Please try again later.</div>";
} else {
    try {
        $stmt = $conn->prepare($query);
        
        // Only bind parameters if we have any
        if (!empty($queryParams)) {
            $types = str_repeat('i', count($queryParams));
            $stmt->bind_param($types, ...$queryParams);
        }
        
        $stmt->execute();
        $result = $stmt->get_result();
        
        // Group results by course
        while ($row = $result->fetch_assoc()) {
            $course_id = $row['course_id'];
            
            // Initialize course if not exists
            if (!isset($courses_with_videos[$course_id])) {
                $courses_with_videos[$course_id] = [
                    'course_id' => $course_id,
                    'course_title' => $row['course_title'],
                    'course_thumbnail' => $row['course_thumbnail'],
                    'course_published' => $row['course_published'],
                    'videos' => []
                ];
            }
            
            // Add video to course if video exists
            if ($row['video_id']) {
                $courses_with_videos[$course_id]['videos'][] = [
                    'video_id' => $row['video_id'],
                    'title' => $row['video_title'],
                    'thumbnail' => $row['video_thumbnail'],
                    'is_published' => $row['video_published'],
                    'created_at' => $row['video_created_at'],
                    'duration' => $row['duration']
                ];
            }
        }
        
        // Add additional debug info after query execution
        if ($debug_mode) {
            echo "<div class='alert alert-info'>Found " . $result->num_rows . " videos for teacher ID: $userId</div>";
        }
        
        // Fetch all videos
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $videos[] = $row;
            }
        }
    } catch (Exception $e) {
        error_log("Error executing video query: " . $e->getMessage());
        $error_message = "<div class='alert alert-danger'>Error retrieving videos. Please try again later.</div>";
    }
}

// Get course details if filtering by course
$courseDetail = null;
if ($course_filter && $conn) {
    try {
        $stmt = $conn->prepare("SELECT title FROM courses WHERE course_id = ? AND teacher_id = ?");
        $stmt->bind_param("ii", $course_filter, $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows > 0) {
            $courseDetail = $result->fetch_assoc();
        }
    } catch (Exception $e) {
        error_log("Error retrieving course details: " . $e->getMessage());
    }
}
?>

<!-- Include DataTables CSS -->
<link rel="stylesheet" href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap5.min.css">
    
<style>
    /* Modern Videos Page Styling - Panadite Theme */
    * {
        font-family: 'Inter', 'Poppins', sans-serif !important;
    }

    .container-fluid {
        padding: 0 30px !important;
    }

    /* Page Header */
    h1.h3 {
        font-size: 28px !important;
        font-weight: 700 !important;
        color: #1f2937 !important;
        letter-spacing: -0.025em !important;
        margin-bottom: 8px !important;
    }

    .text-muted {
        color: #6b7280 !important;
        font-size: 16px !important;
    }

    /* Small Stats Cards */
    .stats-card {
        background: #ffffff !important;
        border-radius: 12px !important;
        padding: 16px !important;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08) !important;
        transition: all 0.2s ease !important;
        border: 1px solid #f1f5f9 !important;
        height: 100% !important;
        display: flex !important;
        align-items: center !important;
        gap: 12px !important;
    }

    .stats-card:hover {
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.12) !important;
        transform: translateY(-2px) !important;
    }

    .stats-icon {
        width: 36px !important;
        height: 36px !important;
        border-radius: 8px !important;
        background: linear-gradient(135deg, #4361ee, #5a72f0) !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        color: white !important;
        font-size: 18px !important;
        flex-shrink: 0 !important;
    }

    .stats-content {
        flex: 1 !important;
    }

    .stats-number {
        font-size: 20px !important;
        font-weight: 700 !important;
        color: #1f2937 !important;
        line-height: 1 !important;
        margin-bottom: 2px !important;
    }

    .stats-label {
        font-size: 11px !important;
        font-weight: 600 !important;
        color: #6b7280 !important;
        text-transform: uppercase !important;
        letter-spacing: 0.5px !important;
    }

    /* Modern Table Styling */
    .videos-table-container {
        background: #ffffff !important;
        border-radius: 16px !important;
        padding: 24px !important;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08) !important;
        margin-top: 24px !important;
    }

    .table {
        width: 100% !important;
        margin-bottom: 0 !important;
        font-size: 14px !important;
    }

    .table th {
        font-weight: 600 !important;
        color: #374151 !important;
        font-size: 12px !important;
        text-transform: uppercase !important;
        letter-spacing: 0.05em !important;
        padding: 12px 8px !important;
        border-bottom: 2px solid #f3f4f6 !important;
        background: #f8fafc !important;
        border-top: none !important;
    }

    .table td {
        padding: 16px 8px !important;
        border-bottom: 1px solid #f1f5f9 !important;
        vertical-align: middle !important;
        border-top: none !important;
    }

    .table tbody tr {
        transition: all 0.2s ease !important;
    }

    .table tbody tr:hover {
        background: #f8fafc !important;
    }

    /* Video Thumbnail - Very Small */
    .video-thumb {
        width: 40px !important;
        height: 30px !important;
        border-radius: 6px !important;
        object-fit: cover !important;
        box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1) !important;
    }

    /* Video Title */
    .video-title {
        font-weight: 600 !important;
        color: #1f2937 !important;
        font-size: 14px !important;
        margin-bottom: 2px !important;
    }

    .video-course {
        font-size: 11px !important;
        color: #6b7280 !important;
    }

    /* Status Badges */
    .badge {
        padding: 4px 8px !important;
        border-radius: 12px !important;
        font-weight: 600 !important;
        font-size: 10px !important;
        text-transform: uppercase !important;
        letter-spacing: 0.05em !important;
    }

    .badge.bg-success {
        background: linear-gradient(135deg, #10b981, #059669) !important;
        color: white !important;
    }

    .badge.bg-warning {
        background: linear-gradient(135deg, #f59e0b, #d97706) !important;
        color: white !important;
    }

    /* Action Buttons - Compact */
    .action-btn {
        width: 28px !important;
        height: 28px !important;
        padding: 0 !important;
        border-radius: 50% !important;
        display: inline-flex !important;
        align-items: center !important;
        justify-content: center !important;
        font-size: 12px !important;
        margin: 0 2px !important;
    }

    .btn-light {
        background: #f8fafc !important;
        border: 1px solid #e2e8f0 !important;
        color: #64748b !important;
    }

    .btn-light:hover {
        background: #e2e8f0 !important;
        border-color: #cbd5e1 !important;
        color: #475569 !important;
        transform: translateY(-1px) !important;
    }

    /* Table Column Widths - Very Compact */
    .col-thumb { width: 60px !important; }
    .col-title { width: auto !important; min-width: 200px !important; }
    .col-course { width: 120px !important; }
    .col-status { width: 80px !important; }
    .col-date { width: 100px !important; }
    .col-views { width: 60px !important; }
    .col-actions { width: 120px !important; }

    /* Search and Filters */
    .filter-container {
        background: #ffffff !important;
        border-radius: 12px !important;
        padding: 20px !important;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.06) !important;
        margin-bottom: 20px !important;
    }

    .form-control, .form-select {
        border: 2px solid #e5e7eb !important;
        border-radius: 8px !important;
        padding: 8px 12px !important;
        font-size: 13px !important;
        height: auto !important;
    }

    .form-control:focus, .form-select:focus {
        border-color: #4361ee !important;
        box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.1) !important;
        outline: none !important;
    }

    /* Button Styling */
    .btn {
        border-radius: 8px !important;
        padding: 8px 16px !important;
        font-weight: 600 !important;
        font-size: 13px !important;
        transition: all 0.2s ease !important;
    }

    .btn-primary {
        background: linear-gradient(135deg, #4361ee, #5a72f0) !important;
        border: none !important;
        color: white !important;
    }

    .btn-primary:hover {
        background: linear-gradient(135deg, #3b4de8, #4361ee) !important;
        transform: translateY(-1px) !important;
        box-shadow: 0 4px 12px rgba(67, 97, 238, 0.3) !important;
    }

    /* Responsive */
    @media (max-width: 768px) {
        .container-fluid {
            padding: 0 20px !important;
        }
        
        .videos-table-container {
            padding: 16px !important;
        }
        
        .table th, .table td {
            padding: 8px 4px !important;
        }
        
        .video-thumb {
            width: 30px !important;
            height: 22px !important;
        }
    }
    
    .hero-content {
        position: relative;
        z-index: 2;
    }
    
    .hero-title {
        font-size: 2.2rem;
        font-weight: 700;
        margin-bottom: 16px;
        display: flex;
        align-items: center;
    }
    
    .hero-subtitle {
        font-size: 1.05rem;
        line-height: 1.6;
        opacity: 0.9;
        margin-bottom: 24px;
        max-width: 600px;
    }
    
    .hero-stats {
        display: flex;
        gap: 3rem;
        margin-bottom: 28px;
        background: rgba(255,255,255,0.1);
        padding: 20px;
        border-radius: 16px;
        backdrop-filter: blur(5px);
        box-shadow: 0 4px 20px rgba(0,0,0,0.1);
    }
    
    .stat-item {
        text-align: center;
        padding: 0 10px;
        position: relative;
    }
    
    .stat-item:not(:last-child):after {
        content: "";
        position: absolute;
        right: -1.5rem;
        top: 50%;
        transform: translateY(-50%);
        height: 70%;
        width: 1px;
        background: rgba(255,255,255,0.2);
    }
    
    .stat-number {
        font-size: 2.4rem;
        font-weight: 700;
        margin-bottom: 8px;
        color: #fff;
        text-shadow: 0 2px 4px rgba(0,0,0,0.1);
        display: block;
    }
    
    .stat-label {
        font-size: 0.8rem;
        font-weight: 600;
        letter-spacing: 1.2px;
        opacity: 0.9;
        text-transform: uppercase;
    }
    
    .hero-image {
        max-width: 100%;
        height: auto;
        filter: drop-shadow(0 4px 20px rgba(0, 0, 0, 0.15));
    }
    
    /* Decorative elements */
    .decoration-shape {
        position: absolute;
        border-radius: 50%;
        background: rgba(255, 255, 255, 0.08);
        z-index: 1;
    }
    
    .shape-1 {
        width: 300px;
        height: 300px;
        top: -100px;
        right: -50px;
    }
    
    .shape-2 {
        width: 200px;
        height: 200px;
        bottom: -80px;
        right: 20%;
    }
    
    .shape-3 {
        width: 150px;
        height: 150px;
        bottom: 50px;
        left: 10%;
        background: rgba(255, 255, 255, 0.05);
    }
    
    .section-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 20px;
    }
    
    .videos-display {
        margin-top: 20px;
    }
    
    .modern-video-grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
        gap: 20px;
    }
    
    .modern-video-card {
        border-radius: 8px;
        overflow: hidden;
        transition: all 0.3s ease;
        background: white;
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
    }
    
    .video-card-inner {
        display: flex;
        flex-direction: column;
        height: 100%;
    }
    
    .video-thumbnail-container {
        position: relative;
        padding-top: 56.25%;
        overflow: hidden;
    }
    
    .video-thumbnail-container img {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        object-fit: cover;
        transition: transform 0.3s ease;
    }
    
    .video-overlay {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0,0,0,0.6);
        display: flex;
        justify-content: center;
        align-items: center;
    }
</style>

<style>
    /* Modern Video Grid Layout */
    .modern-video-grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
        gap: 24px;
        margin-bottom: 24px;
    }
    
    /* Video Card Base Styling */
    .modern-video-card {
        display: flex;
        flex-direction: column;
        background-color: #fff;
        border-radius: 10px;
        overflow: hidden;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
        transition: transform 0.3s ease, box-shadow 0.3s ease;
        height: 100%;
    }
    
    .modern-video-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 10px 20px rgba(0, 0, 0, 0.08);
    }
    
    /* Filter and Search Section Styling */
    .video-filters {
        background-color: #fff;
        border-radius: 10px;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
        padding: 16px 20px;
        margin-bottom: 24px;
    }
    
    .filter-row {
        display: flex;
        flex-wrap: wrap;
        gap: 16px;
        align-items: center;
    }
    
    .search-input-group {
        position: relative;
        flex-grow: 1;
        min-width: 200px;
    }
    
    .search-icon {
        position: absolute;
        left: 12px;
        top: 50%;
        transform: translateY(-50%);
        color: #6c757d;
        z-index: 5;
    }
    
    .search-input-group input {
        padding-left: 40px;
        border-radius: 8px;
        border: 1px solid #e2e8f0;
        height: 42px;
    }
    
    /* View Toggle Buttons */
    .btn-view-toggle {
        padding: 8px 12px;
        border-radius: 4px;
    }
    
    .btn-view-toggle.active {
        background-color: var(--primary-color);
        color: white;
        border-color: var(--primary-color);
    }
    
    /* Video List View Styles */
    .video-list-view {
        margin-top: 1rem;
    }
    
    .video-list-view .table {
        border-collapse: separate;
        border-spacing: 0 8px;
        overflow: hidden;
        height: 100%;
        display: flex;
        flex-direction: column;
    }
    
    .modern-video-card:hover {
        transform: translateY(-4px);
        box-shadow: 0 8px 16px rgba(0,0,0,0.12);
    }
    
    .video-card-inner {
        display: flex;
        flex-direction: column;
        height: 100%;
    }
    
    .video-thumbnail-container {
        position: relative;
        padding-top: 56.25%; /* 16:9 aspect ratio */
        overflow: hidden;
        background-color: #f5f5f5;
    }
    
    .video-thumbnail-img {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        object-fit: cover;
    }
    
    .video-overlay {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.2);
        display: flex;
        align-items: center;
        justify-content: center;
        opacity: 0;
        transition: opacity 0.3s ease;
    }
    
    .modern-video-card:hover .video-overlay {
        opacity: 1;
    }
    
    .play-button {
        width: 50px;
        height: 50px;
        background-color: rgba(255, 255, 255, 0.9);
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.2);
        transition: transform 0.2s ease;
    }
    
    .play-button:hover {
        transform: scale(1.1);
    }
    
    .duration-badge {
        position: absolute;
        bottom: 10px;
        right: 10px;
        background-color: rgba(0, 0, 0, 0.7);
        color: #fff;
        padding: 2px 8px;
        border-radius: 4px;
        font-size: 0.8rem;
        font-weight: 500;
    }
    
    .status-badge {
        position: absolute;
        top: 10px;
        right: 10px;
    }
    
    .video-status {
        display: inline-flex;
        align-items: center;
        padding: 4px 10px;
        border-radius: 20px;
        font-size: 0.75rem;
        font-weight: 500;
    }
    
    .status-published {
        background-color: rgba(25, 135, 84, 0.1);
        color: #198754;
    }
    
    .status-draft {
        background-color: rgba(255, 193, 7, 0.1);
        color: #fd7e14;
    }
    
    .video-card-content {
        padding: 16px;
        display: flex;
        flex-direction: column;
        flex-grow: 1;
    }
    
    .video-card-title {
        font-size: 1rem;
        font-weight: 600;
        margin-bottom: 8px;
        line-height: 1.4;
        color: #333;
    }
    
    .video-card-course {
        font-size: 0.85rem;
        color: #6c757d;
        margin-bottom: 12px;
        display: flex;
        align-items: center;
    }
    
    .video-card-meta {
        display: flex;
        justify-content: space-between;
        margin-bottom: 16px;
        font-size: 0.8rem;
        color: #6c757d;
    }
    
    .video-card-actions {
        display: flex;
        gap: 8px;
        margin-top: auto;
    }
    
    .video-card-actions .btn {
        padding: 0.375rem 0.75rem;
        font-size: 0.875rem;
        border-radius: 6px;
    }
    
    .video-card-actions .dropdown-menu {
        min-width: 200px;
        padding: 8px 0;
        border-radius: 8px;
        box-shadow: 0 4px 16px rgba(0,0,0,0.1);
        border: none;
    }
    
    .video-card-actions .dropdown-item {
    }
    
    .video-list-view .table thead th {
        border: none;
        background-color: #f8f9fa;
        padding: 12px 16px;
        font-weight: 600;
        color: #495057;
    }
    
    .video-list-view .table tbody tr {
        box-shadow: 0 2px 5px rgba(0,0,0,0.05);
        transition: all 0.2s ease;
        border-radius: 8px;
    }
    
    .video-list-view .table tbody tr:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 8px rgba(0,0,0,0.08);
    }
    
    .video-list-view .table tbody td {
        vertical-align: middle;
        padding: 12px 16px;
        background-color: #fff;
        border-top: none;
    }
    
    .video-list-view .table tbody tr td:first-child {
        border-top-left-radius: 8px;
        border-bottom-left-radius: 8px;
    }
    
    .video-list-view .table tbody tr td:last-child {
        border-top-right-radius: 8px;
        border-bottom-right-radius: 8px;
    }
    
    /* Responsive adjustments */
    @media (max-width: 768px) {
        .filter-row {
            flex-direction: column;
            align-items: stretch;
        }
        
        .search-input-group {
            margin-bottom: 12px;
        }
        
        .btn-group {
            align-self: flex-start;
        }
        
        .modern-video-grid {
            grid-template-columns: repeat(auto-fill, minmax(240px, 1fr));
        }
    }
    
    /* Empty state styling */
    .empty-state-modern {
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        padding: 40px 20px;
        background-color: #fff;
        border-radius: 10px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.05);
        margin: 20px 0;
        text-align: center;
    }
    
    .empty-state-icon {
        width: 80px;
        height: 80px;
        background-color: #f8f9fa;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        margin-bottom: 24px;
        color: #adb5bd;
    }
    
    .empty-state-icon svg {
        width: 36px;
        height: 36px;
    }
    
    .empty-state-title {
        font-size: 1.25rem;
        font-weight: 600;
        margin-bottom: 12px;
        color: #343a40;
    }
    
    .empty-state-text {
        color: #6c757d;
        margin-bottom: 24px;
        max-width: 400px;
    }
    
    /* Video Search and Filter Section */
    .video-filters {
        background-color: #fff;
        border-radius: 12px;
        padding: 16px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        margin-bottom: 24px;
    }
    
    .filter-row {
        display: flex;
        flex-wrap: wrap;
        gap: 16px;
        align-items: center;
        margin-bottom: 16px;
    }
    
    .search-input-group {
        flex: 1;
        min-width: 200px;
        position: relative;
    }
    
    .search-icon {
        position: absolute;
        left: 12px;
        top: 50%;
        transform: translateY(-50%);
        color: #adb5bd;
    }
    
    .search-input-group input {
        padding-left: 40px;
        border-radius: 8px;
    }
    
    /* Page title */
    .page-title {
        font-size: 1.75rem;
        font-weight: 600;
        margin-bottom: 10px;
    }
    
    /* View toggle button group */
    .btn-view-toggle {
        border-radius: 6px;
    }
    
    /* Fix navbar overlap issue - ensure content starts below fixed navbar */
    .page-content {
        padding-top: 100px !important; /* Account for fixed navbar height + margin */
    }
    
    /* Additional spacing for mobile devices */
    @media (max-width: 768px) {
        .page-content {
            padding-top: 90px !important;
        }
    }
</style>

        <!-- Small Stats Cards Section -->
        <div class="row mb-4" style="max-width: 50%;">
            <div class="col-md-4 mb-3">
                <div class="stats-card">
                    <div class="d-flex align-items-center">
                        <div class="stats-icon me-3" style="background: rgba(66, 97, 238, 0.1) !important; padding: 12px; border-radius: 8px;">
                            <i class='bx bx-video' style="font-size: 20px !important; color: #4261ee !important;"></i>
                        </div>
                        <div>
                            <div class="stats-number"><?php echo count($videos ?? []); ?></div>
                            <div class="stats-label">Total Videos</div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-4 mb-3">
                <div class="stats-card">
                    <div class="d-flex align-items-center">
                        <div class="stats-icon me-3" style="background: rgba(239, 68, 68, 0.1) !important; padding: 12px; border-radius: 8px;">
                            <i class='bx bx-graduation' style="font-size: 20px !important; color: #ef4444 !important;"></i>
                        </div>
                        <div>
                            <div class="stats-number"><?php echo count($courses ?? []); ?></div>
                            <div class="stats-label">Courses</div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-4 mb-3">
                <div class="stats-card">
                    <div class="d-flex align-items-center">
                        <div class="stats-icon me-3" style="background: rgba(34, 197, 94, 0.1) !important; padding: 12px; border-radius: 8px;">
                            <i class='bx bx-time' style="font-size: 20px !important; color: #22c55e !important;"></i>
                        </div>
                        <div>
                            <div class="stats-number">
                                <?php 
                                $totalMinutes = 0;
                                foreach ($videos ?? [] as $video) {
                                    if (isset($video['duration']) && is_numeric($video['duration'])) {
                                        $totalMinutes += (int)$video['duration'];
                                    }
                                }
                                echo floor($totalMinutes / 60);
                                ?>h
                            </div>
                            <div class="stats-label">Content Hours</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Alert Messages Section -->
            <?php if (!empty($message)): ?>
            <div class="alert alert-info alert-dismissible fade show mb-4 rounded-3" role="alert">
                <div class="d-flex align-items-center">
                    <svg class="svg-inline--fa fa-info-circle me-2" style="width: 20px; height: 20px;" aria-hidden="true" focusable="false" data-prefix="fas" data-icon="info-circle" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path fill="currentColor" d="M256 512A256 256 0 1 0 256 0a256 256 0 1 0 0 512zM216 336h24V272H216c-13.3 0-24-10.7-24-24s10.7-24 24-24h48c13.3 0 24 10.7 24 24v88h8c13.3 0 24 10.7 24 24s-10.7 24-24 24H216c-13.3 0-24-10.7-24-24s10.7-24 24-24zm40-208a32 32 0 1 1 0 64 32 32 0 1 1 0-64z"></path></svg>
                    <?php echo $message; ?>
                </div>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
            <?php endif; ?>
            
            <?php if (isset($error_message)): ?>
            <div class="alert alert-danger alert-dismissible fade show mb-4 rounded-3" role="alert">
                <div class="d-flex align-items-center">
                    <svg class="svg-inline--fa fa-exclamation-triangle me-2" style="width: 20px; height: 20px;" aria-hidden="true" focusable="false" data-prefix="fas" data-icon="exclamation-triangle" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path fill="currentColor" d="M256 32c14.2 0 27.3 7.5 34.5 19.8l216 368c7.3 12.4 7.3 27.7 .2 40.1S486.3 480 472 480H40c-14.3 0-27.6-7.7-34.7-20.1s-7-27.8 .2-40.1l216-368C228.7 39.5 241.8 32 256 32zm0 128c-13.3 0-24 10.7-24 24V296c0 13.3 10.7 24 24 24s24-10.7 24-24V184c0-13.3-10.7-24-24-24zm32 224a32 32 0 1 0 -64 0 32 32 0 1 0 64 0z"></path></svg>
                    <?php echo $error_message; ?>
                </div>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
            <?php endif; ?>

            <!-- Video Management Section -->
            <div class="video-management-section">
                <!-- Section Header with Title -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2 class="section-title d-flex align-items-center">
                        <svg class="svg-inline--fa fa-video me-2" style="width: 24px; height: 24px; color: var(--primary-color);" aria-hidden="true" focusable="false" data-prefix="fas" data-icon="video" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path fill="currentColor" d="M0 128C0 92.7 28.7 64 64 64H320c35.3 0 64 28.7 64 64V384c0 35.3-28.7 64-64 64H64c-35.3 0-64-28.7-64-64V128zM559.1 99.8c10.4 5.6 16.9 16.4 16.9 28.2V384c0 11.8-6.5 22.6-16.9 28.2s-23 5-32.9-1.6L384 336V176L526.2 101.4c9.9-6.5 22.5-7.2 32.9-1.6z"></path></svg>
                        Your Videos
                    </h2>
                    <a href="upload-video.php" class="btn btn-primary rounded-pill">
                        <svg class="svg-inline--fa fa-plus me-2" style="width: 16px; height: 16px;" aria-hidden="true" focusable="false" data-prefix="fas" data-icon="plus" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path fill="currentColor" d="M256 80c0-17.7-14.3-32-32-32s-32 14.3-32 32V224H48c-17.7 0-32 14.3-32 32s14.3 32 32 32H192V432c0 17.7 14.3 32 32 32s32-14.3 32-32V288H400c17.7 0 32-14.3 32-32s-14.3-32-32-32H256V80z"></path></svg>
                        Upload New Video
                    </a>
                </div>

                <!-- Search and Filter Card -->
                <div class="filter-container">
                    <div class="row align-items-center g-3">
                        <div class="col-md-4">
                            <div class="position-relative">
                                <i class='bx bx-search' style="position: absolute; left: 12px; top: 50%; transform: translateY(-50%); color: #9ca3af; font-size: 14px;"></i>
                                <input type="text" class="form-control ps-5" id="videoSearch" placeholder="Search videos...">
                            </div>
                        </div>
                        <?php if (!empty($courses)): ?>
                        <div class="col-md-3">
                            <select class="form-select" id="courseFilter">
                                <option value="">All Courses</option>
                                <?php foreach ($courses as $course): ?>
                                <option value="<?php echo $course['course_id']; ?>" <?php echo ($course_filter == $course['course_id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($course['title']); ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <?php endif; ?>
                        <div class="col-md-2">
                            <select class="form-select" id="statusFilter">
                                <option value="">All Status</option>
                                <option value="published">Published</option>
                                <option value="draft">Draft</option>
                            </select>
                        </div>
                    </div>
                </div>

                <?php if (!empty($courses_with_videos)): ?>
                    <!-- Course Videos Accordion -->
                    <div class="course-videos-accordion">
                        <div class="accordion" id="courseVideosAccordion">
                            <?php foreach ($courses_with_videos as $course): ?>
                            <div class="accordion-item course-accordion-item">
                                <h2 class="accordion-header" id="heading<?php echo $course['course_id']; ?>">
                                    <button class="accordion-button collapsed course-accordion-button" type="button" data-bs-toggle="collapse" data-bs-target="#collapse<?php echo $course['course_id']; ?>" aria-expanded="false" aria-controls="collapse<?php echo $course['course_id']; ?>">
                                        <div class="course-header-content">
                                            <!-- Course Thumbnail -->
                                            <div class="course-thumbnail">
                                                <img src="<?php echo !empty($course['course_thumbnail']) ? $course['course_thumbnail'] : '../assets/images/placeholder.png'; ?>" 
                                                     alt="<?php echo htmlspecialchars($course['course_title']); ?>" 
                                                     class="course-thumb-img">
                                            </div>
                                            
                                            <!-- Course Info -->
                                            <div class="course-info">
                                                <div class="course-title"><?php echo htmlspecialchars($course['course_title']); ?></div>
                                                <div class="course-meta">
                                                    <span class="video-count"><?php echo count($course['videos']); ?> videos</span>
                                                    <span class="course-status">
                                                        <span class="badge <?php echo $course['course_published'] ? 'bg-success' : 'bg-warning'; ?>">
                                                            <?php echo $course['course_published'] ? 'Published' : 'Draft'; ?>
                                                        </span>
                                                    </span>
                                                </div>
                                            </div>
                                        </div>
                                    </button>
                                </h2>
                                <div id="collapse<?php echo $course['course_id']; ?>" class="accordion-collapse collapse" aria-labelledby="heading<?php echo $course['course_id']; ?>" data-bs-parent="#courseVideosAccordion">
                                    <div class="accordion-body">
                                        <?php if (!empty($course['videos'])): ?>
                                            <div class="videos-grid">
                                                <?php foreach ($course['videos'] as $video): ?>
                                                <div class="video-card">
                                                    <div class="video-thumbnail">
                                                        <img src="<?php 
                                                              if (!empty($video['thumbnail'])) {
                                                                  $thumb = $video['thumbnail'];
                                                                  if (strpos($thumb, 'http://') === 0 || strpos($thumb, 'https://') === 0) {
                                                                      echo $thumb;
                                                                  } else {
                                                                      echo '../' . ltrim($thumb, '/');
                                                                  }
                                                              } else {
                                                                  echo 'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" width="160" height="90" viewBox="0 0 160 90"%3E%3Crect width="160" height="90" fill="%23f3f4f6"/%3E%3Cpath d="M60 30l40 15-40 15z" fill="%236b7280"/%3E%3C/svg%3E';
                                                              }
                                                              ?>" 
                                                             alt="<?php echo htmlspecialchars($video['title']); ?>" 
                                                             class="video-thumb-img">
                                                        <div class="video-overlay">
                                                            <i class="fas fa-play play-icon"></i>
                                                        </div>
                                                    </div>
                                                    <div class="video-info">
                                                        <h6 class="video-title"><?php echo htmlspecialchars($video['title']); ?></h6>
                                                        <div class="video-meta">
                                                            <span class="video-date"><?php echo date('M j, Y', strtotime($video['created_at'])); ?></span>
                                                            <span class="video-status">
                                                                <span class="badge <?php echo $video['is_published'] ? 'bg-success' : 'bg-warning'; ?>">
                                                                    <?php echo $video['is_published'] ? 'Published' : 'Draft'; ?>
                                                                </span>
                                                            </span>
                                                        </div>
                                                        <div class="video-actions">
                                                            <a href="view_video.php?id=<?php echo $video['video_id']; ?>" class="btn btn-sm btn-primary" title="View Video">
                                                                <i class="fas fa-play"></i> View
                                                            </a>
                                                            <a href="edit-video.php?id=<?php echo $video['video_id']; ?>" class="btn btn-sm btn-outline-primary" title="Edit Video">
                                                                <i class="fas fa-edit"></i> Edit
                                                            </a>
                                                            <button onclick="deleteVideo(<?php echo $video['video_id']; ?>)" class="btn btn-sm btn-outline-danger" title="Delete Video">
                                                                <i class="fas fa-trash"></i>
                                                            </button>
                                                        </div>
                                                    </div>
                                                </div>
                                                <?php endforeach; ?>
                                            </div>
                                        <?php else: ?>
                                            <div class="no-videos-message">
                                                <div class="text-center py-4">
                                                    <i class="fas fa-video text-muted mb-3" style="font-size: 3rem;"></i>
                                                    <h5>No videos in this course yet</h5>
                                                    <p class="text-muted">Start adding videos to make your course content available to students.</p>
                                                    <a href="upload-video.php?course_id=<?php echo $course['course_id']; ?>" class="btn btn-primary">
                                                        <i class="fas fa-plus me-2"></i>Upload First Video
                                                    </a>
                                                </div>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php else: ?>
                    <!-- Empty State when no videos exist -->
                    <div class="empty-state-modern">
                        <div class="empty-state-icon">
                            <svg class="svg-inline--fa fa-video" style="width: 80px; height: 80px; color: #e2e8f0;" aria-hidden="true" focusable="false" data-prefix="fas" data-icon="video" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path fill="currentColor" d="M0 128C0 92.7 28.7 64 64 64H320c35.3 0 64 28.7 64 64V384c0 35.3-28.7 64-64 64H64c-35.3 0-64-28.7-64-64V128zM559.1 99.8c10.4 5.6 16.9 16.4 16.9 28.2V384c0 11.8-6.5 22.6-16.9 28.2s-23 5-32.9-1.6L384 336V176L526.2 101.4c9.9-6.5 22.5-7.2 32.9-1.6z"></path></svg>
                        </div>
                        <h3 class="empty-state-title">No Videos Yet</h3>
                        <p class="empty-state-message">You haven't uploaded any videos yet. Start creating engaging content for your students.</p>
                        <a href="upload-video.php" class="btn btn-primary btn-lg rounded-pill">
                            <svg class="svg-inline--fa fa-plus me-2" style="width: 16px; height: 16px;" aria-hidden="true" focusable="false" data-prefix="fas" data-icon="plus" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path fill="currentColor" d="M256 80c0-17.7-14.3-32-32-32s-32 14.3-32 32V224H48c-17.7 0-32 14.3-32 32s14.3 32 32 32H192V432c0 17.7 14.3 32 32 32s32-14.3 32-32V288H400c17.7 0 32-14.3 32-32s-14.3-32-32-32H256V80z"></path></svg> Upload Your First Video
                        </a>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Course Videos Accordion Styling -->
    <style>
        /* Course Videos Accordion */
        .course-videos-accordion {
            max-width: 100%;
        }

        .course-accordion-item {
            border: none !important;
            border-radius: 12px !important;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05) !important;
            margin-bottom: 16px !important;
            overflow: hidden !important;
        }

        .course-accordion-button {
            border: none !important;
            background: white !important;
            padding: 16px 20px !important;
            border-radius: 12px !important;
            box-shadow: none !important;
            font-weight: 500 !important;
        }

        .course-accordion-button:focus {
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.1) !important;
            border-color: transparent !important;
        }

        .course-accordion-button:not(.collapsed) {
            background: white !important;
            color: #1f2937 !important;
        }

        .course-header-content {
            display: flex;
            align-items: center;
            gap: 16px;
            width: 100%;
        }

        .course-thumbnail {
            flex-shrink: 0;
        }

        .course-thumb-img {
            width: 50px;
            height: 50px;
            object-fit: cover;
            border-radius: 8px;
            border: 2px solid #f1f5f9;
        }

        .course-info {
            flex: 1;
            text-align: left;
        }

        .course-title {
            font-size: 16px;
            font-weight: 600;
            color: #1f2937;
            margin-bottom: 4px;
        }

        .course-meta {
            display: flex;
            align-items: center;
            gap: 12px;
            font-size: 14px;
            color: #6b7280;
        }

        .video-count {
            font-weight: 500;
        }

        /* Accordion Body */
        .accordion-body {
            padding: 0 !important;
            background: #fafbfc;
        }

        /* Videos Grid - Compact Layout */
        .videos-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
            gap: 12px;
            padding: 16px;
        }

        .video-card {
            background: white;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.08);
            transition: all 0.2s ease;
            border: 1px solid #f1f5f9;
        }

        .video-card:hover {
            transform: translateY(-1px);
            box-shadow: 0 3px 8px rgba(0, 0, 0, 0.12);
        }

        .video-thumbnail {
            position: relative;
            width: 100%;
            height: 110px;
            overflow: hidden;
            background: #f8fafc;
        }

        .video-thumb-img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        .video-overlay {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.3);
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0;
            transition: opacity 0.2s ease;
        }

        .video-card:hover .video-overlay {
            opacity: 1;
        }

        .play-icon {
            color: white;
            font-size: 1.5rem;
        }

        .video-info {
            padding: 12px;
        }

        .video-title {
            font-size: 13px;
            font-weight: 600;
            color: #1f2937;
            margin-bottom: 6px;
            line-height: 1.3;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }

        .video-meta {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 8px;
            font-size: 11px;
        }

        .video-date {
            color: #6b7280;
        }

        .video-actions {
            display: flex;
            gap: 4px;
            flex-wrap: wrap;
        }

        .video-actions .btn {
            border-radius: 6px;
            font-size: 11px;
            padding: 4px 8px;
            font-weight: 500;
            flex: 1;
            text-align: center;
        }

        .video-actions .btn i {
            font-size: 10px;
        }

        /* No Videos Message */
        .no-videos-message {
            padding: 40px 20px;
            text-align: center;
        }

        .no-videos-message i {
            opacity: 0.5;
        }

        .no-videos-message h5 {
            color: #374151;
            margin-bottom: 8px;
        }

        .no-videos-message p {
            margin-bottom: 20px;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .course-header-content {
                gap: 12px;
            }

            .course-thumb-img {
                width: 40px;
                height: 40px;
            }

            .course-title {
                font-size: 14px;
            }

            .course-meta {
                font-size: 12px;
                gap: 8px;
            }

            .videos-grid {
                grid-template-columns: 1fr;
                gap: 12px;
                padding: 12px;
            }

            .video-actions {
                gap: 3px;
            }

            .video-actions .btn {
                font-size: 10px;
                padding: 3px 6px;
            }

            .video-actions {
                justify-content: center;
            }
        }

        /* Badge customizations */
        .badge {
            font-size: 9px !important;
            padding: 2px 6px !important;
            border-radius: 4px !important;
        }
    </style>

    <!-- Layout Fixes CSS -->
    <style>
    /* Fix horizontal scrolling only - let natural vertical scrolling work */
    body {
        overflow-x: hidden !important;
    }

    .main-content {
        padding: 20px !important;
        overflow-x: hidden !important;
    }

    .container-fluid {
        padding-left: 24px !important;
        padding-right: 24px !important;
        max-width: 100% !important;
    }

    .row {
        margin-left: 0 !important;
        margin-right: 0 !important;
    }

    .col-12, .col-md-6, .col-lg-3, .col-lg-4, .col-lg-8 {
        padding-left: 12px !important;
        padding-right: 12px !important;
    }

    /* Videos page specific fixes */
    .video-stats {
        padding: 15px !important;
        margin-bottom: 20px !important;
    }

    .card {
        margin-bottom: 20px !important;
        border-radius: 12px !important;
        box-shadow: 0 2px 10px rgba(0,0,0,0.08) !important;
    }

    .card-body {
        padding: 20px !important;
    }

    /* Table responsive fixes */
    .table-responsive {
        overflow-x: auto !important;
        -webkit-overflow-scrolling: touch !important;
    }

    @media (max-width: 768px) {
        .main-content {
            padding: 15px !important;
        }
        
        .container-fluid {
            padding-left: 15px !important;
            padding-right: 15px !important;
        }
        
        .col-12, .col-md-6, .col-lg-3, .col-lg-4, .col-lg-8 {
            padding-left: 8px !important;
            padding-right: 8px !important;
        }
        
        .card-body {
            padding: 15px !important;
        }
    }
    </style>

    <!-- DataTables Scripts -->
    <script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap5.min.js"></script>
    
    <script>
        $(document).ready(function() {
            // Video-specific functionality

            // Initialize DataTable for videos table
            $('#videosTable').DataTable({
                responsive: true,
                pageLength: 25,
                order: [[4, 'desc']], // Sort by date column by default
                columnDefs: [
                    { orderable: false, targets: [0, 5] } // Disable sorting on thumbnail and actions columns
                ]
            });

            // Enhanced search functionality using DataTables
            $('#videoSearch').on('keyup', function() {
                $('#videosTable').DataTable().search(this.value).draw();
            });

            // Course filter functionality
            $('#courseFilter').on('change', function() {
                const selectedCourse = $(this).val();
                if (selectedCourse) {
                    $('#videosTable').DataTable().column(2).search(selectedCourse).draw();
                } else {
                    $('#videosTable').DataTable().column(2).search('').draw();
                }
            });
            
            // Enhanced keyboard shortcuts
            $(document).keydown(function(e) {
                // Ctrl/Cmd + F for search
                if ((e.ctrlKey || e.metaKey) && e.keyCode === 70) {
                    e.preventDefault();
                    $('#videoSearch').focus();
                }
                
                // Esc to clear search
                if (e.keyCode === 27) {
                    $('#videoSearch').val('');
                    $('#videosTable').DataTable().search('').draw();
                }
            });
        });

        // Global functions for video actions
        function clearFilters() {
            $('#videoSearch').val('');
            $('#courseFilter').val('');
            $('#videosTable').DataTable().search('').column(2).search('').draw();
        }
        
        function deleteVideo(videoId) {
            if (confirm('Are you sure you want to delete this video? This action cannot be undone.')) {
                window.location.href = 'videos.php?delete=' + videoId;
            }
        }
        
        function togglePublish(videoId) {
            window.location.href = 'videos.php?toggle_publish=' + videoId;
        }
        
        function duplicateVideo(videoId) {
            if (confirm('Do you want to create a duplicate of this video?')) {
                window.location.href = 'videos.php?duplicate=' + videoId;
            }
        }

        function showToast(message, type = 'info') {
            const bgColor = type === 'success' ? '#28a745' : type === 'error' ? '#dc3545' : '#007bff';
            
            Toastify({
                text: message,
                duration: 3000,
                gravity: "top",
                position: "right",
                backgroundColor: bgColor,
                stopOnFocus: true,
                onClick: function() {
                    this.hideToast();
                }
            }).showToast();
        }
    </script>

    <?php
    // Include the layout end file
    require_once('components/teacher_layout_end.php');
    ?>
