<?php
/**
 * Email Verification Page - Panadite Academy
 */

require_once __DIR__ . '/config/email_verification_service.php';
require_once __DIR__ . '/auth/functions.php';

// Check if already logged in
if (isLoggedIn()) {
    $role = $_SESSION['user_role'];
    if ($role === 'admin') {
        header('Location: /admin/dashboard.php');
    } else if ($role === 'teacher') {
        header('Location: /teacher/dashboard.php');
    } else {
        header('Location: /student/dashboard.php');
    }
    exit();
}

$message = '';
$success = false;
$error = false;
$result = null; // Initialize result variable

// Handle email verification
if (isset($_GET['token'])) {
    $token = trim($_GET['token']);
    
    if (empty($token)) {
        $error = true;
        $message = 'Invalid verification link.';
    } else {
        try {
            $result = verifyEmailAndCreateAccount($token);
            
            if ($result['success']) {
                $success = true;
                $message = $result['message'];
                
                // Optional: Auto-login the user after verification
                // You can uncomment this if you want automatic login
                /*
                $_SESSION['logged_in'] = true;
                $_SESSION['user_id'] = $result['user_id'];
                $_SESSION['user_role'] = $result['role'];
                */
            } else {
                $error = true;
                $message = $result['message'];
            }
        } catch (Exception $e) {
            $error = true;
            $message = 'System error during verification: ' . $e->getMessage();
        } catch (Error $e) {
            $error = true;
            $message = 'PHP Error during verification: ' . $e->getMessage();
        }
    }
} else {
    $error = true;
    $message = 'No verification token provided.';
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Email Verification - Panadite Academy</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        body {
            background: #f8f9fa;
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
        }
        .verification-section {
            min-height: 80vh;
            display: flex;
            align-items: center;
            padding: 60px 0;
        }
        .verification-card {
            background: white;
            border-radius: 16px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.08);
            border: 1px solid rgba(39, 107, 228, 0.1);
            overflow: hidden;
        }
        .card-header {
            background: linear-gradient(135deg, #276BE4 0%, #1e5bb8 100%);
            padding: 40px;
            text-align: center;
            color: white;
        }
        .success-icon {
            font-size: 4rem;
            color: #28a745;
            animation: bounceIn 0.8s ease-out;
        }
        .error-icon {
            font-size: 4rem;
            color: #dc3545;
            animation: shake 0.6s ease-in-out;
        }
        @keyframes bounceIn {
            0% { transform: scale(0); opacity: 0; }
            50% { transform: scale(1.1); opacity: 0.8; }
            100% { transform: scale(1); opacity: 1; }
        }
        @keyframes shake {
            0%, 100% { transform: translateX(0); }
            25% { transform: translateX(-8px); }
            75% { transform: translateX(8px); }
        }
        .btn-primary {
            background: #276BE4;
            border: none;
            padding: 12px 28px;
            border-radius: 8px;
            font-weight: 600;
            transition: all 0.3s ease;
        }
        .btn-primary:hover {
            background: #1e5bb8;
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(39, 107, 228, 0.3);
        }
        .btn-outline-secondary {
            border-color: #276BE4;
            color: #276BE4;
            font-weight: 600;
        }
        .btn-outline-secondary:hover {
            background: #276BE4;
            border-color: #276BE4;
        }
        .gradient-text {
            background: linear-gradient(135deg, #276BE4, #FFA915);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <section class="verification-section">
        <div class="container">
            <div class="row justify-content-center">
                <div class="col-md-8 col-lg-6">
                    <div class="verification-card">
                        <!-- Card Header -->
                        <div class="card-header">
                            <div class="mb-3">
                                <?php if ($success): ?>
                                    <i class="fas fa-check-circle success-icon"></i>
                                    <h2 class="mt-3 mb-2 fw-bold">✅ Verification Successful!</h2>
                                    <p class="mb-0 opacity-90">Welcome to Panadite Academy</p>
                                <?php else: ?>
                                    <i class="fas fa-times-circle error-icon"></i>
                                    <h2 class="mt-3 mb-2 fw-bold">❌ Verification Failed</h2>
                                    <p class="mb-0 opacity-90">Please try again or contact support</p>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Card Body -->
                        <div class="p-5 text-center">
                            <!-- Message -->
                            <div class="mb-4">
                                <p class="text-muted fs-5 lh-base"><?= htmlspecialchars($message) ?></p>
                            </div>



                            <?php if ($success): ?>
                                <!-- Success Actions -->
                                <div class="d-grid gap-3">
                                    <a href="/auth/login.php" class="btn btn-primary btn-lg">
                                        <i class="fas fa-sign-in-alt me-2"></i> Login to Your Account
                                    </a>
                                    <a href="/" class="btn btn-outline-secondary">
                                        <i class="fas fa-home me-2"></i> Back to Home
                                    </a>
                                </div>
                                
                                <!-- Success Info -->
                                <div class="mt-4 p-4 bg-light rounded-3 border-start border-success border-4">
                                    <div class="d-flex align-items-center">
                                        <div class="d-flex align-items-center justify-content-center me-3" style="width: 40px; height: 40px; background: rgba(40, 167, 69, 0.1); border-radius: 50%;">
                                            <i class="fas fa-info-circle" style="color: #28a745;"></i>
                                        </div>
                                        <div>
                                            <h6 class="mb-1 fw-bold" style="color: #28a745;">Account Created Successfully</h6>
                                            <small class="text-muted">
                                                Your account has been created and is ready to use.
                                                <?php if (isset($result) && is_array($result) && isset($result['role']) && $result['role'] === 'teacher'): ?>
                                                    <br><strong>Note:</strong> Your teacher profile is pending admin approval.
                                                <?php endif; ?>
                                            </small>
                                        </div>
                                    </div>
                                </div>
                                
                            <?php elseif ($error): ?>
                                <!-- Error Actions -->
                                <div class="d-grid gap-3">
                                    <a href="/auth/register.php" class="btn btn-primary btn-lg">
                                        <i class="fas fa-user-plus me-2"></i> Try Registration Again
                                    </a>
                                    <button class="btn btn-outline-secondary" onclick="showResendForm()">
                                        <i class="fas fa-envelope me-2"></i> Resend Verification Email
                                    </button>
                                </div>
                                
                                <!-- Resend Email Form (Initially Hidden) -->
                                <div id="resendForm" class="mt-4" style="display: none;">
                                    <div class="card bg-light border-0">
                                        <div class="card-body">
                                            <h6 class="card-title mb-3"><i class="fas fa-paper-plane me-2 text-primary"></i>Resend Verification Email</h6>
                                            <form method="post" action="resend-verification.php">
                                                <div class="input-group">
                                                    <span class="input-group-text"><i class="fas fa-envelope" style="color: #276BE4;"></i></span>
                                                    <input type="email" name="email" class="form-control" placeholder="Enter your email address" required>
                                                    <button type="submit" class="btn btn-primary">
                                                        <i class="fas fa-send me-1"></i> Send
                                                    </button>
                                                </div>
                                                <small class="text-muted mt-2 d-block">
                                                    Enter the email address you used during registration
                                                </small>
                                            </form>
                                        </div>
                                    </div>
                                </div>
                            <?php endif; ?>

                            <!-- Support Section -->
                            <hr class="my-5">
                            <div class="text-center">
                                <p class="text-muted mb-2">Need assistance?</p>
                                <a href="mailto:admin@panaditeacademy.com" class="text-decoration-none" style="color: #276BE4; font-weight: 600;">
                                    <i class="fas fa-life-ring me-2"></i>Contact Support
                                </a>
                            </div>
                    </div>


                    </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

   <!-- Footer -->
<footer class="mt-5 position-relative">
        <!-- Top gradient bar -->
        <div class="position-absolute top-0 start-0 w-100" style="height: 4px; background: linear-gradient(90deg, #276BE4, #FFA915);"></div>
        
        <!-- Main footer content -->
        <div class="bg-white py-5">
            <div class="container">
                <div class="row gy-4">
                    <!-- Logo and description -->
                    <div class="col-lg-4 col-md-6 mb-4">
                        <div class="mb-4">
                            <img src="/assets/images/logo.jpeg" alt="Panadite Academy" style="height: 50px;">
                        </div>
                        <p class="mb-2" style="color: #333; font-weight: 500;">Empowering learners through high-quality educational content.</p>
                        <p style="color: #333;">Get ahead of the curve with our premium videos and courses.</p>
                    </div>
                    
                    <!-- Quick links -->
                    <div class="col-lg-2 col-md-6 mb-4">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Quick Links
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled">
                            <li class="mb-2"><a href="/index.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Home</a></li>
                            <li class="mb-2"><a href="/courses.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Courses</a></li>
                            <li class="mb-2"><a href="/videos.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Videos</a></li>
                            <li class="mb-2"><a href="/about.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>About Us</a></li>
                            <li class="mb-2"><a href="/contact.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Contact</a></li>
                        </ul>
                    </div>
                    
                    <!-- Facilitator links -->
                    <div class="col-lg-2 col-md-6 mb-4">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Facilitator
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled">
                            <li class="mb-2"><a href="/auth/register.php?role=teacher" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Become a Facilitator</a></li>
                            <li class="mb-2"><a href="/teacher/dashboard.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Teacher Dashboard</a></li>
                            <li class="mb-2"><a href="/teacher/sales.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Your Sales</a></li>
                            <li class="mb-2"><a href="/teacher/help.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Teacher Help</a></li>
                        </ul>
                    </div>
                    
                    <!-- Contact info -->
                    <div class="col-lg-4 col-md-6">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Contact
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled mb-4">
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-map-marker-alt" style="color: #276BE4;"></i>
                                </span>
                                <span style="color: #333;">296 Pretorius Street, Corner Lilian Ngoyi, Pretoria Central, 0001, South Africa</span>
                            </li>
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-envelope" style="color: #276BE4;"></i>
                                </span>
                                <a href="mailto:admin@panaditeacademy.com" class="text-decoration-none" style="color: #333;">admin@panaditeacademy.com</a>
                            </li>
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-phone" style="color: #276BE4;"></i>
                                </span>
                                <a href="tel:+27747223141" class="text-decoration-none" style="color: #333;">+27 74 722 3141</a>
                            </li>
                        </ul>
                        
                        <!-- Social icons -->
                        <div class="d-flex">
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-facebook-f"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-twitter"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-instagram"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-linkedin-in"></i>
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Copyright section -->
        <div style="background: #f8f9fa; border-top: 1px solid #eee;" class="py-3">
            <div class="container">
                <div class="row align-items-center">
                    <div class="col-md-6 mb-3 mb-md-0">
                        <p class="mb-0" style="color: #333; font-weight: 500;">&copy; <?php echo date('Y'); ?> Panadite Academy. All rights reserved.</p>
                    </div>
                    <div class="col-md-6 text-md-end">
                        <a href="/privacy-policy.php" class="text-decoration-none me-3" style="color: #333;">Privacy Policy</a>
                        <a href="/terms.php" class="text-decoration-none me-3" style="color: #333;">Terms of Use</a>
                        <a href="/sitemap.php" class="text-decoration-none" style="color: #333;">Sitemap</a>
                    </div>
                </div>
            </div>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function showResendForm() {
            const form = document.getElementById('resendForm');
            if (form.style.display === 'none') {
                form.style.display = 'block';
                form.scrollIntoView({ behavior: 'smooth', block: 'center' });
            } else {
                form.style.display = 'none';
            }
        }

        // Smooth animations on page load
        document.addEventListener('DOMContentLoaded', function() {
            const card = document.querySelector('.verification-card');
            card.style.opacity = '0';
            card.style.transform = 'translateY(30px)';
            
            setTimeout(() => {
                card.style.transition = 'all 0.6s ease-out';
                card.style.opacity = '1';
                card.style.transform = 'translateY(0)';
            }, 100);
        });

        <?php if ($success): ?>
        // Check for guest cart items and provide appropriate messaging
        document.addEventListener('DOMContentLoaded', function() {
            checkGuestCartOnVerification();
        });
        
        function checkGuestCartOnVerification() {
            const guestCart = localStorage.getItem('panadite_guest_cart');
            if (guestCart) {
                try {
                    const cartItems = JSON.parse(guestCart);
                    if (cartItems && cartItems.length > 0) {
                        // Show cart preservation message
                        const messageDiv = document.querySelector('.verification-card .p-5');
                        if (messageDiv) {
                            const cartMessage = document.createElement('div');
                            cartMessage.className = 'alert alert-info mt-4';
                            cartMessage.innerHTML = `
                                <div class="d-flex align-items-center">
                                    <i class="fas fa-shopping-cart me-3 fs-4"></i>
                                    <div>
                                        <strong>🛒 Your Cart is Preserved!</strong><br>
                                        <small class="text-muted">You have ${cartItems.length} course${cartItems.length > 1 ? 's' : ''} in your cart. They will be available when you log in.</small>
                                    </div>
                                </div>
                            `;
                            messageDiv.appendChild(cartMessage);
                        }
                        
                        // Update login button to emphasize cart transfer and set redirect flag
                        const loginButton = document.querySelector('a[href*="login"]');
                        if (loginButton) {
                            loginButton.innerHTML = '<i class="fas fa-sign-in-alt me-2"></i>Login & View Cart (' + cartItems.length + ' items)';
                            loginButton.className = 'btn btn-success btn-lg px-4 py-2';
                            
                            // Add click handler to set new login flag for cart redirect
                            loginButton.addEventListener('click', function(e) {
                                sessionStorage.setItem('just_logged_in', 'true');
                                sessionStorage.setItem('has_cart_items', cartItems.length.toString());
                                console.log('🔄 Login flag set for cart redirect after login');
                            });
                        }
                        
                        console.log('✅ Guest cart detected for new user:', cartItems.length, 'items');
                    }
                } catch (error) {
                    console.error('Error checking guest cart:', error);
                }
            }
        }
        
        // Optional: Auto-redirect after success (uncomment if desired)
        // setTimeout(() => {
        //     window.location.href = '/auth/login.php';
        // }, 8000);
        <?php endif; ?>
    </script>
</body>
</html>
