<?php
require_once '../auth/functions.php';
require_once '../includes/config.php';
require_once '../includes/db_connection.php';

// Ensure user is logged in as an admin
requireRole('admin');

$userId = $_SESSION['user_id'];
$user = getCurrentUser();

// Function to check if a column exists in a table
function columnExists($conn, $table, $column) {
    $sql = "SHOW COLUMNS FROM `$table` LIKE '$column'";
    $result = $conn->query($sql);
    return $result->num_rows > 0;
}

// Check if category_description column exists
$hasDescriptionColumn = columnExists($conn, 'categories', 'category_description');

// Initialize variables
$successMsg = '';
$errorMsg = '';
$categoryName = '';
$categoryDescription = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate input
    $categoryName = trim($_POST['category_name']);
    $categoryDescription = trim($_POST['category_description']);
    
    // Basic validation
    if (empty($categoryName)) {
        $errorMsg = "Category name is required.";
    } else {
        // Check if category already exists
        $stmt = $conn->prepare("SELECT category_id FROM categories WHERE category_name = ?");
        $stmt->bind_param("s", $categoryName);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $errorMsg = "Category already exists.";
        } else {
            // Insert new category
            if ($hasDescriptionColumn) {
                $stmt = $conn->prepare("INSERT INTO categories (category_name, category_description) VALUES (?, ?)");
                $stmt->bind_param("ss", $categoryName, $categoryDescription);
            } else {
                $stmt = $conn->prepare("INSERT INTO categories (category_name) VALUES (?)");
                $stmt->bind_param("s", $categoryName);
            }
            
            if ($stmt->execute()) {
                $successMsg = "Category added successfully!";
                $categoryName = '';
                $categoryDescription = '';
            } else {
                $errorMsg = "Error adding category: " . $conn->error;
            }
        }
    }
}

// Fetch existing categories for display
$categories = [];
try {
    $query = "SELECT category_id, category_name";
    if ($hasDescriptionColumn) {
        $query .= ", category_description";
    } else {
        $query .= ", '' as category_description";
    }
    $query .= " FROM categories ORDER BY category_name";
    
    $stmt = $conn->prepare($query);
    $stmt->execute();
    $result = $stmt->get_result();
    
    while ($row = $result->fetch_assoc()) {
        $categories[] = $row;
    }
} catch (Exception $e) {
    $errorMsg = "Error fetching categories: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add Category | Panadite Admin</title>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Boxicons CSS -->
    <link href="https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css" rel="stylesheet">
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        /* Modern color variables */
        :root {
            --primary-color: #4e73df;
            --success-color: #1cc88a;
            --info-color: #36b9cc;
            --warning-color: #f6c23e;
            --danger-color: #e74a3b;
            --secondary-color: #858796;
            --light-color: #f8f9fc;
            --dark-color: #5a5c69;
        }
        
        /* Card hover effects */
        .hover-shadow {
            transition: all 0.3s ease;
        }
        .hover-shadow:hover {
            transform: translateY(-5px);
            box-shadow: 0 0.5rem 1.5rem rgba(0, 0, 0, 0.15) !important;
        }
        
        /* Icon wrapper styles */
        .icon-wrapper {
            width: 48px;
            height: 48px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        /* Background subtle colors for icons */
        .bg-primary-subtle {
            background-color: rgba(78, 115, 223, 0.1);
        }
        .bg-success-subtle {
            background-color: rgba(28, 200, 138, 0.1);
        }
        .bg-info-subtle {
            background-color: rgba(54, 185, 204, 0.1);
        }
        .bg-warning-subtle {
            background-color: rgba(246, 194, 62, 0.1);
        }
        .bg-danger-subtle {
            background-color: rgba(231, 74, 59, 0.1);
        }
        
        /* Form styles */
        .form-control:focus {
            border-color: #4e73df;
            box-shadow: 0 0 0 0.25rem rgba(78, 115, 223, 0.25);
        }
        
        .btn-primary {
            background-color: #4e73df;
            border-color: #4e73df;
        }
        
        .btn-primary:hover {
            background-color: #3c5ec9;
            border-color: #3c5ec9;
        }
        
        /* Table styles */
        .table-responsive {
            border-radius: 0.5rem;
            overflow: hidden;
        }
        
        /* Category badge */
        .category-badge {
            font-size: 0.8rem;
            padding: 0.35rem 0.8rem;
            border-radius: 50rem;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="container-fluid p-4">
                <!-- Page Header -->
                <div class="d-sm-flex justify-content-between align-items-center mb-4">
                    <div class="page-header-wrapper">
                        <h2 class="mb-0 fw-bold">
                            <div class="d-flex align-items-center">
                                <div class="icon-wrapper bg-primary-subtle rounded-circle p-3 me-3">
                                    <i class="bx bx-category fs-3 text-primary"></i>
                                </div>
                                Add Category
                            </div>
                        </h2>
                        <p class="text-muted mt-2">Create and manage categories for your courses and videos</p>
                    </div>
                    <div class="d-flex">
                        <button type="button" class="btn btn-sm btn-outline-primary rounded-pill px-3 me-2" data-bs-toggle="modal" data-bs-target="#bulkImportModal">
                            <i class="bx bx-import me-1"></i> Bulk Import
                        </button>
                        <a href="categories.php" class="btn btn-sm btn-primary rounded-pill px-3">
                            <i class="bx bx-list-ul me-1"></i> View All
                        </a>
                    </div>
                </div>
                
                <!-- Display success and error messages -->
                <?php if(!empty($successMsg)): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="bx bx-check-circle me-2"></i>
                        <?php echo $successMsg; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                
                <?php if(!empty($errorMsg)): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="bx bx-error-circle me-2"></i>
                        <?php echo $errorMsg; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                
                <div class="row">
                    <!-- Add Category Form -->
                    <div class="col-lg-5 mb-4">
                        <div class="card border-0 rounded-4 shadow-sm hover-shadow">
                            <div class="card-header bg-white py-4 border-0">
                                <div class="d-flex align-items-center">
                                    <div class="icon-wrapper bg-info-subtle rounded-circle p-3 me-3">
                                        <i class="bx bx-plus-circle fs-3 text-info"></i>
                                    </div>
                                    <h5 class="mb-0 fw-semibold">Create New Category</h5>
                                </div>
                            </div>
                            <div class="card-body p-4">
                                <form action="add_category.php" method="POST">
                                    <div class="mb-4">
                                        <label for="category_name" class="form-label">Category Name <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control rounded-3" id="category_name" name="category_name" value="<?php echo htmlspecialchars($categoryName); ?>" required>
                                    </div>
                                    <div class="mb-4">
                                        <label for="category_description" class="form-label">Category Description</label>
                                        <textarea class="form-control rounded-3" id="category_description" name="category_description" rows="3"><?php echo htmlspecialchars($categoryDescription); ?></textarea>
                                    </div>
                                    <div class="d-grid">
                                        <button type="submit" class="btn btn-primary rounded-3 py-2">
                                            <i class="bx bx-save me-2"></i> Save Category
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Categories List -->
                    <div class="col-lg-7">
                        <div class="card border-0 rounded-4 shadow-sm hover-shadow">
                            <div class="card-header bg-white py-4 border-0">
                                <div class="d-flex align-items-center justify-content-between">
                                    <div class="d-flex align-items-center">
                                        <div class="icon-wrapper bg-primary-subtle rounded-circle p-3 me-3">
                                            <i class="bx bx-list-ul fs-3 text-primary"></i>
                                        </div>
                                        <h5 class="mb-0 fw-semibold">Categories List</h5>
                                    </div>
                                    <div class="input-group" style="max-width: 250px;">
                                        <input type="text" class="form-control rounded-pill rounded-end-0" placeholder="Search..." id="categorySearch">
                                        <span class="input-group-text bg-white border-start-0 rounded-pill rounded-start-0">
                                            <i class="bx bx-search"></i>
                                        </span>
                                    </div>
                                </div>
                            </div>
                            <div class="card-body p-0">
                                <?php if (count($categories) > 0): ?>
                                    <div class="table-responsive">
                                        <table class="table table-hover align-middle mb-0" id="categoriesTable">
                                            <thead class="table-light">
                                                <tr>
                                                    <th class="ps-4">ID</th>
                                                    <th>Category Name</th>
                                                    <th>Description</th>
                                                    <th class="text-end pe-4">Actions</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach($categories as $category): ?>
                                                    <tr>
                                                        <td class="ps-4"><?php echo $category['category_id']; ?></td>
                                                        <td>
                                                            <span class="category-badge bg-info-subtle text-info">
                                                                <?php echo htmlspecialchars($category['category_name']); ?>
                                                            </span>
                                                        </td>
                                                        <td>
                                                            <div class="text-wrap" style="max-width: 250px;">
                                                                <?php echo htmlspecialchars($category['category_description'] ?: 'No description'); ?>
                                                            </div>
                                                        </td>
                                                        <td class="text-end pe-4">
                                                            <button type="button" class="btn btn-sm btn-outline-primary rounded-pill me-1 edit-btn" 
                                                                    data-id="<?php echo $category['category_id']; ?>" 
                                                                    data-name="<?php echo htmlspecialchars($category['category_name']); ?>" 
                                                                    data-description="<?php echo htmlspecialchars($category['category_description']); ?>">
                                                                <i class="bx bx-edit-alt"></i>
                                                            </button>
                                                            <button type="button" class="btn btn-sm btn-outline-danger rounded-pill delete-btn" 
                                                                    data-id="<?php echo $category['category_id']; ?>" 
                                                                    data-name="<?php echo htmlspecialchars($category['category_name']); ?>">
                                                                <i class="bx bx-trash"></i>
                                                            </button>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                <?php else: ?>
                                    <div class="alert alert-info border-0 m-4 rounded-3">
                                        <div class="d-flex align-items-center">
                                            <i class="bx bx-info-circle fs-3 me-3"></i>
                                            <p class="mb-0">No categories found. Add your first category using the form.</p>
                                        </div>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <?php if (count($categories) > 0): ?>
                                <div class="card-footer bg-white border-0 p-3">
                                    <div class="d-flex justify-content-between align-items-center">
                                        <span class="text-muted small">Showing <?php echo count($categories); ?> categories</span>
                                        <button class="btn btn-sm btn-outline-danger rounded-pill px-3" id="clearAllBtn">
                                            <i class="bx bx-trash-alt me-1"></i> Clear All
                                        </button>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>
    
    <!-- Edit Category Modal -->
    <div class="modal fade" id="editCategoryModal" tabindex="-1" aria-labelledby="editCategoryModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content border-0 rounded-4">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title" id="editCategoryModalLabel">Edit Category</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body p-4">
                    <form id="editCategoryForm">
                        <input type="hidden" id="edit_category_id" name="category_id">
                        <div class="mb-3">
                            <label for="edit_category_name" class="form-label">Category Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control rounded-3" id="edit_category_name" name="category_name" required>
                        </div>
                        <div class="mb-3">
                            <label for="edit_category_description" class="form-label">Category Description</label>
                            <textarea class="form-control rounded-3" id="edit_category_description" name="category_description" rows="3"></textarea>
                        </div>
                    </form>
                </div>
                <div class="modal-footer border-0">
                    <button type="button" class="btn btn-outline-secondary rounded-pill px-4" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary rounded-pill px-4" id="updateCategoryBtn">Update Category</button>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Delete Confirmation Modal -->
    <div class="modal fade" id="deleteCategoryModal" tabindex="-1" aria-labelledby="deleteCategoryModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content border-0 rounded-4">
                <div class="modal-header bg-danger text-white">
                    <h5 class="modal-title" id="deleteCategoryModalLabel">Delete Category</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body p-4">
                    <input type="hidden" id="delete_category_id">
                    <p>Are you sure you want to delete the category: <strong id="delete_category_name"></strong>?</p>
                    <p class="text-danger">This action cannot be undone. All courses and videos in this category will be set to uncategorized.</p>
                </div>
                <div class="modal-footer border-0">
                    <button type="button" class="btn btn-outline-secondary rounded-pill px-4" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-danger rounded-pill px-4" id="confirmDeleteBtn">Delete</button>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Bulk Import Modal -->
    <div class="modal fade" id="bulkImportModal" tabindex="-1" aria-labelledby="bulkImportModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content border-0 rounded-4">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title" id="bulkImportModalLabel">Bulk Import Categories</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body p-4">
                    <form id="importForm">
                        <div class="mb-3">
                            <label for="importFile" class="form-label">CSV File <span class="text-danger">*</span></label>
                            <input type="file" class="form-control" id="importFile" accept=".csv" required>
                            <small class="text-muted">CSV should have columns: category_name, category_description</small>
                        </div>
                        <div class="alert alert-info">
                            <i class="bx bx-info-circle me-2"></i>
                            <span>Download <a href="#" class="text-decoration-underline">template file</a> to see the required format.</span>
                        </div>
                    </form>
                </div>
                <div class="modal-footer border-0">
                    <button type="button" class="btn btn-outline-secondary rounded-pill px-4" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary rounded-pill px-4" id="importBtn">Import</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Sidebar toggle functionality
        const sidebarToggle = document.querySelector('.sidebar-toggle');
        const sidebarToggleMobile = document.querySelector('.sidebar-toggle-mobile');
        const sidebar = document.querySelector('.sidebar');
        
        if (sidebarToggle) {
            sidebarToggle.addEventListener('click', () => {
                sidebar.classList.toggle('close');
            });
        }
        
        if (sidebarToggleMobile) {
            sidebarToggleMobile.addEventListener('click', () => {
                sidebar.classList.toggle('show');
            });
        }
        
        // Category search functionality
        const categorySearch = document.getElementById('categorySearch');
        if (categorySearch) {
            categorySearch.addEventListener('keyup', function() {
                const searchValue = this.value.toLowerCase();
                const table = document.getElementById('categoriesTable');
                
                if (table) {
                    const rows = table.getElementsByTagName('tr');
                    
                    for (let i = 1; i < rows.length; i++) {
                        const categoryName = rows[i].getElementsByTagName('td')[1].textContent.toLowerCase();
                        const categoryDescription = rows[i].getElementsByTagName('td')[2].textContent.toLowerCase();
                        
                        if (categoryName.includes(searchValue) || categoryDescription.includes(searchValue)) {
                            rows[i].style.display = '';
                        } else {
                            rows[i].style.display = 'none';
                        }
                    }
                }
            });
        }
        
        // Edit button functionality
        const editButtons = document.querySelectorAll('.edit-btn');
        editButtons.forEach(button => {
            button.addEventListener('click', function() {
                const id = this.getAttribute('data-id');
                const name = this.getAttribute('data-name');
                const description = this.getAttribute('data-description');
                
                document.getElementById('edit_category_id').value = id;
                document.getElementById('edit_category_name').value = name;
                document.getElementById('edit_category_description').value = description;
                
                const editModal = new bootstrap.Modal(document.getElementById('editCategoryModal'));
                editModal.show();
            });
        });
        
        // Delete button functionality
        const deleteButtons = document.querySelectorAll('.delete-btn');
        deleteButtons.forEach(button => {
            button.addEventListener('click', function() {
                const id = this.getAttribute('data-id');
                const name = this.getAttribute('data-name');
                
                document.getElementById('delete_category_id').value = id;
                document.getElementById('delete_category_name').textContent = name;
                
                const deleteModal = new bootstrap.Modal(document.getElementById('deleteCategoryModal'));
                deleteModal.show();
            });
        });
        
        // Update category functionality (placeholder)
        document.getElementById('updateCategoryBtn')?.addEventListener('click', function() {
            // In a real implementation, this would submit data via AJAX
            alert('Update functionality would go here. In a production environment, this would save the changes.');
            
            const modal = bootstrap.Modal.getInstance(document.getElementById('editCategoryModal'));
            modal.hide();
        });
        
        // Delete category functionality (placeholder)
        document.getElementById('confirmDeleteBtn')?.addEventListener('click', function() {
            // In a real implementation, this would submit data via AJAX
            alert('Delete functionality would go here. In a production environment, this would delete the category.');
            
            const modal = bootstrap.Modal.getInstance(document.getElementById('deleteCategoryModal'));
            modal.hide();
        });
        
        // Import functionality (placeholder)
        document.getElementById('importBtn')?.addEventListener('click', function() {
            // In a real implementation, this would process the CSV file
            const fileInput = document.getElementById('importFile');
            
            if (fileInput.files.length === 0) {
                alert('Please select a CSV file first.');
                return;
            }
            
            alert('Import functionality would go here. In a production environment, this would process the CSV file.');
            
            const modal = bootstrap.Modal.getInstance(document.getElementById('bulkImportModal'));
            modal.hide();
        });
        
        // Clear all categories functionality (placeholder)
        document.getElementById('clearAllBtn')?.addEventListener('click', function() {
            if (confirm('Are you sure you want to delete ALL categories? This action cannot be undone.')) {
                alert('Clear All functionality would go here. In a production environment, this would delete all categories.');
            }
        });
    });
    </script>
</body>
</html>
