<?php
// Initialize session if not started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Check if user is logged in and is an admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Include database connection
include_once "../../config/db_connect.php";

// Set content type to JSON
header('Content-Type: application/json');

// Check if required data is provided
if (!isset($_POST['teacher_id']) || empty($_POST['teacher_id']) || !isset($_POST['student_ids']) || empty($_POST['student_ids'])) {
    echo json_encode(['success' => false, 'message' => 'Teacher ID and at least one student ID are required']);
    exit;
}

$teacherId = (int)$_POST['teacher_id'];
$studentIds = $_POST['student_ids'];
$notes = isset($_POST['notes']) ? trim($_POST['notes']) : '';
$adminId = $_SESSION['user_id']; // The admin who is creating the assignment

// Validate student IDs (ensure they are numerical)
if (is_array($studentIds)) {
    foreach ($studentIds as $key => $id) {
        $studentIds[$key] = (int)$id;
    }
} else {
    // If a single ID is sent as a string
    $studentIds = [(int)$studentIds];
}

try {
    // Begin transaction
    $conn->beginTransaction();
    
    // First, verify that the teacher exists and is active
    $teacherQuery = "SELECT user_id FROM users WHERE user_id = :teacher_id AND role = 'teacher' AND status = 'active'";
    $teacherStmt = $conn->prepare($teacherQuery);
    $teacherStmt->bindParam(':teacher_id', $teacherId, PDO::PARAM_INT);
    $teacherStmt->execute();
    
    if (!$teacherStmt->fetch()) {
        echo json_encode(['success' => false, 'message' => 'Invalid or inactive teacher']);
        exit;
    }
    
    // Prepare the insert statement
    $insertQuery = "INSERT INTO teacher_student_assignments (teacher_id, student_id, assigned_by, notes) 
                    VALUES (:teacher_id, :student_id, :assigned_by, :notes)
                    ON DUPLICATE KEY UPDATE 
                    status = 'active', 
                    notes = VALUES(notes), 
                    assigned_date = CURRENT_TIMESTAMP";
    $insertStmt = $conn->prepare($insertQuery);
    
    $successCount = 0;
    $errorCount = 0;
    
    foreach ($studentIds as $studentId) {
        // Verify that the student exists and is active
        $studentQuery = "SELECT user_id FROM users WHERE user_id = :student_id AND role = 'student' AND status = 'active'";
        $studentStmt = $conn->prepare($studentQuery);
        $studentStmt->bindParam(':student_id', $studentId, PDO::PARAM_INT);
        $studentStmt->execute();
        
        if (!$studentStmt->fetch()) {
            $errorCount++;
            continue; // Skip this invalid student
        }
        
        // Check if this student is already assigned to the teacher
        $checkQuery = "SELECT assignment_id FROM teacher_student_assignments 
                       WHERE teacher_id = :teacher_id AND student_id = :student_id AND status = 'active'";
        $checkStmt = $conn->prepare($checkQuery);
        $checkStmt->bindParam(':teacher_id', $teacherId, PDO::PARAM_INT);
        $checkStmt->bindParam(':student_id', $studentId, PDO::PARAM_INT);
        $checkStmt->execute();
        
        // If already assigned, just update. Otherwise, insert new
        try {
            $insertStmt->bindParam(':teacher_id', $teacherId, PDO::PARAM_INT);
            $insertStmt->bindParam(':student_id', $studentId, PDO::PARAM_INT);
            $insertStmt->bindParam(':assigned_by', $adminId, PDO::PARAM_INT);
            $insertStmt->bindParam(':notes', $notes, PDO::PARAM_STR);
            $insertStmt->execute();
            $successCount++;
        } catch (PDOException $e) {
            $errorCount++;
        }
    }
    
    // Commit transaction if there were any successful assignments
    if ($successCount > 0) {
        $conn->commit();
        
        $message = "Successfully assigned $successCount student" . ($successCount > 1 ? "s" : "");
        if ($errorCount > 0) {
            $message .= " ($errorCount failed)";
        }
        
        echo json_encode([
            'success' => true,
            'message' => $message,
            'data' => [
                'success_count' => $successCount,
                'error_count' => $errorCount
            ]
        ]);
    } else {
        $conn->rollBack();
        echo json_encode([
            'success' => false,
            'message' => 'Failed to create any assignments'
        ]);
    }
    
} catch (PDOException $e) {
    // Rollback transaction on error
    $conn->rollBack();
    
    // Return error response
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
