<?php
// Initialize session if not started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Check if user is logged in and is an admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Include database connection
include_once "../../config/db_connect.php";

// Set content type to JSON
header('Content-Type: application/json');

// Get filter parameters
$teacherId = isset($_GET['teacher_id']) ? (int)$_GET['teacher_id'] : null;
$status = isset($_GET['status']) ? $_GET['status'] : null;

try {
    // Start building the query
    $query = "SELECT tsa.assignment_id, tsa.teacher_id, tsa.student_id, 
              tsa.assigned_date, tsa.status, tsa.notes,
              t.first_name as teacher_first_name, t.last_name as teacher_last_name, 
              t.email as teacher_email,
              s.first_name as student_first_name, s.last_name as student_last_name,
              s.email as student_email
              FROM teacher_student_assignments tsa
              JOIN users t ON tsa.teacher_id = t.user_id
              JOIN users s ON tsa.student_id = s.user_id
              LEFT JOIN teacher_profiles tp ON t.user_id = tp.teacher_id
              LEFT JOIN student_profiles sp ON s.user_id = sp.student_id
              WHERE 1=1";
    
    $params = [];
    
    // Add filters if provided
    if ($teacherId) {
        $query .= " AND tsa.teacher_id = :teacher_id";
        $params[':teacher_id'] = $teacherId;
    }
    
    if ($status && in_array($status, ['active', 'inactive'])) {
        $query .= " AND tsa.status = :status";
        $params[':status'] = $status;
    }
    
    // Add ordering
    $query .= " ORDER BY tsa.assigned_date DESC";
    
    $stmt = $conn->prepare($query);
    
    // Bind parameters if any
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    
    $stmt->execute();
    
    $assignments = [];
    
    // Fetch all assignments
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $assignments[] = [
            'assignment_id' => $row['assignment_id'],
            'teacher_id' => $row['teacher_id'],
            'student_id' => $row['student_id'],
            'assigned_date' => $row['assigned_date'],
            'status' => $row['status'],
            'notes' => $row['notes'] ? htmlspecialchars($row['notes']) : null,
            'teacher_first_name' => htmlspecialchars($row['teacher_first_name']),
            'teacher_last_name' => htmlspecialchars($row['teacher_last_name']),
            'teacher_email' => htmlspecialchars($row['teacher_email']),
            'teacher_initials' => strtoupper(substr($row['teacher_first_name'], 0, 1) . substr($row['teacher_last_name'], 0, 1)),
            'student_first_name' => htmlspecialchars($row['student_first_name']),
            'student_last_name' => htmlspecialchars($row['student_last_name']),
            'student_email' => htmlspecialchars($row['student_email']),
            'student_initials' => strtoupper(substr($row['student_first_name'], 0, 1) . substr($row['student_last_name'], 0, 1)),
        ];
    }
    
    // Return success response with assignments data
    echo json_encode([
        'success' => true,
        'data' => $assignments
    ]);
    
} catch (PDOException $e) {
    // Return error response
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
