<?php
// Initialize session if not started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Check if user is logged in and is an admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Include database connection
include_once "../../config/db_connect.php";

// Set content type to JSON
header('Content-Type: application/json');

// Check if teacher_id is provided
if (!isset($_GET['teacher_id']) || empty($_GET['teacher_id'])) {
    echo json_encode(['success' => false, 'message' => 'Teacher ID is required']);
    exit;
}

$teacherId = (int)$_GET['teacher_id'];

try {
    // Query to get students not yet assigned to this teacher
    $query = "SELECT u.user_id, u.first_name, u.last_name, u.email
              FROM users u
              LEFT JOIN student_profiles sp ON u.user_id = sp.student_id
              WHERE u.role = 'student' 
              AND u.status = 'active'
              AND u.user_id NOT IN (
                  SELECT student_id 
                  FROM teacher_student_assignments 
                  WHERE teacher_id = :teacher_id
                  AND status = 'active'
              )
              ORDER BY u.first_name, u.last_name";
    
    $stmt = $conn->prepare($query);
    $stmt->bindParam(':teacher_id', $teacherId, PDO::PARAM_INT);
    $stmt->execute();
    
    $students = [];
    
    // Fetch all unassigned students
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $students[] = [
            'user_id' => $row['user_id'],
            'first_name' => htmlspecialchars($row['first_name']),
            'last_name' => htmlspecialchars($row['last_name']),
            'email' => htmlspecialchars($row['email']),
            'initials' => strtoupper(substr($row['first_name'], 0, 1) . substr($row['last_name'], 0, 1))
        ];
    }
    
    // Return success response with students data
    echo json_encode([
        'success' => true,
        'data' => $students
    ]);
    
} catch (PDOException $e) {
    // Return error response
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
