<?php
require_once __DIR__ . '/../../auth/functions.php';
require_once __DIR__ . '/../../config/db_connect.php';

class CertificateGenerator {
    private $conn;
    private $settings;
    
    public function __construct($connection) {
        $this->conn = $connection;
        $this->loadSettings();
    }
    
    private function loadSettings() {
        $settings_query = "SELECT setting_key, setting_value FROM certificate_settings";
        $result = $this->conn->query($settings_query);
        
        $this->settings = [];
        while ($row = $result->fetch_assoc()) {
            $this->settings[$row['setting_key']] = $row['setting_value'];
        }
    }
    
    public function generateCertificate($student_id, $certificate_type, $course_id = null, $template_id = null) {
        // Get student information
        $student_info = $this->getStudentInfo($student_id);
        if (!$student_info) {
            throw new Exception("Student not found");
        }
        
        // Get course information if applicable
        $course_info = null;
        if ($course_id) {
            $course_info = $this->getCourseInfo($course_id);
        }
        
        // Get template - use specified template or default for type
        $template = $this->getTemplate($certificate_type, $template_id);
        if (!$template) {
            throw new Exception("No template found for certificate type: " . $certificate_type);
        }
        
        // Get template fields
        $fields = $this->getTemplateFields($template['template_id']);
        
        // Generate certificate number
        $certificate_number = $this->generateCertificateNumber($certificate_type);
        
        // Prepare field data
        $field_data = $this->prepareFieldData($student_info, $course_info, $certificate_number);
        
        // Generate certificate image/PDF
        $generated_file = $this->createCertificateImage($template, $fields, $field_data);
        
        // Save certificate record
        $certificate_id = $this->saveCertificateRecord([
            'student_id' => $student_id,
            'template_id' => $template['template_id'],
            'certificate_type' => $certificate_type,
            'course_id' => $course_id,
            'certificate_number' => $certificate_number,
            'generated_file' => $generated_file,
            'metadata' => json_encode([
                'student_info' => $student_info,
                'course_info' => $course_info,
                'generation_date' => date('Y-m-d H:i:s')
            ])
        ]);
        
        return [
            'certificate_id' => $certificate_id,
            'certificate_number' => $certificate_number,
            'file_path' => $generated_file,
            'download_url' => '/uploads/certificates/generated/' . $generated_file
        ];
    }
    
    private function getStudentInfo($student_id) {
        $stmt = $this->conn->prepare("SELECT * FROM users WHERE user_id = ? AND role = 'student'");
        if (!$stmt) {
            error_log("Failed to prepare getStudentInfo query: " . $this->conn->error);
            return null;
        }
        $stmt->bind_param("i", $student_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if (!$result) {
            error_log("Failed to get result from getStudentInfo query: " . $stmt->error);
            return null;
        }
        return $result->fetch_assoc();
    }
    
    private function getCourseInfo($course_id) {
        $stmt = $this->conn->prepare("SELECT * FROM courses WHERE course_id = ?");
        if (!$stmt) {
            error_log("Failed to prepare getCourseInfo query: " . $this->conn->error);
            return null;
        }
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if (!$result) {
            error_log("Failed to get result from getCourseInfo query: " . $stmt->error);
            return null;
        }
        return $result->fetch_assoc();
    }
    
    private function getTemplate($certificate_type, $template_id = null) {
        if ($template_id) {
            $stmt = $this->conn->prepare("SELECT * FROM certificate_templates WHERE template_id = ? AND is_active = 1");
            $stmt->bind_param("i", $template_id);
        } else {
            $stmt = $this->conn->prepare("SELECT * FROM certificate_templates WHERE certificate_type = ? AND is_active = 1 ORDER BY is_default DESC, created_date DESC LIMIT 1");
            $stmt->bind_param("s", $certificate_type);
        }
        
        $stmt->execute();
        return $stmt->get_result()->fetch_assoc();
    }
    
    private function getTemplateFields($template_id) {
        $stmt = $this->conn->prepare("SELECT * FROM certificate_template_fields WHERE template_id = ? ORDER BY field_id");
        $stmt->bind_param("i", $template_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $fields = [];
        while ($field = $result->fetch_assoc()) {
            $fields[] = $field;
        }
        
        return $fields;
    }
    
    private function generateCertificateNumber($certificate_type) {
        $prefix = $this->settings['certificate_prefix'] ?? 'PA';
        $type_code = strtoupper(substr($certificate_type, 0, 2));
        $timestamp = date('Ymd');
        $random = str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
        
        return $prefix . '-' . $type_code . '-' . $timestamp . '-' . $random;
    }
    
    private function prepareFieldData($student_info, $course_info, $certificate_number) {
        return [
            'student_name' => $student_info['first_name'] . ' ' . $student_info['last_name'],
            'student_id' => $student_info['student_id'] ?? $student_info['user_id'],
            'course_title' => $course_info['title'] ?? '',
            'completion_date' => date('F j, Y'),
            'issue_date' => date('F j, Y'),
            'certificate_number' => $certificate_number,
            'director_name' => $this->settings['director_name'] ?? 'Academy Director',
            'director_signature' => '', // Could be an image path
            'academy_name' => $this->settings['academy_name'] ?? 'Panadite Academy',
            'grade' => '', // Can be populated from course completion data
            'duration' => $course_info['duration'] ?? '',
            'instructor_name' => '' // Can be populated from course instructor data
        ];
    }
    
    private function createCertificateImage($template, $fields, $field_data) {
        $template_path = __DIR__ . '/../../uploads/certificates/templates/' . $template['template_file'];
        
        // Check if template file exists
        if (!file_exists($template_path)) {
            throw new Exception("Template file not found: " . $template_path);
        }
        
        // Create image from template
        $image_info = getimagesize($template_path);
        if (!$image_info) {
            throw new Exception("Invalid template image file");
        }
        
        switch ($image_info[2]) {
            case IMAGETYPE_JPEG:
                $template_image = imagecreatefromjpeg($template_path);
                break;
            case IMAGETYPE_PNG:
                $template_image = imagecreatefrompng($template_path);
                break;
            default:
                throw new Exception("Unsupported image format. Use JPG or PNG.");
        }
        
        if (!$template_image) {
            throw new Exception("Failed to create image from template");
        }
        
        // Add text fields to the image
        foreach ($fields as $field) {
            $text = $field_data[$field['field_type']] ?? $field['custom_text'] ?? '';
            
            if (empty($text)) continue;
            
            // Convert hex color to RGB
            $color_rgb = $this->hexToRgb($field['font_color']);
            $text_color = imagecolorallocate($template_image, $color_rgb['r'], $color_rgb['g'], $color_rgb['b']);
            
            // Calculate font size for GD (approximate conversion)
            $font_size = max(8, min(72, $field['font_size'] * 0.75));
            
            // Use built-in fonts for simplicity (you can enhance this with TTF fonts)
            $font = 5; // Built-in font
            
            // Position text based on alignment
            $text_x = $field['x_position'];
            if ($field['text_align'] === 'center') {
                $text_width = imagefontwidth($font) * strlen($text);
                $text_x = $field['x_position'] + ($field['width'] - $text_width) / 2;
            } elseif ($field['text_align'] === 'right') {
                $text_width = imagefontwidth($font) * strlen($text);
                $text_x = $field['x_position'] + $field['width'] - $text_width;
            }
            
            // Center text vertically in the field
            $text_height = imagefontheight($font);
            $text_y = $field['y_position'] + ($field['height'] - $text_height) / 2;
            
            // Add text to image
            imagestring($template_image, $font, $text_x, $text_y, $text, $text_color);
        }
        
        // Generate unique filename
        $filename = 'certificate_' . time() . '_' . uniqid() . '.png';
        $output_path = __DIR__ . '/../../uploads/certificates/generated/' . $filename;
        
        // Save the certificate
        if (!imagepng($template_image, $output_path)) {
            throw new Exception("Failed to save certificate image");
        }
        
        // Clean up memory
        imagedestroy($template_image);
        
        return $filename;
    }
    
    private function hexToRgb($hex) {
        $hex = ltrim($hex, '#');
        return [
            'r' => hexdec(substr($hex, 0, 2)),
            'g' => hexdec(substr($hex, 2, 2)),
            'b' => hexdec(substr($hex, 4, 2))
        ];
    }
    
    private function saveCertificateRecord($data) {
        $stmt = $this->conn->prepare("INSERT INTO student_certificates (student_id, template_id, certificate_type, course_id, certificate_number, generated_file, metadata) VALUES (?, ?, ?, ?, ?, ?, ?)");
        $stmt->bind_param("iisisss", $data['student_id'], $data['template_id'], $data['certificate_type'], $data['course_id'], $data['certificate_number'], $data['generated_file'], $data['metadata']);
        
        if ($stmt->execute()) {
            return $this->conn->insert_id;
        } else {
            throw new Exception("Failed to save certificate record");
        }
    }
}

// API endpoint handling
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
    
    try {
        // Check authentication
        if (!isset($_SESSION['user_id'])) {
            throw new Exception("Authentication required");
        }
        
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            $input = $_POST;
        }
        
        $required_fields = ['student_id', 'certificate_type'];
        foreach ($required_fields as $field) {
            if (!isset($input[$field])) {
                throw new Exception("Missing required field: " . $field);
            }
        }
        
        $generator = new CertificateGenerator($conn);
        
        $result = $generator->generateCertificate(
            $input['student_id'],
            $input['certificate_type'],
            $input['course_id'] ?? null,
            $input['template_id'] ?? null
        );
        
        echo json_encode([
            'success' => true,
            'data' => $result,
            'message' => 'Certificate generated successfully'
        ]);
        
    } catch (Exception $e) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
}
?>
