<?php
require_once __DIR__ . '/../../auth/functions.php';
require_once __DIR__ . '/../../config/db_connect.php';
require_once __DIR__ . '/../../config/fast_email_service.php';
require_once __DIR__ . '/../auto_certificate_generator.php';
require_once __DIR__ . '/../pdf_certificate_generator.php';

// Check if user is logged in and has admin privileges
requireRole('admin');

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

try {
    // Get input data
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }
    
    // Validate required fields
    if (empty($input['recipient_email'])) {
        throw new Exception('Recipient email is required');
    }
    
    $recipientEmail = filter_var($input['recipient_email'], FILTER_VALIDATE_EMAIL);
    if (!$recipientEmail) {
        throw new Exception('Invalid email address');
    }
    
    $studentName = !empty($input['student_name']) ? htmlspecialchars($input['student_name']) : 'John Doe';
    $courseTitle = !empty($input['course_title']) ? htmlspecialchars($input['course_title']) : 'Sample Course';
    
    // Generate test certificate using WORKING PDF approach
    $certificateNumber = 'TEST-' . date('Y') . '-' . strtoupper(substr(uniqid(), -6));
    $issueDate = date('F j, Y');
    
    // Create PDF certificate generator instance
    $pdfGenerator = new PdfCertificateGenerator($conn);
    
    // Generate beautiful PDF certificate (same as working test)
    try {
        $pdfResult = $pdfGenerator->createPrintableCertificate($certificateNumber, $studentName, $courseTitle, $issueDate);
    } catch (Exception $e) {
        // If database version fails, use simplified version without database
        error_log("Database certificate generation failed, using simplified version: " . $e->getMessage());
        $pdfResult = $pdfGenerator->createSimpleCertificate($certificateNumber, $studentName, $courseTitle, $issueDate);
    }
    
    if (!$pdfResult['success']) {
        throw new Exception('Failed to generate certificate: ' . ($pdfResult['error'] ?? 'Unknown error'));
    }
    
    // Create download URL
    $certificateUrl = "https://panaditeacademy.com" . $pdfResult['download_url'];
    
    // Send ULTRA SIMPLE email (same format that just worked perfectly)
    $subject = "Certificate Ready - {$courseTitle} - " . date('H:i:s');
    
    $emailBody = "<h2>🎓 Certificate Ready - {$studentName}</h2>
    <p><strong>Course:</strong> {$courseTitle}</p>
    <p><strong>Certificate Number:</strong> {$certificateNumber}</p>
    <p><strong>Download Link:</strong> <a href='{$certificateUrl}'>Click here to view your certificate</a></p>
    <p><strong>Time sent:</strong> " . date('Y-m-d H:i:s') . "</p>
    <p>Congratulations on completing your course!</p>";
    
    // Send email using your working fast email service
    $emailSent = sendEmailNow($recipientEmail, $subject, $emailBody);
    
    if ($emailSent) {
        echo json_encode([
            'success' => true,
            'message' => 'PDF Certificate generated and email sent successfully!',
            'details' => [
                'recipient' => $recipientEmail,
                'student_name' => $studentName,
                'course_title' => $courseTitle,
                'certificate_number' => $certificateNumber,
                'certificate_file' => $pdfResult['filename'],
                'download_url' => $certificateUrl,
                'timestamp' => date('Y-m-d H:i:s')
            ]
        ]);
    } else {
        throw new Exception('Failed to send email - check server mail configuration');
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>
