<?php
require_once '../auth/functions.php';
require_once '../config/db_connect.php';
require_once '../config/fast_email_service.php';
require_once 'auto_certificate_generator.php';
require_once 'pdf_certificate_generator.php';

// Check if user is admin
requireRole('admin');

$message = '';
$messageType = '';

// Handle form submission
if ($_POST && isset($_POST['send_test'])) {
    try {
        $recipientEmail = filter_var($_POST['recipient_email'], FILTER_VALIDATE_EMAIL);
        $studentName = htmlspecialchars($_POST['student_name']);
        $courseTitle = htmlspecialchars($_POST['course_title']);
        
        if (!$recipientEmail) {
            throw new Exception('Please enter a valid email address');
        }
        
        if (empty($studentName) || empty($courseTitle)) {
            throw new Exception('Please fill in all fields');
        }
        
        // Generate test certificate
        $certificateNumber = 'TEST-' . date('Y') . '-' . strtoupper(substr(uniqid(), -6));
        $issueDate = date('F j, Y');
        
        // Create PDF certificate generator instance
        $pdfGenerator = new PdfCertificateGenerator($conn);
        
        // Try to generate certificate with database first, fallback to simple version
        try {
            $pdfResult = $pdfGenerator->createPrintableCertificate($certificateNumber, $studentName, $courseTitle, $issueDate);
        } catch (Exception $e) {
            // If database version fails, use simplified version without database
            error_log("Database certificate generation failed, using simplified version: " . $e->getMessage());
            $pdfResult = $pdfGenerator->createSimpleCertificate($certificateNumber, $studentName, $courseTitle, $issueDate);
        }
        
        if (!$pdfResult['success']) {
            throw new Exception('Failed to generate certificate: ' . ($pdfResult['error'] ?? 'Unknown error'));
        }
        
        // Create download URL (adjust domain as needed)
        $certificateUrl = "https://panaditeacademy.com" . $pdfResult['download_url'];
        
        // Use ULTRA SIMPLE email format (same as working debug emails)
        $subject = "Certificate Ready - {$courseTitle} - " . date('H:i:s');
        
        $emailBody = "<h2>🎓 Certificate Ready - {$studentName}</h2>
        <p><strong>Course:</strong> {$courseTitle}</p>
        <p><strong>Certificate Number:</strong> {$certificateNumber}</p>
        <p><strong>Download Link:</strong> <a href='{$certificateUrl}'>Click here to view your certificate</a></p>
        <p><strong>Time sent:</strong> " . date('Y-m-d H:i:s') . "</p>
        <p>Congratulations on completing your course!</p>";
        
        // Send email using your existing fast email service
        $emailSent = sendEmailNow($recipientEmail, $subject, $emailBody);
        
        if ($emailSent) {
            $message = "✅ PDF Certificate generated and email sent successfully!<br>";
            $message .= "<strong>Email sent to:</strong> {$recipientEmail}<br>";
            $message .= "<strong>Certificate Number:</strong> {$certificateNumber}<br>";
            $message .= "<strong>Student:</strong> {$studentName}<br>";
            $message .= "<strong>Course:</strong> {$courseTitle}<br>";
            $message .= "<strong>Certificate File:</strong> {$pdfResult['filename']}<br>";
            $message .= "<strong>Download URL:</strong> <a href='{$certificateUrl}' target='_blank'>View Certificate</a><br>";
            $message .= "<strong>Sent at:</strong> " . date('Y-m-d H:i:s');
            $messageType = 'success';
        } else {
            throw new Exception('Failed to send email - check server mail configuration');
        }
        
    } catch (Exception $e) {
        $message = "❌ Error: " . $e->getMessage();
        $messageType = 'danger';
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Certificate Email Test - Panadite Academy</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        .test-container {
            max-width: 600px;
            margin: 50px auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.2);
            overflow: hidden;
        }
        .header {
            background: linear-gradient(135deg, #4169E1 0%, #20B2AA 100%);
            color: white;
            text-align: center;
            padding: 30px;
        }
        .header h1 {
            margin: 0;
            font-size: 2rem;
            font-weight: 600;
        }
        .form-container {
            padding: 40px;
        }
        .form-label {
            font-weight: 600;
            color: #333;
            margin-bottom: 8px;
        }
        .form-control {
            border-radius: 10px;
            border: 2px solid #e9ecef;
            padding: 12px 15px;
            transition: all 0.3s ease;
        }
        .form-control:focus {
            border-color: #4169E1;
            box-shadow: 0 0 0 0.2rem rgba(65, 105, 225, 0.25);
        }
        .btn-test {
            background: linear-gradient(135deg, #4169E1 0%, #20B2AA 100%);
            border: none;
            border-radius: 10px;
            padding: 12px 30px;
            font-weight: 600;
            font-size: 1.1rem;
            color: white;
            transition: all 0.3s ease;
        }
        .btn-test:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(65, 105, 225, 0.3);
            color: white;
        }
        .result-box {
            margin-top: 20px;
            padding: 20px;
            border-radius: 10px;
            border-left: 5px solid;
        }
        .result-success {
            background: #d1edff;
            border-color: #28a745;
            color: #155724;
        }
        .result-error {
            background: #f8d7da;
            border-color: #dc3545;
            color: #721c24;
        }
        .info-box {
            background: #e8f4fd;
            border: 1px solid #bee5eb;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 30px;
        }
        .info-box h5 {
            color: #0c5460;
            margin-bottom: 15px;
        }
        .info-box ul {
            margin-bottom: 0;
            color: #0c5460;
        }
    </style>
</head>
<body>
    <div class="test-container">
        <div class="header">
            <i class="bx bx-envelope bx-lg mb-2"></i>
            <h1>Certificate Email Test</h1>
            <p class="mb-0">Test your certificate generation and email delivery</p>
        </div>
        
        <div class="form-container">
            <div class="info-box">
                <h5><i class="bx bx-info-circle me-2"></i>Test Information</h5>
                <ul>
                    <li>This will generate and send a <strong>real certificate email</strong></li>
                    <li>Uses your existing <strong>auto_certificate_generator.php</strong></li>
                    <li>Creates the same <strong>beautiful certificate design</strong></li>
                    <li>Sends via your current <strong>email server configuration</strong></li>
                </ul>
            </div>
            
            <?php if ($message): ?>
            <div class="result-box result-<?= $messageType === 'success' ? 'success' : 'error' ?>">
                <strong><i class="bx <?= $messageType === 'success' ? 'bx-check-circle' : 'bx-x-circle' ?> me-2"></i>Result:</strong><br>
                <?= $message ?>
            </div>
            <?php endif; ?>
            
            <form method="POST" action="">
                <div class="mb-4">
                    <label for="recipient_email" class="form-label">
                        <i class="bx bx-envelope me-2"></i>Recipient Email Address
                    </label>
                    <input type="email" class="form-control" id="recipient_email" name="recipient_email" 
                           placeholder="Enter email to receive test certificate" required 
                           value="<?= htmlspecialchars($_POST['recipient_email'] ?? '') ?>">
                    <small class="text-muted">Enter any valid email address to test</small>
                </div>
                
                <div class="mb-4">
                    <label for="student_name" class="form-label">
                        <i class="bx bx-user me-2"></i>Test Student Name
                    </label>
                    <input type="text" class="form-control" id="student_name" name="student_name" 
                           placeholder="Enter student name for certificate" required 
                           value="<?= htmlspecialchars($_POST['student_name'] ?? 'John Doe') ?>">
                </div>
                
                <div class="mb-4">
                    <label for="course_title" class="form-label">
                        <i class="bx bx-book me-2"></i>Test Course Title
                    </label>
                    <input type="text" class="form-control" id="course_title" name="course_title" 
                           placeholder="Enter course title for certificate" required 
                           value="<?= htmlspecialchars($_POST['course_title'] ?? 'Sample Course') ?>">
                </div>
                
                <div class="text-center">
                    <button type="submit" name="send_test" class="btn btn-test btn-lg">
                        <i class="bx bx-send me-2"></i>Generate & Send Test Certificate
                    </button>
                </div>
            </form>
            
            <div class="mt-4 text-center">
                <a href="certificate_templates.php" class="btn btn-outline-secondary">
                    <i class="bx bx-arrow-back me-2"></i>Back to Certificate Templates
                </a>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
