<?php
require_once '../auth/functions.php';
require_once '../includes/config.php';
require_once '../includes/db_connection.php';

// Ensure user is logged in as an admin
requireRole('admin');

$userId = $_SESSION['user_id'];
$user = getCurrentUser();

// Initialize variables
$successMsg = '';
$errorMsg = '';

// Process approval/rejection if form submitted
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['approve']) && isset($_POST['teacher_id'])) {
        $teacherId = intval($_POST['teacher_id']);
        
        // Check if the teacher_profiles table has a status column
        $checkColumnQuery = "SHOW COLUMNS FROM teacher_profiles LIKE 'status'"; 
        $columnResult = $conn->query($checkColumnQuery);
        
        if ($columnResult->num_rows > 0) {
            // Status column exists - update it directly
            $stmt = $conn->prepare("UPDATE teacher_profiles SET status = 'approved' WHERE teacher_id = ?");
            $stmt->bind_param("i", $teacherId);
            
            if ($stmt->execute()) {
                // Also update the users table to ensure they have teacher role
                $userStmt = $conn->prepare("UPDATE users SET user_role = 'teacher' WHERE user_id = ?");
                $userStmt->bind_param("i", $teacherId);
                $userStmt->execute();
                
                $successMsg = "Facilitator approved successfully!";
            } else {
                $errorMsg = "Error approving facilitator: " . $conn->error;
            }
        } else {
            // Check if users table has is_active column
            $checkActiveColumn = "SHOW COLUMNS FROM users LIKE 'is_active'"; 
            $activeColumnResult = $conn->query($checkActiveColumn);
            $hasActiveColumn = ($activeColumnResult && $activeColumnResult->num_rows > 0);
            
            // Build the SQL based on available columns
            $updateSQL = "UPDATE users SET user_role = 'teacher'";
            if ($hasActiveColumn) {
                $updateSQL .= ", is_active = 1";
            }
            $updateSQL .= " WHERE user_id = ?";
            
            $userStmt = $conn->prepare($updateSQL);
            $userStmt->bind_param("i", $teacherId);
            
            if ($userStmt->execute()) {
                // Check if there's an is_approved column in teacher_profiles
                $checkApprovalColumn = "SHOW COLUMNS FROM teacher_profiles LIKE 'is_approved'"; 
                $approvalResult = $conn->query($checkApprovalColumn);
                
                if ($approvalResult->num_rows > 0) {
                    // Update is_approved instead
                    $approveStmt = $conn->prepare("UPDATE teacher_profiles SET is_approved = 1 WHERE teacher_id = ?");
                    $approveStmt->bind_param("i", $teacherId);
                    $approveStmt->execute();
                }
                
                $successMsg = "Facilitator approved successfully!";
            } else {
                $errorMsg = "Error approving facilitator: " . $conn->error;
            }
        }
    } 
    else if (isset($_POST['reject']) && isset($_POST['teacher_id'])) {
        $teacherId = intval($_POST['teacher_id']);
        
        // Check if the teacher_profiles table has a status column
        $checkColumnQuery = "SHOW COLUMNS FROM teacher_profiles LIKE 'status'"; 
        $columnResult = $conn->query($checkColumnQuery);
        
        if ($columnResult->num_rows > 0) {
            // Status column exists - update it directly
            $stmt = $conn->prepare("UPDATE teacher_profiles SET status = 'rejected' WHERE teacher_id = ?");
            $stmt->bind_param("i", $teacherId);
            
            if ($stmt->execute()) {
                $successMsg = "Facilitator request rejected!";
            } else {
                $errorMsg = "Error rejecting facilitator: " . $conn->error;
            }
        } else {
            // Check if there's an is_approved column in teacher_profiles
            $checkApprovalColumn = "SHOW COLUMNS FROM teacher_profiles LIKE 'is_approved'"; 
            $approvalResult = $conn->query($checkApprovalColumn);
            
            if ($approvalResult->num_rows > 0) {
                // Update is_approved to 0 instead
                $rejectStmt = $conn->prepare("UPDATE teacher_profiles SET is_approved = 0 WHERE teacher_id = ?");
                $rejectStmt->bind_param("i", $teacherId);
                
                if ($rejectStmt->execute()) {
                    // Check if users table has is_active column
                    $checkActiveColumn = "SHOW COLUMNS FROM users LIKE 'is_active'"; 
                    $activeColumnResult = $conn->query($checkActiveColumn);
                    
                    if ($activeColumnResult && $activeColumnResult->num_rows > 0) {
                        // Mark user as inactive if column exists
                        $userStmt = $conn->prepare("UPDATE users SET is_active = 0 WHERE user_id = ?");
                        $userStmt->bind_param("i", $teacherId);
                        $userStmt->execute();
                    }
                    
                    $successMsg = "Facilitator request rejected!";
                } else {
                    $errorMsg = "Error rejecting facilitator: " . $conn->error;
                }
            } else {
                // Check if users table has is_active column
                $checkActiveColumn = "SHOW COLUMNS FROM users LIKE 'is_active'"; 
                $activeColumnResult = $conn->query($checkActiveColumn);
                
                if ($activeColumnResult && $activeColumnResult->num_rows > 0) {
                    // Mark user as inactive if column exists
                    $userStmt = $conn->prepare("UPDATE users SET is_active = 0 WHERE user_id = ?");
                    $userStmt->bind_param("i", $teacherId);
                    
                    if ($userStmt->execute()) {
                        $successMsg = "Facilitator request rejected!";
                    } else {
                        $errorMsg = "Error rejecting facilitator: " . $conn->error;
                    }
                } else {
                    // If no is_active column, just show success message
                    $successMsg = "Facilitator request rejected!";
                }
            }
        }
    }
    
    // Process video approval
    if (isset($_POST['approve_video']) && isset($_POST['video_id'])) {
        $videoId = intval($_POST['video_id']);
        
        // Check if the videos table has a status column
        $checkColumnQuery = "SHOW COLUMNS FROM videos LIKE 'status'"; 
        $columnResult = $conn->query($checkColumnQuery);
        
        if ($columnResult->num_rows > 0) {
            // Status column exists - update it directly
            $stmt = $conn->prepare("UPDATE videos SET status = 'approved' WHERE video_id = ?");
        } else {
            // No status column - update is_published instead
            $stmt = $conn->prepare("UPDATE videos SET is_published = 1 WHERE video_id = ?");
        }
        
        $stmt->bind_param("i", $videoId);
        
        if ($stmt->execute()) {
            $successMsg = "Video approved successfully!";
        } else {
            $errorMsg = "Error approving video: " . $conn->error;
        }
    }
    else if (isset($_POST['reject_video']) && isset($_POST['video_id'])) {
        $videoId = intval($_POST['video_id']);
        
        // Check if the videos table has a status column
        $checkColumnQuery = "SHOW COLUMNS FROM videos LIKE 'status'"; 
        $columnResult = $conn->query($checkColumnQuery);
        
        if ($columnResult->num_rows > 0) {
            // Status column exists - update it directly
            $stmt = $conn->prepare("UPDATE videos SET status = 'rejected' WHERE video_id = ?");
        } else {
            // No status column - handle rejection via is_published (could also delete or add a separate flag)
            $stmt = $conn->prepare("UPDATE videos SET is_published = 0 WHERE video_id = ?");
        }
        
        $stmt->bind_param("i", $videoId);
        
        if ($stmt->execute()) {
            $successMsg = "Video rejected successfully!";
        } else {
            $errorMsg = "Error rejecting video: " . $conn->error;
        }
    }
}

// Get pending teacher applications
$pendingTeachers = [];
try {
    // Check if the teacher_profiles table has various columns we need
    $hasStatusColumn = false;
    $hasCourseTitle = false;
    $hasIsApproved = false;
    
    // Check for status column
    $checkStatusColumn = "SHOW COLUMNS FROM teacher_profiles LIKE 'status'"; 
    $statusResult = $conn->query($checkStatusColumn);
    if ($statusResult && $statusResult->num_rows > 0) {
        $hasStatusColumn = true;
    }
    
    // Check for is_approved column
    $checkApprovedColumn = "SHOW COLUMNS FROM teacher_profiles LIKE 'is_approved'"; 
    $approvedResult = $conn->query($checkApprovedColumn);
    if ($approvedResult && $approvedResult->num_rows > 0) {
        $hasIsApproved = true;
    }
    
    // Check for course_title column
    $checkCourseColumn = "SHOW COLUMNS FROM teacher_profiles LIKE 'course_title'"; 
    $courseResult = $conn->query($checkCourseColumn);
    if ($courseResult && $courseResult->num_rows > 0) {
        $hasCourseTitle = true;
    }
    

    
    // Build query based on available columns
    $query = "SELECT u.user_id, u.username, u.email, u.first_name, u.last_name, u.created_at, 
             tp.bio, tp.qualifications, tp.expertise";
             
    // Add status column if it exists
    if ($hasStatusColumn) {
        $query .= ", tp.status";
    } else {
        $query .= ", 'pending' as status";
    }
    
    // Add is_approved column if it exists
    if ($hasIsApproved) {
        $query .= ", tp.is_approved";
    } else {
        $query .= ", 0 as is_approved";
    }
    
    // Add course_title if it exists
    if ($hasCourseTitle) {
        $query .= ", tp.course_title";
    } else {
        $query .= ", '' as course_title";
    }
    
    $query .= " FROM users u
               JOIN teacher_profiles tp ON u.user_id = tp.teacher_id";
    
    // For pending facilitators: either they have status=pending OR have user_role=teacher but not yet approved
    if ($hasStatusColumn) {
        $query .= " WHERE (tp.status = 'pending' OR tp.status IS NULL)";
    } else if ($hasIsApproved) {
        $query .= " WHERE (tp.is_approved = 0 OR tp.is_approved IS NULL)";
    } else {
        // Fallback to just looking at users with teacher role
        $query .= " WHERE u.user_role = 'teacher'";
    }
    
    $query .= " ORDER BY u.created_at DESC";
    
    
    $stmt = $conn->prepare($query);
    
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $pendingTeachers[] = $row;
    }
    

} catch (Exception $e) {
    $errorMsg = "Error retrieving pending facilitators: " . $e->getMessage();
}

// Get pending videos for approval
$pendingVideos = [];
try {
    // Check if the videos table has a status column
    $checkColumnQuery = "SHOW COLUMNS FROM videos LIKE 'status'"; 
    $columnResult = $conn->query($checkColumnQuery);
    
    if ($columnResult->num_rows > 0) {
        // Status column exists - use it directly
        $stmt = $conn->prepare("
            SELECT v.video_id, v.title, v.description, v.price, v.thumbnail, v.created_at,
                   v.status, u.first_name, u.last_name, u.email, c.title as course_title
            FROM videos v
            JOIN users u ON v.teacher_id = u.user_id
            JOIN courses c ON v.course_id = c.course_id
            WHERE v.status = 'pending'
            ORDER BY v.created_at DESC
        ");
    } else {
        // No status column - use is_published as a proxy (showing videos awaiting approval)
        $stmt = $conn->prepare("
            SELECT v.video_id, v.title, v.description, v.price, v.thumbnail, v.created_at,
                   CASE WHEN v.is_published = 0 THEN 'pending' ELSE 'approved' END as status,
                   u.first_name, u.last_name, u.email, c.title as course_title
            FROM videos v
            JOIN users u ON v.teacher_id = u.user_id
            JOIN courses c ON v.course_id = c.course_id
            WHERE v.is_published = 0
            ORDER BY v.created_at DESC
        ");
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $pendingVideos[] = $row;
    }
} catch (Exception $e) {
    $errorMsg = "Error retrieving pending videos: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Approve Facilitators - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        /* Custom styles for approval pages */
        .stat-card {
            display: flex;
            padding: 1rem;
            border-radius: 10px;
            min-width: 180px;
            align-items: center;
            justify-content: space-between;
        }
        
        .stat-card-info {
            display: flex;
            flex-direction: column;
        }
        
        .stat-card-label {
            font-size: 0.8rem;
            opacity: 0.8;
        }
        
        .stat-card-value {
            font-size: 1.5rem;
            font-weight: 600;
        }
        
        .stat-card-icon {
            display: flex;
            align-items: center;
            justify-content: center;
            width: 40px;
            height: 40px;
            border-radius: 8px;
            color: #fff;
        }
        
        .bg-light-primary {
            background-color: rgba(13, 110, 253, 0.1);
            color: #0d6efd;
        }
        
        .bg-light-info {
            background-color: rgba(13, 202, 240, 0.1);
            color: #0dcaf0;
        }
        
        .bg-light-success {
            background-color: rgba(25, 135, 84, 0.1);
            color: #198754;
        }
        
        .alert-icon {
            display: flex;
            align-items: center;
            justify-content: center;
            min-width: 40px;
        }
        
        .approval-card {
            border: none;
            transition: all 0.3s ease;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
        }
        
        .approval-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 8px 20px rgba(0,0,0,0.08);
        }
        
        .approval-card .card-header {
            border-bottom: 1px solid rgba(0,0,0,0.05);
        }
        
        .action-btn {
            border-radius: 50px;
            padding: 0.375rem 1rem;
        }
        
        .view-btn {
            background-color: #f8f9fa;
            color: #212529;
            border: 1px solid #dee2e6;
        }
        
        .view-btn:hover {
            background-color: #e9ecef;
        }
        
        .approval-status {
            font-size: 0.8rem;
            padding: 0.25rem 0.75rem;
            border-radius: 50px;
        }
        
        .avatar-container {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            overflow: hidden;
            background-color: #e9ecef;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 12px;
        }
        
        .avatar-container i {
            font-size: 1.5rem;
            opacity: 0.5;
        }
        
        .avatar-circle {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1rem;
            font-weight: 500;
        }
        
        .icon-box {
            width: 28px;
            height: 28px;
            display: flex;
            align-items: center;
            justify-content: center;
            flex-shrink: 0;
        }
        
        .bg-light-secondary {
            background-color: rgba(108, 117, 125, 0.1);
            color: #6c757d;
        }
        
        .bg-primary-light {
            background-color: rgba(13, 110, 253, 0.1);
        }
        
        .bg-light-info {
            background-color: rgba(13, 202, 240, 0.1);
            color: #0dcaf0;
        }
        
        .bg-light-success {
            background-color: rgba(25, 135, 84, 0.1);
            color: #198754;
        }
        
        .card-icon-circle {
            width: 42px;
            height: 42px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            background-color: rgba(13, 110, 253, 0.1);
        }
        
        .empty-state {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            padding: 3rem 1.5rem;
            text-align: center;
        }
        
        .empty-state-img {
            max-width: 120px;
            margin-bottom: 1.5rem;
            opacity: 0.6;
        }
    </style>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Handle form submissions with confirmation
        document.querySelectorAll('form').forEach(form => {
            form.addEventListener('submit', function(e) {
                // Check if this is an approve or reject form
                if (form.querySelector('[name="approve"]') || form.querySelector('[name="approve_video"]')) {
                    if (!confirm('Are you sure you want to approve this item?')) {
                        e.preventDefault();
                        return false;
                    }
                } else if (form.querySelector('[name="reject"]') || form.querySelector('[name="reject_video"]')) {
                    if (!confirm('Are you sure you want to reject this item?')) {
                        e.preventDefault();
                        return false;
                    }
                }
            });
        });
        
        // Enable all tooltips
        const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
        tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });
    });
    </script>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="container-fluid p-4">
                <!-- Page Header with Breadcrumb -->
                <div class="page-header mb-4">
                    <nav aria-label="breadcrumb">
                        <ol class="breadcrumb">
                            <li class="breadcrumb-item"><a href="index.php"><i class='bx bx-home-alt me-1'></i>Dashboard</a></li>
                            <li class="breadcrumb-item active" aria-current="page">Approve Facilitators</li>
                        </ol>
                    </nav>
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <h1 class="h2 mb-0 d-flex align-items-center">
                                <i class='bx bx-check-shield fs-1 me-3 text-primary'></i>
                                Approve Facilitators
                            </h1>
                            <p class="text-muted mb-0 mt-2">Review and manage facilitator and content approval requests</p>
                        </div>
                        <div class="d-flex">
                            <div class="stat-card me-3 bg-light-primary">
                                <div class="stat-card-info">
                                    <span class="stat-card-label">Pending Facilitators</span>
                                    <h4 class="stat-card-value mb-0"><?php echo count($pendingTeachers); ?></h4>
                                </div>
                                <div class="stat-card-icon bg-primary">
                                    <i class='bx bx-user-plus'></i>
                                </div>
                            </div>
                            <div class="stat-card bg-light-info">
                                <div class="stat-card-info">
                                    <span class="stat-card-label">Pending Videos</span>
                                    <h4 class="stat-card-value mb-0"><?php echo count($pendingVideos); ?></h4>
                                </div>
                                <div class="stat-card-icon bg-info">
                                    <i class='bx bx-video'></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Display success and error messages -->
                <?php if(!empty($successMsg)): ?>
                    <div class="alert alert-success alert-dismissible fade show d-flex align-items-center" role="alert">
                        <div class="alert-icon me-3">
                            <i class="bx bx-check-circle fs-4"></i>
                        </div>
                        <div class="alert-content">
                            <strong>Success!</strong> <?php echo $successMsg; ?>
                        </div>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                
                <?php if(!empty($errorMsg)): ?>
                    <div class="alert alert-danger alert-dismissible fade show d-flex align-items-center" role="alert">
                        <div class="alert-icon me-3">
                            <i class="bx bx-error-circle fs-4"></i>
                        </div>
                        <div class="alert-content">
                            <strong>Error!</strong> <?php echo $errorMsg; ?>
                        </div>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                
                <!-- Pending Facilitators Section -->
                <div class="card mb-4 shadow-sm approval-card">
                    <div class="card-header d-flex justify-content-between align-items-center bg-white py-3">
                        <div class="d-flex align-items-center">
                            <div class="card-icon-circle bg-primary-light me-3">
                                <i class="bx bx-user-plus fs-4 text-primary"></i>
                            </div>
                            <div>
                                <h5 class="mb-0 fw-semibold">Pending Facilitator Applications</h5>
                                <p class="text-muted small mb-0">Review and manage facilitator applications</p>
                            </div>
                        </div>
                        <span class="badge bg-primary rounded-pill px-3 py-2"><?php echo count($pendingTeachers); ?></span>
                    </div>
                    <div class="card-body p-0">
                        <?php if (empty($pendingTeachers)): ?>
                            <div class="empty-state">
                                <img src="../assets/img/empty-state.svg" alt="No applications" class="empty-state-img">
                                <h6 class="fw-semibold">No Pending Applications</h6>
                                <p class="text-muted">New facilitator applications will appear here when submitted</p>
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table align-middle mb-0 border-0">
                                    <thead>
                                        <tr class="bg-light">
                                            <th class="ps-4 py-3 text-secondary">APPLICANT</th>
                                            <th class="py-3 text-secondary">EMAIL</th>
                                            <th class="py-3 text-secondary">COURSE</th>
                                            <th class="py-3 text-secondary">APPLICATION DATE</th>
                                            <th class="text-end pe-4 py-3 text-secondary">ACTIONS</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($pendingTeachers as $teacher): ?>
                                            <tr class="border-bottom">
                                                <td class="ps-4 py-3">
                                                    <div class="d-flex align-items-center">
                                                        <div class="avatar-container">
                                                            <?php if (!empty($teacher['profile_image'])): ?>
                                                                <img src="<?php echo htmlspecialchars($teacher['profile_image']); ?>" alt="Profile" class="rounded-circle" width="40" height="40">
                                                            <?php else: ?>
                                                                <div class="avatar-circle bg-primary-light text-primary">
                                                                    <?php echo substr($teacher['first_name'], 0, 1) . substr($teacher['last_name'], 0, 1); ?>
                                                                </div>
                                                            <?php endif; ?>
                                                        </div>
                                                        <div>
                                                            <h6 class="mb-0 fw-semibold"><?php echo htmlspecialchars($teacher['first_name'] . ' ' . $teacher['last_name']); ?></h6>
                                                            <span class="text-muted small">@<?php echo htmlspecialchars($teacher['username'] ?? 'teacher'); ?></span>
                                                        </div>
                                                    </div>
                                                </td>
                                                <td class="py-3">
                                                    <div class="d-flex align-items-center">
                                                        <div class="icon-box bg-light-info rounded-circle me-2">
                                                            <i class='bx bx-envelope text-info'></i>
                                                        </div>
                                                        <span><?php echo htmlspecialchars($teacher['email']); ?></span>
                                                    </div>
                                                </td>
                                                <td class="py-3">
                                                    <?php if (!empty($teacher['course_title'])): ?>
                                                        <div class="d-flex align-items-center">
                                                            <div class="icon-box bg-light-success rounded-circle me-2">
                                                                <i class='bx bx-book-open text-success'></i>
                                                            </div>
                                                            <span class="course-title"><?php echo htmlspecialchars($teacher['course_title']); ?></span>
                                                        </div>
                                                    <?php else: ?>
                                                        <div class="d-flex align-items-center">
                                                            <div class="icon-box bg-light-secondary rounded-circle me-2">
                                                                <i class='bx bx-info-circle text-secondary'></i>
                                                            </div>
                                                            <span class="text-muted">Not specified</span>
                                                        </div>
                                                    <?php endif; ?>
                                                </td>
                                                <td class="py-3">
                                                    <div class="d-flex align-items-center">
                                                        <div class="icon-box bg-light-primary rounded-circle me-2">
                                                            <i class='bx bx-calendar text-primary'></i>
                                                        </div>
                                                        <span class="date-applied"><?php echo date('M d, Y', strtotime($teacher['created_at'])); ?></span>
                                                    </div>
                                                </td>
                                                <td class="text-end pe-4 py-3">
                                                    <div class="d-flex justify-content-end gap-2">
                                                        <!-- View profile button -->
                                                        <button type="button" class="btn action-btn view-btn" data-bs-toggle="modal" data-bs-target="#profileModal<?php echo $teacher['user_id']; ?>">
                                                            <i class='bx bx-user me-1'></i> View
                                                        </button>
                                                        <!-- Approve button -->
                                                        <form method="POST" action="approve_facilitators.php" class="d-inline">
                                                            <input type="hidden" name="teacher_id" value="<?php echo $teacher['user_id']; ?>">
                                                            <button type="submit" name="approve" value="1" class="btn action-btn btn-success">
                                                                <i class='bx bx-check-circle me-1'></i> Approve
                                                            </button>
                                                        </form>
                                                        <!-- Reject button -->
                                                        <form method="POST" action="approve_facilitators.php" class="d-inline">
                                                            <input type="hidden" name="teacher_id" value="<?php echo $teacher['user_id']; ?>">
                                                            <button type="submit" name="reject" value="1" class="btn action-btn btn-danger">
                                                                <i class='bx bx-x-circle me-1'></i> Reject
                                                            </button>
                                                        </form>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            
                            <!-- Teacher Profile Modals -->
                            <?php foreach ($pendingTeachers as $teacher): ?>
                                <div class="modal fade" id="profileModal<?php echo $teacher['user_id']; ?>" tabindex="-1" aria-hidden="true">
                                    <div class="modal-dialog modal-dialog-centered modal-lg">
                                        <div class="modal-content">
                                            <div class="modal-header bg-light">
                                                <h5 class="modal-title">Facilitator Profile</h5>
                                                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                            </div>
                                            <div class="modal-body">
                                                <div class="d-flex align-items-center mb-4">
                                                    <div class="me-3" style="width: 80px; height: 80px;">
                                                        <div class="bg-primary rounded-circle text-white d-flex align-items-center justify-content-center" style="width: 100%; height: 100%; font-size: 1.8rem;">
                                                            <?php echo substr($teacher['first_name'], 0, 1) . substr($teacher['last_name'], 0, 1); ?>
                                                        </div>
                                                    </div>
                                                    <div>
                                                        <h4 class="mb-0"><?php echo $teacher['first_name'] . ' ' . $teacher['last_name']; ?></h4>
                                                        <p class="text-muted mb-0">@<?php echo $teacher['username']; ?></p>
                                                    </div>
                                                </div>
                                                
                                                <div class="card mb-3">
                                                    <div class="card-body">
                                                        <h6 class="card-subtitle mb-2 text-muted">Contact Information</h6>
                                                        <p class="mb-1"><i class="bx bx-envelope me-1"></i> <?php echo $teacher['email']; ?></p>
                                                        <p class="mb-1"><i class="bx bx-calendar me-1"></i> Registered: <?php echo date('F j, Y', strtotime($teacher['created_at'])); ?></p>
                                                    </div>
                                                </div>
                                                
                                                <div class="card mb-3">
                                                    <div class="card-body">
                                                        <h6 class="card-subtitle mb-2 text-muted">Course Information</h6>
                                                        <p class="text-muted mb-0">
                                                            <i class="bx bx-book me-1"></i> 
                                                            <?php echo !empty($teacher['course_title']) ? htmlspecialchars($teacher['course_title']) : 'No course information available'; ?>
                                                        </p>
                                                    </div>
                                                </div>
                                                
                                                <?php if (!empty($teacher['bio'])): ?>
                                                <div class="card mb-3">
                                                    <div class="card-body">
                                                        <h6 class="card-subtitle mb-2 text-muted">Bio</h6>
                                                        <p class="mb-0"><?php echo nl2br(htmlspecialchars($teacher['bio'])); ?></p>
                                                    </div>
                                                </div>
                                                <?php endif; ?>
                                                
                                                <?php if (!empty($teacher['qualifications'])): ?>
                                                <div class="card mb-3">
                                                    <div class="card-body">
                                                        <h6 class="card-subtitle mb-2 text-muted">Qualifications</h6>
                                                        <p class="mb-0"><?php echo nl2br(htmlspecialchars($teacher['qualifications'])); ?></p>
                                                    </div>
                                                </div>
                                                <?php endif; ?>
                                                
                                                <?php if (!empty($teacher['expertise'])): ?>
                                                <div class="card mb-3">
                                                    <div class="card-body">
                                                        <h6 class="card-subtitle mb-2 text-muted">Areas of Expertise</h6>
                                                        <p class="mb-0"><?php echo nl2br(htmlspecialchars($teacher['expertise'])); ?></p>
                                                    </div>
                                                </div>
                                                <?php endif; ?>
                                            </div>
                                            <div class="modal-footer">
                                                <form method="POST" class="me-2">
                                                    <input type="hidden" name="teacher_id" value="<?php echo $teacher['user_id']; ?>">
                                                    <button type="submit" name="approve" class="btn btn-success">
                                                        <i class='bx bx-check-circle me-1'></i> Approve
                                                    </button>
                                                </form>
                                                <form method="POST">
                                                    <input type="hidden" name="teacher_id" value="<?php echo $teacher['user_id']; ?>">
                                                    <button type="submit" name="reject" class="btn btn-danger">
                                                        <i class='bx bx-x-circle me-1'></i> Reject
                                                    </button>
                                                </form>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
                
            <!-- Pending Videos Section -->
            <div class="card shadow-sm mb-4 approval-card">
                <div class="card-header d-flex justify-content-between align-items-center bg-white py-3">
                    <div class="d-flex align-items-center">
                        <div class="card-icon-circle bg-light-info me-3">
                            <i class="bx bx-video fs-4 text-info"></i>
                        </div>
                        <div>
                            <h5 class="mb-0 fw-semibold">Pending Video Approvals</h5>
                            <p class="text-muted small mb-0">Review and approve educational content</p>
                        </div>
                    </div>
                    <span class="badge bg-info rounded-pill px-3 py-2"><?php echo count($pendingVideos); ?></span>
                </div>
                <div class="card-body p-0">
                    <?php if (empty($pendingVideos)): ?>
                        <div class="empty-state">
                            <img src="../assets/img/empty-state.svg" alt="No videos" class="empty-state-img">
                            <h6 class="fw-semibold">No Pending Videos</h6>
                            <p class="text-muted">New video submissions will appear here for approval</p>
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table align-middle mb-0 border-0">
                                <thead>
                                    <tr class="bg-light">
                                        <th class="ps-4 py-3 text-secondary">VIDEO</th>
                                        <th class="py-3 text-secondary">INSTRUCTOR</th>
                                        <th class="py-3 text-secondary">COURSE</th>
                                        <th class="py-3 text-secondary">UPLOAD DATE</th>
                                        <th class="text-end pe-4 py-3 text-secondary">ACTIONS</th>
                                    </tr>
                                </thead>
                                <tbody>
                                        <?php foreach ($pendingVideos as $video): ?>
                                            <tr class="border-bottom">
                                                <td class="ps-4 py-3">
                                                    <div class="d-flex align-items-center">
                                                        <div class="video-thumbnail me-3" style="width: 100px; height: 60px; border-radius: 6px; overflow: hidden; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
                                                            <?php if (!empty($video['thumbnail'])): ?>
                                                                <img src="<?php echo htmlspecialchars($video['thumbnail']); ?>" alt="<?php echo htmlspecialchars($video['title']); ?>" style="width: 100%; height: 100%; object-fit: cover;">
                                                            <?php else: ?>
                                                                <div class="d-flex align-items-center justify-content-center h-100 bg-light-info">
                                                                    <i class='bx bx-video-off text-info fs-3'></i>
                                                                </div>
                                                            <?php endif; ?>
                                                        </div>
                                                        <div>
                                                            <h6 class="mb-1 fw-semibold"><?php echo htmlspecialchars($video['title']); ?></h6>
                                                            <div class="d-flex align-items-center">
                                                                <span class="badge bg-light-secondary text-secondary me-2 approval-status">Pending</span>
                                                                <small class="text-muted">ID: <?php echo $video['video_id']; ?></small>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </td>
                                                <td class="py-3">
                                                    <div class="d-flex align-items-center">
                                                        <div class="icon-box bg-light-primary rounded-circle me-2">
                                                            <i class='bx bx-user text-primary'></i>
                                                        </div>
                                                        <span><?php echo htmlspecialchars($video['first_name'] . ' ' . $video['last_name']); ?></span>
                                                    </div>
                                                </td>
                                                <td class="py-3">
                                                    <div class="d-flex align-items-center">
                                                        <div class="icon-box bg-light-info rounded-circle me-2">
                                                            <i class='bx bx-book-open text-info'></i>
                                                        </div>
                                                        <span><?php echo htmlspecialchars($video['course_title']); ?></span>
                                                    </div>
                                                </td>
                                                <td class="py-3">
                                                    <div class="d-flex align-items-center">
                                                        <div class="icon-box bg-light-secondary rounded-circle me-2">
                                                            <i class='bx bx-calendar text-secondary'></i>
                                                        </div>
                                                        <span><?php echo date('M d, Y', strtotime($video['created_at'])); ?></span>
                                                    </div>
                                                </td>
                                                <td class="text-end pe-4 py-3">
                                                    <div class="d-flex justify-content-end gap-2">
                                                        <!-- View video button -->
                                                        <button type="button" class="btn action-btn view-btn" data-bs-toggle="modal" data-bs-target="#videoModal<?php echo $video['video_id']; ?>">
                                                            <i class='bx bx-play-circle me-1'></i> Preview
                                                        </button>
                                                        <!-- Approve button -->
                                                        <form method="POST" action="approve_facilitators.php" class="d-inline">
                                                            <input type="hidden" name="video_id" value="<?php echo $video['video_id']; ?>">
                                                            <button type="submit" name="approve_video" value="1" class="btn action-btn btn-success">
                                                                <i class='bx bx-check-circle me-1'></i> Approve
                                                            </button>
                                                        </form>
                                                        <!-- Reject button -->
                                                        <form method="POST" action="approve_facilitators.php" class="d-inline">
                                                            <input type="hidden" name="video_id" value="<?php echo $video['video_id']; ?>">
                                                            <button type="submit" name="reject_video" value="1" class="btn action-btn btn-danger">
                                                                <i class='bx bx-x-circle me-1'></i> Reject
                                                            </button>
                                                        </form>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            
                            <!-- Video Preview Modals -->
                            <?php foreach ($pendingVideos as $video): ?>
                                <div class="modal fade" id="videoModal<?php echo $video['video_id']; ?>" tabindex="-1" aria-hidden="true">
                                    <div class="modal-dialog modal-dialog-centered modal-lg">
                                        <div class="modal-content">
                                            <div class="modal-header border-0 pb-0">
                                                <div class="d-flex align-items-center">
                                                    <div class="card-icon-circle bg-light-info me-2">
                                                        <i class='bx bx-video fs-4 text-info'></i>
                                                    </div>
                                                    <h5 class="modal-title fw-semibold">Video Preview</h5>
                                                </div>
                                                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                            </div>
                                            <div class="modal-body pt-2">
                                                <?php if (!empty($video['thumbnail'])): ?>
                                                <div class="video-preview mb-4 position-relative" style="border-radius: 12px; overflow: hidden; box-shadow: 0 4px 20px rgba(0,0,0,0.1);">
                                                    <img src="<?php echo htmlspecialchars($video['thumbnail']); ?>" alt="<?php echo htmlspecialchars($video['title']); ?>" class="img-fluid w-100" style="max-height: 400px; object-fit: cover;">
                                                    <div class="position-absolute top-50 start-50 translate-middle">
                                                        <div class="bg-white rounded-circle d-flex align-items-center justify-content-center" style="width: 64px; height: 64px; opacity: 0.9; box-shadow: 0 2px 12px rgba(0,0,0,0.15);">
                                                            <i class='bx bx-play fs-3 text-info'></i>
                                                        </div>
                                                    </div>
                                                </div>
                                                <?php endif; ?>
                                                
                                                <div class="mb-4">
                                                    <h4 class="fw-bold mb-3"><?php echo htmlspecialchars($video['title']); ?></h4>
                                                    
                                                    <div class="row g-3 mb-4">
                                                        <div class="col-md-6 col-lg-3">
                                                            <div class="d-flex align-items-center p-3 border rounded bg-light-primary bg-opacity-10">
                                                                <div class="icon-box bg-light-primary rounded-circle me-3">
                                                                    <i class='bx bx-user text-primary'></i>
                                                                </div>
                                                                <div>
                                                                    <div class="small text-muted">Instructor</div>
                                                                    <div class="fw-semibold"><?php echo htmlspecialchars($video['first_name'] . ' ' . $video['last_name']); ?></div>
                                                                </div>
                                                            </div>
                                                        </div>
                                                        
                                                        <div class="col-md-6 col-lg-3">
                                                            <div class="d-flex align-items-center p-3 border rounded bg-light-info bg-opacity-10">
                                                                <div class="icon-box bg-light-info rounded-circle me-3">
                                                                    <i class='bx bx-book-open text-info'></i>
                                                                </div>
                                                                <div>
                                                                    <div class="small text-muted">Course</div>
                                                                    <div class="fw-semibold"><?php echo htmlspecialchars($video['course_title']); ?></div>
                                                                </div>
                                                            </div>
                                                        </div>
                                                        
                                                        <div class="col-md-6 col-lg-3">
                                                            <div class="d-flex align-items-center p-3 border rounded bg-light-secondary bg-opacity-10">
                                                                <div class="icon-box bg-light-secondary rounded-circle me-3">
                                                                    <i class='bx bx-calendar text-secondary'></i>
                                                                </div>
                                                                <div>
                                                                    <div class="small text-muted">Upload Date</div>
                                                                    <div class="fw-semibold"><?php echo date('M d, Y', strtotime($video['created_at'])); ?></div>
                                                                </div>
                                                            </div>
                                                        </div>
                                                        
                                                        <div class="col-md-6 col-lg-3">
                                                            <div class="d-flex align-items-center p-3 border rounded bg-light-success bg-opacity-10">
                                                                <div class="icon-box bg-light-success rounded-circle me-3">
                                                                    <i class='bx bx-dollar-circle text-success'></i>
                                                                </div>
                                                                <div>
                                                                    <div class="small text-muted">Price</div>
                                                                    <div class="fw-semibold">$<?php echo number_format($video['price'], 2); ?></div>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    </div>
                                                    
                                                    <div class="card border-0 bg-light">
                                                        <div class="card-body">
                                                            <h6 class="fw-semibold mb-3">Description</h6>
                                                            <p class="mb-0"><?php echo nl2br(htmlspecialchars($video['description'])); ?></p>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="modal-footer border-0 pt-0">
                                                <form method="POST" action="approve_facilitators.php" class="me-2">
                                                    <input type="hidden" name="video_id" value="<?php echo $video['video_id']; ?>">
                                                    <button type="submit" name="approve_video" value="1" class="btn action-btn btn-success px-4">
                                                        <i class='bx bx-check-circle me-1'></i> Approve
                                                    </button>
                                                </form>
                                                <form method="POST" action="approve_facilitators.php">
                                                    <input type="hidden" name="video_id" value="<?php echo $video['video_id']; ?>">
                                                    <button type="submit" name="reject_video" value="1" class="btn action-btn btn-danger px-4">
                                                        <i class='bx bx-x-circle me-1'></i> Reject
                                                    </button>
                                                </form>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
        </div>
    </section>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Sidebar toggle functionality
        const sidebarToggle = document.querySelector('.sidebar-toggle');
        const sidebarToggleMobile = document.querySelector('.sidebar-toggle-mobile');
        const sidebar = document.querySelector('.sidebar');
        
        if (sidebarToggle) {
            sidebarToggle.addEventListener('click', () => {
                sidebar.classList.toggle('close');
            });
        }
        
        if (sidebarToggleMobile) {
            sidebarToggleMobile.addEventListener('click', () => {
                sidebar.classList.toggle('show');
            });
        }
        
        // Responsive sidebar behavior
        function checkScreenSize() {
            if (window.innerWidth < 768) {
                sidebar.classList.remove('close');
                sidebar.classList.remove('show');
            } else if (window.innerWidth < 992) {
                sidebar.classList.add('close');
                sidebar.classList.remove('show');
            }
        }
        
        // Initial check and listen for window resize
        checkScreenSize();
        window.addEventListener('resize', checkScreenSize);
        
        // Dropdown functionality
        document.querySelectorAll('.dropdown-toggle').forEach(item => {
            item.addEventListener('click', event => {
                event.preventDefault();
                const dropdownMenu = item.nextElementSibling;
                if (dropdownMenu && dropdownMenu.classList.contains('dropdown-menu')) {
                    dropdownMenu.classList.toggle('show');
                }
            });
        });
        
        // Close dropdowns when clicking outside
        document.addEventListener('click', event => {
            const isDropdownButton = event.target.classList.contains('dropdown-toggle');
            if (!isDropdownButton) {
                document.querySelectorAll('.dropdown-menu.show').forEach(dropdown => {
                    dropdown.classList.remove('show');
                });
            }
        });
    });
    </script>
</body>
</html>
