<?php
require_once __DIR__ . '/../auth/functions.php';
requireRole('admin');

require_once __DIR__ . '/../config/db_connect.php';

$message = '';
$messageType = '';

// Handle grading, assessment actions, and document approvals
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        $action = $_POST['action'];
        
        if ($action === 'grade_assessment') {
            $assessment_id = intval($_POST['assessment_id']);
            $score = floatval($_POST['score']);
            $feedback = trim($_POST['feedback']);
            $admin_id = $_SESSION['user_id'];
            
            // Validate score
            if ($score < 0 || $score > 100) {
                $message = 'Score must be between 0 and 100.';
                $messageType = 'error';
            } else {
                // Update assessment with score and feedback
                $status = $score >= 70 ? 'passed' : 'failed';
                $grade_stmt = $conn->prepare("UPDATE sita_assessments SET score = ?, feedback = ?, status = ?, graded_by = ?, graded_date = NOW() WHERE assessment_id = ?");
                $grade_stmt->bind_param("dssii", $score, $feedback, $status, $admin_id, $assessment_id);
                
                if ($grade_stmt->execute()) {
                    // Get assessment details for email
                    $assess_stmt = $conn->prepare("SELECT sa.*, u.email, u.first_name, u.last_name, c.title as course_title, ar.request_id FROM sita_assessments sa JOIN users u ON sa.student_id = u.user_id JOIN courses c ON sa.course_id = c.course_id JOIN accreditation_requests ar ON sa.accreditation_request_id = ar.request_id WHERE sa.assessment_id = ?");
                    $assess_stmt->bind_param("i", $assessment_id);
                    $assess_stmt->execute();
                    $assessment_data = $assess_stmt->get_result()->fetch_assoc();
                    
                    // Update accreditation request status and generate certificate if passed
                    if ($status === 'passed') {
                        $status_update = $conn->prepare("UPDATE accreditation_requests SET status = 'assessment_passed' WHERE request_id = ?");
                        $status_update->bind_param("i", $assessment_data['request_id']);
                        $status_update->execute();
                        
                        // Automatically generate SITA accreditation certificate
                        require_once __DIR__ . '/../config/certificate_hooks.php';
                        $cert_result = triggerAccreditationCertificate(
                            $assessment_data['student_id'], 
                            $assessment_data['course_id'], 
                            $assessment_id, 
                            $conn
                        );
                        
                        if ($cert_result['success']) {
                            $message .= " An accreditation certificate has been automatically generated for the student.";
                        }
                    } else {
                        $status_update = $conn->prepare("UPDATE accreditation_requests SET status = 'assessment_failed' WHERE request_id = ?");
                        $status_update->bind_param("i", $assessment_data['request_id']);
                        $status_update->execute();
                    }
                    
                    // Send email notification to student
                    require_once __DIR__ . '/../config/fast_email_service.php';
                    
                    $email_subject = "Assessment Results - {$assessment_data['course_title']}";
                    $email_body = "
                    <h2>📋 Your Assessment Has Been Graded</h2>
                    <p>Dear {$assessment_data['first_name']},</p>
                    <p>Your assessment for <strong>{$assessment_data['course_title']}</strong> has been graded.</p>
                    
                    <div style='background: " . ($status === 'passed' ? '#f0fdf4; border: 2px solid #22c55e;' : '#fef2f2; border: 2px solid #ef4444;') . " border-radius: 8px; padding: 15px; margin: 20px 0;'>
                        <h3 style='color: " . ($status === 'passed' ? '#16a34a;' : '#dc2626;') . " margin-top: 0;'>
                            " . ($status === 'passed' ? '🎉 Congratulations! You Passed!' : '📝 Assessment Not Passed') . "
                        </h3>
                        <p><strong>Score:</strong> {$score}% (Minimum required: 70%)</p>
                        " . ($feedback ? "<p><strong>Feedback:</strong> {$feedback}</p>" : "") . "
                    </div>
                    
                    " . ($status === 'passed' ? 
                        "<p><strong>Next Steps:</strong> You can now proceed to upload your required documents to complete the accreditation process.</p>
                        <p><a href='" . $_SERVER['HTTP_HOST'] . "/student/required-documents.php?id={$assessment_id}' style='background: #16a085; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;'>Upload Documents</a></p>" :
                        "<p><strong>Next Steps:</strong> You may retake the assessment after reviewing the feedback. Please contact support if you need additional guidance.</p>"
                    ) . "
                    
                    <p>Thank you for your dedication to professional development!</p>
                    
                    <hr style='margin: 30px 0;'>
                    <p style='font-size: 0.9em; color: #64748b;'>
                        <strong>Panadite Academy</strong><br>
                        Professional Development & Accreditation<br>
                        This is an automated message.
                    </p>
                    ";
                    
                    queueEmail(
                        $assessment_data['email'],
                        $email_subject,
                        $email_body,
                        'assessment_graded',
                        'high'
                    );
                    
                    // Log audit trail
                    $audit_stmt = $conn->prepare("INSERT INTO sita_assessment_audit_log (assessment_id, student_id, action, performed_by, details) VALUES (?, ?, 'assessment_graded', ?, ?)");
                    $audit_details = "Assessment graded with score: {$score}%, Status: {$status}";
                    $audit_stmt->bind_param("iiis", $assessment_id, $assessment_data['student_id'], $admin_id, $audit_details);
                    $audit_stmt->execute();
                    
                    $message = "Assessment graded successfully. Student has been notified via email.";
                    $messageType = 'success';
                } else {
                    $message = 'Error grading assessment.';
                    $messageType = 'error';
                }
            }
        } elseif ($action === 'approve_documents') {
            // Handle document approval for accreditation requests
            $request_id = intval($_POST['request_id']);
            $admin_id = $_SESSION['user_id'];
            $comments = trim($_POST['approval_comments'] ?? 'Documents approved for SITA accreditation');
            
            // Update all documents to verified status
            $update_docs = $conn->prepare("UPDATE accreditation_documents SET verification_status = 'verified', verified_by = ?, verified_date = NOW() WHERE request_id = ?");
            $update_docs->bind_param("ii", $admin_id, $request_id);
            
            // Update request status to approved_final (ready for certificate)
            $update_request = $conn->prepare("UPDATE accreditation_requests SET status = 'approved_final', final_admin_id = ?, final_review_date = NOW(), final_comments = ? WHERE request_id = ?");
            $update_request->bind_param("isi", $admin_id, $comments, $request_id);
            
            if ($update_docs->execute() && $update_request->execute()) {
                // Get student details for notification
                $student_query = $conn->prepare("SELECT u.email, u.first_name, u.last_name, c.title as course_title FROM accreditation_requests ar JOIN users u ON ar.student_id = u.user_id JOIN courses c ON ar.course_id = c.course_id WHERE ar.request_id = ?");
                $student_query->bind_param("i", $request_id);
                $student_query->execute();
                $student_data = $student_query->get_result()->fetch_assoc();
                
                if ($student_data) {
                    // Send notification email to student
                    $to = $student_data['email'];
                    $subject = 'SITA Accreditation Documents Approved - Certificate Processing';
                    $message_body = "Dear {$student_data['first_name']} {$student_data['last_name']},\n\n";
                    $message_body .= "Great news! Your SITA accreditation documents for {$student_data['course_title']} have been reviewed and approved.\n\n";
                    $message_body .= "Your accreditation certificate is now being processed and will be sent to you within 5-10 business days.\n\n";
                    $message_body .= "Admin Comments: {$comments}\n\n";
                    $message_body .= "Thank you for choosing Panadite Academy!\n\n";
                    $message_body .= "Best regards,\nPanadite Academy Team";
                    
                    $headers = "From: noreply@panaditeacademy.com\r\n";
                    $headers .= "Reply-To: support@panaditeacademy.com\r\n";
                    
                    mail($to, $subject, $message_body, $headers);
                }
                
                $message = 'Documents approved successfully! Student has been notified that their certificate is being processed.';
                $messageType = 'success';
            } else {
                $message = 'Error approving documents. Please try again.';
                $messageType = 'error';
            }
            
        } elseif ($action === 'reject_documents') {
            // Handle document rejection for accreditation requests
            $request_id = intval($_POST['request_id']);
            $admin_id = $_SESSION['user_id'];
            $rejection_reason = trim($_POST['rejection_reason'] ?? 'Documents do not meet accreditation requirements');
            
            // Update all documents to rejected status
            $update_docs = $conn->prepare("UPDATE accreditation_documents SET verification_status = 'rejected', verified_by = ?, verified_date = NOW() WHERE request_id = ?");
            $update_docs->bind_param("ii", $admin_id, $request_id);
            
            // Update request status to rejected
            $update_request = $conn->prepare("UPDATE accreditation_requests SET status = 'rejected', rejection_reason = ?, approved_by = ?, approved_date = NOW() WHERE request_id = ?");
            $update_request->bind_param("sii", $rejection_reason, $admin_id, $request_id);
            
            if ($update_docs->execute() && $update_request->execute()) {
                // Get student details for notification
                $student_query = $conn->prepare("SELECT u.email, u.first_name, u.last_name, c.title as course_title FROM accreditation_requests ar JOIN users u ON ar.student_id = u.user_id JOIN courses c ON ar.course_id = c.course_id WHERE ar.request_id = ?");
                $student_query->bind_param("i", $request_id);
                $student_query->execute();
                $student_data = $student_query->get_result()->fetch_assoc();
                
                if ($student_data) {
                    // Send rejection notification email
                    $to = $student_data['email'];
                    $subject = 'SITA Accreditation Documents - Review Required';
                    $message_body = "Dear {$student_data['first_name']} {$student_data['last_name']},\n\n";
                    $message_body .= "Thank you for submitting your SITA accreditation documents for {$student_data['course_title']}.\n\n";
                    $message_body .= "After careful review, we need you to resubmit your documents with the following corrections:\n\n";
                    $message_body .= "Reason: {$rejection_reason}\n\n";
                    $message_body .= "Please log into your student portal and resubmit the corrected documents.\n\n";
                    $message_body .= "If you have any questions, please contact our support team.\n\n";
                    $message_body .= "Best regards,\nPanadite Academy Team";
                    
                    $headers = "From: noreply@panaditeacademy.com\r\n";
                    $headers .= "Reply-To: support@panaditeacademy.com\r\n";
                    
                    mail($to, $subject, $message_body, $headers);
                }
                
                $message = 'Documents rejected. Student has been notified to resubmit corrected documents.';
                $messageType = 'error';
            } else {
                $message = 'Error rejecting documents. Please try again.';
                $messageType = 'error';
            }
            
        } elseif ($action === 'upload_certificate') {
            // Handle certificate upload for accreditation requests
            $request_id = intval($_POST['request_id']);
            $admin_id = $_SESSION['user_id'];
            $certificate_notes = trim($_POST['certificate_notes'] ?? '');
            
            // Check if file was uploaded
            if (isset($_FILES['certificate_file']) && $_FILES['certificate_file']['error'] === UPLOAD_ERR_OK) {
                $upload_dir = __DIR__ . '/../uploads/certificates/';
                if (!is_dir($upload_dir)) {
                    mkdir($upload_dir, 0755, true);
                }
                
                $file_name = $_FILES['certificate_file']['name'];
                $file_extension = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
                $allowed_extensions = ['pdf', 'doc', 'docx', 'jpg', 'jpeg', 'png'];
                
                if (in_array($file_extension, $allowed_extensions) && $_FILES['certificate_file']['size'] <= 10 * 1024 * 1024) {
                    $safe_filename = 'cert_' . $request_id . '_' . time() . '.' . $file_extension;
                    $upload_path = $upload_dir . $safe_filename;
                    $web_path = '/uploads/certificates/' . $safe_filename;
                    
                    if (move_uploaded_file($_FILES['certificate_file']['tmp_name'], $upload_path)) {
                        // Update accreditation request with certificate
                        $update_cert = $conn->prepare("UPDATE accreditation_requests SET certificate_path = ?, certificate_issued_date = NOW(), status = 'certificate_issued', final_comments = ? WHERE request_id = ?");
                        $final_comments = $certificate_notes ?: 'SITA accreditation certificate has been issued and uploaded';
                        $update_cert->bind_param("ssi", $web_path, $final_comments, $request_id);
                        
                        if ($update_cert->execute()) {
                            // CRITICAL FIX: Insert into accreditation_certificates table
                            $cert_number = 'SITA-' . str_pad($request_id, 6, '0', STR_PAD_LEFT) . '-' . date('Y');
                            $verification_code = 'VER-' . strtoupper(substr(md5($request_id . time()), 0, 8));
                            
                            // Check if accreditation_certificates table exists, create if needed
                            $table_exists = $conn->query("SHOW TABLES LIKE 'accreditation_certificates'");
                            
                            if ($table_exists->num_rows == 0) {
                                // Create table if it doesn't exist
                                $create_table = "CREATE TABLE accreditation_certificates (
                                    certificate_id INT AUTO_INCREMENT PRIMARY KEY,
                                    request_id INT NOT NULL,
                                    certificate_number VARCHAR(100) UNIQUE NOT NULL,
                                    certificate_path VARCHAR(500) NOT NULL,
                                    issue_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                                    verification_code VARCHAR(50) NOT NULL,
                                    issued_by INT,
                                    status ENUM('active', 'revoked') DEFAULT 'active',
                                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                                    FOREIGN KEY (request_id) REFERENCES accreditation_requests(request_id),
                                    FOREIGN KEY (issued_by) REFERENCES users(user_id)
                                )";
                                $conn->query($create_table);
                            }
                            
                            // Insert certificate record
                            $insert_cert = $conn->prepare("INSERT INTO accreditation_certificates (request_id, certificate_number, certificate_path, verification_code, issued_by, status) VALUES (?, ?, ?, ?, ?, 'active')");
                            $insert_cert->bind_param("isssi", $request_id, $cert_number, $web_path, $verification_code, $admin_id);
                            
                            if ($insert_cert->execute()) {
                                error_log("SUCCESS: Created certificate record ID " . $conn->insert_id . " for request $request_id");
                            } else {
                                error_log("ERROR: Failed to create certificate record: " . $insert_cert->error);
                            }
                            
                            // Get student details for notification
                            $student_query = $conn->prepare("SELECT u.email, u.first_name, u.last_name, c.title as course_title FROM accreditation_requests ar JOIN users u ON ar.student_id = u.user_id JOIN courses c ON ar.course_id = c.course_id WHERE ar.request_id = ?");
                            $student_query->bind_param("i", $request_id);
                            $student_query->execute();
                            $student_data = $student_query->get_result()->fetch_assoc();
                            
                            if ($student_data) {
                                // Send notification email to student
                                $to = $student_data['email'];
                                $subject = 'SITA Accreditation Certificate Ready - Download Available';
                                $message_body = "Dear {$student_data['first_name']} {$student_data['last_name']},\n\n";
                                $message_body .= "Congratulations! Your SITA accreditation certificate for {$student_data['course_title']} is now ready.\n\n";
                                $message_body .= "You can download your certificate by logging into your student portal and visiting the course page.\n\n";
                                if ($certificate_notes) {
                                    $message_body .= "Certificate Notes: {$certificate_notes}\n\n";
                                }
                                $message_body .= "Thank you for choosing Panadite Academy!\n\n";
                                $message_body .= "Best regards,\nPanadite Academy Team";
                                
                                $headers = "From: noreply@panaditeacademy.com\r\n";
                                $headers .= "Reply-To: support@panaditeacademy.com\r\n";
                                
                                mail($to, $subject, $message_body, $headers);
                            }
                            
                            $message = 'Certificate uploaded successfully! Student has been notified and can now download their certificate.';
                            $messageType = 'success';
                        } else {
                            $message = 'Error updating database with certificate information.';
                            $messageType = 'error';
                        }
                    } else {
                        $message = 'Error uploading certificate file.';
                        $messageType = 'error';
                    }
                } else {
                    $message = 'Invalid file type or size. Please upload PDF, DOC, DOCX, JPG, or PNG files under 10MB.';
                    $messageType = 'error';
                }
            } else {
                $message = 'No certificate file was uploaded.';
                $messageType = 'error';
            }
            
        } elseif ($action === 'update_questions') {
            $assessment_id = intval($_POST['assessment_id']);
            $questions = $_POST['questions'];
            
            // Update questions
            $updated_count = 0;
            foreach ($questions as $question_id => $question_data) {
                $update_stmt = $conn->prepare("UPDATE sita_assessment_questions SET question_text = ?, max_words = ?, points = ? WHERE question_id = ? AND assessment_id = ?");
                $question_text = trim($question_data['text']);
                $max_words = intval($question_data['max_words']);
                $points = intval($question_data['points']);
                $update_stmt->bind_param("siiii", $question_text, $max_words, $points, $question_id, $assessment_id);
                
                if ($update_stmt->execute()) {
                    $updated_count++;
                }
            }
            
            if ($updated_count > 0) {
                $message = "{$updated_count} questions updated successfully.";
                $messageType = 'success';
            } else {
                $message = 'No questions were updated.';
                $messageType = 'error';
            }
        }
    }
}

// Get all accreditation requests for document review (simplified query without sita_assessments)
$assessments_query = "
    SELECT 
        CONCAT('request_', ar.request_id) as assessment_id,
        CONCAT('SITA Document Review - ', c.title) as title,
        CASE WHEN ar.status = 'assignment_created' THEN 'awaiting_completion' ELSE ar.status END as status,
        COALESCE(ar.documents_submitted_date, ar.assignment_created_date, ar.request_date) as created_date,
        ar.student_id,
        ar.course_id,
        ar.request_id,
        ar.status as request_status,
        u.first_name, u.last_name, u.email,
        c.title as course_title,
        ar.request_date, ar.assignment_instructions,
        ar.assignment_document_path, ar.documents_submitted_date,
        ar.final_comments, ar.rejection_reason,
        NULL as grader_first_name, NULL as grader_last_name,
        NULL as score, NULL as feedback, NULL as graded_date,
        COUNT(DISTINCT ad.document_id) as document_count,
        GROUP_CONCAT(CONCAT(ad.document_type, ':', ad.verification_status) SEPARATOR ',') as document_statuses
    FROM accreditation_requests ar
    JOIN users u ON ar.student_id = u.user_id
    JOIN courses c ON ar.course_id = c.course_id
    LEFT JOIN accreditation_documents ad ON ar.request_id = ad.request_id
    WHERE ar.status IN ('assignment_created', 'documents_submitted', 'assessment_passed', 'assessment_failed', 'approved_final', 'certificate_issued', 'rejected')
    GROUP BY ar.request_id
    ORDER BY 
        CASE ar.status 
            WHEN 'documents_submitted' THEN 1 
            WHEN 'assignment_created' THEN 2 
            WHEN 'approved_final' THEN 3 
            ELSE 4 
        END,
        COALESCE(ar.documents_submitted_date, ar.assignment_created_date, ar.request_date) DESC";

$assessments_result = $conn->query($assessments_query);
$assessments = [];
if ($assessments_result) {
    while ($row = $assessments_result->fetch_assoc()) {
        $assessments[] = $row;
    }
}

// Get current assessment if viewing one
$current_assessment = null;
$assessment_questions = [];
$student_answers = [];

if (isset($_GET['id'])) {
    $assessment_id = $_GET['id'];
    
    // Check if this is a document review request (starts with "request_")
    if (strpos($assessment_id, 'request_') === 0) {
        $request_id = intval(str_replace('request_', '', $assessment_id));
        
        // Get accreditation request details for document review
        $current_stmt = $conn->prepare("SELECT ar.*, u.first_name, u.last_name, u.email, c.title as course_title FROM accreditation_requests ar JOIN users u ON ar.student_id = u.user_id JOIN courses c ON ar.course_id = c.course_id WHERE ar.request_id = ?");
        if ($current_stmt) {
            $current_stmt->bind_param("i", $request_id);
            $current_stmt->execute();
            $current_assessment = $current_stmt->get_result()->fetch_assoc();
            
            // Convert accreditation request to assessment-like format
            if ($current_assessment) {
                $current_assessment['assessment_id'] = $assessment_id;
                $current_assessment['title'] = 'SITA Document Review - ' . $current_assessment['course_title'];
                $current_assessment['status'] = $current_assessment['status']; // Keep original status
                $current_assessment['created_date'] = $current_assessment['documents_submitted_date'] ?? $current_assessment['request_date'];
            }
        } else {
            echo "<div class='alert alert-danger'>Database query failed: " . $conn->error . "</div>";
            $current_assessment = null;
        }
    } else {
        // Handle regular numeric assessment IDs (for future sita_assessments integration)
        $assessment_id = intval($assessment_id);
        
        // For now, redirect back to main list since sita_assessments doesn't exist yet
        header("Location: assessment_management.php");
        exit();
    }
    
    if ($current_assessment) {
        // For document review requests, we don't have assessment questions
        if (strpos($assessment_id, 'request_') === 0) {
            // This is a document review, no questions needed
            $assessment_questions = [];
            $student_answers = [];
        } else {
            // For future sita_assessments integration - get questions and answers
            $questions_stmt = $conn->prepare("SELECT * FROM sita_assessment_questions WHERE assessment_id = ? ORDER BY question_order");
            if ($questions_stmt) {
                $questions_stmt->bind_param("i", $assessment_id);
                $questions_stmt->execute();
                $questions_result = $questions_stmt->get_result();
                
                while ($question = $questions_result->fetch_assoc()) {
                    $assessment_questions[] = $question;
                }
            }
            
            // Get student answers if assessment is submitted
            if ($current_assessment['status'] === 'submitted' || $current_assessment['status'] === 'graded' || $current_assessment['status'] === 'passed' || $current_assessment['status'] === 'failed') {
                $answers_stmt = $conn->prepare("SELECT question_id, answer_text FROM sita_assessment_answers WHERE assessment_id = ? AND student_id = ?");
                if ($answers_stmt) {
                    $answers_stmt->bind_param("ii", $assessment_id, $current_assessment['student_id']);
                    $answers_stmt->execute();
                    $answers_result = $answers_stmt->get_result();
                    
                    while ($answer = $answers_result->fetch_assoc()) {
                        $student_answers[$answer['question_id']] = $answer['answer_text'];
                    }
                }
            }
        }
    }
}

function getStatusBadgeClass($status) {
    switch ($status) {
        case 'created': return 'bg-info';
        case 'awaiting_completion': return 'bg-primary';
        case 'submitted': return 'bg-warning';
        case 'graded': return 'bg-success';
        case 'passed': return 'bg-success';
        case 'failed': return 'bg-danger';
        case 'assignment_created': return 'bg-primary';
        case 'documents_submitted': return 'bg-warning';
        case 'assessment_passed': return 'bg-success';
        case 'assessment_failed': return 'bg-danger';
        case 'certificate_issued': return 'bg-success';
        default: return 'bg-secondary';
    }
}

function getStatusText($status) {
    switch ($status) {
        case 'created': return 'Created';
        case 'awaiting_completion': return 'Awaiting Student';
        case 'submitted': return 'Submitted';
        case 'graded': return 'Graded';
        case 'passed': return 'Passed';
        case 'failed': return 'Failed';
        case 'assignment_created': return 'Assignment Ready';
        case 'documents_submitted': return 'Under Review';
        case 'assessment_passed': return 'Assessment Passed';
        case 'assessment_failed': return 'Assessment Failed';
        case 'certificate_issued': return 'Certificate Issued';
        default: return ucfirst(str_replace('_', ' ', $status));
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $current_assessment ? 'Assessment Details - ' . htmlspecialchars($current_assessment['course_title']) : 'Assessment Management' ?> - Admin Dashboard</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link rel="stylesheet" href="/assets/css/user-cards.css">
    <!-- DataTables CSS -->
    <link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/1.11.5/css/dataTables.bootstrap5.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <style>
        /* Modern Container Styling */
        .dashboard-content {
            padding: 2rem 2.5rem !important;
            background: #f8fafc;
            min-height: 100vh;
        }
        
        .page-header {
            margin-bottom: 2.5rem;
            padding: 1.5rem 2rem;
            background: white;
            border-radius: 16px;
            box-shadow: 0 2px 20px rgba(0, 0, 0, 0.08);
            border: 1px solid #e2e8f0;
        }
        
        .page-header h1 {
            font-size: 1.75rem;
            font-weight: 600;
            color: #1e293b;
            margin-bottom: 0.5rem;
        }
        
        .page-header p {
            color: #64748b;
            font-size: 0.95rem;
            margin-bottom: 0;
        }
        
        .assessment-card {
            background: white;
            border-radius: 16px;
            box-shadow: 0 4px 25px rgba(0, 0, 0, 0.08);
            overflow: hidden;
            margin-bottom: 2rem;
            border: 1px solid #e2e8f0;
        }
        
        .assessment-card .card-header {
            background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
            color: #92400e;
            padding: 1.5rem 2rem;
            border: none;
        }
        
        .assessment-card .card-body {
            padding: 2rem 2.5rem;
        }
        
        /* Document Section Styling */
        .documents-section {
            background: white;
            border-radius: 16px;
            box-shadow: 0 4px 25px rgba(0, 0, 0, 0.08);
            border: 1px solid #e2e8f0;
            overflow: hidden;
        }
        
        .documents-section .card-header {
            background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
            color: #92400e;
            padding: 1.25rem 2rem;
            border: none;
        }
        
        .documents-section .card-body {
            padding: 2rem;
        }
        
        /* Certificate Upload Section */
        .certificate-section {
            background: white;
            border-radius: 16px;
            box-shadow: 0 4px 25px rgba(0, 0, 0, 0.08);
            border: 1px solid #e2e8f0;
            overflow: hidden;
        }
        
        .certificate-section .card-header {
            background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
            color: #92400e;
            padding: 1.25rem 2rem;
            border: none;
        }
        
        .certificate-section .card-body {
            padding: 2rem;
        }
        
        /* Document Items */
        .document-item {
            background: #f8fafc;
            border: 2px solid #e2e8f0 !important;
            border-radius: 12px !important;
            padding: 1.5rem !important;
            margin-bottom: 1rem !important;
            transition: all 0.3s ease;
        }
        
        .document-item:hover {
            border-color: #f59e0b !important;
            box-shadow: 0 4px 15px rgba(245, 158, 11, 0.15);
            transform: translateY(-2px);
        }
        
        .file-icon i {
            font-size: 2.5rem !important;
            color: #f59e0b !important;
        }
        
        /* Form Elements */
        .form-control {
            border-radius: 8px;
            border: 2px solid #e2e8f0;
            padding: 0.75rem 1rem;
            font-size: 0.95rem;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            border-color: #f59e0b;
            box-shadow: 0 0 0 3px rgba(245, 158, 11, 0.1);
        }
        
        .btn {
            border-radius: 8px;
            padding: 0.75rem 1.5rem;
            font-weight: 500;
            font-size: 0.9rem;
            transition: all 0.3s ease;
        }
        
        .btn-success {
            background: linear-gradient(135deg, #10b981 0%, #059669 100%);
            border: none;
            box-shadow: 0 2px 10px rgba(16, 185, 129, 0.3);
            color: white;
        }
        
        .btn-success:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(16, 185, 129, 0.4);
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
            border: none;
            box-shadow: 0 2px 10px rgba(245, 158, 11, 0.3);
            color: white;
        }
        
        .btn-primary:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 15px rgba(245, 158, 11, 0.4);
            color: white;
        }
        
        /* Spacing and Layout */
        .row {
            margin-bottom: 1.5rem;
        }
        
        .col-md-8, .col-md-4 {
            padding: 0 1rem;
        }
        
        /* Badge Styling */
        .badge {
            font-size: 0.8rem;
            font-weight: 500;
            padding: 0.5rem 1rem;
            border-radius: 20px;
        }
        
        .badge.bg-primary {
            background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%) !important;
            color: white !important;
        }
        
        /* Info Text */
        .text-muted {
            color: #64748b !important;
        }
        
        /* Responsive Design */
        @media (max-width: 768px) {
            .dashboard-content {
                padding: 1rem 1.5rem !important;
            }
            
            .page-header {
                padding: 1rem 1.5rem;
                margin-bottom: 1.5rem;
            }
            
            .assessment-card .card-body,
            .documents-section .card-body,
            .certificate-section .card-body {
                padding: 1.5rem;
            }
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="container-fluid">
                <?php if ($message): ?>
                    <div class="alert alert-<?= $messageType === 'success' ? 'success' : 'danger' ?> alert-dismissible fade show" role="alert" style="border-radius: 12px; border: none; box-shadow: 0 2px 15px rgba(0, 0, 0, 0.1);">
                        <i class="bx <?= $messageType === 'success' ? 'bx-check-circle' : 'bx-error-circle' ?> me-2"></i>
                        <?= htmlspecialchars($message) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
            
            <?php if ($current_assessment): ?>
                <!-- Single Assessment Management View -->
                <div class="page-header">
                    <div>
                        <a href="assessment_management.php" class="btn btn-outline-secondary mb-2">
                            <i class="bx bx-arrow-back me-1"></i> Back to Assessments
                        </a>
                        <h1><i class="bx bx-edit me-2"></i>Assessment Management</h1>
                        <p class="text-muted">
                            Student: <?= htmlspecialchars($current_assessment['first_name'] . ' ' . $current_assessment['last_name']) ?> • 
                            Course: <?= htmlspecialchars($current_assessment['course_title']) ?>
                        </p>
                    </div>
                </div>
                
                <div class="assessment-card">
                    <div class="card-header">
                        <div class="d-flex justify-content-between align-items-center">
                            <h5 class="mb-0"><?= htmlspecialchars($current_assessment['title']) ?></h5>
                            <span class="badge <?= getStatusBadgeClass($current_assessment['status']) ?>">
                                <?= getStatusText($current_assessment['status']) ?>
                            </span>
                        </div>
                    </div>
                    
                    <div class="card-body">
                        <div class="row mb-4">
                            <div class="col-md-6">
                                <p><strong>Student:</strong> <?= htmlspecialchars($current_assessment['first_name'] . ' ' . $current_assessment['last_name']) ?></p>
                                <p><strong>Email:</strong> <?= htmlspecialchars($current_assessment['email']) ?></p>
                                <p><strong>Course:</strong> <?= htmlspecialchars($current_assessment['course_title']) ?></p>
                            </div>
                            <div class="col-md-6">
                                <p><strong>Created:</strong> <?= date('F j, Y g:i A', strtotime($current_assessment['created_date'])) ?></p>
                                <?php if ($current_assessment['due_date']): ?>
                                    <p><strong>Due Date:</strong> <?= date('F j, Y g:i A', strtotime($current_assessment['due_date'])) ?></p>
                                <?php endif; ?>
                                <?php if ($current_assessment['submitted_date']): ?>
                                    <p><strong>Submitted:</strong> <?= date('F j, Y g:i A', strtotime($current_assessment['submitted_date'])) ?></p>
                                <?php endif; ?>
                                <?php if ($current_assessment['score']): ?>
                                    <p><strong>Score:</strong> <?= $current_assessment['score'] ?>%</p>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <?php if (strpos($current_assessment['assessment_id'], 'request_') === 0): ?>
                            <!-- Document Review Mode -->
                            <div class="row">
                                <div class="col-md-8">
                                    <div class="documents-section mb-4">
                                        <div class="card-header">
                                            <h5 class="mb-0"><i class="bx bx-file-blank me-2"></i>Submitted Documents</h5>
                                        </div>
                                        <div class="card-body">
                                            <?php
                                            // Get documents for this request
                                            $request_id = intval(str_replace('request_', '', $current_assessment['assessment_id']));
                                            $docs_query = "SELECT * FROM accreditation_documents WHERE request_id = ? ORDER BY document_type";
                                            $docs_stmt = $conn->prepare($docs_query);
                                            $docs_stmt->bind_param("i", $request_id);
                                            $docs_stmt->execute();
                                            $documents = $docs_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
                                            ?>
                                            
                                            <?php if (!empty($documents)): ?>
                                                <div class="d-flex justify-content-between align-items-center mb-3">
                                                    <p class="mb-0 text-muted"><?= count($documents) ?> documents submitted</p>
                                                    <div>
                                                        <a href="/admin/download_all_documents.php?request_id=<?= $request_id ?>" 
                                                           class="btn btn-success btn-sm">
                                                            <i class="bx bx-download"></i> Download All (ZIP)
                                                        </a>
                                                    </div>
                                                </div>
                                                
                                                <div class="document-grid">
                                                    <?php foreach ($documents as $doc): ?>
                                                        <div class="document-item">
                                                            <div class="row align-items-center">
                                                                <div class="col-md-2 text-center">
                                                                    <div class="file-icon">
                                                                        <i class="bx bx-file-blank" style="font-size: 2rem; color: #6366f1;"></i>
                                                                    </div>
                                                                </div>
                                                                <div class="col-md-6">
                                                                    <h6 class="mb-1"><?= ucwords(str_replace('_', ' ', $doc['document_type'])) ?></h6>
                                                                    <p class="mb-1 text-truncate"><?= htmlspecialchars($doc['document_name']) ?></p>
                                                                    <small class="text-muted">
                                                                        <?= round($doc['file_size']/1024/1024, 2) ?>MB | 
                                                                        <?= date('M j, Y H:i', strtotime($doc['upload_date'])) ?>
                                                                    </small>
                                                                </div>
                                                                <div class="col-md-4">
                                                                    <span class="badge bg-<?= $doc['verification_status'] === 'verified' ? 'success' : ($doc['verification_status'] === 'rejected' ? 'danger' : 'warning') ?> w-100">
                                                                        <?= ucwords($doc['verification_status']) ?>
                                                                    </span>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    <?php endforeach; ?>
                                                </div>
                                            <?php else: ?>
                                                <div class="text-center py-4">
                                                    <i class="bx bx-file" style="font-size: 3rem; color: #cbd5e1;"></i>
                                                    <h6 class="mt-2 text-muted">No documents submitted yet</h6>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="col-md-4">
                                    <!-- Document Approval Actions -->
                                    <?php if ($current_assessment['status'] === 'documents_submitted'): ?>
                                        <div class="card mb-4">
                                            <div class="card-header bg-warning text-white">
                                                <h6 class="mb-0"><i class="bx bx-clipboard-check me-2"></i>Review Actions</h6>
                                            </div>
                                            <div class="card-body">
                                                <div class="d-grid gap-2 mb-3">
                                                    <form method="POST" class="d-inline">
                                                        <input type="hidden" name="action" value="approve_documents">
                                                        <input type="hidden" name="request_id" value="<?= $request_id ?>">
                                                        <textarea name="approval_comments" class="form-control form-control-sm mb-2" 
                                                                 rows="2" placeholder="Approval comments...">Documents approved for SITA accreditation</textarea>
                                                        <button type="submit" class="btn btn-success w-100">
                                                            <i class="bx bx-check"></i> Approve Documents
                                                        </button>
                                                    </form>
                                                </div>
                                                
                                                <div class="collapse" id="rejectSection">
                                                    <form method="POST">
                                                        <input type="hidden" name="action" value="reject_documents">
                                                        <input type="hidden" name="request_id" value="<?= $request_id ?>">
                                                        <textarea name="rejection_reason" class="form-control form-control-sm mb-2" 
                                                                 rows="3" placeholder="Reason for rejection..." required></textarea>
                                                        <button type="submit" class="btn btn-danger w-100">
                                                            <i class="bx bx-x"></i> Reject Documents
                                                        </button>
                                                    </form>
                                                </div>
                                                
                                                <button class="btn btn-outline-danger btn-sm w-100" data-bs-toggle="collapse" data-bs-target="#rejectSection">
                                                    <i class="bx bx-x-circle"></i> Reject Documents
                                                </button>
                                            </div>
                                        </div>
                                    <?php elseif ($current_assessment['status'] === 'approved_final'): ?>
                                        <div class="card mb-4 border-success">
                                            <div class="card-header bg-success text-white">
                                                <h6 class="mb-0"><i class="bx bx-check-circle me-2"></i>Documents Approved</h6>
                                            </div>
                                            <div class="card-body">
                                                <div class="text-center">
                                                    <i class="bx bx-shield-check text-success" style="font-size: 3rem;"></i>
                                                    <h6 class="mt-2">Ready for Certificate</h6>
                                                    <p class="text-muted small">Student has been notified</p>
                                                </div>
                                            </div>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <!-- Certificate Upload Section -->
                                    <div class="card">
                                        <div class="card-header">
                                            <h5 class="mb-0"><i class="bx bx-certificate me-2"></i>Upload Certificate</h5>
                                        </div>
                                        <div class="card-body">
                                            <?php
                                            // Check if certificate already uploaded
                                            $cert_check = $conn->prepare("SELECT certificate_path FROM accreditation_requests WHERE request_id = ? AND certificate_path IS NOT NULL");
                                            $cert_check->bind_param("i", $request_id);
                                            $cert_check->execute();
                                            $cert_result = $cert_check->get_result();
                                            $existing_cert = $cert_result->fetch_assoc();
                                            ?>
                                            
                                            <?php if ($existing_cert): ?>
                                                <div class="text-center mb-3">
                                                    <i class="bx bx-file-blank text-success" style="font-size: 2rem;"></i>
                                                    <h6 class="mt-2">Certificate Uploaded</h6>
                                                    <a href="<?= $existing_cert['certificate_path'] ?>" class="btn btn-outline-success btn-sm" target="_blank">
                                                        <i class="bx bx-download"></i> View Certificate
                                                    </a>
                                                </div>
                                            <?php endif; ?>
                                            
                                            <form method="POST" enctype="multipart/form-data">
                                                <input type="hidden" name="action" value="upload_certificate">
                                                <input type="hidden" name="request_id" value="<?= $request_id ?>">
                                                
                                                <div class="mb-3">
                                                    <label class="form-label">Accredited Certificate</label>
                                                    <input type="file" name="certificate_file" class="form-control form-control-sm" 
                                                           accept=".pdf,.doc,.docx,.jpg,.jpeg,.png" required>
                                                    <small class="text-muted">PDF, DOC, DOCX, JPG, PNG (Max: 10MB)</small>
                                                </div>
                                                
                                                <div class="mb-3">
                                                    <label class="form-label">Certificate Notes</label>
                                                    <textarea name="certificate_notes" class="form-control form-control-sm" 
                                                             rows="2" placeholder="Optional notes for the student..."></textarea>
                                                </div>
                                                
                                                <button type="submit" class="btn btn-info w-100">
                                                    <i class="bx bx-upload"></i> Upload Certificate
                                                </button>
                                            </form>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php else: ?>
                            <!-- Traditional Assessment Mode -->
                            <h4>Student Answers</h4>
                            
                            <?php foreach ($assessment_questions as $index => $question): ?>
                                <div class="question-card">
                                    <h6>Question <?= $index + 1 ?> (<?= $question['points'] ?> points)</h6>
                                    <p class="mb-2"><?= nl2br(htmlspecialchars($question['question_text'])) ?></p>
                                    <small class="text-muted">Max words: <?= $question['max_words'] ?></small>
                                    
                                    <div class="answer-box">
                                        <?php if (isset($student_answers[$question['question_id']])): ?>
                                            <?php 
                                            $answer = $student_answers[$question['question_id']];
                                            $word_count = str_word_count($answer);
                                            ?>
                                            <div class="d-flex justify-content-between align-items-start mb-2">
                                                <strong>Student Answer:</strong>
                                                <small class="text-muted"><?= $word_count ?> words</small>
                                            </div>
                                            <p><?= nl2br(htmlspecialchars($answer)) ?></p>
                                        <?php else: ?>
                                            <p class="text-muted fst-italic">No answer provided</p>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                            
                            <?php if ($current_assessment['status'] === 'submitted'): ?>
                                <!-- Grading Panel -->
                                <div class="grading-panel">
                                    <h5><i class="bx bx-award me-2"></i>Grade Assessment</h5>
                                    <form method="POST">
                                        <input type="hidden" name="action" value="grade_assessment">
                                        <input type="hidden" name="assessment_id" value="<?= $current_assessment['assessment_id'] ?>">
                                        
                                        <div class="row">
                                            <div class="col-md-4">
                                                <label class="form-label">Score (%)</label>
                                                <input type="number" name="score" class="form-control" min="0" max="100" step="0.1" required>
                                                <small class="text-muted">Minimum 70% to pass</small>
                                            </div>
                                            <div class="col-md-8">
                                                <label class="form-label">Feedback (Optional)</label>
                                                <textarea name="feedback" class="form-control" rows="3" placeholder="Provide feedback to help the student improve..."></textarea>
                                            </div>
                                        </div>
                                        
                                        <div class="mt-3">
                                            <button type="submit" class="btn btn-success">
                                                <i class="bx bx-check me-2"></i>Submit Grade
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            <?php elseif ($current_assessment['status'] === 'graded' || $current_assessment['status'] === 'passed' || $current_assessment['status'] === 'failed'): ?>
                                <!-- Show Grading Results -->
                                <div class="grading-panel">
                                    <h5><i class="bx bx-award me-2"></i>Grading Results</h5>
                                    <div class="row">
                                        <div class="col-md-4">
                                            <p><strong>Score:</strong> <?= $current_assessment['score'] ?>%</p>
                                            <p><strong>Status:</strong> 
                                                <span class="badge <?= getStatusBadgeClass($current_assessment['status']) ?>">
                                                    <?= getStatusText($current_assessment['status']) ?>
                                                </span>
                                            </p>
                                            <p><strong>Graded:</strong> <?= date('M j, Y g:i A', strtotime($current_assessment['graded_date'])) ?></p>
                                        </div>
                                        <div class="col-md-8">
                                            <p><strong>Feedback:</strong></p>
                                            <div class="bg-white p-3 border rounded">
                                                <?= $current_assessment['feedback'] ? nl2br(htmlspecialchars($current_assessment['feedback'])) : '<em class="text-muted">No feedback provided</em>' ?>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php endif; ?>
                        <?php endif; ?>
                    </div>
                </div>
            <?php else: ?>
                <!-- Assessments List View -->
                <div class="page-header">
                    <h1><i class="bx bx-edit me-2"></i>Assessment Management</h1>
                </div>
                
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="bx bx-list-ul me-2"></i>All Assessments</h5>
                    </div>
                    <div class="card-body">
                        <?php if (count($assessments) > 0): ?>
                            <div class="table-responsive">
                                <table id="assessmentsTable" class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Student</th>
                                            <th>Course</th>
                                            <th>Type</th>
                                            <th>Status</th>
                                            <th>Documents/Score</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($assessments as $assessment): ?>
                                            <tr>
                                                <td>
                                                    <div class="user-info">
                                                        <div class="user-avatar placeholder">
                                                            <?= strtoupper(substr($assessment['first_name'], 0, 1)) ?>
                                                        </div>
                                                        <div class="user-details">
                                                            <h6><?= htmlspecialchars($assessment['first_name'] . ' ' . $assessment['last_name']) ?></h6>
                                                            <small><?= htmlspecialchars($assessment['email']) ?></small>
                                                        </div>
                                                    </div>
                                                </td>
                                                <td><?= htmlspecialchars($assessment['course_title']) ?></td>
                                                <td>
                                                    <?php if ($assessment['request_status'] === 'documents_submitted'): ?>
                                                        <span class="badge bg-info text-white">
                                                            <i class="bx bx-file-blank me-1"></i>Document Review
                                                        </span>
                                                    <?php elseif ($assessment['request_status'] === 'approved_final'): ?>
                                                        <span class="badge bg-success text-white">
                                                            <i class="bx bx-check-circle me-1"></i>Approved
                                                        </span>
                                                    <?php elseif ($assessment['request_status'] === 'rejected'): ?>
                                                        <span class="badge bg-danger text-white">
                                                            <i class="bx bx-x-circle me-1"></i>Rejected
                                                        </span>
                                                    <?php else: ?>
                                                        <span class="badge bg-primary text-white">
                                                            <i class="bx bx-edit me-1"></i>Assessment
                                                        </span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <span class="badge <?= getStatusBadgeClass($assessment['status']) ?>">
                                                        <?= getStatusText($assessment['status']) ?>
                                                    </span>
                                                    
                                                    <?php if ($assessment['documents_submitted_date']): ?>
                                                        <br><small class="text-muted">Submitted: <?= date('M j, Y', strtotime($assessment['documents_submitted_date'])) ?></small>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?php if ($assessment['request_status'] === 'documents_submitted'): ?>
                                                        <small class="text-success fw-bold"><?= $assessment['document_count'] ?>/4 Documents</small>
                                                        <div class="mt-1">
                                                            <?php
                                                            $doc_types = explode(',', $assessment['document_statuses']);
                                                            foreach ($doc_types as $doc_info) {
                                                                list($type, $status) = explode(':', $doc_info);
                                                                $color = $status === 'verified' ? 'success' : ($status === 'rejected' ? 'danger' : 'warning');
                                                                echo "<span class='badge bg-{$color} me-1' style='font-size:0.7rem;'>" . ucwords($type) . "</span>";
                                                            }
                                                            ?>
                                                        </div>
                                                    <?php else: ?>
                                                        <?= $assessment['score'] ? $assessment['score'] . '%' : '-' ?>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <div class="action-buttons">
                                                        <?php if ($assessment['request_status'] === 'documents_submitted'): ?>
                                                            <!-- Document Review Actions -->
                                                            <button type="button" class="btn btn-outline-info btn-sm me-1" 
                                                                    data-bs-toggle="modal" data-bs-target="#reviewModal<?= $assessment['request_id'] ?>">
                                                                <i class="bx bx-file-blank"></i> Review
                                                            </button>
                                                        <?php elseif ($assessment['request_status'] === 'approved_final'): ?>
                                                            <span class="text-success fw-bold"><i class="bx bx-check-circle"></i> Approved</span>
                                                        <?php elseif ($assessment['request_status'] === 'rejected'): ?>
                                                            <span class="text-danger fw-bold"><i class="bx bx-x-circle"></i> Rejected</span>
                                                        <?php else: ?>
                                                            <!-- Regular Assessment Actions -->
                                                            <a href="assessment_management.php?id=<?= $assessment['assessment_id'] ?>" class="btn btn-primary btn-sm">
                                                                <i class="bx bx-edit"></i> Manage
                                                            </a>
                                                        <?php endif; ?>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <div class="text-center py-5">
                                <i class="bx bx-edit" style="font-size: 4rem; color: #cbd5e1;"></i>
                                <h4 class="text-muted mt-3">No Pending Reviews</h4>
                                <p class="text-muted">Document submissions and assessments will appear here for review.</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </section>

    <!-- Document Review Modals -->
    <?php foreach ($assessments as $assessment): ?>
        <?php if ($assessment['request_status'] === 'documents_submitted'): ?>
            <div class="modal fade" id="reviewModal<?= $assessment['request_id'] ?>" tabindex="-1">
                <div class="modal-dialog modal-lg">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title">
                                <i class="bx bx-file-blank me-2"></i>
                                Review Documents - <?= htmlspecialchars($assessment['first_name'] . ' ' . $assessment['last_name']) ?>
                            </h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <div class="row">
                                <div class="col-md-8">
                                    <h6><i class="bx bx-info-circle me-2"></i>Student Information</h6>
                                    <div class="bg-light p-3 rounded mb-4">
                                        <p class="mb-1"><strong>Name:</strong> <?= htmlspecialchars($assessment['first_name'] . ' ' . $assessment['last_name']) ?></p>
                                        <p class="mb-1"><strong>Email:</strong> <?= htmlspecialchars($assessment['email']) ?></p>
                                        <p class="mb-1"><strong>Course:</strong> <?= htmlspecialchars($assessment['course_title']) ?></p>
                                        <p class="mb-0"><strong>Submitted:</strong> <?= date('M j, Y H:i', strtotime($assessment['documents_submitted_date'])) ?></p>
                                    </div>
                                    
                                    <h6><i class="bx bx-file me-2"></i>Submitted Documents (<?= $assessment['document_count'] ?>/4)</h6>
                                    <div class="document-list">
                                        <?php
                                        // Get detailed document info for this request
                                        $docs_query = "SELECT * FROM accreditation_documents WHERE request_id = ? ORDER BY document_type";
                                        $docs_stmt = $conn->prepare($docs_query);
                                        $docs_stmt->bind_param("i", $assessment['request_id']);
                                        $docs_stmt->execute();
                                        $docs_result = $docs_stmt->get_result();
                                        ?>
                                        
                                        <?php while ($doc = $docs_result->fetch_assoc()): ?>
                                            <div class="d-flex justify-content-between align-items-center mb-2 p-2 bg-white border rounded">
                                                <div>
                                                    <i class="bx bx-file me-2"></i>
                                                    <strong><?= ucwords(str_replace('_', ' ', $doc['document_type'])) ?>:</strong>
                                                    <?= htmlspecialchars($doc['document_name']) ?>
                                                    <br><small class="text-muted">
                                                        <?= round($doc['file_size']/1024/1024, 2) ?>MB | 
                                                        <?= date('M j, Y H:i', strtotime($doc['upload_date'])) ?>
                                                    </small>
                                                </div>
                                                <div>
                                                    <span class="badge bg-<?= $doc['verification_status'] === 'verified' ? 'success' : ($doc['verification_status'] === 'rejected' ? 'danger' : 'warning') ?>">
                                                        <?= ucwords($doc['verification_status']) ?>
                                                    </span>
                                                    <a href="/student/download_files_simple.php?file=<?= urlencode(basename($doc['file_path'])) ?>&type=accreditation" 
                                                       class="btn btn-sm btn-outline-primary ms-2" target="_blank">
                                                        <i class="bx bx-download"></i>
                                                    </a>
                                                </div>
                                            </div>
                                        <?php endwhile; ?>
                                    </div>
                                </div>
                                
                                <div class="col-md-4">
                                    <h6><i class="bx bx-clipboard-check me-2"></i>Review Actions</h6>
                                    
                                    <!-- Approve Documents -->
                                    <div class="card border-success mb-3">
                                        <div class="card-header bg-success text-white">
                                            <h6 class="mb-0"><i class="bx bx-check-circle me-2"></i>Approve Documents</h6>
                                        </div>
                                        <div class="card-body">
                                            <form method="POST">
                                                <input type="hidden" name="action" value="approve_documents">
                                                <input type="hidden" name="request_id" value="<?= $assessment['request_id'] ?>">
                                                
                                                <div class="mb-3">
                                                    <label class="form-label">Approval Comments</label>
                                                    <textarea name="approval_comments" class="form-control form-control-sm" 
                                                             rows="3" placeholder="Documents meet SITA accreditation requirements. Certificate processing initiated.">Documents approved for SITA accreditation</textarea>
                                                </div>
                                                
                                                <button type="submit" class="btn btn-success btn-sm w-100">
                                                    <i class="bx bx-check"></i> Approve & Notify Student
                                                </button>
                                            </form>
                                        </div>
                                    </div>
                                    
                                    <!-- Reject Documents -->
                                    <div class="card border-danger">
                                        <div class="card-header bg-danger text-white">
                                            <h6 class="mb-0"><i class="bx bx-x-circle me-2"></i>Reject Documents</h6>
                                        </div>
                                        <div class="card-body">
                                            <form method="POST">
                                                <input type="hidden" name="action" value="reject_documents">
                                                <input type="hidden" name="request_id" value="<?= $assessment['request_id'] ?>">
                                                
                                                <div class="mb-3">
                                                    <label class="form-label">Rejection Reason <span class="text-danger">*</span></label>
                                                    <textarea name="rejection_reason" class="form-control form-control-sm" 
                                                             rows="3" placeholder="Please specify what needs to be corrected..." required></textarea>
                                                </div>
                                                
                                                <button type="submit" class="btn btn-danger btn-sm w-100">
                                                    <i class="bx bx-x"></i> Reject & Notify Student
                                                </button>
                                            </form>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <?php if ($assessment['final_comments']): ?>
                                <div class="mt-4 pt-3 border-top">
                                    <h6><i class="bx bx-message-detail me-2"></i>Previous Comments</h6>
                                    <div class="bg-light p-3 rounded">
                                        <?= nl2br(htmlspecialchars($assessment['final_comments'])) ?>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    <?php endforeach; ?>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <!-- DataTables -->
    <script src="https://cdn.datatables.net/1.11.5/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/dataTables.bootstrap5.min.js"></script>
    
    <script>
        // Initialize DataTables
        $(document).ready(function() {
            $('#assessmentsTable').DataTable({
                responsive: true,
                order: [[2, 'desc']], // Sort by created date descending
                pageLength: 25,
                language: {
                    search: "Search assessments:",
                    lengthMenu: "Show _MENU_ assessments per page",
                    info: "Showing _START_ to _END_ of _TOTAL_ assessments"
                }
            });
        });
    </script>
</body>
</html>
